import calendar
import datetime
from abc import ABC, abstractmethod

import pandas as pd
from dateutil.relativedelta import relativedelta
from django.db.models import Q, Sum, FloatField
from django.db.models.functions import Coalesce

from commons.logger import logger
from loans.models import IntervalChoices, LoanTypes
from transactions.models import Account
from sacco.utils import businessdata

months_with_ids = {
    1: 'January',
    2: 'February',
    3: 'March',
    4: 'April',
    5: 'May',
    6: 'June',
    7: 'July',
    8: 'August',
    9: 'September',
    10: 'October',
    11: 'November',
    12: 'December'
}


def gen_interval(interval, formula):
    year = datetime.datetime.now().year
    month = datetime.datetime.now().month
    if formula == LoanTypes.LOAN_FORMULA.PER_ANNAM:
        if interval == IntervalChoices.YEARS:
            interval = 1
        elif interval == IntervalChoices.MONTHS:
            interval = 12
        elif interval == IntervalChoices.DAYS:
            days_in_year = (datetime.date(year, 12, 31) - datetime.date(year, 1, 1)).days + 1
            interval = days_in_year
        elif interval == IntervalChoices.WEEKS:
            num_weeks = datetime.date(year, 12, 28).isocalendar()[1]
            interval = num_weeks
    elif formula == LoanTypes.LOAN_FORMULA.PER_MONTH:
        if interval == IntervalChoices.DAYS:
            days_in_month = calendar.monthrange(year, month)[1]
            interval = days_in_month
        elif interval == IntervalChoices.WEEKS:
            num_days = calendar.monthrange(year, month)[1]  # num of days in month
            weeks_in_month = (num_days + 6) // 7
            interval = weeks_in_month
        elif interval == IntervalChoices.FORTNIGHT:
            interval = 4
        elif interval == IntervalChoices.MONTHS:
            interval = 1

    elif formula == LoanTypes.LOAN_FORMULA.PER_WEEK:
        if interval == IntervalChoices.DAYS:
            interval = 7
        elif interval == IntervalChoices.WEEKS:
            interval = 1

    elif formula == LoanTypes.LOAN_FORMULA.PER_DAY:
        interval = 1
    return interval

def gen_interval2(interval, period):
    year = datetime.datetime.now().year
    month = datetime.datetime.now().month
    if period == 'Years':
        if interval == IntervalChoices.YEARS:
            interval = 1
        elif interval == IntervalChoices.MONTHS:
            interval = 12
        elif interval == IntervalChoices.DAYS:
            days_in_year = (datetime.date(year, 12, 31) - datetime.date(year, 1, 1)).days + 1
            interval = days_in_year
        elif interval == IntervalChoices.WEEKS:
            num_weeks = datetime.date(year, 12, 28).isocalendar()[1]
            interval = num_weeks
    elif period == 'Months':
        if interval == IntervalChoices.DAYS:
            days_in_month = calendar.monthrange(year, month)[1]
            interval = days_in_month
        elif interval == IntervalChoices.WEEKS:
            num_days = calendar.monthrange(year, month)[1]  # num of days in month
            weeks_in_month = (num_days + 6) // 7
            interval = weeks_in_month
        elif interval == IntervalChoices.FORTNIGHT:
            interval = 4
        elif interval == IntervalChoices.MONTHS:
            interval = 1
    elif period == 'Weeks':
        if interval == IntervalChoices.DAYS:
            interval = 7

        elif interval == IntervalChoices.WEEKS:
            interval = 1

    elif period == 'Days':
        interval = 1
    return interval


def get_dates(loan, interval):
    start_date = loan.approved_on
    if loan.schedule_start is not None:
        start_date = loan.schedule_start
    if loan.interval == IntervalChoices.DAYS:
        new_date = start_date + datetime.timedelta(days=interval) if start_date else None
        # Print the new date
        return new_date
    elif loan.interval == IntervalChoices.WEEKS:
        new_date = start_date + datetime.timedelta(weeks=interval)
        return new_date
    elif loan.interval == IntervalChoices.MONTHS:
        new_date = start_date + relativedelta(months=interval)
        return new_date
    elif loan.interval == IntervalChoices.YEARS:
        new_date = start_date + relativedelta(years=interval)
        return new_date


def paid_amount(obj) -> float:
    paid = obj.loan_trans.filter(
        Q(transaction_type='Loan repayment') | Q(transaction_type='Loan interest')
    ).aggregate(
        paid=Coalesce(Sum('reporting_amount'), 0.0, output_field=FloatField()))['paid']
    return paid


def paid_interest(obj) -> float:
    paid = obj.loan_trans.filter(transaction_type='Loan interest').aggregate(
        paid=Coalesce(Sum('reporting_amount'), 0.0, output_field=FloatField()))['paid']
    return paid


class Amortisation(ABC):
    def __init__(self, rate, amount, period, schedule, formulae, *args, **kwargs):
        self.rate = rate
        self.amount = amount
        self.period = period
        self.schedule = schedule
        self.formulae = formulae
        self.account = kwargs['acc']
        self.loan = kwargs['loan']

    @abstractmethod
    def gen_schedules(self):
        pass

    # @abstractmethod
    # def amount_payable(self):
    #     pass


class ReducingBalance(Amortisation):

    def gen_schedules(self, *args, **kwargs):
        logger.debug('rec')
        loan = self.loan
        loan_amount = self.amount
        interest_rate = self.rate
        loan_term_period = self.period
        loan_formulae = self.formulae
        schedule = self.schedule
        payment_interval = gen_interval(schedule, loan_formulae)
        paid = paid_amount(loan)
        num_payments = loan_term_period * payment_interval
        # Calculate schedule interest rate and payment amount
        schedule_interest_rate = interest_rate / payment_interval
        payment_amount = (loan_amount * schedule_interest_rate) / (1 - (1 + schedule_interest_rate) ** (-num_payments))
        # Create empty DataFrame to hold amortization schedule
        schedule = pd.DataFrame(
            columns=['payment_date', 'payment', 'installment', 'principal_paid', 'interest_paid',
                     'total_interest_paid', 'schedule_balance', 'remaining_balance', 'total_installment', 'pay'])

        # Fill in first row with initial loan information
        schedule.loc[0] = [0, 0, 0, 0, 0, 0, 0, loan_amount, 0, 0]
        installs = []
        # Loop through each payment and calculate values for amortization schedule
        pay = 0

        for i in range(1, int(num_payments) + 1):
            payment_date = get_dates(loan, i)

            # Calculate interest and principal for this payment
            interest_paid = schedule.loc[i - 1, 'remaining_balance'] * schedule_interest_rate
            principal_paid = payment_amount - interest_paid
            installment = payment_amount

            # Update remaining balance
            remaining_balance = schedule.loc[i - 1, 'remaining_balance'] - principal_paid

            total_installment = schedule.loc[i - 1, 'total_installment'] + installment

            paid_here = min(paid, installment)
            paid -= paid_here
            schedule_balance = installment - paid_here
            if schedule_balance > 0:
                pay += 1
            # print(schedule_balance)
            # Add row to schedule DataFrame
            schedule.loc[i] = [payment_date, i, installment, principal_paid, interest_paid,
                               schedule.loc[i - 1, 'total_interest_paid'] + interest_paid, schedule_balance,
                               remaining_balance, total_installment, pay]
        schedule = schedule.drop(0)
        return schedule

    # def amount_payable(self):
    #     schedules = self.gen_schedules()
    #     sum_installment = schedules['installment'].sum()
    #     return round(sum_installment, 2)


class FlatRate(Amortisation):

    def gen_schedules(self):
        logger.debug('flat')
        loan = self.loan
        loan_amount = self.amount
        interest_rate = self.rate
        loan_term_period = self.period
        interest_principle = loan_amount * interest_rate
        total_interest = interest_principle * loan_term_period
        total_payable = loan_amount + total_interest
        paid = paid_amount(loan)

        installment = total_payable / loan_term_period if loan_term_period > 0 else 0
        principal_peri_sched = loan_amount / loan_term_period if loan_term_period > 0 else 0
        # print('payable ', payable_per_sched)
        # print('my sched ', principal_peri_sched)

        schedule = pd.DataFrame(
            columns=['payment_date', 'payment', 'installment', 'principal_paid', 'interest_paid',
                     'total_interest_paid', 'schedule_balance', 'remaining_balance', 'total_installment', 'pay'])
        # Fill in first row with initial loan information
        schedule.loc[0] = [0, 0, 0, 0, 0, 0, 0, loan_amount, 0, 0]
        pay = 0
        for i in range(1, int(loan_term_period) + 1):
            payment_date = get_dates(loan, i)
            # Update remaining balance
            remaining_balance = schedule.loc[i - 1, 'remaining_balance'] - principal_peri_sched
            total_installment = schedule.loc[i - 1, 'total_installment'] + installment
            paid_here = min(paid, installment)
            paid -= paid_here
            schedule_balance = installment - paid_here
            if schedule_balance > 0:
                pay += 1

            # print(schedule_balance)

            # Add row to schedule DataFrame
            schedule.loc[i] = [payment_date, i, installment, principal_peri_sched, interest_principle,
                               schedule.loc[i - 1, 'total_interest_paid'] + interest_principle, schedule_balance,
                               remaining_balance, total_installment, pay]
        schedule = schedule.drop(0)
        return schedule

    # def amount_payable(self):
    #     schedules = self.gen_schedules()
    #     sum_installment = schedules['installment'].sum()
    #     return round(sum_installment, 2)


def loan_payment_details(pay_type: int, *args, **kwargs) -> Amortisation:
    FLAT_RATE = 1
    REDUCING_BALANCE = 2
    if pay_type == REDUCING_BALANCE:
        return ReducingBalance(*args, **kwargs)
    elif pay_type == FLAT_RATE:
        return FlatRate(*args, **kwargs)
    raise ValueError('Invalid rate type')



def get_principal(loan, request) -> float:
    # print('loan id', loan.id)
    # interestspaid = loan.loan_trans.filter(transaction_type='Loan interest',
    #                                        branch__business_id=businessdata(request))
    # print('de interest', interestspaid)

    # loanr = Account.objects.filter(name='Loan receivables', business=businessdata(request))[0]
    loanacct = Account.objects.filter(name='Loan Receivables', business=businessdata(request)).first()
    adjustment_added = loan.loan_trans.filter(
        Q(account_dr=loanacct.id), Q(transaction_type='Principal Adjustment')).aggregate(
        total=Coalesce(Sum('reporting_amount', output_field=FloatField()), 0.00))['total']
    adjustment_down = loan.loan_trans.filter(
        Q(account_cr=loanacct.id), Q(transaction_type='Principal Adjustment')).aggregate(
        total=Coalesce(Sum('reporting_amount', output_field=FloatField()), 0.00))['total']

    # totaladjustments = loan.loan_trans.objects.filter(transaction_type='Loan Adjustment').aggregate(
    #     adjustments=Coalesce(Sum('reporting_amount')))['adjustments']

    # adjustments = loan.loan_trans.objects.filter(transaction_type='Loan Adjustment')
    totaldebit = loan.loan_trans.filter(Q(account_dr_id=loanacct.id),
                                        ~Q(transaction_type='Principal Adjustment')).aggregate(
        total=Coalesce(Sum('reporting_amount', output_field=FloatField()), 0.00))['total']
    # incase the is removal of an extra charge (this is a fix for the extra charge that was being added to loan amt ap
    # approved due to 'added to principal'
    extra_charge_removed = loan.loan_trans.filter(transaction_type='Remove extra loan charge').aggregate(
        ex_charge=Coalesce(Sum('reporting_amount', output_field=FloatField()), 0.00))['ex_charge']

    # print('totalpdebit', float(totaldebit))
    # print('adjustment_down', float(adjustment_down - adjustment_added))
    principal = float(totaldebit) - float(adjustment_down - adjustment_added) - extra_charge_removed

    return principal



