import calendar
import datetime

import pandas as pd
from dateutil.relativedelta import relativedelta
from django.db.models import Q, Sum, FloatField
from django.db.models.functions import Coalesce

from commons.logger import logger
from loans.models import IntervalChoices, LoanTypes, Loans
from sacco.models import LoanSettings
from sacco.utils import businessdata
from transactions.models import Account


def gen_interval(interval, formula):
    year = datetime.datetime.now().year
    month = datetime.datetime.now().month
    # Annum
    if formula == LoanTypes.LOAN_FORMULA.PER_ANNAM and interval == IntervalChoices.YEARS:
        interval = 1
    elif formula == LoanTypes.LOAN_FORMULA.PER_ANNAM and interval == IntervalChoices.MONTHS:
        interval = 12
    elif formula == LoanTypes.LOAN_FORMULA.PER_ANNAM and interval == IntervalChoices.DAYS:
        days_in_year = (datetime.date(year, 12, 31) - datetime.date(year, 1, 1)).days + 1
        interval = days_in_year
    elif formula == LoanTypes.LOAN_FORMULA.PER_ANNAM and interval == IntervalChoices.WEEKS:
        num_weeks = datetime.date(year, 12, 28).isocalendar()[1]
        interval = num_weeks

    elif interval == IntervalChoices.DAYS and formula == LoanTypes.LOAN_FORMULA.PER_MONTH:
        days_in_month = calendar.monthrange(year, month)[1]
        interval = days_in_month
    elif interval == IntervalChoices.WEEKS and formula == LoanTypes.LOAN_FORMULA.PER_MONTH:
        weeks_in_month = 4
        interval = weeks_in_month
    elif interval == IntervalChoices.FORTNIGHT and formula == LoanTypes.LOAN_FORMULA.PER_MONTH:
        interval = 2
    elif interval == IntervalChoices.MONTHS and formula == LoanTypes.LOAN_FORMULA.PER_MONTH:
        interval = 1

    elif formula == LoanTypes.LOAN_FORMULA.PER_WEEK and interval == IntervalChoices.DAYS:
        days_in_week = 7
        interval = days_in_week
    elif formula == LoanTypes.LOAN_FORMULA.PER_WEEK and interval == IntervalChoices.WEEKS:
        interval = 1

    elif formula == LoanTypes.LOAN_FORMULA.PER_DAY and interval == IntervalChoices.DAYS:
        interval = 1
    return interval


def get_dates(loan, interval):
    start_date = loan.approved_on
    if loan.schedule_start is not None:
        start_date = loan.schedule_start
    if loan.interval == IntervalChoices.DAYS:
        new_date = start_date + datetime.timedelta(days=interval) if start_date else None
        # Print the new date
        return new_date
    elif loan.interval == IntervalChoices.WEEKS:
        new_date = start_date + datetime.timedelta(weeks=interval)
        return new_date
    elif loan.interval == IntervalChoices.MONTHS:
        new_date = start_date + relativedelta(months=interval)
        return new_date
    elif loan.interval == IntervalChoices.YEARS:
        new_date = start_date + relativedelta(years=interval)
        return new_date


def get_principal(loan, request) -> float:

    loanacct = Account.objects.filter(name='Loan Receivables', business=businessdata(request)).first()
    adjustment_added = loan.loan_trans.filter(
        Q(account_dr=loanacct.id), Q(transaction_type='Principal Adjustment')).aggregate(
        total=Coalesce(Sum('reporting_amount', output_field=FloatField()), 0.00))['total']
    adjustment_down = loan.loan_trans.filter(
        Q(account_cr=loanacct.id), Q(transaction_type='Principal Adjustment')).aggregate(
        total=Coalesce(Sum('reporting_amount', output_field=FloatField()), 0.00))['total']

    # adjustments = loan.loan_trans.objects.filter(transaction_type='Loan Adjustment')
    totaldebit = loan.loan_trans.filter(Q(account_dr_id=loanacct.id),
                                        ~Q(transaction_type='Principal Adjustment')).aggregate(
        total=Coalesce(Sum('reporting_amount', output_field=FloatField()), 0.00))['total']
    # incase the is removal of an extra charge (this is a fix for the extra charge that was being added to loan amt ap
    # approved due to 'added to principal'
    extra_charge_removed = loan.loan_trans.filter(transaction_type='Remove extra loan charge').aggregate(
        ex_charge=Coalesce(Sum('reporting_amount', output_field=FloatField()), 0.00))['ex_charge']

    # print('totalpdebit', float(totaldebit))
    # print('adjustment_down', float(adjustment_down - adjustment_added))
    principal = float(totaldebit) - float(adjustment_down - adjustment_added) - extra_charge_removed

    return principal


def reducing_amortization(principal, interest_rate, num_periods, *args, **kwargs):
    if num_periods is None:
        num_periods =0
    loan = kwargs['loan']
    loan_formulae = kwargs['formula']
    payment_interval = gen_interval(loan.interval, loan_formulae)
    # num_payments = num_periods * payment_interval
    # num_payments = num_periods * payment_interval
    # print(num_payments)

    # Calculate periodic interest rate
    periodic_interest_rate = interest_rate / payment_interval
    print('my principle', principal)
    logger.info(f'my principle, {principal}')
    print('my periodic interest rate', periodic_interest_rate)
    logger.info(f'my periodic interest rate, {periodic_interest_rate}')
    print('num_periods', num_periods)
    logger.info(f'num_periods, {num_periods}')

    # Calculate periodic payment amount
    try:
        periodic_payment = (principal * periodic_interest_rate) / (1 - (1 + periodic_interest_rate) ** -num_periods)
        # Generate amortization schedule
        remaining_balance = principal
        amortization_schedule = []
        # print('num_payments', num_payments)
        for period in range(1, int(num_periods) + 1):
            payment_date = get_dates(loan, period)
            interest_payment = remaining_balance * periodic_interest_rate
            principal_payment = periodic_payment - interest_payment
            remaining_balance -= principal_payment

            amortization_schedule.append({
                'payment_date': payment_date,
                'Period': period,
                'Payment': periodic_payment,
                'Principal': principal_payment,
                'Interest': interest_payment,
                'Balance': remaining_balance
            })

        return amortization_schedule
    except ZeroDivisionError as e:
        logger.error(f'{loan}')
        logger.error(str(e))
    return []




def flat_amortization(principal, interest_rate, num_periods, *args, **kwargs):
    if num_periods is None:
        num_periods = 0
    loan = kwargs['loan']
    loan_formulae = kwargs['formula']
    payment_interval = gen_interval(loan.interval, loan_formulae)

    # Calculate periodic interest rate
    periodic_interest_rate = interest_rate / payment_interval

    num_payments = num_periods
    # print('num_periods', num_payments)
    # print('pri', principal)
    interest_principle = principal * periodic_interest_rate
    # if interest_principle is None:
    #     interest_principle = 0
    
    total_interest = interest_principle * num_periods
    print('totla int', total_interest)
    total_payable = principal + total_interest
    installment = total_payable / num_payments if num_payments > 0 else 0
    principal_per_sched = principal / num_payments if num_payments > 0 else 0
    remaining_balance = principal
    amortization_schedule = []
    print('we pay', num_periods)
    print('intervak', payment_interval)
    for i in range(1, int(num_payments) + 1):
        payment_date = get_dates(loan, i)
        remaining_balance -= principal_per_sched
        amortization_schedule.append({
            'payment_date': payment_date,
            'Period': i,
            'Payment': installment,
            'Principal': principal_per_sched,
            'Interest': interest_principle,
            'Balance': remaining_balance
        })
    return amortization_schedule


def flat_amortization_aging_report(principal, interest_rate, num_periods, *args, **kwargs):
    loan = Loans.objects.get(id=kwargs['loan'])
    loan_formulae = kwargs['formula']
    payment_interval = gen_interval(loan.interval, loan_formulae)

    # Calculate periodic interest rate
    periodic_interest_rate = interest_rate / payment_interval

    num_payments = num_periods

    interest_principle = principal * periodic_interest_rate

    total_interest = interest_principle * num_periods
    total_payable = principal + total_interest
    installment = total_payable / num_payments if num_payments > 0 else 0
    principal_per_sched = principal / num_payments if num_payments > 0 else 0
    remaining_balance = principal
    amortization_schedule = []
    
    for i in range(1, int(num_payments) + 1):
        payment_date = get_dates(loan, i)
        remaining_balance -= principal_per_sched
        amortization_schedule.append({
            'payment_date': payment_date,
            'Period': i,
            'Payment': installment,
            'Principal': principal_per_sched,
            'Interest': interest_principle,
            'Balance': remaining_balance
        })
    return amortization_schedule


def reducing_amortization_aging_report(principal, interest_rate, num_periods, *args, **kwargs):
    loan = Loans.objects.get(id=kwargs['loan'])
    loan_formulae = kwargs['formula']
    payment_interval = gen_interval(loan.interval, loan_formulae)
    # num_payments = num_periods * payment_interval
    num_payments = num_periods * payment_interval

    # Calculate periodic interest rate
    periodic_interest_rate = interest_rate / payment_interval

    # Calculate periodic payment amount
    periodic_payment = (principal * periodic_interest_rate) / (1 - (1 + periodic_interest_rate) ** -num_periods)

    # Generate amortization schedule
    remaining_balance = principal
    amortization_schedule = []
    print('num_payments', num_payments)
    for period in range(1, int(num_payments) + 1):
        payment_date = get_dates(loan, period)
        interest_payment = remaining_balance * periodic_interest_rate
        principal_payment = periodic_payment - interest_payment
        remaining_balance -= principal_payment

        amortization_schedule.append({
            'payment_date': payment_date,
            'Period': period,
            'Payment': periodic_payment,
            'Principal': principal_payment,
            'Interest': interest_payment,
            'Balance': remaining_balance
        })
    return amortization_schedule
