from django.contrib import messages
from django.contrib.auth.decorators import login_required
from django.http import HttpResponseRedirect
from django.urls import reverse_lazy
from django.utils.decorators import method_decorator
from django.views.generic import TemplateView
from django.views.generic.edit import FormMixin, FormView

from loans.forms.upload_forms import LoanUploadForm, RepaymentUploadForm
from loans.models import UploadLoans, LoanTypes
from sacco.utils import businessdata, branchdata
from transactions.models import Account


@method_decorator(login_required, name='dispatch')
class LoanUploadsView(FormMixin, TemplateView):
    form_class = LoanUploadForm
    template_name = 'new_loans/loan_uploads.html'

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        title = 'Loan Uploads'
        business = businessdata(self.request)
        bankaccounts = Account.objects.filter(business=business, category__name__in=['Bank', 'Mobile Money', 'Cash', 'Reserves'])
        uplaods = UploadLoans.objects.filter(branch_id=branchdata(self.request))
        loantypes = LoanTypes.objects.filter(business=business, is_active=True)
        context['title'] = title
        context['bankaccounts'] = bankaccounts
        context['uplaods'] = uplaods
        context['loantypes'] = loantypes
        return context

    def form_valid(self, form):
        form.submit()
        messages.success(self.request, 'success', extra_tags='Successfully uploaded')
        return super().form_valid(form)

    def form_invalid(self, form):
        return super().form_invalid(form)


class UploadRepyments(TemplateView):
    template_name = 'new_loans/upload_loanrepayments.html'

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        bankaccounts = Account.objects.filter(business=businessdata(self.request),
                                              category__name__in=['Bank', 'Cash', 'Shares', 'Account Receivables', 'Reserves'])
        context['title'] = 'UPLOAD Repayments'
        context['bankaccounts'] = bankaccounts

        return context

class UploadRepaymentFormView(FormView):
    form_class = RepaymentUploadForm
    success_url = reverse_lazy('new_upload_repayment')
    template_name = 'new_loans/upload_loanrepayments.html'

    def get_form_kwargs(self):
        kwargs = super().get_form_kwargs()
        kwargs['request'] = self.request
        return kwargs

    def form_valid(self, form):
        form.save()
        messages.success(self.request, 'success', extra_tags='Successfully uploaded')
        return super().form_valid(form)

    def form_invalid(self, form):
        for field, errors in form.errors.items():
            for error in errors:
                # print(f"Field: {field}, Error: {error}")
                messages.error(self.request, 'error', extra_tags=f'{error}')
        # messages.error(self.request, 'error', extra_tags='Error in uploading')
        return HttpResponseRedirect(reverse_lazy('new_upload_repayment'))
