import datetime

from django.db.models import FloatField, Sum
from django.db.models.functions import Coalesce

from loans.models import Loans


class LoansReportMixin:
    def branch(self):
        pass

    def filter_loans(self) -> {}:
        filter_item = {}
        today_ = datetime.datetime.today()
        from_date = self.request.GET.get('from_date', None)
        to_date = self.request.GET.get('to_date', None)
        if from_date and to_date:
            filter_item.update({'schedule_start__gte': from_date, 'schedule_start__lte': to_date})
        else:
            filter_item['schedule_start'] = today_

        return filter_item

    def filter_transactions(self) -> {}:
        filters = {}
        period = self.request.GET.get('date', None)
        today_ = datetime.datetime.today()
        user = self.request.user
        branch = self.branch()
        filters['branch'] = branch
        if period:
            filters['tx_date__lte'] = period
        else:
            filters['tx_date__lte'] = today_
        return filters

    def range_filters(self) -> dict:
        from_date = self.request.GET.get('from_date', None)
        to_date = self.request.GET.get('to_date', None)
        filters = {}
        user = self.request.user
        today_ = datetime.datetime.today()
        filters['branch'] = self.branch()
        if from_date and to_date:
            filters.update({'tx_date__gte': from_date, 'tx_date__lte': to_date})
        else:
            filters['tx_date'] = today_
        return filters

    def show_loans(self, loans, account, interest_account) -> list:
        loans_list = []
        cleaned_loans = Loans.objects.filter(id__in=list(loans.values_list('loan_id', flat=True)),
                                             loan_status=3).prefetch_related('account').prefetch_related('loan_trans')
        for loan in cleaned_loans:
            debits_ = loan.loan_trans.filter(account_dr=account).filter(**self.filter_transactions()).aggregate(
                debits=Coalesce(Sum('reporting_amount'), 0.0, output_field=FloatField()))['debits']
            credits_ = loan.loan_trans.filter(account_cr=account).filter(**self.filter_transactions()).aggregate(
                credits=Coalesce(Sum('reporting_amount'), 0.0, output_field=FloatField()))['credits']

            debits_int_ = \
                loan.loan_trans.filter(account_dr=interest_account).filter(**self.filter_transactions()).aggregate(
                    total=Coalesce(Sum('reporting_amount'), 0.0, output_field=FloatField()))['total']
            credits_int_ = \
                loan.loan_trans.filter(account_cr=interest_account).filter(**self.filter_transactions()).aggregate(
                    total=Coalesce(Sum('reporting_amount'), 0.0, output_field=FloatField()))['total']

            loans_list.append(
                {
                    'name': loan.account.names,
                    'account': loan.account.acc_number,
                    'principle': "{:0,.2f}".format(float(loan.amount_approved)),
                    'interest': "{:0,.2f}".format(float(credits_int_)),
                    'balance': "{:0,.2f}".format(float(debits_ - credits_)),
                    'disbursed': "{:0,.2f}".format(float(debits_ - credits_)),
                    'duration': loan.approved_duration
                }
            )

        return loans_list