from datetime import timedelta

from django.db import models

from accounts.models import Biodata
from sacco.models import MemberAccount, FinancialYear, AccountTypes, Member
from utils.file_uploads import loan_files


class IntervalChoices(models.IntegerChoices):
    DAYS = 0, 'Days'
    WEEKS = 1, 'Weeks'
    FORTNIGHT = 2, 'Fortnight'
    MONTHS = 3, 'Months'
    YEARS = 4, 'Years'


class LoanTypes(models.Model):
    class LOAN_FORMULA(models.IntegerChoices):
        PER_ANNAM = 1, 'Per Annum'
        PER_MONTH = 2, 'Per Month'
        PER_WEEK = 3, 'Per Week'
        PER_DAY = 4, 'Per Day'

    name = models.CharField(max_length=255)
    max_loan = models.FloatField(verbose_name='Maximum Loan')
    interest_rate = models.FloatField()
    excuse_d = models.IntegerField(verbose_name='Excuse days')
    warning_d = models.IntegerField(verbose_name='Warning days')
    formula = models.IntegerField(null=True, blank=True, default=1, choices=LOAN_FORMULA.choices)
    maximum_period = models.IntegerField(null=True, blank=True, default=None, verbose_name='Maximum Period')
    maximum_securities = models.IntegerField(default=1)
    security_ratio = models.IntegerField(default=150)
    is_active = models.BooleanField(default=True)
    sub_intervals = models.BooleanField(default=False)
    date_added = models.DateField(auto_now_add=True)
    business = models.ForeignKey('accounts.Business', on_delete=models.CASCADE, null=True)
    added_by = models.ForeignKey('accounts.Staff', related_name='sacco_staff', on_delete=models.SET_NULL, null=True)
    interval = models.IntegerField(null=True, blank=True, choices=IntervalChoices.choices, default=IntervalChoices.DAYS)
    synced = models.BooleanField(default=False, verbose_name='Synced to the portal')
    is_standalone = models.BooleanField(default=False)

    def __str__(self):
        return self.name

    class Meta:
        db_table = 'loan_types'
        verbose_name = 'Loan Type'
        verbose_name_plural = 'Loan Types'

    @property
    def get_period(self):
        if self.formula == LoanTypes.LOAN_FORMULA.PER_DAY:
            return "days"
        elif self.formula == LoanTypes.LOAN_FORMULA.PER_WEEK:
            return "weeks"
        elif self.formula == LoanTypes.LOAN_FORMULA.PER_MONTH:
            return "months"
        elif self.formula == LoanTypes.LOAN_FORMULA.PER_ANNAM:
            return "years"

        return


class UploadLoans(models.Model):
    file_name = models.CharField(max_length=255, null=True, blank=True)
    file = models.FileField(upload_to=loan_files, null=True, blank=True)
    total_records = models.IntegerField(null=True, blank=True)
    branch = models.ForeignKey('accounts.Branch', on_delete=models.CASCADE, null=True, blank=True)
    upload_date = models.DateField(null=True, blank=True)
    loan_type = models.ForeignKey(LoanTypes, on_delete=models.SET_NULL, null=True)
    added_by = models.ForeignKey('accounts.Staff', on_delete=models.SET_NULL, null=True)
    created_date = models.DateTimeField(auto_now_add=True)
    failed_loans = models.TextField(blank=True, null=True)
    total_failed_loans = models.IntegerField(null=True, blank=True)

    class Meta:
        db_table = 'upload_loans'
        verbose_name = 'Upload Loan'
        verbose_name_plural = 'Uploads Loans'


class Loans(models.Model):
    class LOAN_STATUS(models.IntegerChoices):
        APPLIED = 0, 'Applied'
        APPRAISED = 1, 'Appraised'
        APPROVED = 2, 'Approved'
        DISBURSED = 3, 'Disbursed'
        CLOSED = 4, 'Closed'
        REJECTED = 5, 'Rejected'
        WAIVED_OFF = 6, 'WaivedOff'
        WRITTEN_OFF = 7, 'WrittenOff'
        TOPPED_UP = 8, 'ToppedUp'


    class INTEREST_TYPES(models.IntegerChoices):
        FLAT_RATE = 1, 'Flat rate'
        REDUCING_BALANCE = 2, 'Reducing Balance'

    class REPAYMENT_METHODS(models.IntegerChoices):
        BY_CASH = 1, 'By Cash'
        BY_MEMBER_ACCOUNT = 2, 'By member Account'

    account = models.ForeignKey(MemberAccount, on_delete=models.SET_NULL, related_name='members', null=True)
    branch = models.ForeignKey('accounts.Branch', on_delete=models.CASCADE)
    applicant = models.ForeignKey(Member, on_delete=models.SET_NULL, related_name='applicant', null=True)
    amount_requested = models.FloatField(null=True, blank=True)
    amount_approved = models.FloatField(default=0.00)
    rate = models.FloatField(default=0.00)
    rate_type = models.IntegerField(default=1, choices=INTEREST_TYPES.choices)
    repayment_method = models.IntegerField(default=1, choices=REPAYMENT_METHODS.choices)
    loan_type = models.ForeignKey(LoanTypes, on_delete=models.SET_NULL, null=True)
    requested_on = models.DateField(null=True, blank=True)
    approved_on = models.DateField(null=True, blank=True)
    schedule_start = models.DateField(null=True, blank=True)
    charges = models.TextField(null=True, blank=True)
    requested_duration = models.FloatField(null=True, blank=True)
    approved_duration = models.FloatField(null=True, blank=True)
    deductibles = models.FloatField(null=True, blank=True)
    financial_year = models.ForeignKey(FinancialYear, on_delete=models.SET_NULL, null=True, blank=True)
    loan_status = models.IntegerField(default=0, choices=LOAN_STATUS.choices)
    is_topup = models.BooleanField(default=False, null=True, blank=True)
    charges_paid = models.BooleanField(default=False, null=True, blank=True)
    balance = models.FloatField(null=True, blank=True)
    interest = models.FloatField(null=True, blank=True)
    principal = models.FloatField(null=True, blank=True)
    interval = models.IntegerField(null=True, blank=True, choices=IntervalChoices.choices)
    approved_by = models.ForeignKey('accounts.Staff', related_name='approvedBy', on_delete=models.SET_NULL, null=True)
    added_by = models.ForeignKey('accounts.Staff', related_name='addedBy', on_delete=models.SET_NULL, null=True)
    created_date = models.DateTimeField(auto_now_add=True)
    upload = models.ForeignKey(UploadLoans, related_name='upload', on_delete=models.CASCADE, null=True, blank=True)
    voucher = models.CharField(max_length=255, verbose_name='Loan Voucher Number', null=True, blank=True)
    sub_intervals = models.BooleanField(default=False)
    loan_number = models.CharField(max_length=255, verbose_name='Loan number', null=True, blank=True)
    topped_up_loan = models.BigIntegerField(null=True, blank=True)
    group_guaratorship = models.BooleanField(default=False)
    #if person belongs to a group - the guarantor
    group_guarantor = models.ForeignKey(Member, related_name='individual_group_guarantor', on_delete=models.SET_NULL, null=True, blank=True)
    officer = models.ForeignKey('accounts.Staff', on_delete=models.SET_NULL, null=True, blank=True)

    #Field to store json for loan summary eg principal, interest, total payment
    loan_summary_details = models.TextField(null=True, blank=True)

    #balance of savings that shouldnt be withdrawn until loan is paid off
    savings_rem_balance = models.FloatField(null=True, blank=True)
    savings_percentage_balance = models.FloatField(null=True, blank=True)
    # is_old = models.BooleanField(default=True)


    def __unicode__(self):
        return f'Requested: {self.amount_requested} by: {self.applicant.biodata.name}'

    class Meta:
        db_table = 'loans'
        verbose_name = 'Loan'
        verbose_name_plural = 'Loans'

    @property
    def ending_date(self):
        try:
            enddate = self.approved_on + timedelta(days=self.approved_duration)
            return enddate
        except Exception as e:
            return 'None'


class LoanUpload(models.Model):
    loan = models.ForeignKey(Loans, related_name='docs', on_delete=models.CASCADE)
    file_name = models.CharField(max_length=255, null=True, blank=True)
    file = models.FileField(upload_to=loan_files, null=True, blank=True)
    added_by = models.ForeignKey('accounts.Staff', on_delete=models.SET_NULL, null=True)
    created_date = models.DateTimeField(auto_now_add=True)

    class Meta:
        db_table = 'loan_uploads'
        verbose_name = 'Loan Upload'
        verbose_name_plural = 'Loan Uploads'


class LoanTopups(models.Model):
    closed_loan = models.ForeignKey(Loans, related_name='closed_loan', on_delete=models.CASCADE)
    topup_loan = models.ForeignKey(Loans, related_name='topup_loan', on_delete=models.CASCADE, blank=True, null=True)
    branch = models.ForeignKey('accounts.Branch', on_delete=models.CASCADE, blank=True, null=True)
    added_by = models.ForeignKey('accounts.Staff', on_delete=models.SET_NULL, null=True)
    approved = models.BooleanField(default=True, null=True, blank=True)
    created_date = models.DateTimeField(auto_now_add=True)

    class Meta:
        db_table = 'loan_topups'
        verbose_name = 'Loan Topup'
        verbose_name_plural = 'Loan Topups'


class Loansecurities(models.Model):
    loan = models.ForeignKey(Loans, related_name='securities', on_delete=models.CASCADE)
    name = models.CharField(max_length=255, null=True, blank=True)
    value = models.FloatField(null=True, blank=True)
    description = models.TextField(null=True, blank=True)
    file = models.FileField(upload_to='loan_uploads')
    added_by = models.ForeignKey('accounts.Staff', on_delete=models.SET_NULL, null=True)
    created_date = models.DateTimeField(auto_now_add=True)

    class Meta:
        db_table = 'loan_securities'
        verbose_name = 'Loan Security'
        verbose_name_plural = 'Loan Securities'

    @property
    def file_url(self):
        if self.file and hasattr(self.file, 'url'):
            return self.file.url


class Loanguarantors(models.Model):
    loan = models.ForeignKey(Loans, related_name='guarantors', on_delete=models.CASCADE)
    biodata = models.ForeignKey(Biodata, on_delete=models.CASCADE, blank=True, null=True)
    added_by = models.ForeignKey('accounts.Staff', on_delete=models.SET_NULL, null=True)
    created_date = models.DateTimeField(auto_now_add=True)
    group = models.ForeignKey(Member, related_name='group_guarantors', on_delete=models.SET_NULL, blank=True, null=True)

    class Meta:
        db_table = 'loan_guarantors'
        verbose_name = 'Loan Guarantor'
        verbose_name_plural = 'Loan Guarantors'


class LoanOrAccountOrShareBroker(models.Model):
    THETYPE = (
        ('l', 'Loan type'),
        ('a', 'account type'),
        ('s', 'share'),)
    account_type = models.ForeignKey(AccountTypes, on_delete=models.CASCADE, null=True, blank=True)
    loan_type = models.ForeignKey(LoanTypes, on_delete=models.CASCADE, null=True, blank=True)
    the_type = models.CharField(max_length=1, choices=THETYPE, default='d')

    class Meta:
        db_table = 'loan_or_account_or_share_broker'
        verbose_name = 'Loan or account or share broker'
        verbose_name_plural = 'Loan or account or share brokers'

    def __str__(self):
        return str(self.the_type)


class GeneralCharge(models.Model):
    class APPLICATION(models.TextChoices):
        LOANS = 'l', 'Loans'
        REGISTRATION = 'r', 'Registration'
        SHARES = 's', 'Shares'
        OTHERS = 'o', 'Others'

    EXEC_PERIOD = (
        ('d', 'Daily'),
        ('w', 'Weekly'),
        ('m', 'Monthly'),
        ('y', 'Yearly'),
    )
    REVENUE_STATUS = (
        ('YES', 'Yes'),
        ('NO', 'No'),
    )

    charge = models.CharField(max_length=255, blank=False, null=False, verbose_name='Charge Name')
    application = models.CharField(max_length=1, choices=APPLICATION.choices, default='d')
    # the_type = models.ForeignKey(LoanOrAccountOrShareBroker, on_delete=models.CASCADE, null=True, blank=True)
    amount = models.FloatField(default=0, null=False, blank=False)
    is_percentage = models.BooleanField(default=False)
    is_fine = models.BooleanField(default=False)
    status = models.BooleanField(default=False)
    date_added = models.DateField(auto_now_add=True)
    added_by = models.ForeignKey('accounts.Staff', on_delete=models.SET_NULL, null=True)
    business = models.ForeignKey('accounts.Business', related_name='general_charges', on_delete=models.CASCADE, null=True)
    execution_period = models.CharField(max_length=1, choices=EXEC_PERIOD, default='d', blank=True, null=True)
    execution_period_number = models.IntegerField(default=0, blank=True, null=True)
    is_revenue = models.CharField(max_length=4, default='YES', choices=REVENUE_STATUS)


    # @property
    # def get_id(self):
    #     if self.the_type.the_type == 'l':
    #         return self.the_type.loan_type.id
    #     elif self.the_type.the_type == 'a':
    #         return self.the_type.account_type.id
    #     else:
    #         return 'null'
    def __str__(self):
        return str(self.charge) + '--' + str(self.business)


class ApplicationAccountOrLoanType(models.Model):
    general_charge = models.ForeignKey(GeneralCharge, related_name='applications', on_delete=models.CASCADE, null=True, blank=True)
    account_type = models.ForeignKey(AccountTypes, on_delete=models.CASCADE, null=True, blank=True)
    loan_type = models.ForeignKey(LoanTypes, related_name='application_account_loan_type', on_delete=models.CASCADE, null=True, blank=True)

    class Meta:
        db_table = 'application_account_or_loan_type'
        verbose_name = 'application account or loan or share type'
        verbose_name_plural = 'application account or loan or share types'

    def __str__(self):
        return str(self.general_charge) + '--' + str(self.account_type) + '--' + str(self.loan_type)


class LoanCharges(models.Model):
    amount = models.FloatField()
    charge = models.ForeignKey(GeneralCharge, on_delete=models.CASCADE, null=True, blank=True)
    loan = models.ForeignKey(Loans, on_delete=models.CASCADE, related_name='loan_charges', null=True, blank=True)

    class Meta:
        db_table = 'loan_charges'
        verbose_name = 'Loan charge'
        verbose_name_plural = 'Loan charges'

    def __str__(self):
        return str(self.charge)

class LoanRepaymentUpload(models.Model):
    narration = models.TextField(blank=False, null=False)
    total_records_added = models.FloatField(blank=False, null=False)
    total_records_failed = models.FloatField(blank=False, null=False)
    transactions_created = models.TextField(blank=False, null=False)
    failed_loans = models.TextField(blank=False, null=False)
    date_added = models.DateField(auto_now_add=True)
    branch = models.ForeignKey('accounts.Branch', on_delete=models.CASCADE, related_name='repayment_branch', null=False, blank=False)
    file = models.FileField(upload_to='loan repayments', null=False, blank=False)
    class Meta:
        db_table = 'repayment_upload'
        verbose_name = 'Repayment upload'
        verbose_name_plural = 'Repayment uploads'

    def __str__(self):
        return str(self.narration)


class BouncedLoansUpload(models.Model):
    narration = models.TextField(blank=False, null=False)
    total_records_added = models.FloatField(blank=False, null=False)
    total_records_failed = models.FloatField(blank=False, null=False)
    transactions_created = models.TextField(blank=False, null=False)
    failed_loans = models.TextField(blank=False, null=False)
    date_added = models.DateField(auto_now_add=True)
    branch = models.ForeignKey('accounts.Branch', on_delete=models.CASCADE, related_name='bounced_branch', null=False, blank=False)
    file = models.FileField(upload_to='bounced loans', null=False, blank=False)
    class Meta:
        db_table = 'bounced_loan_upload'
        verbose_name = 'bounced loan upload'
        verbose_name_plural = 'bounced loan uploads'

    def __str__(self):
        return str(self.narration)

class LoanFines(models.Model):
    loan = models.ForeignKey(Loans, on_delete=models.CASCADE, related_name='loan_fines_custom')
    amount = models.FloatField(default=0, null=False, blank=False)
    narration = models.TextField(null=True)
    added_by = models.ForeignKey('accounts.Staff', on_delete=models.SET_NULL, null=True)
    created_date = models.DateTimeField(auto_now_add=True)

    class Meta:
        db_table = 'loan_fines'
        verbose_name = 'Loan Fines'
        verbose_name_plural = 'Loan Fines'

class LoanFinePayments(models.Model):
    loan = models.ForeignKey(Loans, on_delete=models.CASCADE, related_name='loan_fine_pay')
    amount = models.FloatField(default=0, null=False, blank=False)
    added_by = models.ForeignKey('accounts.Staff', on_delete=models.SET_NULL, null=True)
    created_date = models.DateTimeField(auto_now_add=True)

    class Meta:
        db_table = 'loan_fine_payments'
        verbose_name = 'Loan Fine Payments'
        verbose_name_plural = 'Loan Fine Payments'
