import json
import traceback
from datetime import time, datetime

import xlwt as xlwt
from django.db.models import ExpressionWrapper, Value, Func, CharField, F, Case, When, Sum, FloatField
from django.db.models.functions import Concat, Coalesce
from django.views import View
from django.http import HttpResponse
from django.template import loader

from accounts.models import Business
from loans.models import *
from sacco.models import CurrencySetting, OtherSettings, WithdrawUpload, SavingUpload

from sacco.utils import businessdata, branchdata, biz_staff_branch
from transactions.downloads import render_to_pdf
from transactions.models import Transactions, SharesTransactions
from transactions.reports import today

style = xlwt.easyxf('font: color-index black, bold on; pattern: pattern solid, fore_colour white; align: wrap on,'
                    'vert centre, horiz center; borders: top_color black, bottom_color black,right_color black, '
                    'left_color black, left thin, right thin, top thin, bottom thin;')

style0 = xlwt.easyxf(
    'font: color-index white, bold on; pattern: pattern solid, fore_colour black; align: wrap off,vert centre, '
    'horiz center; borders: top_color black, bottom_color black,right_color black, '
    'left_color black, left thin, right thin, top thin, bottom thin')

style1 = xlwt.easyxf(
    'font: color-index black; pattern: fore_colour white; align: wrap off; borders: top_color black, bottom_color '
    'black,right_color black, left_color black, left thin, right thin, top thin, bottom thin')


class ApplicationsPdf(View):
    template_name = 'loans/loan_applications_pdf.html'

    def get(self, request, *args, **kwargs):
        title = 'Loan application'
        loanApplications = Loans.objects.raw(
            "SELECT l. *, b.the_account_id as accountid, b.members_id as memberid, a.acc_number, m.biodata_id, m.id, c.name AS member_name FROM account_broker b, member_accounts a, sacco_member m, biodata c, loans l where b.the_account_id = a.id and b.business_id = %s and b.members_id = m.id and m.biodata_id = c.id and l.account_id = a.id and l.branch_id = %s and (l.loan_status = 0 or l.loan_status = 1 or l.loan_status = 2)" % (
                businessdata(request), branchdata(request)))

        file = render_to_pdf(self.template_name, locals())
        resp = HttpResponse(file.getvalue(), content_type="application/pdf")
        resp['Content-Disposition'] = 'attachment; filename=Loanapplications'
        return resp


class Runningpdf(View):
    template_name = 'loans/running_pdf.html'

    def get(self, request, *args, **kwargs):
        title = 'Running Loans'
        business = Business.objects.filter(id=businessdata(request)).first()

        # print('account running', running_loans.account.account_type)

        running_loans = Loans.objects.raw(
            "SELECT l. *, b.the_account_id as accountid, b.members_id as memberid, a.acc_number, m.biodata_id, m.id, c.name AS member_name FROM account_broker b, member_accounts a, sacco_member m, biodata c, loans l where b.the_account_id = a.id and b.business_id = %s and b.members_id = m.id and m.biodata_id = c.id and l.account_id = a.id and l.branch_id = %s and (l.loan_status = 3)" % (
                businessdata(request), branchdata(request)))

        file = render_to_pdf(self.template_name, locals())
        resp = HttpResponse(file.getvalue(), content_type="application/pdf")
        d4 = today.strftime("%b-%d-%Y")
        file_name = f'{business.short_name} Running loans - {d4}'
        resp['Content-Disposition'] = 'attachment; filename="{}.pdf"'.format(file_name.upper())
        return resp


class Runningpffdf(View):
    template_name = 'loans/running_pdf.html'
    title = 'Running Loans'

    def get(self, request, *args, **kwargs):
        loanApplications = Loans.objects.raw(
            "SELECT l. *, b.the_account_id as accountid, b.members_id as memberid, a.acc_number, m.biodata_id, m.id, c.name AS member_name FROM account_broker b, member_accounts a, sacco_member m, biodata c, loans l where b.the_account_id = a.id and b.business_id = %s and b.members_id = m.id and m.biodata_id = c.id and l.account_id = a.id and l.branch_id = %s and (l.loan_status = 3)" % (
                businessdata(request), branchdata(request)))

        file = render_to_pdf(self.template_name, locals())
        resp = HttpResponse(file.getvalue(), content_type="runningloans/pdf")
        resp[('Content-Dis'
              'position')] = 'attachment; filename=Runningloans'
        return resp


class SomeView(View):
    template_name = 'loans/loan_applications_pdf.html'
    title = 'Running Loans'

    def get(self, request, *args, **kwargs):
        template_name = 'fdf'
        loanApplications = Loans.objects.raw(
            "SELECT l. *, b.the_account_id as accountid, b.members_id as memberid, a.acc_number, m.biodata_id, m.id, c.name AS member_name FROM account_broker b, member_accounts a, sacco_member m, biodata c, loans l where b.the_account_id = a.id and b.business_id = %s and b.members_id = m.id and m.biodata_id = c.id and l.account_id = a.id and l.branch_id = %s and (l.loan_status = 0 or l.loan_status = 1 or l.loan_status = 2)" % (
                businessdata(request), branchdata(request)))
        # Create the HttpResponse object with the appropriate CSV header.
        response = HttpResponse(
            content_type='text/csv',
            headers={'Content-Disposition': 'attachment; filename="somefilename.csv"'},
        )

        # The data is hard-coded here, but you could load it from a database or
        # some other source.
        csv_data = (
            ('First row', 'Foo', 'Bar', 'Baz'),
            ('Second row', 'A', 'B', 'C', '"Testing"', "Here's a quote"),
        )

        t = loader.get_template(template_name)
        c = {'data': csv_data}
        response.write(t.render(c))
        return response


def export_application(request):
    if request.method == "POST":
        pass
    else:
        loanApplications = Loans.objects.raw(
            "SELECT l. *, b.the_account_id as accountid, b.members_id as memberid, a.acc_number, m.biodata_id, m.id, c.name AS member_name FROM account_broker b, member_accounts a, sacco_member m, biodata c, loans l where b.the_account_id = a.id and b.business_id = %s and b.members_id = m.id and m.biodata_id = c.id and l.account_id = a.id and l.branch_id = %s and (l.loan_status = 0 or l.loan_status = 1 or l.loan_status = 2)" % (
                businessdata(request), branchdata(request)))
        try:
            response = HttpResponse(content_type='application/ms-excel')
            response['Content-Disposition'] = 'attachment; filename="Loan applications.xls"'

            wb = xlwt.Workbook(encoding='utf-8')
            ws = wb.add_sheet('Loan applications')

            ws.col(1).width = int(20 * 500)
            ws.col(2).width = int(20 * 500)
            ws.col(3).width = int(20 * 400)
            ws.col(4).width = int(20 * 360)
            ws.col(5).width = int(20 * 260)
            ws.col(6).width = int(20 * 260)
            ws.col(7).width = int(20 * 360)
            ws.col(8).width = int(20 * 260)

            style1 = xlwt.easyxf(
                'font: color-index black, bold on; pattern: pattern solid, fore_colour white; ')

            row_num = 1
            ronum = 0
            date_format = xlwt.XFStyle()
            date_format.num_format_str = 'yyyy-mm-dd'
            font_style = xlwt.XFStyle()
            font_style.font.bold = True

            font_stylet = xlwt.XFStyle()
            font_stylet.font.bold = True
            # font_stylet.height = 320

            columns = [
                'SN',
                'MEMBER NAME',
                'DATE',
                'AMOUNT REQUESTE',
                'INTEREST RATE',
                'DURATION',
                'STATUS'
            ]
            ws.write_merge(0, 0, 0, 7, "applications", font_stylet)
            for col_num in range(len(columns)):
                ws.write(row_num, col_num, columns[col_num], font_style)

            font_style = xlwt.XFStyle()
            date_format.num_format_str = 'yyyy-mm-dd'
            font_style.alignment.wrap = False
            font_style.alignment.vert = xlwt.Alignment.HORZ_RIGHT
            for c in loanApplications:
                row_num += 1
                ronum = ronum + 1
                if c.loan_status == 0:
                    loanstate = 'Applied'

                if c.loan_status == 1:
                    loanstate = 'Appraised'

                if c.loan_status == 2:
                    loanstate = 'Approved'

                row = [
                    ronum,
                    c.member_name,
                    c.requested_on.strftime("%Y-%m-%d"),
                    c.amount_requested,
                    c.rate,
                    c.requested_duration,
                    loanstate

                ]

                for col_num in range(len(row)):
                    ws.write(row_num, col_num, row[col_num], font_style)

            wb.save(response)

            return response

        except Exception as ex:
            print(str(ex))
            traceback.print_exc()

            return HttpResponse(str(ex))


def RunningLoansExcel(request):
    response = HttpResponse(content_type='application/ms-excel')
    business = Business.objects.filter(id=businessdata(request)).first()
    # branch = biz_staff_branch(request.user)

    d1 = today.strftime("%b-%d-%Y")
    file_name = f'{business.short_name} Running loans - {d1}'
    response['Content-Disposition'] = 'attachment; filename="{}.xls"'.format(file_name.capitalize())
    heading = f'{business.name} Running loans as of {d1}'
    sheet_name = f'{d1} Running loans'
    wb = xlwt.Workbook(encoding='utf-8')
    ws = wb.add_sheet(sheet_name.capitalize())

    # Sheet header, first row
    row_num = 2

    columns = ['Name', 'Loan number','Sacco Number', 'Total Principal', 'Rate (%)', 'Duration', 'Total Payable']

    the_len = len(columns)
    ws.write_merge(0, 1, 0, the_len - 1, '{}'.format(heading.upper()), style)
    for col_num in range(len(columns)):
        ws.write(row_num, col_num, columns[col_num], style0)
        ws.row(2).height_mismatch = True
        ws.row(2).height = 256 * 2
        ws.col(0).width = int(20 * 350)
        ws.col(1).width = int(20 * 200)
        ws.col(2).width = int(20 * 250)
        ws.col(3).width = int(20 * 120)
        ws.col(4).width = int(20 * 200)
        ws.col(5).width = int(20 * 200)
        ws.col(5).width = int(20 * 200)

    running_loans = Loans.objects.filter(branch=branchdata(request), loan_status=3,
                                       ) \
        .annotate(principal_paid=Sum('loan_trans__reporting_amount')).annotate(
        applicant_name=F('applicant__biodata__name'),
        acc_number = F('account__acc_number'),
        applicant_phone=F('applicant__biodata__contact'),
        applicant_email=F('applicant__biodata__email')) \
        .annotate(principal_balance=F('amount_approved') - F('principal_paid')) \
        .values('applicant_name','loan_number', 'acc_number','applicant_phone', 'applicant_email', 'amount_approved', 'principal_balance',
                'balance', 'schedule_start', 'approved_duration', 'amount_requested', 'rate')
    saving_rows = running_loans.annotate(
        started=ExpressionWrapper(
            Func(F('schedule_start'), Value('%d/%m/%Y'), function='DATE_FORMAT'),
            output_field=CharField()),
        duration=Concat(F('approved_duration'), Value(' '), Case(
            When(interval=0, then=Value('Day(s)')),
            When(interval=1, then=Value('Week(s)')),
            When(interval=2, then=Value('Fortnight(s)')),
            When(interval=4, then=Value('Year(s)')),
            When(interval=3, then=Value('Month(s)')),

        ), output_field=CharField())

    ).values_list('applicant_name','loan_number','acc_number', 'amount_approved', 'rate', 'duration', 'balance')
    for saving_row in saving_rows:
        row_num += 1
        for col_num in range(len(saving_row)):
            ws.write(row_num, col_num, saving_row[col_num])

    wb.save(response)
    return response


def ExpensesExcel(request):
    response = HttpResponse(content_type='application/ms-excel')
    business = Business.objects.filter(id=businessdata(request)).first()
    branch = biz_staff_branch(request.user)

    d1 = today.strftime("%b-%d-%Y")
    file_name = f'{business.short_name} Expenses - {d1}'
    response['Content-Disposition'] = 'attachment; filename="{}.xls"'.format(file_name.capitalize())
    heading = f'{business.name} Expenses as of {d1}'
    sheet_name = f'{d1} Expenses'
    wb = xlwt.Workbook(encoding='utf-8')
    ws = wb.add_sheet(sheet_name.capitalize())

    # Sheet header, first row
    row_num = 2

    columns = ['Date', 'Reference', 'Amount', 'Expense Account', 'Narration']

    the_len = len(columns)
    ws.write_merge(0, 1, 0, the_len - 1, '{}'.format(heading.upper()), style)
    for col_num in range(len(columns)):
        ws.write(row_num, col_num, columns[col_num], style0)
        ws.row(2).height_mismatch = True
        ws.row(2).height = 256 * 2

        ws.col(0).width = int(20 * 200)
        ws.col(1).width = int(20 * 200)
        ws.col(2).width = int(20 * 200)
        ws.col(3).width = int(20 * 250)
        ws.col(4).width = int(20 * 500)

    expense_transactions = Transactions.objects.filter(branch=branch, transaction_type='Expense Invoice')
    saving_rows = expense_transactions.annotate(
        txn_date=ExpressionWrapper(Func(F('tx_date'), Value('%d/%m/%Y'), function='DATE_FORMAT'),
                                   output_field=CharField()),

    ).values_list('txn_date', 'receipt', 'reporting_amount', 'account_dr__name', 'narration')
    for saving_row in saving_rows:
        row_num += 1
        for col_num in range(len(saving_row)):
            ws.write(row_num, col_num, saving_row[col_num])

    wb.save(response)
    return response


def ShareHoldersExcel(request):
    response = HttpResponse(content_type='application/ms-excel')
    date_input = request.GET.get('date')
    from_date = request.GET.get('from_date')
    to_date = request.GET.get('to_date')
    if from_date and to_date:
        from_date = datetime.strptime(from_date, "%Y-%m-%d").date()
        to_date = datetime.strptime(to_date, "%Y-%m-%d").date()
    business = Business.objects.filter(id=businessdata(request)).first()
    branch = biz_staff_branch(request.user)
    business_setting = OtherSettings.objects.filter(business=businessdata(request)).first()

    d1 = today.strftime("%b-%d-%Y")
    file_name = f'{business.short_name} Share Holders - {d1}'
    sheet_name = f'{d1} Share Holders'

    if from_date and to_date:
        d1 = f'{from_date} to {to_date}'
        file_name = 'selected range'
        sheet_name = f'{d1}'
    response['Content-Disposition'] = 'attachment; filename="{}.xls"'.format(file_name.capitalize())
    heading = f'{business.name} Share Holders as of {d1}'
    wb = xlwt.Workbook(encoding='utf-8')
    ws = wb.add_sheet(sheet_name.capitalize(), cell_overwrite_ok=True)

    # Sheet header, first row
    row_num = 2

    columns = ['Name', 'Telephone', 'Account Number', 'Shares', 'Share Capital']

    the_len = len(columns)
    ws.write_merge(0, 1, 0, the_len - 1, '{}'.format(heading.upper()), style)
    for col_num in range(len(columns)):
        ws.write(row_num, col_num, columns[col_num], style0)
        ws.row(2).height_mismatch = True
        ws.row(2).height = 256 * 2

        ws.col(0).width = int(20 * 400)
        ws.col(1).width = int(20 * 250)
        ws.col(2).width = int(20 * 100)
        ws.col(3).width = int(20 * 250)
        ws.col(4).width = int(20 * 200)

    share_holders = Member.objects.filter(member_type='s', biodata__business=businessdata(request))

    shared_member = []

    branch_trans = SharesTransactions.objects.filter(branch=branch)
    if date_input:
        branch_trans = SharesTransactions.objects.filter(branch=branch, date__lte=date_input)
    if from_date and to_date:
        branch_trans = SharesTransactions.objects.filter(branch=branch, date__range=[from_date, to_date])

    for member in share_holders:
        share_purchases = branch_trans.filter(buyer=member).values('buyer').aggregate(
            purchases=Coalesce(Sum('shares'), 0.0, output_field=FloatField()))['purchases']
        share_sale = branch_trans.filter(seller=member).values('seller').aggregate(
            sales=Coalesce(Sum('shares'), 0.0, output_field=FloatField()))['sales']
        shared_member.append(
            {
                'Name': member.biodata.name,
                'Telephone': member.biodata.contact,
                'Account Number': member.accounts.all().first().acc_number,
                'Shares': share_purchases - share_sale,
                'Share Capital': (share_purchases - share_sale) * business_setting.share_price,
            }

        )
    columns = list(shared_member[0].keys())

    # write columns, start from row 1
    for i, row in enumerate(shared_member, 4):
        for j, col in enumerate(columns):
            ws.write(i, j, row[col])

    wb.save(response)
    return response


def FailedRepaymentExcel(request, pk):
    response = HttpResponse(content_type='application/ms-excel')
    try:
        upload = LoanRepaymentUpload.objects.get(id=pk)
    except Exception as e:
        print('Error', e)
        return 'Records no longer exist'

    file_name = f'Failed repayments registered on {str(upload.date_added)}'
    sheet_name = f'failed repayments details'


    response['Content-Disposition'] = 'attachment; filename="{}.xls"'.format(file_name.capitalize())
    heading = f'FAILED REPAYMENTS'
    wb = xlwt.Workbook(encoding='utf-8')
    ws = wb.add_sheet(sheet_name.capitalize(), cell_overwrite_ok=True)

    # Sheet header, first row
    row_num = 2

    columns = ['Loan number', 'Details']

    the_len = len(columns)
    ws.write_merge(0, 1, 0, the_len - 1, '{}'.format(heading.upper()), style)
    for col_num in range(len(columns)):
        ws.write(row_num, col_num, columns[col_num], style0)
        ws.row(2).height_mismatch = True
        ws.row(2).height = 256 * 2

        ws.col(0).width = int(20 * 400)
        ws.col(1).width = int(20 * 250)
        ws.col(2).width = int(20 * 100)
        ws.col(3).width = int(20 * 250)
        ws.col(4).width = int(20 * 200)


    rows = json.loads(upload.failed_loans)['failed_loans']

    for row in rows:
        row_num += 1
        ws.write(row_num, 0, row['loan number'])
        ws.write(row_num, 1, row['reason'])

    wb.save(response)
    return response

def FailedLoansUploadedExcel(request, pk):
    response = HttpResponse(content_type='application/ms-excel')
    try:
        upload = UploadLoans.objects.get(id=pk)
    except Exception as e:
        print('Error', e)
        return 'Records no longer exist'

    file_name = f'Failed loans registered on {str(upload.upload_date)}'
    sheet_name = f'failed loans upload details'


    response['Content-Disposition'] = 'attachment; filename="{}.xls"'.format(file_name.capitalize())
    heading = f'FAILED LOANS UPLOAD'
    wb = xlwt.Workbook(encoding='utf-8')
    ws = wb.add_sheet(sheet_name.capitalize(), cell_overwrite_ok=True)

    # Sheet header, first row
    row_num = 2

    columns = ['Account number', 'Details']

    the_len = len(columns)
    ws.write_merge(0, 1, 0, the_len - 1, '{}'.format(heading.upper()), style)
    for col_num in range(len(columns)):
        ws.write(row_num, col_num, columns[col_num], style0)
        ws.row(2).height_mismatch = True
        ws.row(2).height = 256 * 2

        ws.col(0).width = int(20 * 400)
        ws.col(1).width = int(20 * 250)
        ws.col(2).width = int(20 * 100)
        ws.col(3).width = int(20 * 250)
        ws.col(4).width = int(20 * 200)


    rows = json.loads(upload.failed_loans)['failed_loans']

    for row in rows:
        row_num += 1
        ws.write(row_num, 0, row['account number'])
        ws.write(row_num, 1, row['reason'])

    wb.save(response)
    return response

def FailedWithdrawExcel(request, pk):
    response = HttpResponse(content_type='application/ms-excel')
    try:
        upload = WithdrawUpload.objects.get(id=pk)
    except Exception as e:
        print('Error', e)
        return 'Records no longer exist'

    file_name = f'Failed withdraws registered on {str(upload.date_added)}'
    sheet_name = f'failed withdraw details'


    response['Content-Disposition'] = 'attachment; filename="{}.xls"'.format(file_name.capitalize())
    heading = f'FAILED WITHDRAWS'
    wb = xlwt.Workbook(encoding='utf-8')
    ws = wb.add_sheet(sheet_name.capitalize(), cell_overwrite_ok=True)

    # Sheet header, first row
    row_num = 2

    columns = ['Account number', 'Details']

    the_len = len(columns)
    ws.write_merge(0, 1, 0, the_len - 1, '{}'.format(heading.upper()), style)
    for col_num in range(len(columns)):
        ws.write(row_num, col_num, columns[col_num], style0)
        ws.row(2).height_mismatch = True
        ws.row(2).height = 256 * 2

        ws.col(0).width = int(20 * 400)
        ws.col(1).width = int(20 * 250)
        ws.col(2).width = int(20 * 100)
        ws.col(3).width = int(20 * 250)
        ws.col(4).width = int(20 * 200)


    rows = json.loads(upload.failed_withdraws)['failed_withdraw']

    for row in rows:
        row_num += 1
        ws.write(row_num, 0, row['account number'])
        ws.write(row_num, 1, row['reason'])

    wb.save(response)
    return response


def FailedSavingExcel(request, pk):
    response = HttpResponse(content_type='application/ms-excel')
    try:
        upload = SavingUpload.objects.get(id=pk)
    except Exception as e:
        print('Error', e)
        return 'Records no longer exist'

    file_name = f'Failed Savings registered on {str(upload.date_added)}'
    sheet_name = f'failed savings details'


    response['Content-Disposition'] = 'attachment; filename="{}.xls"'.format(file_name.capitalize())
    heading = f'FAILED SAVINGS'
    wb = xlwt.Workbook(encoding='utf-8')
    ws = wb.add_sheet(sheet_name.capitalize(), cell_overwrite_ok=True)

    # Sheet header, first row
    row_num = 2

    columns = ['Account number', 'Details']

    the_len = len(columns)
    ws.write_merge(0, 1, 0, the_len - 1, '{}'.format(heading.upper()), style)
    for col_num in range(len(columns)):
        ws.write(row_num, col_num, columns[col_num], style0)
        ws.row(2).height_mismatch = True
        ws.row(2).height = 256 * 2

        ws.col(0).width = int(20 * 400)
        ws.col(1).width = int(20 * 250)
        ws.col(2).width = int(20 * 100)
        ws.col(3).width = int(20 * 250)
        ws.col(4).width = int(20 * 200)


    rows = json.loads(upload.failed_savings)['failed_saving']

    for row in rows:
        row_num += 1
        ws.write(row_num, 0, row['account number'])
        ws.write(row_num, 1, row['reason'])

    wb.save(response)
    return response





class OtherLoansReport(View):
    def get(self, request, *args, **kwargs):
        return
