import json
import traceback
from urllib.parse import unquote

from django.contrib import messages
from django.contrib.auth.decorators import login_required
from django.core import serializers
from django.db.models import Q
from django.http import HttpResponse, HttpResponseRedirect
from django.shortcuts import render
from django.urls import reverse_lazy

from loans.utils.loan_details import get_business_loans_context_data
from sacco.models import AccountBroker
from transactions.models import AccountCategory, Account, Transactions
from transactions.reports import this_business, this_branch, this_financial_year


def generate_json(request):
    objectQuerySet = AccountCategory.objects.filter(business=this_business(request.user))
    data = serializers.serialize('json', list(objectQuerySet))
    return HttpResponse(data)


def AccountCategoryJson():
    json = [
        {
            "fields": {
                "name": "Reserves",
                "dr_cr": "cr",
                "cat_type": "liability",
                "default_accounts": [
                    {
                        "name": "Opening Reserves",
                        "display_name": "Opening Reserves",
                    }
                ]
            }
        },
        {
            "fields": {
                "name": "Cash",
                "dr_cr": "dr",
                "cat_type": "assets",
                "default_accounts": [
                    {
                        "name": "Cash Account",
                        "display_name": "Cash Account"
                    }
                ]
            }
        },
        {
            "fields": {
                "name": "Bank",
                "dr_cr": "dr",
                "cat_type": "assets",
                "default_accounts": []
            }
        },
        {
            "fields": {
                "name": "Expense Accounts",
                "dr_cr": "dr",
                "cat_type": "expenses",
                "default_accounts": [
                    {
                        "name": "PAYE Expenses",
                        "display_name": "PAYE Expenses",
                    },
                    {
                        "name": "Expenses",
                        "display_name": "Expenses",
                    },
                    {
                        "name": "NSSF Expenses",
                        "display_name": "NSSF Expenses",
                    },
                    {
                        "name": "Salary Expenses",
                        "display_name": "Salary Expenses",
                    },
                    {
                        "name": "Allowance Expenses",
                        "display_name": "Allowance Expenses",
                    },
                    {
                        "name": "Staff welfare",
                        "display_name": "Staff welfare",
                    },
                    {
                        "name": "Depreciation Charge",
                        "display_name": "Depreciation Charge",
                    },
                    {
                        "name": "Salaries & wages",
                        "display_name": "Salaries & wages",
                    },
                    {
                        "name": "NSSF 10% Employer",
                        "display_name": "NSSF 10% Employer",
                    },
                    {
                        "name": "Rent Expense",
                        "display_name": "Rent Expense",
                    },
                    {
                        "name": "Transport Expenses",
                        "display_name": "Transport Expenses",
                    },
                    {
                        "name": "Stationery Expenses",
                        "display_name": "Stationery Expenses",
                    },
                    {
                        "name": "Utilities",
                        "display_name": "Utilities",
                    },
                    {
                        "name": "Staff training Costs",
                        "display_name": "Staff training Costs",
                    },
                    {
                        "name": "Internet and Telecommunication expenses",
                        "display_name": "Internet and Telecommunication expenses",
                    },
                    {
                        "name": "Allowances",
                        "display_name": "Allowances",
                    },
                    {
                        "name": "Advertising & Marketing",
                        "display_name": "Advertising & Marketing",
                    },
                ]
            }
        },
        {
            "fields": {
                "name": "Members",
                "dr_cr": "cr",
                "cat_type": "liability",
                "default_accounts": []
            }
        },
        {
            "fields": {
                "name": "Account Receivables",
                "dr_cr": "dr",
                "cat_type": "assets",
                "default_accounts": [
                    {
                        "name": "Loan Receivables",
                        "display_name": "Loan Receivables",
                    }
                ]
            }
        },
        {
            "fields": {
                "name": "Accounts Payable",
                "dr_cr": "cr",
                "cat_type": "liability",
                "default_accounts": [
                    {
                        "name": "NSSF Payable",
                        "display_name": "NSSF Payable",
                    },
                    {
                        "name": "PAYE Payable",
                        "display_name": "PAYE Payable",
                    },
                    {
                        "name": "Salary Payable",
                        "display_name": "Salary Payable",
                    },
                    {
                        "name": "Allowance Payable",
                        "display_name": "Allowance Payable",
                    },
                ]
            }
        },
        {
            "fields": {
                "name": "Income",
                "dr_cr": "cr",
                "cat_type": "income",
                "default_accounts": [
                    {
                        "name": "Sales",
                        "display_name": "Sales",
                    },
                    {
                        "name": "Shares Income",
                        "display_name": "Shares Income",
                    }
                ]
            }
        },
        {
            "fields": {
                "name": "Registration Income",
                "dr_cr": "cr",
                "cat_type": "income",
                "default_accounts": [
                    {
                        "name": "Registration Charges",
                        "display_name": "Registration Charges",
                    }
                ]
            }
        },
        {
            "fields": {
                "name": "Deposit Income",
                "dr_cr": "cr",
                "cat_type": "income",
                "default_accounts": [
                    {
                        "name": "Deposit Charges",
                        "display_name": "Deposit Charges",
                    }
                ]
            }
        },
        {
            "fields": {
                "name": "Withdraw Income",
                "dr_cr": "cr",
                "cat_type": "income",
                "default_accounts": [
                    {
                        "name": "Withdraw Charges",
                        "display_name": "Withdraw Charges",
                    }
                ]
            }
        },
        {
            "fields": {
                "name": "Transfer Income",
                "dr_cr": "cr",
                "cat_type": "income",
                "default_accounts": [
                    {
                        "name": "Account Transfer Charges",
                        "display_name": "Account Transfer Charges",
                    }
                ]
            }
        },
        {
            "fields": {
                "name": "Loan incomes",
                "dr_cr": "cr",
                "cat_type": "income",
                "default_accounts": [
                    {
                        "name": "Loan Charges",
                        "display_name": "Loan Charges",
                    },
                    {
                        "name": "Loan Interest",
                        "display_name": "Loan Interest",
                    }
                ]
            }
        },
        {
            "fields": {
                "name": "Fixed Assets",
                "dr_cr": "dr",
                "cat_type": "assets",
                "default_accounts": []
            }
        },
        {
            "fields": {
                "name": "Depreciation Account",
                "dr_cr": "dr",
                "cat_type": "expenses",
                "default_accounts": []
            }
        },
        {
            "fields": {
                "name": "Asset Disposal Account",
                "dr_cr": "dr",
                "cat_type": "assets",
                "default_accounts": []
            }
        },
        {
            "model": "transactions.accountcategory",
            "pk": 17,
            "fields": {
                "name": "Accumulating Depreciation Account",
                "dr_cr": "dr",
                "cat_type": "assets",
                "default_accounts": []
            }
        },
        {
            "fields": {
                "name": "Asset Disposal Income",
                "dr_cr": "cr",
                "cat_type": "income",
                "default_accounts": []
            }
        },
        {
            "fields": {
                "name": "Suppliers",
                "dr_cr": "cr",
                "cat_type": "liability",
                "default_accounts": []
            }
        },
        {
            "fields": {
                "name": "Customers",
                "dr_cr": "dr",
                "cat_type": "assets",
                "default_accounts": []
            }
        },
        {
            "fields": {
                "name": "Mobile Money",
                "dr_cr": "dr",
                "cat_type": "assets",
                "default_accounts": []
            }
        },
        {
            "fields": {
                "name": "Shares",
                "dr_cr": "cr",
                "cat_type": "liability",
                "default_accounts": [
                    {
                        "name": "Share Capital",
                        "display_name": "Share Capital",
                    }
                ]
            }
        },
        {
            "fields": {
                "name": "Bad debts",
                "dr_cr": "dr",
                "cat_type": "expenses",
                "default_accounts": [
                    {
                        "name": "Bad loans written off",
                        "display_name": "Bad loans written off",
                    }
                ]
            }
        }
    ]
    return json


@login_required
def chart_of_accounts(request):
    business_context = get_business_loans_context_data(request)
    if business_context['branch'] is None:
        messages.error(
            request, "error",
            extra_tags="Action not allowed in the central view. Please logout and login into the respective branch"
        )
        return HttpResponseRedirect(reverse_lazy('index'))

    if request.method == "POST":

        code = request.POST['code']
        name = request.POST['name']
        cat = request.POST['category']
        eul = request.POST['eul']

        try:
            category = AccountCategory.objects.get(id=cat)
        except AccountCategory.DoesNotExist:
            # messages.error(request, 'Selected Category doesn't Exist')
            return HttpResponse("1")

        # try to edit or pass
        try:
            id = request.POST['id']
            cc = Account.objects.get(id=id)
            cc.code = code
            cc.display_name = name
            cc.save()
            messages.success(request, 'Account Edited Successfully')

            return HttpResponseRedirect(request.META.get("HTTP_REFERER"))
        except:
            pass

        try:
            # try to check if such account exists
            Account.objects.get(Q(name=name), Q(business=business_context['branch'].business))
            # messages.error(request, 'Account with this Name already exists')
            return HttpResponse("2")
        except:
            try:
                n = Account.objects.create(
                    code=code,
                    name=name,
                    display_name=name,
                    category=category,
                    business=business_context['branch'].business,
                    added_by=request.user.staff
                )
                if category.name == "Fixed Assets":
                    n.eul = eul
                    n.save()
                    # start with Depreciation Account
                    # get category
                    dep_category = AccountCategory.objects.filter(business=business_context['branch'].business,
                                                                  name='Depreciation Account').first()
                    # create nw
                    dep_acc = Account.objects.create(
                        name="%s depreciation Account" % name,
                        display_name="%s depreciation Account" % name,
                        category=dep_category,
                        related_to=n.id,
                        code=code,
                        business=business_context['branch'].business,
                        added_by=request.user.staff
                    )

                    # go to Accumulated depreciation
                    accu_dep_acc = AccountCategory.objects.filter(business=business_context['branch'].business,
                                                                  name='Accumulating Depreciation Account').first()
                    # create now
                    dep_acc = Account.objects.create(
                        name="%s accumulating depreciation Account" % name,
                        display_name="%s accumulating depreciation Account" % name,
                        code=code,
                        category=accu_dep_acc,
                        related_to=n.id,
                        business=business_context['branch'].business,
                        added_by=request.user.staff
                    )
                    # go to asset disposal account
                    asset_disposal = AccountCategory.objects.filter(business=business_context['branch'].business,
                                                                    name='Asset Disposal Account').first()
                    # create now
                    asst_disp = Account.objects.create(
                        name="%s disposal Account" % name,
                        display_name="%s disposal Account" % name,
                        code=code,
                        category=asset_disposal,
                        related_to=n.id,
                        business=business_context['branch'].business,
                        added_by=request.user.staff
                    )

                    # go to income account
                    disposal_income = AccountCategory.objects.filter(business=business_context['branch'].business,
                                                                     name='Asset Disposal Income').first()
                    # create now
                    asst_disp = Account.objects.create(
                        name="%s disposal income Account" % name,
                        display_name="%s disposal income Account" % name,
                        code=code,
                        category=disposal_income,
                        related_to=n.id,
                        business=business_context['branch'].business,
                        added_by=request.user.staff
                    )

                # if category.cat_type != "expenses" and category.cat_type != "income":
                if 'opening_balance' in request.POST:
                    tx_date = request.POST['date']
                    opening_balance = request.POST['opening_balance'].replace(',', '')
                    if opening_balance is None:
                        opening_balance = 0
                    reserves_acc = request.POST['reserves']
                    reserves_account = Account.objects.get(id=reserves_acc)
                    if category.cat_type == "liability" or category.cat_type == "income":
                        action = "Opening Balances - %s" % category.cat_type
                        acc_dr = reserves_account
                        acc_cr = n
                    else:
                        action = "Opening Balances - %s" % category.cat_type
                        acc_dr = n
                        acc_cr = reserves_account
                    narative = action

                    newtx = Transactions.objects.create(
                        branch=business_context['branch'],
                        transaction_type=action,
                        account_dr=acc_dr,
                        account_cr=acc_cr,
                        financial_year=this_financial_year(request),
                        reporting_amount=opening_balance,
                        narration=narative,
                        tx_date=tx_date,
                        added_by=request.user.staff
                    )

                return HttpResponse("/transactions/chart-of-accounts/")
            except Exception as ex:
                traceback.print_exc()
                return HttpResponse("3")
    else:
        accounts = Account.objects.filter(Q(business=business_context['branch'].business), ~Q(category__name="Members"))
        categories = AccountCategory.objects.filter(business=business_context['branch'].business).order_by("name")
        return render(request, 'transactions/chart_of_accounts.html',
                      {'accounts': accounts, "categories": categories})


def get_reserve_list(request, id):
    try:
        category = AccountCategory.objects.get(id=id)
    except Exception as ex:
        return HttpResponse("1")
    if category.cat_type == "expenses" or category.cat_type == "income" or category.name == "Reserves":
        # return HttpResponse("0")
        pass
    else:
        pass

    accountx = Account.objects.filter(business=this_business(request.user),
                                      category=AccountCategory.objects.filter(
                                          Q(business=this_business(request.user)), Q(name='Reserves')).first()).first()

    resp = "<option value='%s' selected>%s</option>" % (accountx.id, accountx.name)

    return HttpResponse(resp)

def get_eul(request, id):
    try:
        cat = AccountCategory.objects.get(id=id)
        if cat.name == "Fixed Assets":
            return HttpResponse("10")
        else:
            return HttpResponse("2")
    except Exception as ex:
        return HttpResponse("1")


def TestAccount(request):
    ac = Account.objects.all()
    for i in ac:
        member = i.member.member.biodata.name


@login_required
def new_account_popup(request, category, name, js_id):
    if request.method == "POST":
        name = request.POST['name']
        cat = request.POST['category']
        eul = request.POST['eul']

        try:
            category = AccountCategory.objects.get(id=cat)
        except:
            # messages.error(request, 'Selected Category doesnot Exist')
            return HttpResponse("1")

        try:
            # try to check if such account exists
            Account.objects.get(Q(name=name), Q(business=this_business(request.user)))
            # messages.error(request, 'Account with this Name already exists')
            return HttpResponse("2")
        except:
            try:
                n = Account.objects.create(
                    name=name,
                    category=category,
                    business=this_business(request.user),
                    added_by=request.user.staff
                )
                if category.name == "Fixed Assets":
                    n.eul = eul
                    n.save()
                    # create three accounts
                    # start with Depreciation Account
                    # get category
                    dep_category = AccountCategory.objects.filter(business=this_business(request.user),
                                                                  name='Depreciation Account').first()
                    # create nw
                    dep_acc = Account.objects.create(
                        name="%s depreciation Account" % name,
                        category=dep_category,
                        related_to=n.id,
                        business=this_business(request.user),
                        added_by=request.user.staff
                    )

                    # go to Accumulated depreciation
                    accu_dep_acc = AccountCategory.objects.filter(business=this_business(request.user),
                                                                  name='Accumulating Depreciation Account').first()
                    # create now
                    dep_acc = Account.objects.create(
                        name="%s accumulating depreciation Account" % name,
                        category=accu_dep_acc,
                        related_to=n.id,
                        business=this_business(request.user),
                        added_by=request.user.staff
                    )
                    # go to asset disposal account
                    asset_disposal = AccountCategory.objects.filter(business=this_business(request.user),
                                                                    name='Asset Disposal Account').first()
                    # create now
                    asst_disp = Account.objects.create(
                        name="%s disposal Account" % name,
                        category=asset_disposal,
                        related_to=n.id,
                        business=this_business(request.user),
                        added_by=request.user.staff
                    )

                    # go to income account
                    disposal_income = AccountCategory.objects.filter(business=this_business(request.user),
                                                                     name='Asset Disposal Income').first()
                    # create now
                    asst_disp = Account.objects.create(
                        name="%s disposal income Account" % name,
                        category=disposal_income,
                        related_to=n.id,
                        business=this_business(request.user),
                        added_by=request.user.staff
                    )

                if category.cat_type != "expenses" and category.cat_type != "income":
                    if 'opening_balance' in request.POST:
                        opening_balance = request.POST['opening_balance'].replace(',', '')
                        if opening_balance is None:
                            opening_balance = 0
                        reserves_acc = request.POST['reserves']
                        reserves_account = Account.objects.get(id=reserves_acc)
                        if category.cat_type == "liability":
                            action = "Opening Balances - Liabilities"
                            acc_dr = reserves_account
                            acc_cr = n
                        else:
                            action = "Opening Balances - Assets"
                            acc_dr = n
                            acc_cr = reserves_account
                        narative = action

                        newtx = Transactions.objects.create(
                            branch=this_branch(request.user),
                            transaction_type=action,
                            account_dr=acc_dr,
                            account_cr=acc_cr,
                            financial_year=this_financial_year(request),
                            reporting_amount=opening_balance,
                            narration=narative,
                            added_by=request.user.staff
                        )

                return HttpResponse("<option value='%s' selected>%s</option>" % (n.id, n.name))
            except Exception as ex:
                traceback.print_exc()
                return HttpResponse("3")
    else:
        selected = []
        cats = unquote(category)
        name = unquote(name)
        for x in cats.split("|"):
            selected_category = AccountCategory.objects.filter(name=x, business=this_business(request.user)).first()
            if selected_category:
                selected.append(selected_category)

        return render(request, 'transactions/pop_new_gl_account.html',
                      {"selected_category": selected, "name": name, "id": js_id})


def search_members(request):
    business_context = get_business_loans_context_data(request)
    current_business_ids = business_context['data_context']

    if request.method == "GET":
        search_text = request.GET['search_text']
        accounts = []
        members = []

        if search_text and search_text != u"":
            search_text = request.GET['search_text']
            # members = Account.objects.filter(Q(member__member_name__contains=search_text) | Q(member__acc_number__contains=search_text))
            members = AccountBroker.objects.filter(Q(business_id__in=current_business_ids, the_account__acc_number__contains=search_text) | Q(business_id__in=current_business_ids, members__biodata__name__icontains=search_text))
            for m in members:
                #get actual account
                acc = Account.objects.filter(member=m.the_account)
                for ac in acc:
                    accounts.append(ac)

        return render(request, 'transactions/exports/member_search_results.html', {'accounts': accounts})
