from datetime import datetime

from django.db import models

# from sacco.models import Business, Branch, Staff, FinancialYear
from django.db.models.signals import post_save
from django.dispatch import receiver

from loans.models import Loans
from sacco.models import FinancialYear, Member, MemberAccount, AccountBroker


class AccountCategory(models.Model):
    DR_CR = (("dr", "Debit"), ("cr", "Credit"))
    CAT_CHOICES = (
        ("liability", "Liability"),
        ("assets", "Assets"),
        ("expenses", "Expenses"),
        ("income", "Income"),
        ("equity", "Equity"),
    )

    name = models.CharField(max_length=255)
    dr_cr = models.CharField(max_length=20, choices=DR_CR, null=True, blank=True)
    cat_class = models.CharField(max_length=255, null=True, blank=True, default="")
    cat_type = models.CharField(
        max_length=20, choices=CAT_CHOICES, null=True, blank=True
    )
    business = models.ForeignKey("accounts.Business", on_delete=models.PROTECT)
    date_added = models.DateTimeField(auto_now=True)
    added_by = models.ForeignKey("accounts.Staff", on_delete=models.SET_NULL, null=True)

    def __str__(self):
        return "%s -- %s (%s) -> %s" % (
            self.business.name,
            self.name,
            self.dr_cr,
            self.cat_type,
        )

    class Meta:
        db_table = "account_category"
        verbose_name_plural = "Account Categories"


class Account(models.Model):
    code = models.CharField(max_length=20, null=True, blank=True)
    name = models.CharField(max_length=225)
    display_name = models.CharField(max_length=255, null=True, blank=True, default="")
    category = models.ForeignKey(AccountCategory, on_delete=models.PROTECT)
    business = models.ForeignKey(
        "accounts.Business", related_name="accounts", on_delete=models.PROTECT
    )
    member = models.ForeignKey(
        MemberAccount,
        related_name="account",
        on_delete=models.SET_NULL,
        null=True,
        blank=True,
    )
    eul = models.IntegerField(null=True, blank=True)
    related_to = models.IntegerField(null=True, blank=True)
    date_added = models.DateTimeField(auto_now=True)
    added_by = models.ForeignKey("accounts.Staff", on_delete=models.SET_NULL, null=True)

    def __unicode__(self):
        if self.member is None:
            return "%s - %s" % (self.name, self.category.name)
        else:
            membername = self.name

            return membername

    @property
    def bio_name(self):
        # Assuming there is a ForeignKey relationship from Account to Biodata named 'biodata'
        if self.member is not None:
            the_broker = AccountBroker.objects.filter(the_account=self.member, business=self.business)
            if the_broker.exists():
                if the_broker.first().members.biodata is not None:
                    return the_broker.first().members.biodata.name
                else:
                    if the_broker.first().members.is_group:
                        return the_broker.first().members.name
                    return ''
            else:
                return ''
        else:
            return ''

    class Meta:
        db_table = "accounts"

    @property
    def member_name(self):
        if self.member is not None:
            return self.member
        return None

    def __str__(self):
        return self.name + "--" + str(self.member)


class SharesTransactions(models.Model):
    buyer = models.ForeignKey(
        Member, related_name="shares_buyer", on_delete=models.SET_NULL, null=True
    )
    seller = models.ForeignKey(
        Member,
        related_name="shares_seller",
        on_delete=models.SET_NULL,
        null=True,
        blank=True,
    )
    shares = models.FloatField(default=0)
    date = models.DateField(blank=True, null=True)
    narration = models.TextField(null=True)
    branch = models.ForeignKey("accounts.Branch", on_delete=models.CASCADE)
    synced = models.BooleanField(default=False, verbose_name="Synced to the portal")

    @property
    def transaction(self):
        try:
            tx = Transactions.objects.filter(shares=self).first()
            return tx
        except:
            return None

    class Meta:
        db_table = "shares_transaction"
        verbose_name = "Shares Transactions"

    def __str__(self):
        try:
            return "%s from %s to %s" % (
                self.shares,
                self.buyer.biodata.name,
                self.seller.biodata.name,
            )
        except:
            return f"Shares for {str(self.buyer)}"


class Transactions(models.Model):
    txno = models.CharField(max_length=225, null=True, blank=True)
    daily_no = models.IntegerField(null=True, blank=True)
    receipt = models.CharField(max_length=255, null=True, blank=True)
    branch = models.ForeignKey("accounts.Branch", related_name="transactions", on_delete=models.PROTECT)
    financial_year = models.ForeignKey(FinancialYear, on_delete=models.PROTECT, blank=True, null=True)
    fixed_asset = models.ForeignKey("FixedAsset", on_delete=models.SET_NULL, blank=True, null=True)
    loan = models.ForeignKey(Loans, on_delete=models.CASCADE, related_name="loan_trans", null=True, blank=True)
    shares = models.ForeignKey(
        SharesTransactions, on_delete=models.CASCADE, null=True, blank=True
    )
    transaction_type = models.CharField(max_length=255, null=True)
    account_dr = models.ForeignKey(
        Account, related_name="account_dr", on_delete=models.PROTECT, null=True
    )
    account_cr = models.ForeignKey(
        Account, related_name="account_cr", on_delete=models.PROTECT, null=True
    )
    narration = models.TextField(null=True)
    tx_date = models.DateField(blank=True, null=True)
    reporting_amount = models.FloatField(null=True, blank=True)
    reference = models.CharField(
        max_length=255, verbose_name="Transaction Ref No.", blank=True, null=True
    )
    # reconciliation
    original_narration = models.TextField(null=True, blank=True)
    original_date = models.DateTimeField(null=True, blank=True)
    original_amount = models.FloatField(null=True, blank=True)
    reconciled_by = models.ForeignKey(
        "accounts.Staff",
        related_name="reconciled_by",
        on_delete=models.SET_NULL,
        null=True,
    )
    date_added = models.DateTimeField(auto_now=True)
    added_by = models.ForeignKey("accounts.Staff", related_name="txn_made_by",on_delete=models.SET_NULL, null=True)
    paid_by = models.CharField(max_length=250, blank=True, null=True)
    is_reversed = models.BooleanField(default=False)
    is_final_transaction = models.BooleanField(default=False)

    def __unicode__(self):
        return self.txno

    class Meta:
        db_table = "transactions"
        verbose_name = "Transaction"
        verbose_name_plural = "Transactions"
        ordering = ["tx_date"]

    def belongs_to_financial_year(self, input_date, *args, **kwargs):
        current_date = input_date
        financial_year = self.financial_year
        if financial_year.start_date <= current_date <= financial_year.end_date:
            return True
        return False


@receiver(post_save, sender=Transactions)
def update_tx(sender, created, instance, **kwargs):
    if created:
        if instance.tx_date is None:
            date_now = datetime.now().date()
        else:
            date_now = instance.tx_date
        datee = datetime.strptime(str(date_now), "%Y-%m-%d")
        # datee = datetime.date(date_now.year, date_now.month, date_now.day)
        # print("gotten date %s"%datee)

        now2 = datee.strftime("%Y-%m-%d")
        last_id = Transactions.objects.filter(
            branch=instance.branch, tx_date=now2
        ).count()
        branch_id = str(instance.branch.id).zfill(3)
        now = datee.strftime("%d%m%y")
        new_id = last_id
        ntxn = str(new_id).zfill(4)
        instance.txno = "%s%s%s" % (branch_id, ntxn, now)
        instance.daily_no = new_id
        instance.tx_date = datee
        instance.save()
        # super().save(*args, **kwargs)


class AuditTrail(models.Model):
    staff = models.ForeignKey("accounts.Staff", on_delete=models.SET_NULL, null=True)
    branch = models.ForeignKey("accounts.Branch", on_delete=models.SET_NULL, null=True)
    date_added = models.DateTimeField(auto_now_add=True)
    log_action = models.CharField(max_length=255)
    description = models.TextField(blank=True, null=True)
    link = models.TextField()

    class Meta:
        db_table = "audit_trails"
        verbose_name = "Audit Trails"

    def __str__(self):
        return self.log_action


class AccountLogs(models.Model):
    timestamp = models.DateTimeField(auto_now_add=True)
    account = models.ForeignKey(Account, related_name='logs', on_delete=models.CASCADE)
    narration = models.TextField()

    class Meta:
        db_table = 'account_logs'


class FixedAsset(models.Model):
    name = models.CharField(max_length=255)
    branch = models.ForeignKey("accounts.Branch", on_delete=models.CASCADE)
    account = models.ForeignKey(
        Account, related_name="gl_acc", on_delete=models.CASCADE
    )
    gl = models.ForeignKey(Transactions, on_delete=models.SET_NULL, null=True)
    acquisition_date = models.DateField()
    particulars = models.TextField(null=True, blank=True)
    supplier = models.ForeignKey(
        Account, related_name="supplierx", on_delete=models.SET_NULL, null=True
    )
    asset_number = models.CharField(max_length=255, null=True, blank=True)
    purchase_price = models.FloatField(default=0)
    working_condition = models.CharField(max_length=255, default="Excellent")
    date_added = models.DateTimeField(auto_now=True)
    added_by = models.ForeignKey("accounts.Staff", on_delete=models.SET_NULL, null=True)

    def __str__(self):
        return self.name


class FixedAssetLocation(models.Model):
    branch = models.ForeignKey("accounts.Branch", on_delete=models.PROTECT)
    name = models.CharField(max_length=255, null=True, blank=True)
    address = models.TextField(null=True, blank=True)
    details = models.TextField(null=True, blank=True)
    date_added = models.DateTimeField(auto_now=True)
    added_by = models.ForeignKey("accounts.Staff", on_delete=models.SET_NULL, null=True)

    def __str__(self):
        return "%s %s" % (self.name, self.branch.name)


class FixedAssetMovement(models.Model):
    branch = models.ForeignKey("accounts.Branch", on_delete=models.PROTECT)
    asset = models.ForeignKey(FixedAsset, on_delete=models.PROTECT, null=True)
    date = models.DateField(null=True, blank=True)
    is_purchase = models.BooleanField(default=False)
    is_disposal = models.BooleanField(default=False)
    is_transfer = models.BooleanField(default=False)
    is_update = models.BooleanField(default=False)
    is_depreciation = models.BooleanField(default=False)
    gl = models.ForeignKey(
        Transactions, on_delete=models.SET_NULL, null=True, blank=True
    )
    location_from = models.ForeignKey(
        FixedAssetLocation,
        related_name="location_from",
        on_delete=models.PROTECT,
        null=True,
        blank=True,
    )
    location_to = models.ForeignKey(
        FixedAssetLocation,
        related_name="location_to",
        on_delete=models.PROTECT,
        null=True,
        blank=True,
    )
    buyer = models.ForeignKey(Account, on_delete=models.PROTECT, null=True, blank=True)
    sale_price = models.FloatField(default=0)
    condition = models.CharField(max_length=255, null=True, blank=True)
    description = models.TextField(null=True, blank=True)
    date_added = models.DateTimeField(auto_now=True)
    added_by = models.ForeignKey("accounts.Staff", on_delete=models.SET_NULL, null=True)

    def __str__(self):
        return self.asset.name
    

class StaffLedgerAssociated(models.Model):
    branch = models.ForeignKey("accounts.Branch", on_delete=models.PROTECT)
    account_asscociated = models.ForeignKey(
        Account, on_delete=models.PROTECT, null=True
    )
    
    staff = models.ForeignKey(
        "accounts.Staff",
        related_name="staff_associated_with_account",
        on_delete=models.SET_NULL,
        null=True,
    )
    date_added = models.DateTimeField(auto_now=True)
    added_by = models.ForeignKey("accounts.Staff", related_name="added_ledger_associated_staff",on_delete=models.SET_NULL, null=True)

    def __unicode__(self):
        return str(self.account_asscociated) + '--' + str(self.staff)

    class Meta:
        db_table = "staff_ledger_associated"
        verbose_name = "Staff ledger associated"
        verbose_name_plural = "Staff ledgers associated"
