import datetime

from django.contrib import messages
from django.core.paginator import Paginator, PageNotAnInteger, EmptyPage
from django.db.models import Q, Sum, F, Case, When, FloatField, Value, Subquery, OuterRef
from django.db.models.functions import Coalesce
from django.http import HttpResponseRedirect
from django.shortcuts import render, redirect
from django.urls import reverse_lazy
from django.views import View

from accounts.models import Business, Branch
from accounts.permission_mixin import BusinessUserMixin
from loans.utils.loan_details import get_business_loans_context_data
from sacco.constants import SHARE
from sacco.models import OtherSettings, CurrencySetting, BusinessShares, Member, FinancialYear, ActivityLog, \
    AccountBroker, MemberAccount, NotiSettings, SaccoSmsSettings, SavingsSharesRecalculate
from sacco.utils import businessdata, branch_id, biz_staff_branch, biz_data, biz_id, checkAndSendMessage, sendTransEmail
from transactions.models import Transactions, Account, SharesTransactions, StaffLedgerAssociated
from transactions.reports import this_branch


class ManageShares(BusinessUserMixin, View):
    template_name = 'sacco/shares_center.html'

    def get(self, request, *args, **kwargs):
        business_context = get_business_loans_context_data(request)
        business_filter = business_context['business_filter']

        if business_context['branch'] is None:
            for biz_id in business_context['data_context']:
                the_biz_branch=Branch.objects.filter(business_id=biz_id).first()
                SavingsSharesRecalculate.objects.create(
                    branch=the_biz_branch,
                    update_type = 'shares'
                )
                SavingsSharesRecalculate.objects.create(
                    branch=the_biz_branch,
                    update_type='savings'
                )
        else:
            SavingsSharesRecalculate.objects.create(
                branch=business_context['branch'],
                update_type='shares'
            )
            SavingsSharesRecalculate.objects.create(
                branch=business_context['branch'],
                update_type='savings'
            )

        # original ---
        # business_setting = OtherSettings.objects.filter(business=businessdata(request)).first()
        # -- new--
        business_setting = OtherSettings.objects.filter(business_id__in=business_context['data_context'])

        business_currency = CurrencySetting.objects.filter(business=businessdata(request)).first()

        # original ---
        # business_shares = BusinessShares.objects.filter(business=businessdata(request)).first()
        # -- new --
        business_shares = BusinessShares.objects.filter(business_id__in=business_context['data_context'])


        trans = SharesTransactions.objects.filter(branch__business_id__in=business_context['data_context']).order_by('-date')

        page = request.GET.get('page', 1)
        paginator = Paginator(trans, 15)
        try:
            transactions = paginator.page(page)
        except PageNotAnInteger:
            transactions = paginator.page(1)
        except EmptyPage:
            transactions = paginator.page(paginator.num_pages)

        # finance_accounts = Account.objects.filter(business=businessdata(request),
        #                                           category__name__in=['Mobile Money', 'Cash', 'Bank'])
        
        #determine if the staff is attached to any accounts
        check_associated_acc = StaffLedgerAssociated.objects.filter(staff=request.user.staff)
        if check_associated_acc.exists():
            acc_ids = list(check_associated_acc.values_list('account_asscociated__id', flat=True))
            finance_accounts = Account.objects.filter(id__in=acc_ids)
        else:
            finance_accounts = Account.objects.filter(business_id__in=business_context['data_context'],
                                           category__name__in=['Mobile Money', 'Cash', 'Bank'])

        # title = 'Shares Center'
        if business_context['the_current_branch_name'] == 'all branches':
            title = "Shares Center for all branches"
        else:
            the_current_branch_name = business_context['the_current_branch_name']
            title = f"Shares Center for {the_current_branch_name}"

        if len(business_context['data_context']) > 1:
            cap = business_shares.aggregate(total=Sum('market_cap'))['total']
        else:
            cap= business_shares.first().market_cap
        if cap is None:
            cap = 0
        # -- oriiginal --
        # price = business_setting.share_price
        # -- new -- get the average
        if len(business_context['data_context']) > 1:
            price = (business_setting.aggregate(total=Sum('share_price'))['total'])
        else:
            price = business_setting.first().share_price
        if price is None:
            price = 0
        price = price/len(business_context['data_context'])
        price =round(price, 2)

        if len(business_context['data_context']) > 1:
            sold = business_shares.aggregate(total=Sum('sold'))['total']
        else:
            sold = business_shares.first().sold
        if sold is None:
            sold = 0
        currency = business_currency.currency
        share_holders = Member.objects.filter(member_type='s', biodata__business_id__in=business_context['data_context'])

        #acc brokers for share holders
        all_shareholders = AccountBroker.objects.filter(business_id__in=business_context['data_context'], members__member_type='s').values(
            'members__id',
            'members__biodata__name',
            'the_account__acc_number'
            )


        # share_limit = business_setting.share_limit
        if len(business_context['data_context']) > 1:
            share_limit = business_setting.aggregate(total=Sum('share_limit'))['total']
        else:
            share_limit = business_setting.first().share_limit
        if share_limit is None:
            share_limit = 0
        share_limit = share_limit/len(business_context['data_context'])

        if cap < 1 or price < 1:
            shares = 0
        else:
            shares = cap / price

        # --- original ------
        # share_capital = sold * price
        # ----- new ----
        # get all accounts for share capital
        share_cap_acc = list(Account.objects.filter(business_id__in=business_context['data_context'], name='Share Capital').values_list('id', flat=True))
        share_capital_cr = Transactions.objects.filter(account_cr_id__in=share_cap_acc).aggregate(total=Sum('reporting_amount'))['total'] or 0
        share_capital_dr = \
        Transactions.objects.filter(account_dr_id__in=share_cap_acc).aggregate(total=Sum('reporting_amount'))[
            'total'] or 0
        share_capital = share_capital_cr - share_capital_dr

        if cap < 1 or price < 1:
            share_balance = 0
        else:
            share_balance = cap / price - sold
        # details to use to sell and transfer shares
        share_price_selling_at = business_setting.first().share_price
        market_cap_at = business_shares.first().market_cap
        sold_at = business_shares.first().sold
        share_limit_at = business_setting.first().share_limit

        return render(request, self.template_name, locals())


class RemoveSharesTransaction(BusinessUserMixin, View):

    def get(self, request, pk, *args, **kwargs):
        the_trans = SharesTransactions.objects.filter(id=pk)
        if not the_trans.exists():
            messages.error(request, 'Failed', extra_tags='Share transation no longer exists')
            return HttpResponseRedirect(reverse_lazy('shares'))

        #get the seller
        the_buyer = the_trans.first().buyer
        the_seller = the_trans.first().seller
        acc_to_update =[]
        if the_buyer is not None:
            acc_to_update.append(the_buyer.id)
        if the_seller is not None:
            acc_to_update.append(the_seller.id)

        # get the actual transaction
        Transactions.objects.filter(shares=the_trans.first()).delete()
        the_trans.delete()

        # print(acc_to_update)

        # update the member shares
        for mem_id in acc_to_update:
            bought = SharesTransactions.objects.filter(buyer_id=mem_id).aggregate(total=Sum('shares'))['total']
            if bought is None:
                bought = 0
            sold = SharesTransactions.objects.filter(seller_id=mem_id).aggregate(total=Sum('shares'))['total']
            if sold is None:
                sold = 0

            actual_shares = bought - sold

            # actual update
            Member.objects.filter(id=mem_id).update(
                shares=actual_shares
            )


        messages.success(request, 'success', extra_tags='Share transation removed successfully')
        return HttpResponseRedirect(reverse_lazy('shares'))

class RemoveSharesTransactionMember(BusinessUserMixin, View):

    def get(self, request, pk, mem_id,  *args, **kwargs):
        the_trans = SharesTransactions.objects.filter(id=pk)
        if not the_trans.exists():
            messages.error(request, 'Failed', extra_tags='Share transaction no longer exists')
            return redirect(request.META.get('HTTP_REFERER', '/'))
        # get the actual transaction
        Transactions.objects.filter(shares=the_trans.first()).delete()
        the_trans.delete()

        #update the member shares
        bought = SharesTransactions.objects.filter(buyer_id=mem_id).aggregate(total=Sum('shares'))['total']
        if bought is None:
            bought = 0
        sold = SharesTransactions.objects.filter(seller_id=mem_id).aggregate(total=Sum('shares'))['total']
        if sold is None:
            sold = 0

        actual_shares = bought - sold

        # actual update
        Member.objects.filter(id=mem_id).update(
            shares=actual_shares
        )
        messages.success(request, 'success', extra_tags='Share transaction removed successfully')
        return redirect(request.META.get('HTTP_REFERER', '/'))



def updatebalance(memberacct, themember):
    getmember = MemberAccount.objects.filter(id=themember)
    for getmemberid in getmember:
        newbalance = Transactions.objects.filter(
            Q(account_dr=memberacct) | Q(account_cr=memberacct)).aggregate(
            bal=Sum(
                Case(
                    When(account_cr=memberacct, then=F('reporting_amount')),
                    default=0, output_field=FloatField()))
                - Sum(
                Case(
                    When(account_dr=memberacct, then=F('reporting_amount')),
                    default=0, output_field=FloatField()))
        )

        print('NEW BALANCE IS ', newbalance['bal'])
        mbal = newbalance['bal']
        getmemberid.balance = float(mbal)
        getmemberid.save()


class SellShares(BusinessUserMixin, View):
    template_name = 'sacco/shares_center.html'

    def get(self, request, *args, **kwargs):
        return HttpResponseRedirect(reverse_lazy('shares'))

    def post(self, request, *args, **kwargs):
        business_context = get_business_loans_context_data(request)
        if business_context['branch'] is None:
            messages.error(
                request, "error",
                extra_tags="Action not allowed in the central view. Please logout and login into the respective branch"
            )
            return HttpResponseRedirect(request.META.get("HTTP_REFERER"))
        if request.method == 'POST':
            buyer = Member.objects.filter(id=request.POST['buyer']).first()
            pay_mode = request.POST['pay_mode']
            usemember_acct = 0
            if pay_mode == "m":
                usemember_acct = 1
                buyeracct = AccountBroker.objects.filter(members=buyer.id)[0]
                memberacct = Account.objects.filter(member_id=buyeracct.the_account)[0]
                pay_mode = memberacct.id

            seller = None
            selling_amount = request.POST['selling_amount']
            selling_date = request.POST['selling_date']
            business_setting = OtherSettings.objects.filter(business=business_context['branch'].business).first()
            business_shares = BusinessShares.objects.filter(business=business_context['branch'].business).first()
            fy = FinancialYear.objects.filter(start_date__lte=selling_date,
                                              end_date__gte=selling_date, business=business_context['branch'].business)
            share_limit = business_setting.share_limit
            if business_shares.market_cap < 1 or business_setting.share_price < 1:
                shares = 0
            else:
                shares = business_shares.market_cap / business_setting.share_price

            if business_shares.market_cap < 1 or business_setting.share_price < 1:
                share_balance = 0
            else:
                share_balance = business_shares.market_cap / business_setting.share_price - business_shares.sold

            # Calculate shares ==============
            if business_setting.share_price < 1:
                messages.error(request, 'Share Price', extra_tags='Please Set the share price and try again')
                return HttpResponseRedirect(reverse_lazy('shares'))
            else:
                bought = int(selling_amount) / int(business_setting.share_price)

            # Check selected member is eligible to buy the these shares ==============
            if int(bought) + int(buyer.shares) > int(share_limit):
                messages.error(request, 'Share Limits', extra_tags='This member cannot buy ' + str(
                    bought) + ' shares. It violets the maximum share limit of ' + str(share_limit))
                return HttpResponseRedirect(reverse_lazy('shares'))

            # Check sacco has these shares
            if int(share_balance) < int(bought):
                messages.error(request, 'Insufficient shares',
                               extra_tags='This transaction can not be completed. You only have ' + str(
                                   share_balance) + ' shares available for issue')
                return HttpResponseRedirect(reverse_lazy('shares'))
            narrative = 'Sale of ' + str(bought) + ' shares to ' + str(buyer.biodata.name) + ' at ' + str(
                business_setting.share_price) + ' each.'
            if fy.exists():
                fYear = fy.first()
                shares_record = SharesTransactions.objects.create(
                    buyer_id=buyer.id,
                    shares=bought,
                    date=selling_date,
                    narration=narrative,
                    branch=business_context['branch']
                )

                # query buyer shares transactions
                buyer_bought_shares = SharesTransactions.objects.filter(
                    buyer=buyer, branch=business_context['branch']
                ).aggregate(total=Sum('shares'))['total'] or 0
                sold_bought_shares = SharesTransactions.objects.filter(
                    seller=buyer, branch=business_context['branch']
                ).aggregate(total=Sum('shares'))['total'] or 0

                current_shares = buyer_bought_shares - sold_bought_shares

                # set the first time a member buys shares
                if buyer.share_holder_join_date is None:
                    Member.objects.filter(id=buyer.id).update(
                        share_holder_join_date=selling_date
                    )


                # buyer_shares = SharesTransactions.objects.filter(
                #     Q(buyer=buyer, branch=business_context['branch']) | Q(seller=buyer,
                #                                                                branch=business_context['branch']))
                # # print("results = %s" % buyer_shares.count())
                # total_bought = 0
                # total_sold = 0
                # for b in buyer_shares:
                #     if b.buyer == buyer:
                #         total_bought += b.shares
                #     if b.seller == buyer:
                #         total_sold += b.shares
                #
                # current_shares = total_bought - total_sold

                # update buyer shares balance
                buyer.shares = current_shares
                buyer.save()

                # query seller shares transactions
                seller_shares = SharesTransactions.objects.filter(
                    seller=seller, branch=business_context['branch']
                ).aggregate(total=Sum('shares'))['total'] or 0
                total_sold = seller_shares

                # for s in seller_shares:
                #     total_sold += s.shares

                # update business shares sold
                business_shares.sold = total_sold
                business_shares.save()

                # Record a finance transaction
                # shares account
                shares_account = Account.objects.filter(business=business_context['branch'].business, category__name='Shares').first()
                # payment account
                payment_account = Account.objects.filter(business=business_context['branch'].business, id=pay_mode).first()

                # transaction
                new_tx = Transactions.objects.create(
                    branch=business_context['branch'],
                    financial_year=fYear,
                    transaction_type="Sell Shares",
                    account_dr=payment_account,
                    account_cr=shares_account,
                    shares=shares_record,
                    narration=narrative,
                    tx_date=selling_date,
                    reporting_amount=selling_amount,
                    added_by=request.user.staff
                )

                # print('VALUE IS ', usemember_acct)
                if usemember_acct == 1:
                    updatebalance(pay_mode, buyeracct.the_account.id)

                messages.success(request, 'success', extra_tags='Sale of ' + str(bought) + ' Shares ' + str(
                    buyer.biodata) + ' Done Successfully')
                message = f"{request.user.staff.biodata.name} has made sale of {bought} to {buyer.biodata}"
                ActivityLog.objects.create(actor=request.user,
                                           action_type=SHARE,
                                           title='Share sale',
                                           branch=business_context['branch'].id,
                                           remarks=message)
                if buyer.biodata.contact:
                    message = f'Dear {buyer.biodata.name},\nThis is to inform you have successfully bought {bought} shares.'
                    checkAndSendMessage(self.request.user, message, buyer.biodata.contact, buyer.biodata.name)
                return HttpResponseRedirect(reverse_lazy('shares'))
            else:
                messages.error(request, 'Financial Year', extra_tags='Please Set current financial year')
                return HttpResponseRedirect(reverse_lazy('shares'))


class TransferShares(BusinessUserMixin, View):
    template_name = 'sacco/shares_center.html'

    def get(self, request, *args, **kwargs):
        return HttpResponseRedirect(reverse_lazy('shares'))

    def post(self, request, *args, **kwargs):
        business_context = get_business_loans_context_data(request)
        if business_context['branch'] is None:
            messages.error(
                request, "error",
                extra_tags="Action not allowed in the central view. Please logout and login into the respective branch"
            )
            return HttpResponseRedirect(request.META.get("HTTP_REFERER"))
        if request.method == 'POST':
            sender = Member.objects.filter(id=request.POST['transfer']).first()
            receiver = Member.objects.filter(id=request.POST['receiver']).first()
            transfer_date = request.POST['transfer_date']
            no_of_shares = request.POST['shares']

            business_setting = OtherSettings.objects.filter(business=business_context['branch'].business).first()
            business_shares = BusinessShares.objects.filter(business=business_context['branch'].business).first()
            share_limit = business_setting.share_limit

            # Check selected receiver is eligible to buy the these shares ==============
            if int(receiver.shares) + int(no_of_shares) > int(share_limit):
                messages.error(request, 'Share Limits', extra_tags=receiver.biodata.name + ' cannot receive ' + str(
                    no_of_shares) + ' shares. It violets the maximum share limit of ' + str(share_limit))
                return HttpResponseRedirect(reverse_lazy('shares'))

            # Check sender has these shares
            if int(sender.shares) < int(no_of_shares):
                messages.error(request, 'Insufficient shares',
                               extra_tags='This transaction can not be completed. ' + sender.biodata.name + ' only have ' + str(
                                   sender.shares) + ' shares')
                return HttpResponseRedirect(reverse_lazy('shares'))
            narrative = 'Transfer of ' + str(no_of_shares) + ' shares from ' + sender.biodata.name + ' to ' + str(
                receiver.biodata.name)
            SharesTransactions.objects.create(
                buyer_id=receiver.id,
                shares=no_of_shares,
                seller_id=sender.id,
                date=transfer_date,
                narration=narrative,
                branch=business_context['branch']
            )

            # query buyer shares transactions
            sender_shares = SharesTransactions.objects.filter(
                Q(buyer=sender, branch__business=business_context['branch'].business) | Q(seller=sender,
                                                                            branch__business=business_context['branch'].business))
            # print("results = %s" % sender_shares.count())
            total_sent = 0
            total_received = 0
            for s in sender_shares:
                if s.buyer == sender:
                    total_received += s.shares
                if s.seller == sender:
                    total_sent += s.shares

            new_shares = total_received - total_sent

            # update buyer shares balance
            sender.shares = new_shares
            sender.save()

            # query seller shares transactions
            receiver_shares = SharesTransactions.objects.filter(
                Q(buyer=receiver, branch__business=business_context['branch'].business) | Q(seller=receiver,
                                                                              branch__business=business_context['branch'].business))
            # print("results = %s" % receiver_shares.count())
            total_out = 0
            total_in = 0
            for r in receiver_shares:
                if r.buyer == receiver:
                    total_in += r.shares
                if r.seller == receiver:
                    total_out += r.shares

            all_shares = total_in - total_out

            # update buyer shares balance
            receiver.shares = all_shares
            receiver.save()

            messages.success(request, 'success',
                             extra_tags='Transfer of ' + str(no_of_shares) + ' Shares from ' + str(
                                 sender.biodata.name) + ' to ' + receiver.biodata.name + ' Successful')
            message = f"{request.user.staff.biodata.name} has made share transfer of" \
                      f" {no_of_shares} from {sender.biodata.name} to {receiver.biodata.name}"
            ActivityLog.objects.create(actor=request.user,
                                       action_type=SHARE,
                                       title='Share Transfer',
                                       branch=business_context['branch'].id,
                                       remarks=message)

            template_file_link = 'sacco/emails/transactional.html'
            currency = CurrencySetting.objects.filter(business=business_context['branch'].business).first()
            notify_type = NotiSettings.objects.filter(business=business_context['branch'].business).first()
            smsSettingsObj = SaccoSmsSettings.objects.get(when_to_send='on new member registration',
                                                          business=business_context['branch'].business)
            # bizz_id = business_context['branch'].business.id
            this_biz = business_context['branch'].business

            html_body = f'<p>Dear {sender.biodata.name},</p> <p>This is to inform you have successfully transferred shares to {receiver.biodata.name}. <br>We hope that you have a good experience with our services and <b>{this_biz.name}</b> will do it’s best to please you. We will be looked forward to serving you in adherence to our quality standards.</p><p>Thanking you and with profound regards.</p>'
            if smsSettingsObj.status:
                if notify_type.notification_type == 1:  # Send by SMS only
                    if sender.biodata.contact:
                        message = f'Dear {sender.biodata.name},\nThis is to inform you have successfully transferred shares to {receiver.biodata.name}. '
                        checkAndSendMessage(self.request.user, message, sender.biodata.contact, sender.biodata.name)
                    if receiver.biodata.contact:
                        message = f'Dear {receiver.biodata.name},\nThis is to inform you have successfully received shares from {sender.biodata.name}.'
                        checkAndSendMessage(self.request.user, message, receiver.biodata.contact, receiver.biodata.name)
                elif notify_type.notification_type == 2:  # Send by Email only
                    if sender.biodata.email:
                        sendTransEmail(template_file_link, html_body, this_biz.name, 'Account Opening Successful',
                                       sender.biodata.email)
                    if receiver.biodata.email:
                        sendTransEmail(template_file_link, html_body, this_biz.name, 'Account Opening Successful',
                                       receiver.biodata.email)


                elif notify_type.notification_type == 3:  # Send by both SMS and Email
                    if sender.biodata.contact:
                        message = f'Dear {sender.biodata.name},\nThis is to inform you have successfully transferred shares to {receiver.biodata.name}. '
                        checkAndSendMessage(self.request.user, message, sender.biodata.contact, sender.biodata.name)
                    if receiver.biodata.contact:
                        message = f'Dear {receiver.biodata.name},\nThis is to inform you have successfully received shares from {sender.biodata.name}.'
                        checkAndSendMessage(self.request.user, message, receiver.biodata.contact, receiver.biodata.name)

                    if sender.biodata.email:
                        sendTransEmail(template_file_link, html_body, this_biz.name, 'Account Opening Successful',
                                       sender.biodata.email)


            return HttpResponseRedirect(reverse_lazy('shares'))
        else:
            messages.error(request, 'Financial Year', extra_tags='Please Set current financial year')
            return HttpResponseRedirect(reverse_lazy('shares'))


# class ShareReversals()


class ShareHolders(BusinessUserMixin, View):
    template_name = 'sacco/shares_holders.html'

    def get_filters(self) -> {}:
        filters = {}
        user = self.request.user
        branch = biz_staff_branch(user)
        date_input = self.request.GET.get('date')
        from_date = self.request.GET.get('from_date')
        to_date = self.request.GET.get('to_date')
        # filters['branch'] = branch
        if date_input:
            filters['date__lte'] = date_input
        if from_date and to_date:
            filters['date__range'] = [from_date, to_date]

        return filters

    def get(self, request, *args, **kwargs):
        business_context = get_business_loans_context_data(request)
        business_filter = business_context['business_filter']

        if business_context['the_current_branch_name'] == 'all branches':
            title = "Shares Holders for all branches"
        else:
            the_current_branch_name = business_context['the_current_branch_name']
            title = f"Shares Holders for {the_current_branch_name}"

        start_date_str = self.request.GET.get('date_from', '')
        end_date_str = self.request.GET.get('date_to', '')

        # Optional: parse to date objects
        if start_date_str:
            start_date = datetime.datetime.strptime(start_date_str, "%Y-%m-%d").date()
        if end_date_str:
            end_date = datetime.datetime.strptime(end_date_str, "%Y-%m-%d").date()

        # Start building filters
        buyer_filter = Q(shares_buyer__branch__business_id__in=business_context['data_context'])
        seller_filter = Q(shares_seller__branch__business_id__in=business_context['data_context'])

        if start_date_str:
            buyer_filter &= Q(shares_buyer__date__gte=start_date)
            seller_filter &= Q(shares_seller__date__gte=start_date)

        if end_date_str:
            buyer_filter &= Q(shares_buyer__date__lte=end_date)
            seller_filter &= Q(shares_seller__date__lte=end_date)

        member_filter = Q(
            member_type='s',
            biodata__business_id__in=business_context['data_context']
        )
        if start_date_str:
            member_filter &= Q(share_holder_join_date__gte=start_date)
        if end_date_str:
            member_filter &= Q(share_holder_join_date__lte=end_date)

        business_setting = OtherSettings.objects.filter(business=businessdata(request)).first()

        # Optimized query
        shared_member = (Member.objects.filter(member_filter).select_related('biodata').prefetch_related('accounts').annotate(
            total_purchased=Coalesce(
                Sum('shares_buyer__shares', filter=buyer_filter),
                Value(0.0), output_field=FloatField()
            ),
            total_sold=Coalesce(
                Sum('shares_seller__shares', filter=seller_filter),
                Value(0.0), output_field=FloatField()
            )
        ).annotate(
            net_shares=F('total_purchased') - F('total_sold'),
            account_number=Subquery(
                AccountBroker.objects.filter(
                    members_id=OuterRef('id')
                ).values('the_account__acc_number')[:1]
            )
        ).annotate(
            member_name=F('biodata__name'),
            member_contact=F('biodata__contact'),
            shares_value=F('net_shares')*business_setting.share_price
        )
        .values(
            'member_name', 'member_contact', 'net_shares', 'account_number', 'shares_value', 'share_holder_join_date'
        ))

        # print(shared_member)

        # # title = 'Shares Holders'
        # current_date = datetime.date.today()
        # share_holders = Member.objects.filter(member_type='s',
        #                                       biodata__business_id__in=business_context['data_context']).prefetch_related(
        #     'biodata').prefetch_related('accounts')
        # shared_member = []
        # branch_trans = SharesTransactions.objects.filter(**self.get_filters())
        # branch_trans = branch_trans.filter(branch__business_id__in=business_context['data_context'])
        #
        # for member in share_holders:
        #     share_purchases = branch_trans.filter(buyer=member).values('buyer').aggregate(
        #         purchases=Coalesce(Sum('shares'), 0.0, output_field=FloatField()))['purchases']
        #     share_sale = branch_trans.filter(seller=member).values('seller').aggregate(
        #         sales=Coalesce(Sum('shares'), 0.0, output_field=FloatField()))['sales']
        #     shared_member.append(
        #         {
        #             'name': member.biodata.name,
        #             'contact': member.biodata.contact,
        #             'accounts': member.accounts.all(),
        #             'shares': share_purchases - share_sale,
        #             'shares_capital': share_purchases - share_sale,
        #         }
        #     )

        context = {'title': title, 'share_holders': shared_member, 'business_setting': business_setting, 'start_date_str': start_date_str, 'end_date_str':end_date_str
                   }
        return render(request, self.template_name, context)
