import datetime
import json
import random
from typing import List, Dict, Union

import pandas as pd
from dateutil.parser import parse
from django.db import transaction
from django.db.models import Sum, Q, Case, When, ExpressionWrapper, F, FloatField
from django_rq import job

from accounts.models import Biodata
from loans.models import ApplicationAccountOrLoanType, Loans
from loans.utils.loan_uploads_utils import determine_amount_paid_principal_and_interest, \
    get_receivable_acc_and_loan_interest_receivables_acc, get_or_create_excess_payment_acc
from sacco.models import Member, AccountBroker, MemberAccount, TransactionCharge, BusinessShares, \
    SaccoSmsSettings, OtherSettings, MemberUpload, CurrencySetting, NotiSettings
from sacco.utils import biz_staff, biz_data, biz_staff_branch, checkAndSendMessage, gen_account, sendTransEmail, \
    branchdata, branch_id
from transactions.models import AccountCategory, Account, Transactions, SharesTransactions
from transactions.reports import this_branch
from utils.general import random_with_N_digits, ug_dial


class DepositUtil:

    def __init__(self, amount):
        self.amount = amount

    def calculate_deposit_charges(self, depo_cgs, user, member_acc, this_account, date_created, rec_no, ref_no):
        amount = self.amount
        income_acc = Account.objects.filter(category__name="Deposit Income", business=biz_data(user)).first()
        depo_charges = 0
        transactions = []
        # print(depo_cgs.)
        for d in depo_cgs:
            if d.account_type.deposit_charge_vary:
                if d.start <= amount <= d.end:
                    charge_amount = d.charge if not d.is_charge_percentage else amount * (d.charge / 100)
                    depo_charges += charge_amount
                    transactions.append(
                        Transactions(
                            branch=biz_staff_branch(user),
                            transaction_type='Charge on deposit',
                            account_dr=member_acc,
                            account_cr=income_acc,
                            narration=f'Charge on deposit of {amount} on {this_account.acc_number}',
                            reporting_amount=charge_amount,
                            added_by=biz_staff(user),
                            tx_date=date_created,
                            receipt=rec_no,
                            reference=ref_no
                        )
                    )

            else:
                charge_amount = d.charge if not d.is_charge_percentage else amount * (d.charge / 100)
                depo_charges += charge_amount
                # print('am herer', amount)
                transactions.append(
                    Transactions(
                        branch=biz_staff_branch(user),
                        transaction_type='Charge on deposit',
                        account_dr=member_acc,
                        account_cr=income_acc,
                        narration=f'Charge on deposit of {amount} on {this_account.acc_number}',
                        reporting_amount=charge_amount,
                        added_by=biz_staff(user),
                        tx_date=date_created,
                        receipt=rec_no,
                        reference=ref_no
                    )
                )

            # if charge_amount > 0:

        Transactions.objects.bulk_create(transactions)

        return depo_charges

    def create_deposit_transaction(self, user, date_created, narration, rec_no, ref_no, this_account,
                                   member_acc, acc_involved):
        amount = self.amount
        msg = f'Deposit of {amount} to {this_account.acc_number}' if not narration else narration

        return Transactions.objects.create(
            branch=biz_staff_branch(user),
            transaction_type='Deposit',
            account_dr=acc_involved,
            account_cr=member_acc,
            narration=msg,
            reporting_amount=amount,
            added_by=biz_staff(user),
            tx_date=date_created,
            receipt=rec_no,
            reference=ref_no
        )

    def update_account_balance(self, this_account, member_acc):
        member_debit = Transactions.objects.filter(account_dr=member_acc).aggregate(total=Sum('reporting_amount'))[
                           'total'] or 0
        member_credit = Transactions.objects.filter(account_cr=member_acc).aggregate(total=Sum('reporting_amount'))[
                            'total'] or 0
        this_account.balance = member_credit - member_debit
        this_account.save()

    def send_notification(self, zemember, notify_type, this_biz, trans, dep_with, user, template_file_link,
                          member_acc, this_account):
        amount = self.amount
        name = zemember.name
        member_email = zemember.email
        contact = zemember.contact
        if not zemember.is_group:
            name = zemember.biodata.name
            member_email = zemember.biodata.email
            contact = zemember.biodata.contact

        # print('de balance', balance)
        currency = CurrencySetting.objects.filter(business=this_biz).first()
        smsSettingsObj = SaccoSmsSettings.objects.get(when_to_send='On deposit transaction', business=biz_data(user))
        subject = "Deposit request processed successfully"
        business = this_biz.name
        email = member_email
        message = f'Dear {name}, a deposit of {currency.currency} {amount} has been made on your account. Fee {currency.currency} {dep_with}. Account Balance is '
        html_body = f'<p>Dear {name},</p>  Your Deposit request with the following details has been processed successfully.<br/>Transaction Number: {trans.txno}<br/>Amount: {currency.currency}{amount} <br/>Description: {trans.narration} <br/>Date: {trans.tx_date} <p>Please call our contact centre on +{this_biz.contact} in case you require further details.</p>'

        if smsSettingsObj.status:
            if notify_type.notification_type == 1:  # Send by SMS only
                if zemember.biodata.contact:
                    checkAndSendMessage(user, message, contact, name,
                                        account=member_acc, this_account=this_account)
            elif notify_type.notification_type == 2:  # Send by Email only
                if zemember.biodata.email:
                    sendTransEmail(template_file_link, html_body, business, subject, email)

            elif notify_type.notification_type == 3:  # Send by both SMS and Email
                if zemember.biodata.contact:
                    # print('send message')
                    checkAndSendMessage(user, message, zemember.biodata.contact, zemember.biodata.name,
                                        account=member_acc, this_account=this_account)

                if zemember.biodata.email:
                    sendTransEmail(template_file_link, html_body, business, subject, email)



@job
@transaction.atomic
def upload_member(file, user, biz, account_type, financial_year, balo_date):
    upload = pd.read_excel(file, comment='#', dtype={"Account Number": str})
    upload.fillna('', inplace=True)
    upload.columns = upload.columns.str.title()

    open_reserve = Account.objects.get(name='Opening Reserves', business=biz_data(user))
    business_shares = BusinessShares.objects.filter(business=biz_data(user)).first()
    member_upload = MemberUpload.objects.create(business=biz_data(user), file=file)
    count = 0
    existing_accs = []
    message = ""

    for obj in upload.to_dict(orient='records'):
        full_names = str(obj.get('Full Name', '')).strip()
        contact = str(obj.get('Contact', '')).strip()
        other_contact = str(obj.get('Other Contact', '')).strip()
        email = str(obj.get('Email Address', '')).strip()
        shares = obj.get('Shares', 0)
        raw_savings = str(obj.get('Savings Balance', '0')).replace(',', '')
        join_date = obj.get('Join Date', None)
        gender = str(obj.get('Gender', '')).strip()
        acc_no = str(obj.get('Account Number', '')).strip()
        employee_no = str(obj.get('Employee Number', '')).strip()
        address = ''
        if 'Address' in obj:
            address = str(obj.get('Address', '')).strip()
        bd=None
        if 'Birth Date' in obj:
            bd = str(obj.get('Birth Date', '')).strip()
        
        acc_number = acc_no

        # Handle gender abbreviation
        if len(gender) > 1:
            gender = gender[0].upper()
        if join_date:
            try:
                # join_day = parse(join_date).date()
                join_day = pd.Timestamp(join_date)
                join_day = join_day.to_pydatetime().date()
            except Exception as e:
                join_day = None
                print(str(e))
        else:
            join_day = None

        if bd:
            try:
                bd = pd.Timestamp(bd)
                bd = bd.to_pydatetime().date()
            except Exception as e:
                bd = None
                print(str(e))
        else:
            bd = None
        


        # Handle .0 suffix in account number and convert savings to an integer
        if acc_no.endswith('.0'):
            acc_no = acc_no.replace('.0', '')

        try:
            my_raw_savings = raw_savings.replace(',', '')
            savings = int(float(my_raw_savings))
        except ValueError:
            savings = 0

        # Handle employee number if it ends with .0
        if employee_no.endswith('.0'):
            employee_no = employee_no.replace('.0', '')

        if biz.auto_acc_no:
            if acc_no is None or acc_no == '':
                acc_number = gen_account(user)
        if not MemberAccount.objects.filter(acc_number=acc_no, member_account__business=biz_data(user)).exists():

            # Create biodata and member objects
            bio = Biodata.objects.create(
                name=full_names.title(),
                contact=ug_dial(contact) if contact else None,
                other_contact=ug_dial(other_contact) if other_contact else None,
                gender=gender,
                created_by=biz_staff(user),
                email=email,
                business=biz_data(user),
                employee_no=employee_no, member_upload=member_upload.uuid,
                dob=bd,
                location=address
            )

            #original
            # if shares:
            if False:
                print('it ran')
                member = Member.objects.create(
                    biodata=bio,
                    created_by=biz_staff(user),
                    shares=int(shares),
                    branch=biz_staff_branch(user),
                    date_joined=join_day,
                    member_type='s',
                    member_upload=member_upload
                )

                # Shares transactions here
                SharesTransactions.objects.create(
                    buyer_id=member.id,
                    shares=member.shares,
                    date=datetime.datetime.today(),
                    narration='Opening shares',
                    branch=biz_staff_branch(user),
                )
            else:
                member = Member.objects.create(
                    biodata=bio,
                    created_by=biz_staff(user),
                    branch=biz_staff_branch(user),
                    date_joined=join_day,
                    member_type='o', member_upload=member_upload
                )

            account = MemberAccount.objects.create(
                acc_number=acc_number,
                account_type_id=account_type,
                balance=savings,
            )

            AccountBroker.objects.create(
                members=member,
                the_account=account,
                business=biz_data(user),
            )

            cat = AccountCategory.objects.filter(business=biz_data(user), name="Members").first()

            # Chart of accounts
            member_acc = Account.objects.create(
                category=cat,
                business=biz_data(user),
                member=account,
                added_by=biz_staff(user),
            )

            Transactions.objects.create(
                branch=biz_staff_branch(user),
                transaction_type='Opening reserve',
                account_dr=open_reserve,
                account_cr=member_acc,
                narration='Opening balance',
                # reporting_amount=savings,
                reporting_amount=savings,
                added_by=biz_staff(user),
                financial_year=financial_year,
                tx_date=balo_date,
            )

            # Query seller shares transactions
            seller_shares = SharesTransactions.objects.filter(seller__isnull=True, branch__business=biz_data(user))
            total_sold = seller_shares.aggregate(total_sold=Sum('shares'))['total_sold'] or 0

            # Update business shares sold
            business_shares.sold = total_sold
            business_shares.save()
        else:
            message = 'Account exist'
            existing_accs.append(acc_no)
    member_upload.failed = str(existing_accs)
    member_upload.reason = message
    member_upload.save()

    return member_upload


@job
@transaction.atomic
def upload_new_members(file, user, biz, account_type, the_account):
    upload = pd.read_excel(file, comment='#', dtype={"Account Number": str})
    upload.fillna('', inplace=True)
    upload.columns = upload.columns.str.title()
    member_upload = MemberUpload.objects.create(business=biz_data(user), file=file)
    existing_accs = []
    message = ''

    for index, row in upload.iterrows():
        full_names = row['Full Name'].strip()
        contact = str(row['Contact']).strip()
        other_contact = str(row['Other Contact']).strip()
        email = str(row.get('Email Address', None)).strip()
        shares = row.get('Shares', 0)
        raw_savings = str(row['Savings Balance'])
        join_date = row['Join Date']
        gender = str(row['Gender']).strip().upper()
        acc_no = str(row['Account Number']).strip()
        employee_no = str(row.get('Employee Number', None)).strip()
        address = ''
        if 'Address' in row:
            address = str(row.get('Address', '')).strip()
        the_bd=None
        if 'Birth Date' in row:
            the_bd = str(row.get('Birth Date', '')).strip()

        setting_status = OtherSettings.objects.filter(business=biz_data(user)).first()
        currency = CurrencySetting.objects.filter(business=biz_data(user)).first()
        # print(shares)
        
        # if shares == "":
        #     shares = 0

        if employee_no.endswith('.0'):
            employee_no = employee_no.replace('.0', '')

        if join_date:
            try:
                # join_day = parse(join_date).date()
                join_day = pd.Timestamp(join_date)
                join_day = join_day.to_pydatetime().date()
            except Exception as e:
                join_day = None
                print(str(e))
        else:
            join_day = None

        if the_bd:
            try:
                the_bd = pd.Timestamp(the_bd)
                the_bd = the_bd.to_pydatetime().date()
            except Exception as e:
                the_bd = None
                print(str(e))
        else:
            the_bd = None

        if acc_no.endswith('.0'):
            acc_no = acc_no.replace('.0', '')

        savings = int(raw_savings.replace(',', '')) if raw_savings else 0
        
        myc = int(float(contact)) if contact else None
        contact = ug_dial(str(myc)) if contact else None
        myo = int(float(other_contact)) if other_contact else None
        other_contact = ug_dial(str(myo)) if other_contact else None

        if not email:
            email = None

        if len(gender) > 1:
            gender = gender[0]

        if biz.auto_acc_no:
            if acc_no is None or acc_no == '':
                acc_no = gen_account(user)
        amount = 0

        if full_names:
            # print('de acc', acc_no)
            if not MemberAccount.objects.filter(acc_number=acc_no, member_account__business=biz_data(user)).exists():

                bio = Biodata.objects.create(name=full_names.title(), contact=contact, other_contact=other_contact,
                                             gender=gender, created_by=biz_staff(user), email=email,
                                             business=biz_data(user), location=address, dob=the_bd,
                                             employee_no=employee_no, member_upload=member_upload.uuid)
                if shares:
                    setting_status = OtherSettings.objects.filter(business=biz_data(user)).first()
                    member = Member.objects.create(biodata=bio, created_by=biz_staff(user), shares=int(shares),
                                                   branch=biz_staff_branch(user), date_joined=join_day, member_type='s',
                                                   member_upload=member_upload)
                else:
                    member = Member.objects.create(biodata=bio, created_by=biz_staff(user),
                                                   branch=biz_staff_branch(user), date_joined=join_day,
                                                   member_type='o', member_upload=member_upload)
                account = MemberAccount.objects.create(acc_number=acc_no, account_type_id=account_type, balance=savings)
                AccountBroker.objects.create(members=member, the_account=account, business=biz_data(user))
                cat = AccountCategory.objects.filter(business=biz_data(user), name="Members").first()
                equity_cat, created = AccountCategory.objects.get_or_create(business=biz_data(user),
                                                                            name="Shares", dr_cr='cr',
                                                                            cat_type='liability')
                member_acc = Account.objects.create(category=cat, business=biz_data(user), member=account,
                                                    added_by=biz_staff(user))

                acc_involved = Account.objects.filter(id=the_account, business=biz_data(user)).first()
                account_dr = acc_involved

                charges_acc = Account.objects.get(category__name="Registration Income", business=biz_data(user))
                gen_cgs = ApplicationAccountOrLoanType.objects.filter(account_type_id=account_type,
                                                                      general_charge__application='r',
                                                                      general_charge__status=True)

                general_charges = gen_cgs.annotate(
                    amount=Case(
                        When(general_charge__is_percentage=True,
                             then=ExpressionWrapper(savings * (F('general_charge__amount') / 100),
                                                    output_field=FloatField())),
                        default=F('general_charge__amount'),
                        output_field=FloatField()
                    )
                ).annotate(rev_status=F('general_charge__is_revenue'))
                if general_charges.exists():

                    for ch in gen_cgs:
                        if ch.general_charge.is_revenue == 'YES':
                            Transactions.objects.create(
                                branch=biz_staff_branch(user),
                                transaction_type='Registration',
                                account_dr=member_acc,
                                account_cr=charges_acc,
                                narration="{}".format(ch.general_charge.charge),
                                reporting_amount=ch.general_charge.amount,
                                added_by=biz_staff(user),
                            )
                        else:
                            charge = ch.general_charge.charge
                            if Account.objects.filter(name=charge).exists():
                                charge_account = charge
                            else:
                                reserves = AccountCategory.objects.filter(name='Reserves').first()
                                charge_account, created = Account.objects.create(name=ch.general_charge.charge,
                                                                                 category_id=reserves.id,
                                                                                 business=biz_data(user))
                            Transactions.objects.create(
                                branch=biz_staff_branch(user),
                                transaction_type='Registration',
                                account_dr=member_acc,
                                account_cr=charge_account,
                                narration="{}".format(ch.general_charge.charge),
                                reporting_amount=ch.general_charge.amount,
                                added_by=biz_staff(user),
                            )
                dep_cgs = TransactionCharge.objects.filter(account_type_id=account_type, charge_type='d', status=True)
                # print('my dep charged', dep_cgs)
                # Calculate deposit charges total
                depo_charges = 0
                if dep_cgs.exists():
                    for d in dep_cgs:
                        if d.account_type.deposit_charge_vary:
                            if d.start <= savings <= d.end:
                                amount = savings * (d.charge / 100) if d.is_charge_percentage else d.charge
                        else:
                            amount = 0 if d.account_type.deposit_charge_vary else (
                                savings * (d.charge / 100) if d.is_charge_percentage else d.charge)

                        depo_charges += amount


                        Transactions.objects.create(
                            branch=biz_staff_branch(user),
                            transaction_type='Charge',
                            account_dr=member_acc,
                            account_cr=charges_acc,
                            narration=f"Charge on deposit of {raw_savings}",
                            reporting_amount=amount,
                            added_by=biz_staff(user),
                        )

                if not setting_status.set_ordinary:
                    narrative = f'Sale of {shares} shares to {member.biodata.name} at {setting_status.share_price} each.'
                    business_shares = BusinessShares.objects.filter(business=biz_data(user)).first()
                    # print(type(shares))
                    # shares_record = SharesTransactions.objects.create(
                    #     buyer_id=member.id,
                    #     shares=shares,
                    #     date=datetime.datetime.today(),
                    #     narration=narrative,
                    #     branch=this_branch(user)
                    # )
                    # query buyer shares transactions
                    buyer_shares = SharesTransactions.objects.filter(Q(buyer=member) | Q(seller=member))
                    # print("results = %s" % buyer_shares.count())
                    total_bought = 0
                    total_sold = 0
                    for b in buyer_shares:
                        if b.buyer == member:
                            total_bought += b.shares
                        if b.seller == member:
                            total_sold += b.shares

                    current_shares = total_bought - total_sold

                    # update buyer shares balance
                    member.shares = current_shares
                    member.save()

                    # query seller shares transactions
                    seller_shares = SharesTransactions.objects.filter(seller=None)
                    total_sold = 0
                    for s in seller_shares:
                        total_sold += s.shares

                    # update business shares sold
                    business_shares.sold = total_sold
                    business_shares.save()

                    # shares account
                    shares_account = Account.objects.filter(business=biz_data(user),
                                                            category__name='Shares').first()

                    share_cost = 0
                    if shares != '':
                        share_cost = int(setting_status.share_price) * int(shares)

                    if share_cost > 0:

                        Transactions.objects.create(
                            branch=biz_staff_branch(user),
                            transaction_type='Sale of shares',
                            account_dr_id=member_acc.id,
                            account_cr=shares_account,
                            narration=narrative,
                            reporting_amount=share_cost,
                            added_by=biz_staff(user),
                            shares=shares_record,
                        )

                Transactions.objects.create(
                    branch=biz_staff_branch(user),
                    transaction_type="Deposit",
                    account_dr=account_dr,
                    account_cr=member_acc,
                    narration="Initial deposit on registration",
                    reporting_amount=savings,
                    added_by=biz_staff(user),
                )

                member_debits = list(Transactions.objects.filter(account_dr=member_acc).aggregate(
                    total=Sum('reporting_amount')).values())[0]
                if member_debits is None:
                    member_debits = 0
                member_credits = list(Transactions.objects.filter(account_cr=member_acc).aggregate(
                    total=Sum('reporting_amount')).values())[0]
                account.balance = member_credits - member_debits
                account.save()
                smsSettingsObj = SaccoSmsSettings.objects.get(when_to_send='on new member registration',
                                                              business=biz_data(user))
                message = f'Dear Client,\nThis is to inform you that your account in {biz_data(user).name} has been opened with A/C No: {account.acc_number} & deposit of {currency.currency} {savings} '
                html_body = f'<p>Dear {bio.name},</p> <p>This is to inform you that your account in {biz_data(user).name} has been opened with A/C No: {account.acc_number} and an initial deposit of {currency.currency} {savings}. <br>We hope that you have a good experience with our services and <b>{biz_data(user).name}</b> will do it’s best to please you. We will be looked forward to serving you in adherence to our quality standards.</p><p>Thanking you and with profound regards.</p>'
                if smsSettingsObj.status and bio.contact:
                    checkAndSendMessage(user, message, bio.contact, bio.name,
                                        account=member_acc, this_account=account)
                    # print(theResponse)
            else:
                existing_accs.append(acc_no)
                message = 'Account exist'

        else:
            existing_accs.append(acc_no)
            message = 'Full Name not provided'

    member_upload.failed = str(existing_accs)
    member_upload.reason = message
    member_upload.save()


    return member_upload

AccountDict = Dict[str, Union[int, str]] # Assuming your dictionary has keys of type 'str' and values of type 'int' or 'str'

@job
@transaction.atomic
def bulk_member_deposit(user, acounts_list: List[AccountDict], the_account, this_biz, *args, **kwargs):

    date_created = datetime.datetime.now().date()
    print("the_account,", the_account)
    #check if the account is a models obj
    if isinstance(the_account, Account):
        the_account = the_account.id

    try:
        acc_involved = Account.objects.get(id=the_account, business=biz_data(user))
    except Account.DoesNotExist:
        acc_involved = Account.objects.filter(id=the_account, business=biz_data(user)).first()

    template_file_link = 'sacco/emails/transactional.html'
    notify_type = NotiSettings.objects.filter(business=this_biz).first()

    for account in acounts_list:
        acc = account.get('account_number')
        print('account_number', acc)
        if acc is not None:
            if AccountBroker.objects.filter(business=this_biz, the_account__acc_number=acc).exists():
                date_trans = account.get('date')
                if date_trans:
                    date_created = parse(str(date_trans)).date()
                from_initial = account.get('amount')
                from_initial = from_initial.replace(',', '')
                initial = int(float(from_initial))
                rec_no = datetime.datetime.now().strftime("%y%m%d%H%M%S")
                ref_no = rec_no
                narration = ''
                deposit = DepositUtil(initial) # Initializing deposit object, doing everything deposit
                # this_account = MemberAccount.objects.get(acc_number=acc)
                member_account = AccountBroker.objects.filter(the_account__acc_number=acc, business=biz_data(user)).select_related('the_account').first()
                member = AccountBroker.objects.get(the_account=member_account.the_account).members
                member_acc = Account.objects.filter(member=member_account.the_account).first()
                depo_cgs = TransactionCharge.objects.filter(account_type_id=member_account.the_account.account_type,
                                                            charge_type='d', status=True)

                # getting deposit charges
                depo_charges = deposit.calculate_deposit_charges(depo_cgs, user, member_acc,
                                                                 member_account.the_account, date_created, rec_no, ref_no)

                # Recording transactions
                trans = deposit.create_deposit_transaction(user, date_created, narration, rec_no, ref_no,
                                                           member_account.the_account, member_acc, acc_involved)
                # Updating balance
                deposit.update_account_balance(member_account.the_account, member_acc)

               # sending messages or emails
                deposit.send_notification(member, notify_type, this_biz, trans, depo_charges, user, template_file_link,
                                       member_acc, member_account.the_account)

def bulk_member_share_purchase(user, acounts_list: List[AccountDict], the_account, this_biz, *args, **kwargs):
    business_setting = OtherSettings.objects.filter(business=this_biz).first()
    share_price = business_setting.share_price

    for account in acounts_list:
        acc = account.get('account_number')
        share_no = account.get('shares', None)
        if share_no is None:
            share_no = 0
        else:
            share_no = float(share_no)
        amount = share_no * share_price
        reference = ''
        branch = branch_id(user)
        narration = f'{share_no} have been bought by {acc}'
        if acc is not None:
            get_member = AccountBroker.objects.filter(the_account__acc_number=acc,
                                                         business=this_biz).first()
            if get_member:
                # member_ledger = Account.objects.filter(member_id=get_member_id.the_account_id).first()
                if isinstance(the_account, Account):
                    the_account = the_account.id

                try:
                    acc_involved = Account.objects.get(id=the_account)
                except Account.DoesNotExist:
                    acc_involved = Account.objects.filter(id=the_account, business=biz_data(user)).first()

                share_cap = Account.objects.filter(name='Share Capital', business=this_biz)[0]
                trans_date = datetime.datetime.now()

                save_shares = SharesTransactions.objects.create(
                    buyer_id=get_member.members_id,
                    shares=share_no,
                    date=trans_date.date(),
                    narration=f'Sale of {share_no} by the SACCO',
                    branch_id=branch
                )

                # ==========================GET TRANSACTION CHARGES===========================
                Transactions.objects.create(reporting_amount=amount,
                                            narration=narration,
                                            receipt=reference,
                                            account_cr_id=share_cap.id,
                                            account_dr=acc_involved,
                                            tx_date=trans_date.date(),
                                            transaction_type='Sell of shares',
                                            shares_id=save_shares.id,
                                            branch_id=branch)


@transaction.atomic
def repayments(user, acounts_list: List[AccountDict], the_account, this_biz, *args, **kwargs):
    branch_data = branch_id(user)
    get_running_loan = []
    for account in acounts_list:
        acc = account.get('account_number')

        # get members loan
        member_loans = Loans.objects.filter(account__acc_number=acc)
        for loan in member_loans:
            if loan.loan_status == 3: # Disbursed
                get_running_loan.append(loan)
        if len(get_running_loan) > 0:
            current_loan = get_running_loan[0]
        else:
            current_loan = None

        if current_loan is not None:
            if acc is not None:
                if AccountBroker.objects.filter(business=this_biz, the_account__acc_number=acc).exists():
                    date_trans = datetime.datetime.now().date()

                    from_initial = account.get('repayments')
                    from_initial = from_initial.replace(',', '')
                    initial = int(float(from_initial))
                    rec_no = datetime.datetime.now().strftime("%y%m%d%H%M%S")
                    ref_no = rec_no
                    narration = ''
                    # deposit = DepositUtil(initial) # Initializing deposit object, doing everything deposit
                    this_account = AccountBroker.objects.filter(the_account__acc_number=acc, business=biz_data(user)).first().the_account

                    debitted_acc = Account.objects.filter(member_id=this_account.id).first()
                    excel_loan = {
                        'loan number': current_loan.loan_number,
                        'date': date_trans,
                        'fines': account.get('fines', None),
                        'reference': ref_no,
                        'amount paid': initial
                    }

                    payment_info = determine_amount_paid_principal_and_interest(this_biz, current_loan, excel_loan)

                    excel_date = excel_loan['date']

                    with transaction.atomic():
                        try:
                            if payment_info['principal_trans_amt'] > 0:
                                trans_principal = Transactions.objects.create(
                                    account_cr=get_receivable_acc_and_loan_interest_receivables_acc(this_biz)[
                                        'receivable_acc'],
                                    account_dr_id=the_account,
                                    reporting_amount=round(payment_info['principal_trans_amt'], 2),
                                    branch_id=branch_data,
                                    loan=current_loan,
                                    transaction_type='Loan repayment',
                                    tx_date=excel_date,
                                    receipt=rec_no,
                                    reference=excel_loan['reference'],
                                    narration=narration
                                )
                                # transactions_added.append(trans_principal.id)
                                # print('trans principal')

                            if payment_info['interest_trans_amt'] > 0:
                                trans_interest = Transactions.objects.create(
                                    account_cr=get_receivable_acc_and_loan_interest_receivables_acc(this_biz)[
                                        'loan_interest_receivables_acc'],
                                    account_dr_id=the_account,
                                    reporting_amount=round(payment_info['interest_trans_amt'], 2),
                                    branch_id=branch_data,
                                    loan=current_loan,
                                    transaction_type='Loan interest',
                                    tx_date=excel_date,
                                    receipt=rec_no,
                                    reference=excel_loan['reference'],
                                    narration=narration
                                )
                                # create also the expected interest transaction
                                # business_data = businessdata(request)
                                ln_interest_rec_acc = Account.objects.filter(name='loan interest receivables',
                                                                             business=this_biz)
                                if ln_interest_rec_acc:
                                    ln_interest_rec_acc = ln_interest_rec_acc.first()
                                else:
                                    print('Loan interest acc')
                                    ln_interest_rec_acc = Account.objects.create(
                                        name='loan interest receivables', business=this_biz,
                                        display_name='loan interest receivables',
                                        category=AccountCategory.objects.get(name='Account Receivables',
                                                                             business=this_biz),
                                    )

                                ln_interest_revenue_acc = Account.objects.filter(name='Loan Interest',
                                                                                 business=this_biz).first()

                                Exp_interest = Transactions.objects.create(
                                    reporting_amount=round(payment_info['interest_trans_amt'], 2),
                                    narration=f'Expected interest of {str(round(payment_info["interest_trans_amt"], 2))}',
                                    account_dr_id=ln_interest_rec_acc,
                                    account_cr=ln_interest_revenue_acc,
                                    loan=current_loan, tx_date=excel_date,
                                    transaction_type='Expected interest',
                                    added_by=user,
                                    branch_id=branch_data)
                                # print('Exp_interest', Exp_interest)

                                # transactions_added.append(trans_interest.id)
                                # transactions_added.append(Exp_interest.id)
                                # print('trans_interest', trans_interest)
                            if payment_info['excess_amt'] > 0:
                                suspense_acc = get_or_create_excess_payment_acc(this_biz)

                                trans_excess_payment = Transactions.objects.create(
                                    account_cr=suspense_acc,
                                    account_dr_id=the_account,
                                    reporting_amount=round(payment_info['excess_amt'], 2),
                                    branch_id=branch_data,
                                    loan=current_loan,
                                    transaction_type='Excess payment',
                                    narration=f'Excess payment received on the loan: {current_loan.loan_number}',
                                    tx_date=excel_date
                                )
                                # transactions_added.append(trans_excess_payment.id)
                                # loans_closed.append(the_loan.id)

                            # if fines are there
                            # get fines acc
                            if excel_loan['fines'].strip() != '':
                                if float(excel_loan['fines'].strip().replace(',', '')) > 0:
                                    fines_acc = Account.objects.filter(name='Loan Fines', business=this_biz)
                                    if fines_acc.exists():
                                        fines_acc = fines_acc.first()
                                    else:
                                        fines_acc = Account.objects.create(
                                            name='Loan Fines', business=this_biz,
                                            display_name='Loan Fines',
                                            category=AccountCategory.objects.get(name='Loan incomes',
                                                                                 business=this_biz),
                                        )

                                    trans_fines = Transactions.objects.create(
                                        account_cr=fines_acc,
                                        account_dr_id=the_account,
                                        reporting_amount=float(excel_loan['fines'].strip().replace(',', '')),
                                        branch_id=branch_data,
                                        loan=current_loan,
                                        transaction_type='Loan fines',
                                        tx_date=excel_date,
                                        receipt=rec_no,
                                        reference=excel_loan['reference'],
                                        narration=narration
                                    )
                                    # transactions_added.append(trans_fines.id)
                                    # print('RECORED FINE', trans_fines)
                            # loans_added.append(excel_loan['loan number'])
                        except Exception as e:
                            print('EXCEPTION', e)
                            # loans_failed.append(
                            #     {
                            #         'reason': 'Please check if all fields are provided correctly or Contact system administrators',
                            #         'loan number': excel_loan['loan number']
                            #     }
                            # )





