import datetime

import pandas as pd
from django.utils import timezone
from django.core.paginator import Paginator, PageNotAnInteger, EmptyPage
from django.db.models import Sum, F, Case, When, FloatField
from django.db.models.functions import Coalesce
from django.shortcuts import render
from django.views import View

from accounts.permission_mixin import BusinessUserMixin
from commons.logger import logger
from loans.models import Loans
from loans.utils.loan_details import get_business_loans_context_data
from sacco.models import CurrencySetting, LoanSettings, AccountBroker, ActivityLog
from sacco.utils import businessdata, biz_staff_branch
from sacco.views import PermView
from transactions.models import Account, Transactions, AuditTrail, StaffLedgerAssociated


class RunningLoansReport(PermView):
    perm_name = 'view_running_loans_report'
    template_name = 'sacco/reports/running_loans.html'

    def get(self, request, *args, **kwargs):
        title = 'Running Loans Report'
        currency = CurrencySetting.objects.filter(business=businessdata(request)).first()

        loanacct = Account.objects.filter(name='Loan Receivables', business=businessdata(request)).first()

        loans = Loans.objects.filter(branch__business=businessdata(request), loan_status=3).annotate(
            paid=Sum(
                Case(
                    When(loan_trans__account_cr_id=loanacct.id, then=F('loan_trans__reporting_amount')),
                    default=0, output_field=FloatField()
                )
            ),
            given=Sum(
                Case(
                    When(loan_trans__account_dr_id=loanacct.id, then=F('loan_trans__reporting_amount')),
                    default=0, output_field=FloatField()
                )
            )
        ).annotate(principal_paid=Sum('loan_trans__reporting_amount')).annotate(
            applicant_name=F('applicant__biodata__name'),
            applicant_phone=F('account__acc_number')).annotate(principal_balance=F('given') - F('paid')) \
            .values('applicant_name', 'applicant_phone', 'amount_approved', 'principal_balance', 'balance')
        # print(loans)
        loan_settings = LoanSettings.objects.filter(business=businessdata(request)).first()
        decimals = loan_settings.decimals

        return render(request, self.template_name, locals())


class SavingsReport(PermView):
    perm_name = 'view_savings_report'
    template_name = 'sacco/reports/savings_report.html'

    def get_filters(self) -> {}:
        filters = {}
        user = self.request.user
        branch = biz_staff_branch(user)
        date_input = self.request.GET.get('date')
        from_date = self.request.GET.get('from_date')
        to_date = self.request.GET.get('to_date')
        filters['branch'] = branch
        if date_input:
            filters['tx_date__lte'] = date_input
        if from_date and to_date:
            filters['tx_date__range'] = [from_date, to_date]

        return filters

    def get(self, request, *args, **kwargs):
        business_context = get_business_loans_context_data(request)
        business_filter = business_context['business_filter']

        if business_context['the_current_branch_name'] == 'all branches':
            title = "Members Savings Report for all branches"
        else:
            the_current_branch_name = business_context['the_current_branch_name']
            title = f"Members Savings Report for {the_current_branch_name}"

        # title = 'Members Savings Report'
        current_date = datetime.date.today()
        user = request.user
        branch = biz_staff_branch(user)
        accounts = []
        if 'data' in request.session:
            del request.session['data']
        biz_accounts = AccountBroker.objects.filter(members__branch__business_id__in=business_context['data_context'],
                                                    members__is_active=True
                                                    ).prefetch_related('the_account').prefetch_related('members')
        branch_trans = Transactions.objects.filter(branch__business_id__in=business_context['data_context']).filter(**self.get_filters())
        for member in biz_accounts:
            debits = branch_trans.filter(account_dr__member=member.the_account
                                         ).values('account_dr').aggregate(
                debits=Coalesce(Sum('reporting_amount'), 0.0, output_field=FloatField()))['debits']

            credits_ = branch_trans.filter(account_cr__member=member.the_account
                                          ).values('account_cr').aggregate(
                credits=Coalesce(Sum('reporting_amount'), 0.0, output_field=FloatField()))['credits']
            try:
                accounts.append(
                    {
                        'name': member.members.biodata.name,
                        'acc_number': member.the_account.acc_number,
                        'acc_type': member.the_account.account_type.name,
                        'balance': (credits_ if not None else 0) - (debits if not None else 0)
                    }
                )
            except AttributeError as e:
                logger.error(str(e))
                accounts = []
        # accounts_ = Account.objects.filter(business=businessdata(request), category__name__in=['Mobile Money', 'Cash',
        #                                                                                        'Bank'])
        context = {'accounts': accounts, 'title': title, 'today': current_date}
        return render(request, self.template_name, context)


class ExpensesReport(PermView):
    perm_name = 'view_expense_report'
    template_name = 'sacco/reports/expenses_report.html'

    def get(self, request, *args, **kwargs):
        business_context = get_business_loans_context_data(request)
        business_filter = business_context['business_filter']

        if business_context['the_current_branch_name'] == 'all branches':
            title = "Expenses Report for all branches"
        else:
            the_current_branch_name = business_context['the_current_branch_name']
            title = f"Expenses Report for {the_current_branch_name}"
        # title = 'Expenses Report'
        user = request.user
        branch = biz_staff_branch(user)
        currency = CurrencySetting.objects.filter(business=businessdata(request)).first()
        transactions = Transactions.objects.filter(branch__business_id__in=business_context['data_context'], transaction_type='Expense Invoice')
        # page = request.GET.get('page', 1)
        # paginator = Paginator(expense_transactions, 25)
        # try:
        #     transactions = paginator.page(page)
        # except PageNotAnInteger:
        #     transactions = paginator.page(1)
        # except EmptyPage:
        #     transactions = paginator.page(paginator.num_pages)

        return render(request, self.template_name, locals())


class AuditTrialsView(PermView):
    perm_name = 'view_audit_trails'
    template_name = 'sacco/audit_trials.html'

    def get(self, request, *args, **kwargs):
        title = 'System Users Audit'
        branch = biz_staff_branch(request.user)
        audit_trials = ActivityLog.objects.filter(branch=branch.id)
        page = request.GET.get('page', 1)
        paginator = Paginator(audit_trials, 25)
        try:
            audits = paginator.page(page)
        except PageNotAnInteger:
            audits = paginator.page(1)
        except EmptyPage:
            audits = paginator.page(paginator.num_pages)
        return render(request, self.template_name, locals())
    

class DailyLedgerSummaryView(View):
    template_name = 'sacco/reports/daily_ledger_summary.html'

    def get(self, request, *args, **kwargs):
        business_context = get_business_loans_context_data(request)
        business_filter = business_context['business_filter']

        if business_context['the_current_branch_name'] == 'all branches':
            title = "Your daily Ledger Summary for all branches"
        else:
            the_current_branch_name = business_context['the_current_branch_name']
            title = f"Your daily Ledger Summary for {the_current_branch_name}"

        # title = 'Your daily Ledger Summary'
        branch = biz_staff_branch(request.user)

        # Get filter date from request, default to today if not provided
        filter_date = request.GET.get('filter_date')
        if filter_date:
            current_date = datetime.datetime.strptime(filter_date, '%Y-%m-%d').date()
        else:
            current_date = datetime.date.today()

        # print('current_date', current_date)
        the_staff = request.user.staff
        
        #get the ledger account for cashier
        # check if person has a ledger connected to them
        #determine if the staff is attached to any accounts
        check_associated_acc = StaffLedgerAssociated.objects.filter(staff=request.user.staff)
        acc_ids = list(check_associated_acc.values_list('account_asscociated__id', flat=True))

        view_all_daily_transactions = False
        if request.user.staff.position.permissions.filter(item_name='view_all_daily_transactions').exists():
            view_all_daily_transactions = True

        if not view_all_daily_transactions:
            all_tran_today = Transactions.objects.filter(date_added__gte=current_date, added_by=the_staff , branch__business_id__in=business_context['data_context'])
        else:
            all_tran_today = Transactions.objects.filter(date_added__gte=current_date, branch__business_id__in=business_context['data_context'])
        if check_associated_acc.exists():
            cred_accounts = Account.objects.filter(id__in=acc_ids)
        else:
            cred_accounts = Account.objects.filter(business_id__in=business_context['data_context'],
                                           category__name__in=['Mobile Money', 'Cash', 'Bank'])
            
        debit_accounts = Account.objects.filter(business_id__in=business_context['data_context'],
                                           category__name__in=['Mobile Money', 'Cash', 'Bank']).exclude(id__in=acc_ids)
        
        # check if person has final transaction today
        check_final = Transactions.objects.filter(added_by=the_staff, is_final_transaction=True, date_added__gte=current_date)
        if check_final.exists():
            check_final = True
        else:
            check_final = False

        current_server_time = timezone.now()
        # print('curre', current_server_time)
        #if its past 3.30pm show that it about time to be logged out automatically
        target_time = current_server_time.replace(hour=12, minute=30, second=0, microsecond=0)
        # target_time = current_server_time.replace(hour=11, minute=0, second=0, microsecond=0)

        # Check if the current server time is past 5:00 PM
        is_past_3_30_pm = current_server_time >= target_time

        can_login_overtime = False
        if request.user.staff.position.permissions.filter(item_name='can_login_overtime').exists():
            can_login_overtime = True

        # Pagination
        page_number = request.GET.get('page', 1)
        paginator = Paginator(all_tran_today, 20)  # Show 10 transactions per page
        page_obj = paginator.get_page(page_number)
        all_tran_today = page_obj
        # if check_associated_acc:
        #     #get the totals

        return render(request, self.template_name, locals())