from datetime import datetime

from django.contrib import messages
from django.db.models import Sum
from django.http import HttpResponseRedirect, HttpResponse
from django.shortcuts import render
from django.urls import reverse_lazy
from django.views import View

from accounts.models import Staff, Payroll, PayrollPayment
from accounts.permission_mixin import BusinessUserMixin
from sacco.constants import PAYROLL
from sacco.models import ActivityLog
from sacco.utils import branch_id, businessdata
from sacco.views import PermView
from transactions.models import Account, AccountCategory, Transactions

# Staff members on payroll
from transactions.reports import this_financial_year


class StaffPayroll(PermView):
    perm_name = 'view_staff'
    template_name = 'sacco/staff_payroll.html'

    def get(self, request, *args, **kwargs):
        user = request.user
        title = 'Staff Payroll'

        staff_list = Staff.objects.filter(branch=request.user.staff.branch, is_active=True, on_payroll=False)
        object_list = Payroll.objects.filter(branch=request.user.staff.branch, is_deleted=False)

        return render(request, self.template_name, locals())

    def post(self, request, *args, **kwargs):
        form_type = request.POST['form_type']
        staff_id = request.POST['staff']
        # this_staff = Staff.objects.get(id=staff_id)
        # print(staff_id)
        this_staff = Staff.objects.get(id=staff_id)
        gross = str(request.POST['gross']).replace(',', '')
        pay_paye = request.POST['pay_paye']
        pay_nssf = request.POST['pay_nssf']

        # Payroll calculations
        gross_pay = float(gross)
        paye = 0

        if gross_pay > 10000000:
            taxable = gross_pay - 10000000
            one = 0 / 100 * 235000
            two = 10 / 100 * 100000
            three = 20 / 100 * 75000
            four = 30 / 100 * 9590000
            five = 40 / 100 * taxable

            tax = one + two + three + four + five
            paye = round(paye + tax)

        if 10000000 >= gross_pay >= 410001:
            taxable = gross_pay - 410000
            one = 0 / 100 * 235000
            two = 10 / 100 * 100000
            three = 20 / 100 * 75000
            four = 30 / 100 * taxable
            tax = one + two + three + four
            paye = round(paye + tax)

        if 410000 >= gross_pay >= 335001:
            taxable = gross_pay - 335000
            one = 0 / 100 * 235000
            two = 10 / 100 * 100000
            three = 20 / 100 * taxable

            tax = one + two + three
            paye = round(paye + tax)

        if 335000 >= gross_pay >= 235001:
            taxable = gross_pay - 235000
            one = 0 / 100 * 235000
            two = 10 / 100 * taxable
            tax = one + two
            paye = round(paye + tax)

        if 235000 >= gross_pay > 1:
            taxable = 0
            one = 0 / 100 * taxable
            tax = one
            paye = round(paye + tax)

        nssf = 5 / 100 * gross_pay

        # IF THE STAFF DOES NOT PAY P.A.Y.E
        if pay_paye == 0 or pay_paye == "0":
            paye = 0

        # IF THE STAFF DOES NOT PAY N.S.S.F
        if pay_nssf == 0 or pay_nssf == "0":
            nssf = 0

        net_pay = round(gross_pay - paye - nssf)

        if form_type == 'enroll':
            existing_record = Payroll.objects.filter(staff_id=staff_id).first()
            if existing_record is not None:
                existing_record.gross_pay = gross_pay
                existing_record.paye = paye
                existing_record.net_pay = net_pay
                existing_record.nssf = nssf
                existing_record.is_deleted = False
                existing_record.save()

                # Update staff
                this_staff.on_payroll = True
                this_staff.save()
            else:
                Payroll.objects.create(staff_id=staff_id, gross_pay=gross_pay, paye=paye, net_pay=net_pay, nssf=nssf, branch_id=branch_id(request.user))
                this_staff.on_payroll = True
                this_staff.save()

            # Audit trial
            message = f"{request.user.staff.biodata.name} enrolled {this_staff.biodata.name} on payroll successfully with Gross pay: {gross_pay}, " \
                      f"NSSF: {nssf}, PAYE: {paye} and Net pay: {net_pay} "
            ActivityLog.objects.create(actor=request.user,
                                       action_type=PAYROLL,
                                       title='Payroll Enrollment',
                                       branch=branch_id(request.user),
                                       remarks=message)

            messages.success(request, 'success', extra_tags=f'{this_staff.biodata.name} enrolled on payroll successfully')

        if form_type == 'Update_payroll':
            existing_record = Payroll.objects.filter(id=staff_id).first()
            if existing_record:
                existing_record.gross_pay = gross_pay
                existing_record.paye = paye
                existing_record.net_pay = net_pay
                existing_record.nssf = nssf
                existing_record.update_on = datetime.today()
                existing_record.save()
            else:
                messages.success(request, 'Not Found', extra_tags='Payroll record not found to update')
                return HttpResponseRedirect(reverse_lazy('staff_payroll'))

            # Audit trial
            message = f"{request.user.staff.biodata.name} has updated the pay records of {existing_record.staff.biodata.name} to Gross pay: {gross_pay}, NSSF: {nssf}, PAYE: {paye} and Net pay: {net_pay} "
            ActivityLog.objects.create(actor=request.user,
                                       action_type=PAYROLL,
                                       title='Payroll Update',
                                       branch=branch_id(request.user),
                                       remarks=message)

            messages.success(request, 'success', extra_tags=f'{this_staff.biodata.name} enrolled on payroll successfully')

        return HttpResponseRedirect(reverse_lazy('staff_payroll'))


# Pay Salary
class StaffSalary(PermView):
    perm_name = 'view_staff'
    template_name = 'sacco/staff_salaries.html'

    def get(self, request, *args, **kwargs):
        user = request.user
        title = 'Staff Salaries'

        salary_payments = PayrollPayment.objects.filter(branch=request.user.staff.branch, type=1)
        payroll_list = Payroll.objects.filter(branch=request.user.staff.branch, is_deleted=False)
        accounts_ = Account.objects.filter(business=businessdata(request), category__name__in=['Mobile Money', 'Cash', 'Bank'])
        return render(request, self.template_name, locals())

    def post(self, request, *args, **kwargs):
        this_payroll = Payroll.objects.get(id=request.POST['staff'])
        gross = request.POST['staff_gross_pay']
        paye = request.POST['staff_paye']
        nssf = request.POST['staff_nssf']
        net_pay = request.POST['staff_net_pay']
        year_month = request.POST['month']
        from_account = request.POST['from_account']
        pay_date = request.POST['pay_date']
        month = year_month[-2:]
        year = year_month[:4]
        paid_month = year_month + '-28'  # End of the selected month

        # Check payment for this staff in the provided month and year such that you don't pay NSSF and PAYE again
        # Also keep track of the GROSS PAY as it must be the through the that month
        this_month_payments = PayrollPayment.objects.filter(month__month=month, month__year=year, payroll_id=this_payroll.id, type=1)
        if this_month_payments.count() < 1:  # It will mean this is the first payment in the selected month
            # Record the deductions transactions by debiting them (NSSF & PAYE) accordingly
            # credit paye and nssf payable accounts    Then credit PAYE and NSSF expenses
            # ************************* ACCOUNTS PAYABLE **************************
            # PAYE PAYABLE == Get it if is there already ==========================
            paye_payable = Account.objects.filter(name='PAYE Payable', business_id=businessdata(request)).first()
            if paye_payable is None:  # If is not there we create it
                # let us get it's parent category
                account_payable = AccountCategory.objects.filter(name='Accounts Payable', business_id=businessdata(request)).first()
                if account_payable is None:  # if the parent account is not there, we create it as well
                    # let us create the parent account category
                    account_payable = AccountCategory.objects.create(name='Accounts Payable', dr_cr='cr', cat_type='liability',
                                                                     added_by=request.user.staff, business_id=businessdata(request))

                # Now create the child account
                paye_payable = Account.objects.create(name='PAYE Payable', added_by=request.user.staff, business_id=businessdata(request),
                                                      category=account_payable)

            # /PAYE PAYABLE =======================================================

            # NSSF PAYABLE == Get it if is there already ==========================
            nssf_payable = Account.objects.filter(name='NSSF Payable', business_id=businessdata(request)).first()
            if nssf_payable is None:  # If is not there we create it
                # let us get it's parent category
                account_payable = AccountCategory.objects.filter(name='Accounts Payable', business_id=businessdata(request)).first()
                if account_payable is None:  # if the parent account is not there, we create it as well
                    # let us create the parent account category
                    account_payable = AccountCategory.objects.create(name='Accounts Payable', dr_cr='cr', cat_type='liability',
                                                                     added_by=request.user.staff, business_id=businessdata(request))

                # Now create the child account
                nssf_payable = Account.objects.create(name='NSSF Payable', added_by=request.user.staff, business_id=businessdata(request),
                                                      category=account_payable)

            # /NSSF PAYABLE =======================================================

            # ************************* EXPENSE ACCOUNTS **************************
            # PAYE EXPENSE == Get it if is there already ==========================
            paye_expense = Account.objects.filter(name='PAYE Expenses', business_id=businessdata(request)).first()
            if paye_expense is None:  # If is not there we create it
                # let us get it's parent category
                account_expense = AccountCategory.objects.filter(name='Expense Accounts', business_id=businessdata(request)).first()
                if account_expense is None:  # if the parent account is not there, we create it as well
                    # let us create the parent account category
                    account_expense = AccountCategory.objects.create(name='Expense Accounts', dr_cr='dr', cat_type='expenses',
                                                                     added_by=request.user.staff, business_id=businessdata(request))

                # Now create the child account
                paye_expense = Account.objects.create(name='PAYE Expenses', added_by=request.user.staff, business_id=businessdata(request),
                                                      category=account_expense)

            # /PAYE EXPENSE =======================================================

            # NSSF EXPENSE == Get it if is there already ==========================
            nssf_expense = Account.objects.filter(name='NSSF Expenses', business_id=businessdata(request)).first()
            if nssf_expense is None:  # If is not there we create it
                # let us get it's parent category
                account_expense = AccountCategory.objects.filter(name='Expense Accounts', business_id=businessdata(request)).first()
                if account_expense is None:  # if the parent account is not there, we create it as well
                    # let us create the parent account category
                    account_expense = AccountCategory.objects.create(name='Expense Accounts', dr_cr='dr', cat_type='expenses',
                                                                     added_by=request.user.staff, business_id=businessdata(request))

                # Now create the child account
                nssf_expense = Account.objects.create(name='NSSF Expenses', added_by=request.user.staff, business_id=businessdata(request),
                                                      category=account_expense)

            # /PAYE EXPENSE =======================================================

            # Now a transactions - PAYE - >>>>>>>>>>>>>>>>
            Transactions.objects.create(branch_id=branch_id(request.user),
                                        transaction_type=paye_payable.name,
                                        account_dr=paye_expense,
                                        account_cr=paye_payable,
                                        financial_year=this_financial_year(request),
                                        reporting_amount=paye,
                                        tx_date=pay_date,
                                        narration='PAYE realization',
                                        added_by=request.user.staff)
            # Now a transactions - NSSF - >>>>>>>>>>>>>>>>
            Transactions.objects.create(branch_id=branch_id(request.user),
                                        transaction_type=nssf_payable.name,
                                        account_dr=nssf_expense,
                                        account_cr=nssf_payable,
                                        financial_year=this_financial_year(request),
                                        reporting_amount=nssf,
                                        tx_date=pay_date,
                                        narration='NSSF realization',
                                        added_by=request.user.staff)

            # transaction from bank to staff (net pay) and it is an expense (Credit Bank/Cash and debt salary expense)
            # ************************* NET PAY PAYMENT ***************************************************
            # Let us get the salary expense account
            # SALARY EXPENSE == Get it if is there already ==========================
            salary_expense = Account.objects.filter(name='Salary Expenses', business_id=businessdata(request)).first()
            if salary_expense is None:  # If is not there we create it
                # let us get it's parent category
                account_expense = AccountCategory.objects.filter(name='Expense Accounts', business_id=businessdata(request)).first()
                if account_expense is None:  # if the parent account is not there, we create it as well
                    # let us create the parent account category
                    account_expense = AccountCategory.objects.create(name='Expense Accounts', dr_cr='dr', cat_type='expenses',
                                                                     added_by=request.user.staff, business_id=businessdata(request))

                # Now create the child account
                salary_expense = Account.objects.create(name='Salary Expenses', added_by=request.user.staff, business_id=businessdata(request),
                                                        category=account_expense)

            # /SALARY EXPENSE =======================================================

            # Now a transactions - NET PAID - >>>>>>>>>>>>>>>>
            Transactions.objects.create(branch_id=branch_id(request.user),
                                        transaction_type="Salary Payment",
                                        account_dr=salary_expense,
                                        account_cr_id=from_account,
                                        financial_year=this_financial_year(request),
                                        reporting_amount=net_pay,
                                        tx_date=pay_date,
                                        narration=f'Transaction of {net_pay} as salary payment for {month}, {year}',
                                        added_by=request.user.staff)

            # Record a payroll payment
            PayrollPayment.objects.create(gross_pay=gross, paye=paye, net_pay=net_pay, nssf=nssf, month=paid_month, branch_id=this_payroll.branch_id,
                                          payroll_id=this_payroll.id, type=1)
        else:
            # Just make the payment of the balance of that month
            PayrollPayment.objects.create(net_pay=net_pay, month=paid_month, branch_id=this_payroll.branch_id, payroll_id=this_payroll.id, type=1)

            # Credit the Account => Where the money is coming from e.g. Bank, Cash, Mobile Money etc. with net paid
            # ************************* NET PAY PAYMENT ***************************************************
            # Let us get the salary expense account
            # SALARY EXPENSE == Get it if is there already ==========================
            salary_expense = Account.objects.filter(name='Salary Expenses', business_id=businessdata(request)).first()
            if salary_expense is None:  # If is not there we create it
                # let us get it's parent category
                account_expense = AccountCategory.objects.filter(name='Expense Accounts', business_id=businessdata(request)).first()
                if account_expense is None:  # if the parent account is not there, we create it as well
                    # let us create the parent account category
                    account_expense = AccountCategory.objects.create(name='Expense Accounts', dr_cr='dr', cat_type='expenses',
                                                                     added_by=request.user.staff, business_id=businessdata(request))

                # Now create the child account
                salary_expense = Account.objects.create(name='Salary Expenses', added_by=request.user.staff, business_id=businessdata(request),
                                                        category=account_expense)

            # /SALARY EXPENSE =======================================================

            # Now a transactions - NET PAID - >>>>>>>>>>>>>>>>
            Transactions.objects.create(branch_id=branch_id(request.user),
                                        transaction_type="Salary Payment",
                                        account_dr=salary_expense,
                                        account_cr_id=from_account,
                                        financial_year=this_financial_year(request),
                                        reporting_amount=net_pay,
                                        tx_date=pay_date,
                                        narration=f'Transaction of {net_pay} as salary payment for {month}, {year}',
                                        added_by=request.user.staff)

        # Audit trial
        message = f"{request.user.staff.biodata.name} recorded salary payment of {net_pay} to {this_payroll.staff.biodata.name}"
        ActivityLog.objects.create(actor=request.user,
                                   action_type=PAYROLL,
                                   title='Salary Payment',
                                   branch=branch_id(request.user),
                                   remarks=message)

        messages.success(request, 'success', extra_tags=f'Transaction of {net_pay} recorded successfully')

        return HttpResponseRedirect(reverse_lazy('staff_salaries'))


class StaffAdvance(PermView):
    perm_name = 'view_staff'
    template_name = 'sacco/staff_advances.html'

    def get(self, request, *args, **kwargs):
        user = request.user
        title = 'Staff Advances'

        advance_payments = PayrollPayment.objects.filter(branch=request.user.staff.branch, type=3)
        payroll_list = Payroll.objects.filter(branch=request.user.staff.branch, is_deleted=False)
        accounts_ = Account.objects.filter(business=businessdata(request), category__name__in=['Mobile Money', 'Cash', 'Bank'])
        return render(request, self.template_name, locals())

    def post(self, request, *args, **kwargs):
        this_payroll = Payroll.objects.get(id=request.POST['staff'])
        net_pay = request.POST['staff_net_pay']
        year_month = request.POST['month']
        from_account = request.POST['from_account']
        pay_date = request.POST['pay_date']
        month = year_month[-2:]
        year = year_month[:4]
        paid_month = year_month + '-28'  # End of the selected month

        # Just make the payment of the balance of that month
        PayrollPayment.objects.create(net_pay=net_pay, month=paid_month, branch_id=this_payroll.branch_id, payroll_id=this_payroll.id, type=3)

        # Credit the Account => Where the money is coming from e.g. Bank, Cash, Mobile Money etc. with net paid
        # ************************* NET PAY PAYMENT ***************************************************
        # Let us get the salary expense account
        # SALARY EXPENSE == Get it if is there already ==========================
        salary_expense = Account.objects.filter(name='Salary Expenses', business_id=businessdata(request)).first()
        if salary_expense is None:  # If is not there we create it
            # let us get it's parent category
            account_expense = AccountCategory.objects.filter(name='Expense Accounts', business_id=businessdata(request)).first()
            if account_expense is None:  # if the parent account is not there, we create it as well
                # let us create the parent account category
                account_expense = AccountCategory.objects.create(name='Expense Accounts', dr_cr='dr', cat_type='expenses',
                                                                 added_by=request.user.staff, business_id=businessdata(request))
            # Now create the child account
            salary_expense = Account.objects.create(name='Salary Expenses', added_by=request.user.staff, business_id=businessdata(request),
                                                    category=account_expense)
        # /SALARY EXPENSE =======================================================

        # Now a transactions - NET PAID - >>>>>>>>>>>>>>>>
        Transactions.objects.create(branch_id=branch_id(request.user),
                                    transaction_type="Salary Payment",
                                    account_dr=salary_expense,
                                    account_cr_id=from_account,
                                    financial_year=this_financial_year(request),
                                    reporting_amount=net_pay,
                                    tx_date=pay_date,
                                    narration=f'Transaction of {net_pay} as salary advance for {month}, {year}',
                                    added_by=request.user.staff)

        # Audit trial
        message = f"{request.user.staff.biodata.name} recorded salary advance payment of {net_pay} to {this_payroll.staff.biodata.name}"
        ActivityLog.objects.create(actor=request.user,
                                   action_type=PAYROLL,
                                   title='Salary advance payment',
                                   branch=branch_id(request.user),
                                   remarks=message)

        messages.success(request, 'success', extra_tags=f'Transaction of {net_pay} as advance recorded successfully')

        return HttpResponseRedirect(reverse_lazy('staff_advances'))


class StaffAllowance(PermView):
    perm_name = 'view_staff'
    template_name = 'sacco/staff_allowances.html'

    def get(self, request, *args, **kwargs):
        user = request.user
        title = 'Staff Allowances'

        allowance_payments = PayrollPayment.objects.filter(branch=request.user.staff.branch, type=2)
        payroll_list = Payroll.objects.filter(branch=request.user.staff.branch, is_deleted=False)
        accounts_ = Account.objects.filter(business=businessdata(request), category__name__in=['Mobile Money', 'Cash', 'Bank'])
        return render(request, self.template_name, locals())

    def post(self, request, *args, **kwargs):
        this_payroll = Payroll.objects.get(id=request.POST['staff'])
        net_pay = request.POST['staff_net_pay']
        year_month = request.POST['month']
        from_account = request.POST['from_account']
        pay_date = request.POST['pay_date']
        month = year_month[-2:]
        year = year_month[:4]
        paid_month = year_month + '-28'  # End of the selected month

        # Just make the payment of the balance of that month
        PayrollPayment.objects.create(net_pay=net_pay, month=paid_month, branch_id=this_payroll.branch_id, payroll_id=this_payroll.id, type=2)

        # Credit the Account => Where the money is coming from e.g. Bank, Cash, Mobile Money etc. with net paid
        # ************************* NET PAY PAYMENT ***************************************************
        # Let us get the salary expense account
        # Allowance EXPENSE == Get it if is there already ==========================
        allowance_expense = Account.objects.filter(name='Allowance Expenses', business_id=businessdata(request)).first()
        if allowance_expense is None:  # If is not there we create it
            # let us get it's parent category
            account_expense = AccountCategory.objects.filter(name='Expense Accounts', business_id=businessdata(request)).first()
            if account_expense is None:  # if the parent account is not there, we create it as well
                # let us create the parent account category
                account_expense = AccountCategory.objects.create(name='Expense Accounts', dr_cr='dr', cat_type='expenses',
                                                                 added_by=request.user.staff, business_id=businessdata(request))
            # Now create the child account
            allowance_expense = Account.objects.create(name='Allowance Expenses', added_by=request.user.staff, business_id=businessdata(request),
                                                       category=account_expense)
        # /Allowance EXPENSE =======================================================

        # Now a transactions - NET PAID - >>>>>>>>>>>>>>>>
        Transactions.objects.create(branch_id=branch_id(request.user),
                                    transaction_type="Allowance Payment",
                                    account_dr=allowance_expense,
                                    account_cr_id=from_account,
                                    financial_year=this_financial_year(request),
                                    reporting_amount=net_pay,
                                    tx_date=pay_date,
                                    narration=f'Transaction of {net_pay} as Allowance for {month}, {year}',
                                    added_by=request.user.staff)

        # Audit trial
        message = f"{request.user.staff.biodata.name} recorded allowance payment of {net_pay} to {this_payroll.staff.biodata.name}"
        ActivityLog.objects.create(actor=request.user,
                                   action_type=PAYROLL,
                                   title='Allowance payment',
                                   branch=branch_id(request.user),
                                   remarks=message)

        messages.success(request, 'success', extra_tags=f'Transaction of {net_pay} as Allowance recorded successfully')

        return HttpResponseRedirect(reverse_lazy('staff_allowances'))


class PayrollDelete(BusinessUserMixin, View):
    perm_name = 'view_staff'
    template_name = 'sacco/staff_payroll.html'

    def get(self, request, pk, *args, **kwargs):
        payroll_record = Payroll.objects.get(id=pk)

        # remove payroll status from a staff member
        payroll_record.staff.on_payroll = False
        payroll_record.staff.save()

        # delete payroll record
        payroll_record.is_deleted = True
        payroll_record.save()

        # Audit trial
        message = f"{request.user.staff.biodata.name} removed {payroll_record.staff.biodata.name} from the payroll list"
        ActivityLog.objects.create(actor=request.user,
                                   action_type=PAYROLL,
                                   title='Payroll deleting',
                                   branch=branch_id(request.user),
                                   remarks=message)

        messages.success(request, 'success', extra_tags=f'{payroll_record.staff.biodata.name} removed from payroll successfully')

        return HttpResponseRedirect(reverse_lazy('staff_payroll'))


class MonthPayments(View):
    def get(self, request, date, pk, *args, **TYkwargs):
        month = date[-2:]
        year = date[:4]
        # first payment in the provided month and year. # To keep the gross pay uniform for this month and other figures (NSSF, PAYE, NET PAY)
        first_payment = PayrollPayment.objects.filter(month__month=month, month__year=year, payroll_id=pk, type=1).first()
        if first_payment is None:  # if there is no payment for the staff in that month of the year
            # Let us use the original payroll set for the month
            # Get the payroll settings. They are used only on the first payment in each month
            first_payment = Payroll.objects.get(id=pk)
            print('NO PAYMENT')

        # ======= Limits for the provided month of the year ==========
        this_month_gross = first_payment.gross_pay
        this_month_net = first_payment.gross_pay - (first_payment.nssf + first_payment.paye)
        this_month_nssf = first_payment.nssf
        this_month_paye = first_payment.paye

        paid_query = PayrollPayment.objects.filter(month__month=month, month__year=year, payroll_id=pk, type__in =[1,3]).aggregate(
            Sum('net_pay'), Sum('paye'), Sum('nssf'))
        net_paid = paid_query['net_pay__sum'] or 0
        paye_paid = paid_query['paye__sum'] or 0
        nssf_paid = paid_query['nssf__sum'] or 0

        # Now we get the balance between the paid and the set values
        paye = float(this_month_paye) - float(paye_paid)
        nssf = float(this_month_nssf) - float(nssf_paid)
        net_pay = float(this_month_net) - float(net_paid)

        return HttpResponse('%s|%s|%s|%s' % (this_month_gross, paye, nssf, net_pay))
