import traceback
from datetime import date

from django.contrib import messages
from django.http import JsonResponse, HttpResponseRedirect
from django.urls import reverse_lazy
from django.views.generic import FormView

from accounts.models import Salutation, Business
from accounts.permission_mixin import BusinessUserMixin
from commons.logger import logger
from loans.utils.loan_details import get_business_loans_context_data
from sacco.constants import CREATE
from sacco.models import FinancialYear, ActivityLog, AccountTypes, OtherSettings
from sacco.new_member_forms import RegisterMemberForm
from sacco.utils import branch_id, businessdata
from transactions.models import Account


class AddMemberView(BusinessUserMixin, FormView):
    template_name = 'sacco/add_new_member.html'
    form_class = RegisterMemberForm

    def get_form_kwargs(self):
        business_context = get_business_loans_context_data(self.request)
        kwargs = super().get_form_kwargs()
        kwargs.update({
            'user': self.request.user,
            'branch': business_context['branch']  # Pass the branch as a parameter
        })
        return kwargs

    def get_context_data(self, **kwargs):
        business_context = get_business_loans_context_data(self.request)
        if business_context['branch'] is None:
            messages.error(
                self.request, "error",
                extra_tags="Action not allowed in the central view. Please logout and login into the respective branch"
            )
            return HttpResponseRedirect(self.request.META.get("HTTP_REFERER"))

        context = super().get_context_data(**kwargs)
        membership = OtherSettings.objects.get(business=business_context['branch'].business).set_ordinary
        if membership is True:
            member = 'True'
        else:
            member = 'False'
        context['title'] = 'Add member'
        context['form'] = self.form_class
        context['salutations'] = Salutation.objects.all()
        context['acc_types'] = AccountTypes.objects.filter(status=True, business=business_context['branch'].business)
        context['biz'] = business_context['branch'].business
        context['accounts_'] = Account.objects.filter(business=business_context['branch'].business,
                                           category__name__in=['Mobile Money', 'Cash', 'Bank'])
        context['onRegistration'] = member
        return context

    def form_valid(self, form):
        business_context = get_business_loans_context_data(self.request)
        user = self.request.user
        try:
            # logger.error('Form is valid')
            current_transaction = date.today()
            form.save()

            log_title = 'New Member'
            message = f"{user.staff.biodata.name} added a new member"
            ActivityLog.objects.create(actor=user, title=log_title,
                                       action_type=CREATE,
                                       remarks=message,
                                       branch=business_context['branch'].id)
            # return JsonResponse({'msg': 'success', 'extra': 'Successfully Added member'})
            messages.success(self.request, 'success', extra_tags='Successfully added a member')
            # return HttpResponseRedirect(reverse_lazy('add_member_new'))
            return HttpResponseRedirect(reverse_lazy('add_member_new'))
        except Exception as e:
            # logger.error(str(e))
            logger.error(traceback.print_exc())
            # return JsonResponse({'msg': 'error', 'extra': 'Error occurred while submitting form'})
            messages.error(self.request, 'error', extra_tags='Error occurred while submitting form')
            return HttpResponseRedirect(reverse_lazy('add_member_new'))

    def form_invalid(self, form):
        form_errors = form.errors
        # print('am here', form_errors)
        # logger.error(str(form_errors))
        # return JsonResponse({'msg': 'error', 'extra': str(form_errors)})
        messages.error(self.request, 'error', extra_tags=form_errors)
        return HttpResponseRedirect(reverse_lazy('add_member_new'))