import datetime
import traceback

from django import forms
from django.core.exceptions import ValidationError
from django.db import transaction
from django.db.models import Case, When, ExpressionWrapper, F, FloatField, Q, Sum
from django.utils.translation import gettext as _
from django_countries.fields import CountryField

from accounts.models import Biodata
from commons.logger import logger
from loans.models import ApplicationAccountOrLoanType
from loans.utils.error_handlers import MemberCreationError
from sacco.models import MemberAccount, AccountBroker, OtherSettings, Member, AccountTypes, BusinessShares, \
    TransactionCharge, CurrencySetting, NotiSettings, SaccoSmsSettings
from sacco.utils import biz_data, gen_account, check_account, biz_staff, biz_staff_branch, check_charges, \
    checkAndSendMessage, sendTransEmail
from transactions.models import AccountCategory, Account, Transactions, SharesTransactions
from transactions.reports import this_branch
from utils.general import proper_dial


class RegisterMemberForm(forms.Form):
    title = forms.CharField(required=False)
    name = forms.CharField(required=False)
    gender = forms.CharField(required=False)
    marital_status = forms.CharField(required=False)
    upload_date = forms.DateField(required=False)
    dob = forms.DateField(required=False,
                          widget=forms.SelectDateWidget(
                              attrs={'class': 'form-control',
                                     'style': 'width:33.33333333333333%; ' 'display:inline' '-block;'},
                              years=range(1930, datetime.date.today().year + 1)))

    nin = forms.CharField(required=False)
    code = forms.CharField(required=False)
    code1 = forms.CharField(required=False)
    code2 = forms.CharField(required=False)
    contact = forms.CharField(required=False)
    other_contact = forms.CharField(required=False)
    email = forms.EmailField(required=False)
    nok = forms.CharField(required=False)
    nok_contacts = forms.CharField(required=False)
    share_select = forms.CharField(required=False)
    initial_shares = forms.FloatField(required=False)
    account_type = forms.CharField(required=False)
    member_type = forms.CharField(required=False)
    the_account = forms.CharField(required=False)
    mm = forms.CharField(required=False)
    country = forms.CharField(required=False)
    initial = forms.CharField(required=False)
    location = forms.CharField(widget=forms.TextInput, required=False)
    account_number = forms.CharField(widget=forms.TextInput, required=False)

    def __init__(self, *args, **kwargs):
        self.user = kwargs.pop('user', None)
        self.initial_depo = kwargs.pop('initial', None)
        self.branch = kwargs.pop('branch', None)
        super(RegisterMemberForm, self).__init__(*args, **kwargs)

    def the_business(self):
        biz = self.branch.business
        return biz

    def clean_code(self):
        cleaned_data = self.cleaned_data
        code = cleaned_data.get('code')
        return code

    def clean_code1(self):
        cleaned_data = self.cleaned_data
        code = cleaned_data.get('code1')
        return code

    def clean_code2(self):
        cleaned_data = self.cleaned_data
        code = cleaned_data.get('code2')
        return code

    def clean_contact(self):
        cleaned_data = self.cleaned_data
        contact = cleaned_data.get('contact')
        code = cleaned_data.get('code')
        if contact:
            primary_c = proper_dial(contact, code)
        else:
            primary_c = contact
        return primary_c

    def clean_other_contact(self):
        cleaned_data = self.cleaned_data
        other_contact = cleaned_data.get('other_contact')
        code = cleaned_data.get('code1')
        if other_contact:
            other_c = proper_dial(other_contact, code)
        else:
            other_c = other_contact
        return other_c

    def clean_mm(self):
        cleaned_data = self.cleaned_data
        mm = cleaned_data.get('mm')
        code = cleaned_data.get('code2')
        if mm:
            mm_c = proper_dial(mm, code)
        else:
            mm_c = mm
        return mm_c

    def clean_account_number(self):
        cleaned_data = self.cleaned_data
        account_number = cleaned_data.get('account_number')
        # print('aa', type(account_number))
        # try:
        if self.the_business().auto_acc_no is True:

            if account_number == '' or account_number is None:
                account_number = gen_account(self.user)
        else:
            if account_number != '' or account_number is not None:
                if check_account(account_number, self.user) is True:
                    raise ValidationError(_('Account already exists'), code='AccountNumber')
        return account_number

    def clean_initial(self):
        cleaned_data = self.cleaned_data
        account_type = cleaned_data.get('account_type')
        member_type = cleaned_data.get('member_type')
        initial = cleaned_data.get('initial')
        share_select = cleaned_data.get('share_select')
        initial_shares = cleaned_data.get('initial_shares')
        initial_before = float(initial.replace(',', ''))
        initial = float(initial_before)
        the_account_type = AccountTypes.objects.filter(id=account_type).first()
        min_balance = the_account_type.min_balance
        charge = int(check_charges(account_type, member_type, initial, self.user, initial_shares, share_select, self.branch))
        if initial and account_type:
            # print('fine')
            if initial >= charge + min_balance:
                print('charge', charge)
                print('mi ba', min_balance)
                return initial
            else:
                raise ValidationError(
                    _(f'Insufficient amount to cater for charges of {charge} and minimum balance of {min_balance}'),
                    code='Amount')
        return initial

    def clean_initial_shares(self):
        cleaned_data = self.cleaned_data
        shares = cleaned_data.get('initial_shares', 0.0)
        return shares

    def gen_member(self, bio, member_type, setting_status, user, share_select, upload_date, initial_shares, branch):
        """Creating new member"""
        try:
            if member_type == '2':  # If this is a new member
                if setting_status.set_ordinary:  # If this is a new member & all member are ordinary on reg
                    member = Member.objects.create(biodata=bio, created_by=biz_staff(user), member_type='o',
                                                   branch=branch, shares=initial_shares,
                                                   date_joined=upload_date if upload_date else datetime.date.today())
                else:  # If this is new but all member must be shareholders on reg
                    if share_select == '1':  # this means is a shareholder
                        member = Member.objects.create(biodata=bio, created_by=biz_staff(user), member_type='s',
                                                       shares=initial_shares, branch=branch,
                                                       date_joined=upload_date if upload_date else datetime.date.today())
                    else:  # this means is an ordinary
                        member = Member.objects.create(biodata=bio, created_by=biz_staff(user), member_type='o',
                                                       branch=branch, date_joined=upload_date if upload_date else datetime.date.today())
            else:  # If this is an existing member
                if share_select == '1':  # If this is an existing member and a shareholder
                    member = Member.objects.create(biodata=bio, created_by=biz_staff(user), member_type='s',
                                                   shares=initial_shares, branch=branch, date_joined=upload_date if upload_date else datetime.date.today())

                else:  # If this is an existing member but ordinary
                    member = Member.objects.create(biodata=bio, created_by=biz_staff(user), member_type='o',
                                                   branch=branch, date_joined=upload_date if upload_date else datetime.date.today())
            return member
        except Exception as e:
            logger.error(str(e))
            raise ValidationError(_("Error creating member: " + str(e)), code='member_creation')

    def save(self, commit=True):
        cleaned_data = self.cleaned_data
        user = self.user
        title = cleaned_data.get('title')
        upload_date = cleaned_data.get('upload_date')
        name = cleaned_data.get('name')
        gender = cleaned_data.get('gender')
        marital_status = cleaned_data.get('marital_status')
        dob = cleaned_data.get('dob')
        member_type = cleaned_data.get('member_type')
        share_select = cleaned_data.get('share_select')
        initial_shares = cleaned_data.get('initial_shares')
        initial = cleaned_data.get('initial')
        nin = cleaned_data.get('nin')
        contact = cleaned_data.get('contact')
        other_contact = cleaned_data.get('other_contact')
        mm = cleaned_data.get('mm')
        email = cleaned_data.get('email')
        nok = cleaned_data.get('nok')
        nok_contacts = cleaned_data.get('nok_contacts')
        account_type = cleaned_data.get('account_type')
        the_account = cleaned_data.get('the_account')
        location = cleaned_data.get('location')
        country = cleaned_data.get('country')
        account_number = cleaned_data.get('account_number')

        rec_no = datetime.datetime.now().strftime("%y%m%d%H%M%S")
        ref_no = datetime.datetime.now().strftime("%y%m%d%H%M%S")
        amount = 0

        try:
            with transaction.atomic():
                transaction_type = 'Opening balance'

                setting_status = OtherSettings.objects.filter(business=self.branch.business).first()
                logger.error(f'Settings status {str(setting_status)}')

                bio = Biodata.objects.create(
                    name=name,
                    gender=gender,
                    marital_status=marital_status,
                    dob=dob,
                    nin=nin,
                    contact=contact,
                    other_contact=other_contact,
                    mobile_money=mm,
                    title_id=title,
                    country=country,
                    email=email,
                    nok=nok,
                    location=location,
                    nok_contacts=nok_contacts,
                    created_by=biz_staff(user),
                    business=self.branch.business,
                    date_created=upload_date if upload_date else datetime.date.today()
                )
                logger.error(f'Biodata has been created')
                member = self.gen_member(bio, member_type, setting_status, user, share_select, upload_date, initial_shares, self.branch)
                logger.error(f'Member has been created')
                account = MemberAccount.objects.create(account_type_id=account_type, acc_number=account_number)
                logger.error(f'Account created')
                broker = AccountBroker.objects.create(members=member, the_account=account, business=self.branch.business)
                logger.error(f'Broker created')
                cat, created = AccountCategory.objects.get_or_create(business=self.branch.business, name="Members",
                                                                     cat_type='liability', dr_cr='cr')
                logger.error(f'Cat {str(cat)}')
                equity_cat, created = AccountCategory.objects.get_or_create(business=self.branch.business,
                                                                            name="Shares", dr_cr='cr',
                                                                            cat_type='liability')
                logger.error(f'Equity cat {str(equity_cat)}')
                # chart of accounts
                member_acc = Account.objects.create(category=cat, business=self.branch.business, member=account,
                                                    added_by=biz_staff(user))
                logger.error(f'Member acc {str(member_acc)} created')

                if the_account:
                    acc_involved = Account.objects.filter(id=the_account, business=self.branch.business).first()

                opening_acct = Account.objects.get(name='Opening Reserves', business=self.branch.business)

                account_dr = opening_acct

                # equity, created = Account.objects.get_or_create(name='Share Capital', business=biz_data(user),
                #                                                 category=equity_cat)
                logger.error(f'ACCOUNT CREATION DONE')

                if member_type == '2':  # If this is a new member
                    transaction_type = 'Deposit'
                    account_dr = acc_involved

                    charges_acc = Account.objects.get(category__name="Registration Income", business=self.branch.business)
                    gen_cgs = ApplicationAccountOrLoanType.objects.filter(account_type_id=account_type,
                                                                          general_charge__application='r',
                                                                          general_charge__status=True, general_charge__business=self.branch.business)
                    # print('TOTAL CHARGES', gen_cgs)

                    general_charges = gen_cgs.annotate(
                        amount=Case(
                            When(general_charge__is_percentage=True,
                                 then=ExpressionWrapper(initial * (F('general_charge__amount') / 100),
                                                        output_field=FloatField())),
                            default=F('general_charge__amount'),
                            output_field=FloatField()
                        )
                    ).annotate(rev_status=F('general_charge__is_revenue'))

                    # gen_charges = general_charges.values().aggregate(total=Sum('amount', default=0.00))
                    if general_charges.exists():

                        for ch in gen_cgs:
                            if ch.general_charge.is_revenue == 'YES':
                                Transactions.objects.create(
                                    branch=self.branch,
                                    transaction_type='Registration',
                                    account_dr=member_acc,
                                    account_cr=charges_acc,
                                    narration="{}".format(ch.general_charge.charge),
                                    reporting_amount=ch.general_charge.amount,
                                    added_by=biz_staff(user),
                                    receipt=rec_no + 'c',
                                    reference=ref_no + 'c'
                                )
                            else:
                                charge = ch.general_charge.charge
                                if Account.objects.filter(name=charge).exists():
                                    charge_account = charge
                                else:
                                    reserves = AccountCategory.objects.filter(name='Reserves').first()
                                    charge_account, created = Account.objects.create(name=ch.general_charge.charge,
                                                                                     category_id=reserves.id,
                                                                                     business=self.branch.business)
                                Transactions.objects.create(
                                    branch=self.branch,
                                    transaction_type='Registration',
                                    account_dr=member_acc,
                                    account_cr=charge_account,
                                    narration="{}".format(ch.general_charge.charge),
                                    reporting_amount=ch.general_charge.amount,
                                    added_by=biz_staff(user),
                                    receipt=rec_no + 'c',
                                    reference=ref_no + 'c'
                                )

                    # logger.error(f'ACCOUNT CHARGESSSSSS DONE')

                    # Get deposit charges that match the specified criteria
                    dep_cgs = TransactionCharge.objects.filter(account_type_id=account_type, charge_type='d', status=True)
                    # print('my dep charged', dep_cgs)

                    # Calculate deposit charges total
                    depo_charges = 0
                    if dep_cgs.exists():
                        for d in dep_cgs:
                            if d.account_type.deposit_charge_vary:
                                if d.start <= initial <= d.end:
                                    amount = initial * (d.charge / 100) if d.is_charge_percentage else d.charge
                            else:
                                amount = 0 if d.account_type.deposit_charge_vary else (
                                    initial * (d.charge / 100) if d.is_charge_percentage else d.charge)

                            depo_charges += amount

                            if member_type == '2' and amount > 0:
                                Transactions.objects.create(
                                    branch=self.branch,
                                    transaction_type='Charge',
                                    account_dr=member_acc,
                                    account_cr=charges_acc,
                                    narration=f"Charge on deposit of {initial}",
                                    reporting_amount=amount,
                                    added_by=biz_staff(user),
                                    receipt=f"{rec_no}c",
                                    reference=f"{ref_no}c"
                                )
                    # logger.info(f'DEPOSIT CHARGESSSSSS DONE')

                    if not setting_status.set_ordinary:
                        if initial_shares is None:
                            initial_shares = 0
                        narrative = f'Sale of {initial_shares} shares to {member.biodata.name} at {setting_status.share_price} each.'
                        business_shares = BusinessShares.objects.filter(business=self.branch.business).first()
                        shares_record = SharesTransactions.objects.create(
                            buyer_id=member.id,
                            shares=initial_shares,
                            date=datetime.datetime.today(),
                            narration=narrative,
                            branch=self.branch
                        )
                        # query buyer shares transactions
                        buyer_shares = SharesTransactions.objects.filter(Q(buyer=member) | Q(seller=member))
                        # print("results = %s" % buyer_shares.count())
                        total_bought = 0
                        total_sold = 0
                        for b in buyer_shares:
                            if b.buyer == member:
                                total_bought += b.shares
                            if b.seller == member:
                                total_sold += b.shares

                        current_shares = total_bought - total_sold

                        # update buyer shares balance
                        member.shares = current_shares
                        member.save()

                        # query seller shares transactions
                        seller_shares = SharesTransactions.objects.filter(seller=None)
                        total_sold = 0
                        for s in seller_shares:
                            total_sold += s.shares

                        # update business shares sold
                        business_shares.sold = total_sold
                        business_shares.save()

                        # Record a finance transaction
                        # shares account
                        shares_account = Account.objects.filter(business=self.branch.business,
                                                                category__name='Shares').first()

                        share_cost = 0
                        if initial_shares != '':
                            share_cost = int(setting_status.share_price) * int(initial_shares)
                        if share_cost > 0:

                            Transactions.objects.create(
                                branch=self.branch,
                                transaction_type='Sale of shares',
                                account_dr_id=member_acc.id,
                                account_cr=shares_account,
                                narration=narrative,
                                reporting_amount=share_cost,
                                added_by=biz_staff(user),
                                shares=shares_record,
                                receipt=rec_no + 's',
                                reference=ref_no + 's'
                            )

                    logger.info(f'BUYING SHARES DONE')
                Transactions.objects.create(
                    branch=self.branch,
                    transaction_type=transaction_type,
                    account_dr=account_dr,
                    account_cr=member_acc,
                    narration="Initial deposit on registration" if member_type == '2' else 'Opening balance',
                    reporting_amount=initial,
                    added_by=biz_staff(user),
                    receipt=rec_no,
                    reference=ref_no
                )

                logger.info(f'ACTUAL DEPOSIT DONE')

                member_debits = Transactions.objects.filter(account_dr=member_acc).aggregate(
                    total=Sum('reporting_amount', default=0.0))['total']
                if member_debits is None:
                    member_debits = 0

                member_credits = Transactions.objects.filter(account_cr=member_acc).aggregate(
                    total=Sum('reporting_amount', default=0.0))['total']
                if member_credits is None:
                    member_credits = 0

                logger.error(f'Mem cr: {str(member_credits)} and mem_debits: {str(member_debits)}')
                account.balance = member_credits - member_debits
                account.save()

                logger.error(f'ACCOUNT BALANCE UPDATE DONE')

                if member_type == '2':  # If this is a new member
                    # ==== SEND NOTIFICATIONS ======
                    template_file_link = 'sacco/emails/transactional.html'
                    currency = CurrencySetting.objects.filter(business=self.branch.business).first()
                    notify_type = NotiSettings.objects.filter(business=self.branch.business).first()
                    smsSettingsObj = SaccoSmsSettings.objects.get(when_to_send='on new member registration',
                                                                  business=self.branch.business)
                    message = f'Dear Client,\nThis is to inform you that your account in {biz_data(user).name} has been opened with A/C No: {account.acc_number} & deposit of {currency.currency} {initial} '
                    html_body = f'<p>Dear {bio.name},</p> <p>This is to inform you that your account in {biz_data(user).name} has been opened with A/C No: {account.acc_number} and an initial deposit of {currency.currency} {initial}. <br>We hope that you have a good experience with our services and <b>{biz_data(user).name}</b> will do it’s best to please you. We will be looked forward to serving you in adherence to our quality standards.</p><p>Thanking you and with profound regards.</p>'
                    if smsSettingsObj.status:
                        if notify_type.notification_type == 1:  # Send by SMS only
                            if bio.contact:
                                checkAndSendMessage(user, message, bio.contact, bio.name, account=member_acc,
                                                    rec_no=rec_no + 'c', this_account=account)
                        elif notify_type.notification_type == 2:  # Send by Email only
                            if bio.email:
                                sendTransEmail(template_file_link, html_body, biz_data(user).name,
                                               'Account Opening Successful', email)

                        elif notify_type.notification_type == 3:  # Send by both SMS and Email
                            if bio.contact:
                                checkAndSendMessage(user, message, bio.contact, bio.name,
                                                    account=member_acc, this_account=account)

                            if bio.email:
                                sendTransEmail(template_file_link, html_body, biz_data(user).name,
                                               'Account Opening Successful', email)

                logger.info(f'EVERYTHING IS DONE')
                return member
        except Exception as e:
            logger.error(traceback.print_exc())
            logger.error(f'Error in reg forms: {e}')
            # print(traceback.print_exc())
            # print('exception', str(e))
            raise e


class AddGroupForm(forms.Form):
    name = forms.CharField(required=False)
    dob = forms.DateField()
    contact = forms.CharField(required=False)
    other_contact = forms.CharField(required=False)
    email = forms.EmailField(required=False)
    account_type = forms.CharField(required=False)
    desc = forms.CharField(required=False)
    code = CountryField().formfield(required=False)
    code1 = CountryField().formfield(required=False)
    initial = forms.CharField(required=False)
    initial_shares = forms.IntegerField(required=False)
    address = forms.CharField(widget=forms.TextInput, required=False)
    account_number = forms.CharField(widget=forms.TextInput, required=False)

    def __init__(self, *args, **kwargs):
        self.user = kwargs.pop('user', None)
        self.group = kwargs.pop('group', None)
        super(AddGroupForm, self).__init__(*args, **kwargs)

    def the_business(self):
        biz = biz_data(self.user)
        return biz

    def clean_account_number(self):
        cleaned_data = self.cleaned_data
        account_number = cleaned_data.get('account_number')
        # print('aa', type(account_number))
        # try:
        if self.the_business().auto_acc_no is True:

            if account_number == '' or account_number is None:
                account_number = gen_account(self.user)
        else:
            if account_number != '' or account_number is not None:
                if check_account(account_number, self.user) is True:
                    raise ValidationError(_('Account already exists'), code='AccountNumber')
        return account_number

    def clean_initial(self):
        cleaned_data = self.cleaned_data
        account_type = cleaned_data.get('account_type')
        member_type = cleaned_data.get('member_type')
        initial = cleaned_data.get('initial')
        share_select = cleaned_data.get('share_select')
        initial_shares = cleaned_data.get('initial_shares')
        initial_before = float(initial.replace(',', ''))
        initial = float(initial_before)
        the_account_type = AccountTypes.objects.filter(id=account_type).first()
        min_balance = the_account_type.min_balance
        charge = int(check_charges(account_type, member_type, initial, self.user, initial_shares, share_select))
        if initial and account_type:
            # print('fine')
            if initial >= charge + min_balance:
                return initial
            else:
                raise ValidationError(
                    _(f'Insufficient amount to cater for charges of {charge} and minimum balance of {min_balance}'),
                    code='Amount')
        return initial

    def clean_contact(self):
        cleaned_data = self.cleaned_data
        code = cleaned_data.get('code')
        contact = cleaned_data.get('contact')
        if contact:
            primary_c = proper_dial(contact, code)
        else:
            primary_c = contact
        return primary_c

    def clean_other_contact(self):
        cleaned_data = self.cleaned_data
        code = cleaned_data.get('code')
        contact = cleaned_data.get('other_contact')
        if contact:
            primary_c = proper_dial(contact, code)
        else:
            primary_c = contact
        return primary_c

    def gen_member(self, name, contact, email, location, member_type, setting_status, user, share_select, upload_date, initial_shares):
        """Creating new member"""
        try:
            if member_type == '2':  # If this is a new member
                if setting_status.set_ordinary:  # If this is a new member & all member are ordinary on reg
                    member = Member.objects.create(name=name, contact=contact, email=email, created_by=biz_staff(user),
                                                   member_type='o', is_group=True, location=location,
                                                   branch=biz_staff_branch(user), shares=initial_shares,
                                                   date_joined=upload_date if upload_date else datetime.date.today())
                else:  # If this is new but all member must be shareholders on reg
                    if share_select == '1':  # this means is a shareholder
                        member = Member.objects.create(name=name, contact=contact, email=email, created_by=biz_staff(user),
                                                       member_type='s', is_group=True, location=location,
                                                       shares=initial_shares, branch=biz_staff_branch(user), date_joined=upload_date if upload_date else datetime.date.today())
                    else:  # this means is an ordinary
                        member = Member.objects.create(name=name, contact=contact, email=email, created_by=biz_staff(user),
                                                       member_type='o', is_group=True, branch=biz_staff_branch(user),
                                                       location=location, date_joined=upload_date if upload_date else datetime.date.today())
            else:  # If this is an existing member
                if share_select == '1':  # If this is an existing member and a shareholder
                    member = Member.objects.create(name=name, contact=contact, email=email, created_by=biz_staff(user),
                                                   member_type='s', is_group=True, location=location,
                                                   shares=initial_shares, branch=biz_staff_branch(user), date_joined=upload_date if upload_date else datetime.date.today())

                else:  # If this is an existing member but ordinary
                    member = Member.objects.create(name=name, contact=contact, email=email, created_by=biz_staff(user), member_type='o',
                                                   branch=biz_staff_branch(user), is_group=True, location=location,
                                                   date_joined=upload_date if upload_date else datetime.date.today())
            return member
        except Exception as e:
            logger.error(str(e))
            raise ValidationError(_("Error creating member: " + str(e)), code='member_creation')

    @transaction.atomic
    def save(self, commit=True):
        user = self.user
        cleaned_data = super().clean()
        name = cleaned_data.get('name')
        dob = cleaned_data.get('dob')
        email = cleaned_data.get('email')
        location = cleaned_data.get('location')
        the_account = cleaned_data.get('the_account')
        contact = cleaned_data.get('contact')
        account_type = cleaned_data.get('account_type')
        member_type = cleaned_data.get('member_type')
        share_select = cleaned_data.get('share_select')
        initial_shares = cleaned_data.get('initial_shares')
        initial = cleaned_data.get('initial')
        account_number = cleaned_data.get('account_number')

        rec_no = datetime.datetime.now().strftime("%y%m%d%H%M%S")
        ref_no = datetime.datetime.now().strftime("%y%m%d%H%M%S")

        try:
            with transaction.atomic():
                transaction_type = 'Opening balance'

                setting_status = OtherSettings.objects.filter(business=biz_data(user)).first()
                logger.error(f'Settings status {str(setting_status)}')

                logger.error(f'Biodata has been created')
                member = self.gen_member(name, contact, email, location, member_type, setting_status, user, share_select, dob,
                                         initial_shares)
                logger.error(f'Member has been created')
                account = MemberAccount.objects.create(account_type_id=account_type, acc_number=account_number)
                logger.error(f'Account created')
                broker = AccountBroker.objects.create(members=member, the_account=account, business=biz_data(user))
                logger.error(f'Broker created')
                cat, created = AccountCategory.objects.get_or_create(business=biz_data(user), name="Members",
                                                                     cat_type='liability', dr_cr='cr')
                logger.error(f'Cat {str(cat)}')
                equity_cat, created = AccountCategory.objects.get_or_create(business=biz_data(user),
                                                                            name="Shares", dr_cr='cr',
                                                                            cat_type='liability')
                logger.error(f'Equity cat {str(equity_cat)}')
                # chart of accounts
                member_acc = Account.objects.create(category=cat, business=biz_data(user), member=account,
                                                    added_by=biz_staff(user))
                logger.error(f'Member acc {str(member_acc)} created')

                if the_account:
                    acc_involved = Account.objects.filter(id=the_account, business=biz_data(user)).first()

                opening_acct = Account.objects.get(name='Opening Reserves', business=biz_data(user))

                account_dr = opening_acct

                # equity, created = Account.objects.get_or_create(name='Share Capital', business=biz_data(user),
                #                                                 category=equity_cat)
                logger.error(f'ACCOUNT CREATION DONE')

                if member_type == '2':  # If this is a new member
                    transaction_type = 'Deposit'
                    account_dr = acc_involved

                    charges_acc = Account.objects.get(category__name="Registration Income", business=biz_data(user))
                    gen_cgs = ApplicationAccountOrLoanType.objects.filter(account_type_id=account_type,
                                                                          general_charge__application='r',
                                                                          general_charge__status=True)

                    general_charges = gen_cgs.annotate(
                        amount=Case(
                            When(general_charge__is_percentage=True,
                                 then=ExpressionWrapper(initial * (F('general_charge__amount') / 100),
                                                        output_field=FloatField())),
                            default=F('general_charge__amount'),
                            output_field=FloatField()
                        )
                    ).annotate(rev_status=F('general_charge__is_revenue'))

                    # gen_charges = general_charges.values().aggregate(total=Sum('amount', default=0.00))
                    if general_charges.exists():

                        for ch in gen_cgs:
                            if ch.general_charge.is_revenue == 'YES':
                                Transactions.objects.create(
                                    branch=biz_staff_branch(user),
                                    transaction_type='Registration',
                                    account_dr=member_acc,
                                    account_cr=charges_acc,
                                    narration="{}".format(ch.general_charge.charge),
                                    reporting_amount=ch.general_charge.amount,
                                    added_by=biz_staff(user),
                                    receipt=rec_no + 'c',
                                    reference=ref_no + 'c'
                                )
                            else:
                                charge = ch.general_charge.charge
                                if Account.objects.filter(name=charge).exists():
                                    charge_account = charge
                                else:
                                    reserves = AccountCategory.objects.filter(name='Reserves').first()
                                    charge_account, created = Account.objects.create(name=ch.general_charge.charge,
                                                                                     category_id=reserves.id,
                                                                                     business=biz_data(user))
                                Transactions.objects.create(
                                    branch=biz_staff_branch(user),
                                    transaction_type='Registration',
                                    account_dr=member_acc,
                                    account_cr=charge_account,
                                    narration="{}".format(ch.general_charge.charge),
                                    reporting_amount=ch.general_charge.amount,
                                    added_by=biz_staff(user),
                                    receipt=rec_no + 'c',
                                    reference=ref_no + 'c'
                                )

                    logger.error(f'ACCOUNT CHARGES DONE')

                    # Get deposit charges that match the specified criteria
                    dep_cgs = TransactionCharge.objects.filter(account_type_id=account_type, charge_type='d',
                                                               status=True)
                    # print('my dep charged', dep_cgs)

                    # Calculate deposit charges total
                    depo_charges = 0
                    if dep_cgs.exists():
                        for d in dep_cgs:
                            if d.account_type.deposit_charge_vary:
                                if d.start <= initial <= d.end:
                                    amount = initial * (d.charge / 100) if d.is_charge_percentage else d.charge
                            else:
                                amount = 0 if d.account_type.deposit_charge_vary else (
                                    initial * (d.charge / 100) if d.is_charge_percentage else d.charge)

                            depo_charges += amount

                            if member_type == '2' and amount > 0:
                                Transactions.objects.create(
                                    branch=biz_staff_branch(user),
                                    transaction_type='Charge',
                                    account_dr=member_acc,
                                    account_cr=charges_acc,
                                    narration=f"Charge on deposit of {initial}",
                                    reporting_amount=amount,
                                    added_by=biz_staff(user),
                                    receipt=f"{rec_no}c",
                                    reference=f"{ref_no}c"
                                )
                    logger.info(f'DEPOSIT CHARGES DONE')

                    if not setting_status.set_ordinary:
                        narrative = f'Sale of {initial_shares} shares to {member.biodata.name} at {setting_status.share_price} each.'
                        business_shares = BusinessShares.objects.filter(business=biz_data(user)).first()
                        if initial_shares != '' or initial_shares is not None and initial_shares > 0:
                            shares_record = SharesTransactions.objects.create(
                                buyer_id=member.id,
                                shares=initial_shares,
                                date=datetime.datetime.today(),
                                narration=narrative,
                                branch=this_branch(user)
                            )
                            # query buyer shares transactions
                            buyer_shares = SharesTransactions.objects.filter(Q(buyer=member) | Q(seller=member))
                            # print("results = %s" % buyer_shares.count())
                            total_bought = 0
                            total_sold = 0
                            for b in buyer_shares:
                                if b.buyer == member:
                                    total_bought += b.shares
                                if b.seller == member:
                                    total_sold += b.shares

                            current_shares = total_bought - total_sold

                            # update buyer shares balance
                            member.shares = current_shares
                            member.save()

                            # query seller shares transactions
                            seller_shares = SharesTransactions.objects.filter(seller=None)
                            total_sold = 0
                            for s in seller_shares:
                                total_sold += s.shares

                            # update business shares sold
                            business_shares.sold = total_sold
                            business_shares.save()

                            # Record a finance transaction
                            # shares account
                            shares_account = Account.objects.filter(business=biz_data(user),
                                                                    category__name='Shares').first()

                            share_cost = 0
                            if initial_shares != '':
                                share_cost = int(setting_status.share_price) * int(initial_shares)
                            if share_cost > 0:
                                Transactions.objects.create(
                                    branch=biz_staff_branch(user),
                                    transaction_type='Sale of shares',
                                    account_dr_id=member_acc.id,
                                    account_cr=shares_account,
                                    narration=narrative,
                                    reporting_amount=share_cost,
                                    added_by=biz_staff(user),
                                    shares=shares_record,
                                    receipt=rec_no + 's',
                                    reference=ref_no + 's'
                                )

                    logger.info(f'BUYING SHARES DONE')
                if float(initial) > 0:
                    Transactions.objects.create(
                        branch=biz_staff_branch(user),
                        transaction_type=transaction_type,
                        account_dr=account_dr,
                        account_cr=member_acc,
                        narration="Initial deposit on registration" if member_type == '2' else 'Opening balance',
                        reporting_amount=initial,
                        added_by=biz_staff(user),
                        receipt=rec_no,
                        reference=ref_no
                    )

                logger.info(f'ACTUAL DEPOSIT DONE')

                member_debits = Transactions.objects.filter(account_dr=member_acc).aggregate(
                    total=Sum('reporting_amount', default=0.0))['total']
                if member_debits is None:
                    member_debits = 0

                member_credits = Transactions.objects.filter(account_cr=member_acc).aggregate(
                    total=Sum('reporting_amount', default=0.0))['total']
                if member_credits is None:
                    member_credits = 0

                logger.error(f'Mem cr: {str(member_credits)} and mem_debits: {str(member_debits)}')
                account.balance = member_credits - member_debits
                account.save()

                logger.error(f'ACCOUNT BALANCE UPDATE DONE')

                if member_type == '2':  # If this is a new member
                    # ==== SEND NOTIFICATIONS ======
                    template_file_link = 'sacco/emails/transactional.html'
                    currency = CurrencySetting.objects.filter(business=biz_data(user)).first()
                    notify_type = NotiSettings.objects.filter(business=biz_data(user)).first()
                    smsSettingsObj = SaccoSmsSettings.objects.get(when_to_send='on new member registration',
                                                                  business=biz_data(user))
                    message = f'Dear Client,\nThis is to inform you that your account in {biz_data(user).name} has been opened with A/C No: {account.acc_number} & deposit of {currency.currency} {initial} '
                    html_body = f'<p>Dear {member.name},</p> <p>This is to inform you that your account in {biz_data(user).name} has been opened with A/C No: {account.acc_number} and an initial deposit of {currency.currency} {initial}. <br>We hope that you have a good experience with our services and <b>{biz_data(user).name}</b> will do it’s best to please you. We will be looked forward to serving you in adherence to our quality standards.</p><p>Thanking you and with profound regards.</p>'
                    if smsSettingsObj.status:
                        if notify_type.notification_type == 1:  # Send by SMS only
                            if member.contact:
                                checkAndSendMessage(user, message, member.contact, member.name, account=member_acc,
                                                    rec_no=rec_no + 'c', this_account=account)
                        elif notify_type.notification_type == 2:  # Send by Email only
                            if member.email:
                                sendTransEmail(template_file_link, html_body, biz_data(user).name,
                                               'Account Opening Successful', email)

                        elif notify_type.notification_type == 3:  # Send by both SMS and Email
                            if member.contact:
                                checkAndSendMessage(user, message, member.contact, member.name,
                                                    account=member_acc, this_account=account)

                            if member.email:
                                sendTransEmail(template_file_link, html_body, biz_data(user).name,
                                               'Account Opening Successful', email)

                logger.info(f'EVERYTHING IS DONE')
                return member
        except Exception as e:
            logger.error(traceback.print_exc())
            logger.error(f'Error in reg forms: {e}')
            # print(traceback.print_exc())
            # print('exception', str(e))
            raise e


# class EditGroupForm(forms.Form):
#     name = forms.CharField(required=False)
#     dob = forms.DateField(
#         widget=forms.SelectDateWidget(
#             attrs={'class': 'form-control', 'style': 'width:33.33333333333333%; ' 'display:inline' '-block;'},
#             years=range(1920, datetime.date.today().year + 1)), initial=timezone.now())
#
#     contact = forms.CharField(required=False)
#     other_contact = forms.CharField(required=False)
#     email = forms.EmailField(required=False)
#     account_type = forms.CharField(required=False)
#     desc = forms.CharField(required=False)
#     code = CountryField().formfield(required=False)
#     code1 = CountryField().formfield(required=False)
#     location = forms.CharField(widget=forms.TextInput, required=False)
#     account_number = forms.CharField(widget=forms.TextInput, required=False)

