import random
import time
import uuid
from datetime import date

from django.contrib.contenttypes.fields import GenericForeignKey
from django.contrib.contenttypes.models import ContentType
from django.core.exceptions import ValidationError
from django.core.validators import MinValueValidator, MaxValueValidator
from django.db import models
from django.db.models.signals import post_save
from django.dispatch import receiver
from django.urls import reverse

from sacco.constants import ACTION_TYPES, ACTION_STATUS, SUCCESS, SMSCHARGE, BUSINESS
from utils.file_uploads import bio_photo


def gen_txt_ref() -> str:

    timestamp = str(int(time.time()))  # get current timestamp as a string
    rand_num = str(random.randint(100000, 999999))  # generate a random 6-digit number as a string
    transaction_id = timestamp + rand_num  # concatenate the timestamp and random number to create a unique ID
    x = transaction_id.zfill(12)
    return x

class ActivityLog(models.Model):
    actor = models.ForeignKey('accounts.User', on_delete=models.CASCADE, null=True)
    action_type = models.CharField(choices=ACTION_TYPES, max_length=15)
    action_time = models.DateTimeField(auto_now_add=True)
    remarks = models.TextField(blank=True, null=True)
    status = models.CharField(choices=ACTION_STATUS, max_length=7, default=SUCCESS)
    data = models.JSONField(default=dict, blank=True, null=True)
    branch = models.BigIntegerField(null=True, blank=True)
    title = models.CharField(max_length=255, blank=True, null=True)
    # for generic relations
    content_type = models.ForeignKey(
        ContentType, models.SET_NULL, blank=True, null=True
    )
    object_id = models.PositiveIntegerField(blank=True, null=True)
    content_object = GenericForeignKey()

    def __str__(self) -> str:
        return f"{self.action_type} by {self.actor} on {self.action_time}"

    class Meta:
        verbose_name = 'Activity Log'
        db_table = 'activity_logs'
        ordering = ['-action_time']


class BusinessShares(models.Model):
    market_cap = models.FloatField(default=0)
    sold = models.FloatField(default=0)
    business = models.ForeignKey('accounts.Business', on_delete=models.CASCADE)
    business_no = models.CharField(max_length=8, blank=True, null=True)

    class Meta:
        db_table = 'business_shares'
        verbose_name = 'Business Shares'

    def __str__(self):
        return str(self.market_cap)


class LicenseKey(models.Model):
    license_key = models.UUIDField(unique=True, blank=True, null=True)
    business = models.OneToOneField('accounts.Business',related_name='licence', on_delete=models.CASCADE, blank=True, null=True)
    expiry_date = models.DateField(blank=True, null=True)
    date_created = models.DateField(auto_now_add=True)
    status = models.BooleanField(default=True)

    class Meta:
        db_table = 'license_keys'
        verbose_name = 'License Key'
        verbose_name_plural = 'Licenses Keys'

    def __str__(self):
        return self.license_key.__str__()

    def save(self, *args, **kwargs):
        if not self.id:
            self.license_key = uuid.uuid4().hex
        return super(LicenseKey, self).save(*args, **kwargs)

    @property
    def remaining_days(self):
        exp = self.expiry_date
        today = date.today()
        remaining = exp - today
        return remaining.days

    @property
    def is_active(self):
        if self.expiry_date >= date.today():
            return True
        return False


class MemberAccount(models.Model):
    acc_number = models.CharField(blank=True, null=True, max_length=255)
    balance = models.FloatField(null=True, blank=True)
    synced = models.BooleanField(default=False, verbose_name='Synced to the portal')
    account_type = models.ForeignKey('AccountTypes', on_delete=models.CASCADE, blank=True, null=True)
    is_active = models.BooleanField(default=True)

    def __str__(self):
        return f'{self.acc_number}'

    class Meta:
        db_table = 'member_accounts'
        verbose_name = 'Member Account'
        verbose_name_plural = 'Member Accounts'

    @property
    def names(self):
        try:
            data = AccountBroker.objects.get(the_account_id=self.id)
            if data.members.is_group:
                return "%s" % data.members.name
            else:
                return "%s" % data.members.biodata.name
        except AccountBroker.DoesNotExist:
            return ''

    @property
    def member_names(self):
        try:
            data = AccountBroker.objects.get(the_account_id=self.id)
            if data.members.is_group:
                return "%s (%s)" % (data.members.name, self.acc_number)
            else:
                return "%s (%s)" % (data.members.biodata.name, self.acc_number)
        except AccountBroker.DoesNotExist:
            return ''


class AccountTypes(models.Model):
    name = models.CharField(max_length=255)
    is_fixed = models.BooleanField(default=False)
    min_balance = models.FloatField(default=0)
    maturity = models.IntegerField(default=0)
    dormancy_period = models.IntegerField(default=0)
    deposit_charge = models.BooleanField(default=True)
    withdraw_charge = models.BooleanField(default=True)
    transfer_charge = models.BooleanField(default=True)
    business = models.ForeignKey('accounts.Business', on_delete=models.CASCADE, blank=True, null=True)
    status = models.BooleanField(default=True)
    deposit_charge_vary = models.BooleanField(default=True)
    withdraw_charge_vary = models.BooleanField(default=True)
    transfer_charge_vary = models.BooleanField(default=True)
    synced = models.BooleanField(default=False, verbose_name='Synced to the portal')

    class Meta:
        db_table = 'account_types'
        verbose_name = 'Account Type'
        verbose_name_plural = 'Account Types'

    def __str__(self):
        return self.name


class AccountBroker(models.Model):
    members = models.ForeignKey('Member', related_name='owners', on_delete=models.CASCADE)
    the_account = models.ForeignKey(MemberAccount, related_name='member_account', on_delete=models.CASCADE)
    created_date = models.DateField(auto_now_add=True)
    business = models.ForeignKey('accounts.Business', related_name='member_business', on_delete=models.CASCADE)

    class Meta:
        db_table = 'account_broker'
        verbose_name = 'Account Broker'

    def save_related(self):
        self.the_account.save()
        self.save()


class MemberUpload(models.Model):
    uuid = models.UUIDField()
    business = models.ForeignKey('accounts.Business', on_delete=models.CASCADE)
    created_at = models.DateTimeField(auto_now_add=True)
    failed = models.CharField(max_length=255, blank=True, null=True)
    reason = models.TextField(null=True, blank=True)
    file = models.FileField(upload_to='member_upload_files', blank=True, null=True)

    def __str__(self):
        return str(self.uuid)

    class Meta:
        db_table = 'member_upload'

    def save(self, *args, **kwargs):
        if not self.id:
            value = uuid.uuid4().hex
            self.uuid = str(value)
        return super(MemberUpload, self).save(*args, **kwargs)

    @property
    def file_url(self):
        if self.file and hasattr(self.file, 'url'):
            return self.file.url

class Member(models.Model):
    MEMBER_TYPE = (
        ('o', 'Ordinary'),
        ('s', 'Shareholder')
    )
    biodata = models.ForeignKey('accounts.Biodata', blank=True, null=True, on_delete=models.CASCADE)
    name = models.CharField(max_length=255, blank=True, null=True)
    email = models.EmailField(blank=True, null=True)
    contact = models.CharField(null=True, blank=True, max_length=40)
    other_contact = models.CharField(null=True, blank=True, max_length=45)
    photo = models.ImageField(upload_to=bio_photo, blank=True, null=True)
    location = models.TextField(blank=True, null=True)
    branch = models.ForeignKey('accounts.Branch', on_delete=models.CASCADE, null=True, blank=True)
    member_type = models.CharField(max_length=1, choices=MEMBER_TYPE, default='o')
    shares = models.FloatField(default=0.00, null=True, blank=True)
    created_date = models.DateField(auto_now_add=True)
    created_by = models.ForeignKey('accounts.Staff', on_delete=models.SET_NULL, null=True, blank=True)
    accounts = models.ManyToManyField(MemberAccount, related_name='members_account', through=AccountBroker, blank=True)
    referee = models.ForeignKey('Member', related_name='referral', null=True, blank=True, on_delete=models.CASCADE)
    is_group = models.BooleanField(default=False)
    is_active = models.BooleanField(default=True)
    desc = models.TextField(blank=True, null=True, verbose_name='Description')
    date_joined = models.DateField(blank=True, null=True)
    portal = models.BooleanField(default=False, verbose_name='Synced to the portal')
    member_upload = models.ForeignKey(MemberUpload, related_name='members', on_delete=models.SET_NULL, blank=True, null=True)
    share_holder_join_date = models.DateField(null=True, blank=True)
    group_officer = models.ForeignKey('accounts.Staff', on_delete=models.SET_NULL, null=True, blank=True, related_name='group_loans_officer')

    def __str__(self):
        if self.is_group is False:
            return self.biodata.name
        return self.name

    def clean(self):
        super().clean()
        if self.is_group and not (self.name and self.contact and self.other_contact and self.location):
            raise ValidationError("Name, contact, and other_contact are required for group members.")

    @property
    def photo_url(self):
        if self.is_group:
            if self.photo and hasattr(self.photo, 'url'):
                return self.photo.url
        else:
            if self.biodata.photo and hasattr(self.biodata.photo, 'url'):
                return self.biodata.photo.url

    def get_absolute_url(self):
        if self.is_group:
            return reverse('group_detail', kwargs={"pk": self.id})
        return reverse('member_detail', kwargs={"pk": self.id})

    @property
    def active_status(self):
        if self.is_active:
            return 'Active'
        return 'Deactivated'

    class Meta:
        ordering = ['-id', '-biodata__name']

    def save_related(self):
        self.biodata.save()
        self.save()


class UserMember(models.Model):
    user = models.OneToOneField('accounts.User', on_delete=models.CASCADE)
    member = models.OneToOneField(Member, on_delete=models.CASCADE)

    class Meta:
        db_table = 'user_member'


class FinancialYear(models.Model):
    name = models.CharField(max_length=20, null=True, blank=True)
    start_date = models.DateField()
    end_date = models.DateField()
    business = models.ForeignKey('accounts.Business', on_delete=models.CASCADE)
    added_by = models.ForeignKey('accounts.Staff', on_delete=models.SET_NULL, null=True)
    created_date = models.DateTimeField(auto_now_add=True)
    status = models.BooleanField(default=True)

    @property
    def is_past(self):
        return date.today() < self.end_date

    def __unicode__(self):
        return self.name

    class Meta:
        db_table = 'financial_years'
        verbose_name = 'Financial Year'
        ordering = ['-id', '-name']

    # @property
    def is_within_financial_year(self, input_date, *args, **kwargs):

        if self.start_date <= input_date.date() <= self.end_date:
            return True
        return False


class GroupMember(models.Model):
    MEMBER_TYPE = (
        ('Chair', 'ChairPerson'),
        ('Secretary', 'Secretary'),
        ('Treasurer', 'Treasurer'),
        ('Ordinary', 'Ordinary'),
    )
    group = models.ForeignKey(Member, related_name='members', on_delete=models.CASCADE)
    member = models.ForeignKey(Member, related_name='groups', on_delete=models.CASCADE)
    is_signatory = models.BooleanField(default=False)
    role = models.CharField(max_length=20, default='Ordinary', choices=MEMBER_TYPE)
    added_by = models.ForeignKey('accounts.Staff', on_delete=models.SET_NULL, null=True)
    created_date = models.DateTimeField(auto_now_add=True)


    class Meta:
        db_table = 'group_members'
        verbose_name = 'Group Members'
        unique_together = ['group', 'member']

    @property
    def signatory_status(self):
        if self.is_signatory:
            return 'Yes'
        return 'No'


class LoanSettings(models.Model):
    class TOPUPS(models.IntegerChoices):
        CLOSE_RUNNING_LOAN = 1, 'Close running loan'
        LOANS_RUN_CONCURRENTLY = 2, 'Loans run concurrently'
        NOT_ALLOWED = 3, 'Not allowed'

    business = models.ForeignKey('accounts.Business', on_delete=models.CASCADE, null=True, blank=True)
    minimum_guarantors = models.IntegerField(default=1, null=True, blank=True, )
    decimals = models.IntegerField(null=True, blank=True, default=2)
    top_up_limit = models.IntegerField(null=True, blank=True, default=0, verbose_name="Percentage paid on the running loan")
    min_approvals = models.IntegerField(null=True, blank=True, default=1, verbose_name="Minimum number of loan approvals")
    allow_others = models.BooleanField(default=False, null=True, blank=True, verbose_name="Allow non-member Guarantors")
    upload_minutes = models.BooleanField(default=False, null=True, blank=True, verbose_name="Approving committee minutes must be uploaded")
    loan_topup = models.IntegerField(default=TOPUPS.CLOSE_RUNNING_LOAN, choices=TOPUPS.choices, null=True, blank=True)
    fines_on = models.BooleanField(default=True, null=True, blank=True, verbose_name="Automatically impose loan fines")
    use_shares = models.BooleanField(default=True, null=True, blank=True, verbose_name="Consider savings and shares value")
    use_public_holidays = models.BooleanField(default=True, null=True, blank=True, verbose_name="Consider Public Holidays")

    class Meta:
        db_table = 'loan_settings'
        verbose_name = 'Loan Setting'
        verbose_name_plural = 'Loan Settings'


class GroupSettings(models.Model):
    business = models.ForeignKey('accounts.Business', on_delete=models.CASCADE, null=True, blank=True)
    minimum_members = models.IntegerField(default=1)
    minimum_signatories = models.IntegerField(default=1)
    maximum_signatories = models.IntegerField(default=1)
    maximum_members = models.IntegerField(default=1)

    class Meta:
        db_table = 'group_settings'
        verbose_name = 'Group Setting'
        verbose_name_plural = 'Group Settings'


class NotiSettings(models.Model):
    NONE = 0
    SMS = 1
    EMAIL = 2
    BOTH = 3
    NOTIF_TYPES = (
        (SMS, 'SMS'),
        (EMAIL, 'Email'),
        (BOTH, 'Both'),
        (NONE, 'None'),
    )
    business = models.ForeignKey('accounts.Business', on_delete=models.CASCADE, null=True, blank=True)
    notification_type = models.IntegerField(choices=NOTIF_TYPES, default=SMS)
    faces_charge = models.CharField(max_length=20, default=BUSINESS, choices=SMSCHARGE)

    class Meta:
        db_table = 'notification_settings'
        verbose_name = 'Notification Setting'
        verbose_name_plural = 'Notification Settings'


class OtherSettings(models.Model):
    business = models.ForeignKey('accounts.Business', on_delete=models.CASCADE, null=True, blank=True)
    dividends_sharing = models.BooleanField(verbose_name="Dividends are shared to only shareholders", default=True)
    set_ordinary = models.BooleanField(verbose_name="Set ordinary(Default Ordinary user)", default=True)
    share_price = models.FloatField(null=True, blank=True, default=0)
    share_limit = models.IntegerField(null=True, blank=True, default=0)

    class Meta:
        db_table = 'other_settings'
        verbose_name = 'Other Setting'
        verbose_name_plural = 'Other Settings'


class Sms(models.Model):
    branch = models.ForeignKey('accounts.Branch', on_delete=models.CASCADE, null=True, blank=True)
    quantity = models.IntegerField(default=0, null=True, blank=True)

    class Meta:
        db_table = 'sms'
        verbose_name = 'SMS'
        verbose_name_plural = 'SMSs'


class TransactionCharge(models.Model):
    CHARGE_TYPE = (
        ('d', 'Deposit'),
        ('w', 'Withdraw'),
        ('t', 'Transfer'),
    )

    start = models.FloatField(default=0, null=True, blank=True)
    end = models.FloatField(default=0, null=True, blank=True)
    charge = models.FloatField(default=0, null=False, blank=False)
    account_type = models.ForeignKey(AccountTypes, on_delete=models.CASCADE, null=False, blank=False)
    charge_type = models.CharField(max_length=1, choices=CHARGE_TYPE, default='d')
    status = models.BooleanField(default=True)
    is_charge_percentage = models.BooleanField(default=False)
    created_date = models.DateTimeField(auto_now_add=True)
    deletion_date = models.DateTimeField(blank=True, null=True)

    class Meta:
        db_table = 'transaction_charge'
        verbose_name = 'Transaction charge'
        verbose_name_plural = 'Transaction charges'

    def __str__(self):
        return str(self.charge) + '--' + str(self.charge_type) + '--' + str(self.start) + '--' + str(self.end)


class SmsTransaction(models.Model):
    PURCHASE = 0
    USAGE = 1
    PENDING = 1
    SENT = 2
    FAILED = 3

    Tx_TYPES = (
        (PURCHASE, 'Purchase'),
        (USAGE, 'Usage'),
    )
    SMS_STATUS = (
        (PENDING, 'Pending'),
        (SENT, 'Sent'),
        (FAILED, 'Failed'),

    )
    branch = models.ForeignKey('accounts.Branch', on_delete=models.CASCADE, null=True, blank=True)
    quantity = models.IntegerField(default=0, null=True, blank=True)
    trans_type = models.IntegerField(default=0, null=True, blank=True, choices=Tx_TYPES)
    price = models.FloatField(default=0, null=True, blank=True)
    body = models.TextField(null=True, blank=True)
    receiver = models.CharField(max_length=255, null=True, blank=True)
    is_broadcast = models.BooleanField(default=False)
    delivery_id = models.IntegerField(null=True, blank=True)
    status = models.IntegerField(null=True, blank=True, choices=SMS_STATUS, default=PENDING)
    added_by = models.ForeignKey('accounts.Staff', on_delete=models.SET_NULL, null=True, blank=True)
    created_date = models.DateTimeField(auto_now_add=True)
    message_id = models.IntegerField(default=0, null=True, blank=True)

    class Meta:
        db_table = 'sms_transaction'
        verbose_name = 'SMS transaction'
        verbose_name_plural = 'SMS transactions'


class SaccoSmsSettings(models.Model):
    when_to_send = models.CharField(max_length=255, blank=False)
    section = models.CharField(max_length=255, blank=False)
    status = models.BooleanField(default=True)
    business = models.ForeignKey('accounts.Business', on_delete=models.CASCADE, null=True, blank=True)

    class Meta:
        db_table = 'sacco_sms_setting'
        verbose_name = 'Sacco sms setting'
        verbose_name_plural = 'Sacco sms settings'

    def __str__(self):
        return str(self.section) + '--' + str(self.status)


class GeneratedPassword(models.Model):
    created_date = models.DateTimeField(auto_now_add=True)
    is_used = models.BooleanField(default=False)
    staff = models.OneToOneField('accounts.Staff', on_delete=models.CASCADE)
    password = models.CharField(max_length=8)
    can_login = models.BooleanField(default=False)

    class Meta:
        db_table = 'generated_password'


class CurrencySetting(models.Model):
    currency = models.CharField(max_length=8, blank=True)
    business = models.ForeignKey('accounts.Business', on_delete=models.CASCADE, blank=False, null=False)
    show_currency = models.BooleanField(default=True)

    class Meta:
        db_table = 'currency_setting'
        verbose_name = 'currency setting'
        verbose_name_plural = 'currency settings'

    def __str__(self):
        return str(self.currency) + '--' + str(self.business)


class ChargeSchedule(models.Model):
    account = models.ForeignKey(MemberAccount, on_delete=models.CASCADE)
    charge_date = models.DateField(blank=True, null=True)

    class Meta:
        db_table = 'charge_schedule'

    @receiver(post_save, sender=MemberAccount)
    def create_charge(sender, created, instance, **kwargs):
        # x = instance.member_account.business
        # print(f'ze me here {x}')
        # execu = self.event_date.date() + datetime.timedelta(days=2)
        if created:
            account_charge = ChargeSchedule(account=instance)
            account_charge.save()


class MessagePackage(models.Model):
    name = models.CharField(default="", null=True, blank=True, max_length=255)
    unit_cost = models.FloatField(default=0.0)
    price = models.FloatField(default=500.0)
    sms = models.IntegerField(default=10)
    status = models.BooleanField(default=True)

    class Meta:
        db_table = 'message_package'
        verbose_name = 'Message Package'
        verbose_name_plural = 'Message Packages'

    def __str__(self):
        return str(self.name)

class SmsBusinessPackage(models.Model):
    branch = models.ForeignKey('accounts.Branch', on_delete=models.CASCADE, null=True, blank=True)
    package = models.ForeignKey(MessagePackage, on_delete=models.CASCADE, null=True, blank=True)
    db_table = 'sms_business_package'
    verbose_name = 'SMS Business Package'
    verbose_name_plural = 'SMS Business Packages'

class PaymentTransaction(models.Model):
    PENDING = 'PENDING'
    SUCCESS = 'SUCCESS'
    FAILED = 'FAILED'

    Tx_STATUS = (
        (PENDING, 'Pending'),
        (SUCCESS, 'Success'),
        (FAILED, 'Failed'),
    )
    txt_id = models.CharField(max_length=200)
    status = models.CharField(max_length=200, choices=Tx_STATUS, default=PENDING)
    amount = models.FloatField()
    phone = models.CharField(max_length=100)
    description = models.TextField()
    created_by = models.ForeignKey('accounts.Staff', null=True, blank=True, on_delete=models.SET_NULL)
    created_at = models.DateTimeField(auto_now_add=True)
    jpesaid = models.CharField(max_length=200, blank=True, null=True)

    def __str__(self):
        return str(self.txt_id)

    class Meta:
        db_table = 'subscription_transaction'

    # def save(self, *args, **kwargs):
    #     if not self.id:
    #         self.txt_id = gen_txt_ref()
    #     return super(SubscriptionTransaction, self).save(*args, **kwargs)


# class SMSPurchases(models.Model):
#     created_by = models.ForeignKey('accounts.Staff', null=True, blank=True, on_delete=models.SET_NULL)
#     created_at = models.DateTimeField(auto_now_add=True)

class SMSCost(models.Model):
    business = models.ForeignKey('accounts.Business', on_delete=models.CASCADE)
    cost = models.FloatField(default=0.00)

    class Meta:
        db_table = 'sms_cost'


class WithdrawUpload(models.Model):
    narration = models.TextField(blank=False, null=False)
    total_records_added = models.FloatField(blank=False, null=False)
    total_records_failed = models.FloatField(blank=False, null=False)
    transactions_created = models.TextField(blank=False, null=False)
    failed_withdraws = models.TextField(blank=False, null=False)
    date_added = models.DateField(auto_now_add=True)
    branch = models.ForeignKey('accounts.Branch', on_delete=models.CASCADE, related_name='withdraw_branch', null=False, blank=False)
    file = models.FileField(upload_to='withdraws', null=False, blank=False)
    class Meta:
        db_table = 'withdraw_upload'
        verbose_name = 'Withdraw upload'
        verbose_name_plural = 'Withdraw uploads'

    def __str__(self):
        return str(self.narration)


class CustomBusiness(models.Model):
    NAMAMYINGO = 'NAMAMYINGO'
    OTHERS = 'OTHERS'

    BUSINESS_MODULE = (
        (NAMAMYINGO, 'Namayingo Modules'),
        (OTHERS, 'For others Modules'),

    )
    business = models.ForeignKey('accounts.Business', on_delete=models.CASCADE)
    business_module = models.CharField(max_length=255, null=True, blank=True, choices=BUSINESS_MODULE)

    def __str__(self):
        return self.business.name

    class Meta:
        db_table = 'custom_business'

class ReasonMemberRemoval(models.Model):
    reason = models.TextField()
    member = models.ForeignKey(GroupMember, on_delete=models.CASCADE)
    def __str__(self):
        return self.reason


class BulkDepositTracker(models.Model):
    created_at = models.DateField(auto_now_add=True)
    group = models.IntegerField(null=False, blank=False)
    external_id = models.UUIDField(unique=True)

    def __str__(self):
        return self.external_id

    def save(self, *args, **kwargs):
        if not self.id:
            value = uuid.uuid4().hex
            self.uuid = str(value)
        return super(BulkDepositTracker, self).save(*args, **kwargs)

class SavingUpload(models.Model):
    narration = models.TextField(blank=False, null=False)
    total_records_added = models.FloatField(blank=False, null=False)
    total_records_failed = models.FloatField(blank=False, null=False)
    transactions_created = models.TextField(blank=False, null=False)
    failed_savings = models.TextField(blank=False, null=False)
    date_added = models.DateField(auto_now_add=True)
    branch = models.ForeignKey('accounts.Branch', on_delete=models.CASCADE, related_name='saving_branch', null=False, blank=False)
    file = models.FileField(upload_to='savings', null=False, blank=False)
    class Meta:
        db_table = 'saving_upload'
        verbose_name = 'Saving upload'
        verbose_name_plural = 'Saving uploads'

    def __str__(self):
        return str(self.narration)

class AutomatedCharge(models.Model):
    created_at = models.DateField(auto_now_add=True)
    added_transactions = models.TextField()
    charge_record = models.UUIDField(unique=True, default=uuid.uuid4)

    def __str__(self):
        return str(self.charge_record)


class PublicHoliday(models.Model):
    class MonthChoices(models.IntegerChoices):
        JANUARY = 1, 'January'
        FEBRUARY = 2, 'February'
        MARCH = 3, 'March'
        APRIL = 4, 'April'
        MAY = 5, 'May'
        JUNE = 6, 'June'
        JULY = 7, 'July'
        AUGUST = 8, 'August'
        SEPTEMBER = 9, 'September'
        OCTOBER = 10, 'October'
        NOVEMBER = 11, 'November'
        DECEMBER = 12, 'December'

    name = models.CharField(max_length=50, null=True)
    day = models.IntegerField(validators=[MinValueValidator(1),MaxValueValidator(31)], null=True)
    month = models.IntegerField(choices=MonthChoices.choices, null=True)
    branch = models.ForeignKey('accounts.Branch', on_delete=models.CASCADE, null=True, blank=True)

    class Meta:
        db_table = 'public_holiday'

    @property
    def month_str(self):
        if self.month == self.MonthChoices.JANUARY:
            return "January"

        if self.month == self.MonthChoices.FEBRUARY:
            return "February"

        if self.month == self.MonthChoices.MARCH:
            return "March"

        if self.month == self.MonthChoices.APRIL:
            return "April"

        if self.month == self.MonthChoices.MAY:
            return "May"

        if self.month == self.MonthChoices.JUNE:
            return "June"

        if self.month == self.MonthChoices.JULY:
            return "July"

        if self.month == self.MonthChoices.AUGUST:
            return "August"

        if self.month == self.MonthChoices.SEPTEMBER:
            return "September"

        if self.month == self.MonthChoices.OCTOBER:
            return "October"

        if self.month == self.MonthChoices.NOVEMBER:
            return "November"

        if self.month == self.MonthChoices.DECEMBER:
            return "December"


    def __str__(self):
        return str(self.name) + " - " + str(date)


class SavingsSharesRecalculate(models.Model):
    UPDATE_CHOICES = (
        ('shares', 'shares'),
        ('savings', 'savings'),
    )

    created_at = models.DateField(auto_now_add=True)
    branch = models.ForeignKey('accounts.Branch', on_delete=models.CASCADE, null=True, blank=True)
    update_type = models.CharField(choices=UPDATE_CHOICES, max_length=20)


    def __str__(self):
        return str(self.update_type) + '--' + str(self.branch)
