import datetime
from decimal import Decimal


import pandas as pd
from io import BytesIO

import pandas as pd
import xlwt
from django.db.models import Q, Sum, Case, When, Value, ExpressionWrapper, Func, F, CharField, Count, IntegerField, \
    FloatField, OuterRef, Subquery
from django.db.models.functions import Coalesce
from django.http import JsonResponse, HttpResponse
from django.shortcuts import get_object_or_404
from django.utils import timezone
from django.utils.dateparse import parse_date
from openpyxl.styles import Alignment

from openpyxl import Workbook
from openpyxl.styles import PatternFill, Font
from openpyxl.utils import get_column_letter

from accounts.models import Branch
from loans.models import ApplicationAccountOrLoanType, Loans
from loans.utils.loan_details import get_business_loans_context_data
from sacco.models import Member, TransactionCharge, AccountBroker, GroupMember, OtherSettings
from sacco.utils import biz_data, businessdata, biz_staff_branch, session_business_data, determine_business_id_list
from transactions.models import Transactions, SharesTransactions
from transactions.reports import today

style = xlwt.easyxf('font: color-index black, bold on; pattern: pattern solid, fore_colour white; align: wrap on,'
                    'vert centre, horiz center; borders: top_color black, bottom_color black,right_color black, '
                    'left_color black, left thin, right thin, top thin, bottom thin;')

style0 = xlwt.easyxf(
    'font: color-index white, bold on; pattern: pattern solid, fore_colour black; align: wrap off,vert centre, '
    'horiz center; borders: top_color black, bottom_color black,right_color black, '
    'left_color black, left thin, right thin, top thin, bottom thin')

style1 = xlwt.easyxf(
    'font: color-index black; pattern: fore_colour white; align: wrap off; borders: top_color black, bottom_color '
    'black,right_color black, left_color black, left thin, right thin, top thin, bottom thin')


def check_charge_range(request):
    if request.POST:
        account_type = request.POST.get('account_type')
        if account_type:
            try:
                # charges = TransactionCharge.objects.filter(account_type_id=account_type, charge_type='d',
                #                                            status=True).aggregate()
                charges = list(ApplicationAccountOrLoanType.objects.filter(account_type_id=account_type,
                                                                           general_charge__application='r'
                                                                           ).aggregate(
                    total=Sum('general_charge__amount')).values())[0]

                return JsonResponse({'msg': 'success', 'charge': charges})
            except Exception as e:
                # print(str(e))
                charges = 'Not set'
                return JsonResponse({'msg': 'success', 'charge': charges})
        else:
            return JsonResponse({'msg': 'Transaction type required'})

    return JsonResponse({'msg': 'error'})


def check_charge_sum(request):
    if request.POST:
        account_type = request.POST.get('account_type')
        initial = request.POST.get('initial')

        if account_type and initial:
            try:
                try:
                    depo_charges = list(TransactionCharge.objects.filter(account_type_id=account_type, charge_type='d',
                                                                         status=True).aggregate(
                        total=Sum('charge')).values())[0]
                    if depo_charges is None:
                        depo_charges = 0
                except IndexError:
                    depo_charges = 0
                except (Exception,) as e:
                    depo_charges = 0
                try:
                    gen_charges = list(ApplicationAccountOrLoanType.objects.filter(account_type_id=account_type,
                                                                                   general_charge__application='r'
                                                                                   ).aggregate(
                        total=Sum('general_charge__amount')).values())[0]
                    if gen_charges is None:
                        gen_charges = 0
                except IndexError:
                    gen_charges = 0
                except (Exception,) as e:
                    gen_charges = 0

                total_charges = depo_charges + gen_charges

                return JsonResponse({'msg': 'success', 'charge': total_charges})
            except (Exception,) as e:
                charges = 'Not set'
                return JsonResponse({'msg': 'success', 'charge': charges})
        else:
            return JsonResponse({'msg': 'Transaction type required'})

    return JsonResponse({'msg': 'error'})


def export_members(request):
    response = HttpResponse(content_type='application/ms-excel')

    # --------------------------------------------------------------------------------
    business_context = session_business_data(request)
    data_context = determine_business_id_list(request.user.staff, business_context)

    the_current_branch_name = ''
    if business_context is None:
        the_current_branch_name = 'all branches'
    else:
        # get the current branch
        the_current_branch_name = Branch.objects.filter(business=business_context).first().name

    the_filter_branch = ''
    if business_context is None:
        business_filter = request.GET.get('business_filter', '')
        if business_filter:
            if business_filter != 'all_branches':
                # get the branch
                the_filter_branch = Branch.objects.filter(business_id=business_filter).first().name
                # update the data context businesses
                data_context['all_ids'] = [business_filter]
            else:
                the_filter_branch = 'all branches'

    if data_context['title'] == 'all branches':
        title = 'Members for all branches'
    else:
        title = f'Members for {the_current_branch_name}'

    # get the start date and end date
    start_date_str = request.GET.get('start_date', None)
    end_date_str = request.GET.get('end_date', None)
    member_name = request.GET.get('member_name', '')
    # ----------------------------------------------------------------------------------


    biz = biz_data(request.user)
    if biz.type == 1:
        client = "Members"
    else:
        client = "Clients"
    d4 = today.strftime("%b-%d-%Y")
    file_name = f'{title} as of {d4}'
    response['Content-Disposition'] = 'attachment; filename="{}.xls"'.format(file_name)
    sheet_name = f'{biz.short_name} {client} Data'
    wb = xlwt.Workbook(encoding='utf-8')
    ws = wb.add_sheet(sheet_name)

    # Sheet header, first row
    row_num = 2

    columns = ['Name', 'Account No.', 'Gender', 'NIN', 'DOB', 'Marital Status', 'Contact', 'Other Contact', 'email',
               'Date Joined',
               'Nationality', 'Branch']

    the_len = len(columns)
    heading = f'{biz.name} {client}'
    if start_date_str is not None or end_date_str is not None:
        heading = f'{heading} registered'
        if start_date_str is not None and start_date_str !='':
            heading = f'{heading} from {start_date_str}'
        if end_date_str is not None and end_date_str !='':
            heading = f'{heading} up to {end_date_str}'
    if member_name:
        heading = f'{heading} with names containing {member_name}'


    ws.write_merge(0, 1, 0, the_len - 1, '{}'.format(heading.upper()), style)
    for col_num in range(len(columns)):
        ws.write(row_num, col_num, columns[col_num], style0)
        ws.row(2).height_mismatch = True
        ws.row(2).height = 256 * 2

    # get the start date and end date
    # start_date_str = request.GET.get('start_date', None)
    # end_date_str = request.GET.get('end_date', None)
    # member_name = request.GET.get('member_name', None)
    start_date = parse_date(start_date_str) if start_date_str else None
    end_date = parse_date(end_date_str) if end_date_str else None

    # font_style = xlwt.XFStyle()
    members = Member.objects.filter(biodata__business_id__in=data_context['all_ids'], is_active=True, is_group=False)

    if member_name:
        members = members.filter(biodata__name__icontains=member_name)

    if start_date:
        members = members.filter(date_joined__gte=start_date)
    if end_date:
        members = members.filter(date_joined__lte=end_date)

    members_rows = members.annotate(
        marital_s=Case(
            When(biodata__marital_status=1, then=Value('Single')),
            When(biodata__marital_status=2, then=Value('Married')),
            When(biodata__marital_status=3, then=Value('Divorced')),
            When(biodata__marital_status=4, then=Value('Cohabitation')),
            When(biodata__marital_status=5, then=Value('Widowed'))),
        cl=ExpressionWrapper(
            Func(F('date_joined'), Value('%d/%m/%Y'), function='DATE_FORMAT'),
            output_field=CharField()),
        bd=ExpressionWrapper(
            Func(F('biodata__dob'), Value('%d/%m/%Y'), function='DATE_FORMAT'),
            output_field=CharField()),
    ).values_list(
        'biodata__name', 'accounts__acc_number', 'biodata__gender', 'biodata__nin', 'bd', 'marital_s',
        'biodata__contact',
        'biodata__other_contact', 'biodata__email', 'cl', 'biodata__country', 'branch__name'
    )
    for members_row in members_rows:
        row_num += 1
        for col_num in range(len(members_row)):
            ws.write(row_num, col_num, members_row[col_num], style1)

    wb.save(response)
    return response


def export_groups(request):
    response = HttpResponse(content_type='application/ms-excel')
    biz = biz_data(request.user)
    # ------------------------------------------------------------
    business_context = session_business_data(request)
    data_context = determine_business_id_list(request.user.staff, business_context)

    the_current_branch_name = ''
    if business_context is None:
        the_current_branch_name = 'all branches'
    else:
        # get the current branch
        the_current_branch_name = Branch.objects.filter(business=business_context).first().name

    the_filter_branch = ''
    business_filter = ''
    if business_context is None:
        business_filter = request.GET.get('business_filter', '')
        if business_filter:
            if business_filter != 'all_branches':
                # get the branch
                the_filter_branch = Branch.objects.filter(business_id=business_filter).first().name
                # update the data context businesses
                data_context['all_ids'] = [business_filter]
            else:
                the_filter_branch = 'all branches'

    if data_context['title'] == 'all branches':
        title = 'Groups for all branches'
    else:
        title = f'Groups for {the_current_branch_name}'
    # ----------------------------------------------------------
    # title = 'groups'
    response['Content-Disposition'] = 'attachment; filename="{}.xls"'.format(biz.short_name + f' {title}')
    wb = xlwt.Workbook(encoding='utf-8')
    ws = wb.add_sheet('SACCO Group Data')

    # Sheet header, first row
    row_num = 2

    columns = ['Name', 'Members', 'Accounts', 'Date Incorporated', 'Signatories', 'Contact', 'Other Contact', 'email',
               'Date Added', 'Branch']

    the_len = len(columns)

    group_name = request.GET.get('group_name', '')
    the_excel_title = f'{biz.short_name} {title}'
    if group_name:
        the_excel_title = f'{the_excel_title} - with group name containing {group_name}'
    if business_filter:
        the_excel_title = f'{the_excel_title} for {the_filter_branch}'

    ws.write_merge(0, 1, 0, the_len - 1, the_excel_title, style)
    for col_num in range(len(columns)):
        ws.write(row_num, col_num, columns[col_num], style0)
        ws.row(2).height_mismatch = True
        ws.row(2).height = 256 * 2


    # font_style = xlwt.XFStyle()
    groups = Member.objects.filter(branch__business__in=data_context['all_ids'], is_active=True, is_group=True)
    if group_name:
        groups = groups.filter(name__icontains=group_name)
    rows = groups.annotate(
        mems=Count('members'),
        accts=Count('accounts'),
        sign_count=Count(
            Case(
                When(members__is_signatory=True, then=1),
                output_field=IntegerField(),
            )

        ),
        created_on=ExpressionWrapper(
            Func(F('created_date'), Value('%d/%m/%Y'), function='DATE_FORMAT'),
            output_field=CharField()),
        inc=ExpressionWrapper(
            Func(F('biodata__dob'), Value('%d/%m/%Y'), function='DATE_FORMAT'),
            output_field=CharField()),
    ).values_list(
        'name', 'mems', 'accts', 'inc', 'sign_count', 'biodata__contact',
        'biodata__other_contact', 'biodata__email', 'created_on', 'branch__name'
    )
    for row in rows:
        row_num += 1
        for col_num in range(len(row)):
            ws.write(row_num, col_num, row[col_num], style1)

    wb.save(response)
    return response


def export_savings(request):
    response = HttpResponse(content_type='application/ms-excel')
    business_context = get_business_loans_context_data(request)
    business_filter = business_context['business_filter']

    biz = biz_data(request.user)
    branch = biz_staff_branch(request.user)
    date_input = request.GET.get('date')
    from_date = request.GET.get('from_date')
    to_date = request.GET.get('to_date')
    if date_input:
        date_input = datetime.datetime.strptime(date_input, "%Y-%m-%d").date()
    if from_date and to_date:
        from_date = datetime.datetime.strptime(from_date, "%Y-%m-%d").date()
        to_date = datetime.datetime.strptime(to_date, "%Y-%m-%d").date()
    if biz.type == 1:
        client = "Members"
    else:
        client = "Clients"
    today_ = datetime.datetime.today().date()
    savings_date = date_input if date_input is not None else today_
    # x = datetime.datetime.strptime(savings_date, "%b-%d-%Y").date()
    d1 = savings_date.strftime("%b-%d-%Y")
    if from_date and to_date:
        d1 = f'From {from_date} to {to_date}'
        savings_date = d1
    file_name = f'{biz.short_name} {client} Savings Report - {d1}'
    response['Content-Disposition'] = 'attachment; filename="{}.xls"'.format(file_name.capitalize())
    sheet_name = f'{biz.short_name} {client} Savings'
    wb = xlwt.Workbook(encoding='utf-8')
    ws = wb.add_sheet(sheet_name.capitalize(), cell_overwrite_ok=True)

    # Sheet header, first row
    row_num = 2

    columns = ['Name', 'Account Number', 'Saving Product', 'Balance']

    the_len = len(columns)
    heading = f'{biz.name} {client} Savings Report as of {savings_date}'
    ws.write_merge(0, 1, 0, the_len - 1, '{}'.format(heading.upper()), style)
    for col_num in range(len(columns)):
        ws.write(row_num, col_num, columns[col_num], style0)
        ws.row(2).height_mismatch = True
        ws.row(2).height = 256 * 2
    biz_savings = AccountBroker.objects.filter(members__branch__business_id__in=business_context['data_context'], members__is_active=True)
    accounts = []
    for member in biz_savings:
        branch_trans = Transactions.objects.filter(branch__business_id__in=business_context['data_context'])
        if date_input:
            branch_trans = Transactions.objects.filter(branch__business_id__in=business_context['data_context'], tx_date__lte=date_input)
        if from_date and to_date:
            branch_trans = Transactions.objects.filter(branch__business_id__in=business_context['data_context'], tx_date__range=[from_date, to_date])
        debits = branch_trans.filter(account_dr__member=member.the_account
                                     ).values('account_dr').aggregate(
            debits=Coalesce(Sum('reporting_amount'), 0.0, output_field=FloatField()))['debits']
        credits_ = branch_trans.filter(account_cr__member=member.the_account
                                       ).values('account_cr').aggregate(
            credits=Coalesce(Sum('reporting_amount'), 0.0, output_field=FloatField()))['credits']

        accounts.append(
            {
                'Name': member.members.biodata.name,
                'Account Number': member.the_account.acc_number,
                'Saving Product': member.the_account.account_type.name,
                'Balance': (credits_ if not None else 0) - (debits if not None else 0)
            }
        )
    columns = list(accounts[0].keys())

    # write columns, start from row 1
    for i, row in enumerate(accounts, 4):
        for j, col in enumerate(columns):
            ws.write(i, j, row[col])

    wb.save(response)
    return response


#
# def export_group_members(request, pk):
#     try:
#         group = Member.objects.filter(is_group=True).prefetch_related('members').get(id=pk)
#     except Member.DoesNotExist:
#         return render(request, 'errors/404.html', status=404)
#     except Member.MultipleObjectsReturned:
#         group = Member.objects.filter(is_group=True, is_active=True, id=pk).prefetch_related('members').first()
#     except Exception:
#         return render(request, 'errors/500.html', status=500)
#
#     response = HttpResponse(content_type='application/ms-excel')
#     biz = biz_data(request.user)
#     if biz.type == 1:
#         client = "Members"
#     else:
#         client = "Clients"
#     nama = CustomBusiness.objects.filter(business_module='NAMAMYINGO', business=businessdata(request)).first()
#
#     d4 = today.strftime("%b-%d-%Y")
#     file_name = f'{group.name} {client} - {d4}'
#     response['Content-Disposition'] = 'attachment; filename="{}.xls"'.format(file_name)
#     sheet_name = f'{group.name} {client} Data'
#     wb = xlwt.Workbook(encoding='utf-8')
#     ws = wb.add_sheet('All_members')
#
#     # Sheet header, first row
#     row_num = 2
#
#     columns = ['Name', 'Account No.', 'Gender', 'NIN', 'DOB', 'Marital Status', 'Contact', 'Other Contact', 'email', 'Date Joined',
#                'Nationality']
#     if nama:
#         print('yea')
#         columns.append('Loan Status')
#
#     the_len = len(columns)
#     heading = f'{biz.name} {client}'
#     ws.write_merge(0, 1, 0, the_len - 1, '{}'.format(heading.upper()), style)
#     for col_num in range(len(columns)):
#         ws.write(row_num, col_num, columns[col_num], style0)
#         ws.row(2).height_mismatch = True
#         ws.row(2).height = 256 * 2
#
#
#     members = Member.objects.filter(groups__group=group).order_by('-groups__is_signatory')
#     deloan_subquery = Loans.objects.filter(applicant=OuterRef('pk')).filter(loan_status=3).annotate(cc=Count(F('guarantors__group'))).values('cc')[:1]
#
#     # font_style = xlwt.XFStyle()
#     if nama:
#         members = Member.objects.filter(groups__group=group).order_by('-groups__is_signatory').annotate(df=Subquery(deloan_subquery))
#
#         # Now you can access the annotated field 'df' in each member
#         for member in members:
#             print(f"Member {member.id} has {member.df} loans.")
#     # print(me)
#
#     if 'search_member' in request.session:
#         search_text = request.session.get('search_member')
#         members = Member.objects.filter(biodata__name__icontains=search_text, is_active=True,
#                                        is_group=False)
#     members_rows = members.annotate(
#         marital_s=Case(
#             When(biodata__marital_status=1, then=Value('Single')),
#             When(biodata__marital_status=2, then=Value('Married')),
#             When(biodata__marital_status=3, then=Value('Divorced')),
#             When(biodata__marital_status=4, then=Value('Cohabitation')),
#             When(biodata__marital_status=5, then=Value('Widowed'))),
#         cl=ExpressionWrapper(
#             Func(F('date_joined'), Value('%d/%m/%Y'), function='DATE_FORMAT'),
#             output_field=CharField()),
#         bd=ExpressionWrapper(
#             Func(F('biodata__dob'), Value('%d/%m/%Y'), function='DATE_FORMAT'),
#             output_field=CharField()),
#     ).values_list(
#         'biodata__name', 'accounts__acc_number', 'biodata__gender', 'biodata__nin', 'bd','marital_s',
#         'biodata__contact', 'biodata__other_contact', 'biodata__email', 'cl', 'biodata__country',
#     )
#     print(members.values())
#     for members_row in members_rows:
#         row_num += 1
#         for col_num in range(len(members_row)):
#             ws.write(row_num, col_num, members_row[col_num], style1)
#
#     wb.save(response)
#     return response


def export_group_members(request, pk):
    """
    Export group members' information to Excel file using pandas
    Args:
        request: HTTP request object
        pk: Primary key of the group
    Returns:
        HttpResponse with Excel file attachment
    """

    # Define marital status mapping
    MARITAL_STATUS = {
        1: 'Single',
        2: 'Married',
        3: 'Divorced',
        4: 'Cohabitation',
        5: 'Widowed'
    }

    # Get the group and its members
    group = get_object_or_404(Member, pk=pk, is_group=True)
    group_members = GroupMember.objects.filter(group=group).select_related(
        'member',
        'member__biodata'
    ).prefetch_related(
        'member__accounts',
        'member__accounts__account_type',
        'member__members'  # This is the related name for loans
    )

    member_ids = list(group_members.values_list('member_id', flat=True))
    # print('member_ids', member_ids)

    # update running loans balances
    # all_running_loans_for_members = Loans.objects.filter(applicant_id__in=member_ids, loan_status=3)
    #
    # if all_running_loans_for_members.exists():
    #     for ln in all_running_loans_for_members:
    #         update_loan_balance_field(ln.id, ln.branch.business.id, ln.branch.id)
    #         # TODO: improve this part - REASON - This only works for small data

    # Prepare data for DataFrame
    data = []
    for group_member in group_members:
        member = group_member.member
        biodata = member.biodata

        # Get member's accounts
        member_accounts = member.accounts.all()
        account_balance = sum(acc.balance or 0 for acc in member_accounts)
        account_types = ', '.join(acc.account_type.name if acc.account_type else 'N/A'
                                  for acc in member_accounts)

        # Get loan information
        loans = member.applicant.all()
        total_loans = loans.count()
        active_loans = loans.filter(
            loan_status__in=[
                # Loans.LOAN_STATUS.APPROVED,
                Loans.LOAN_STATUS.DISBURSED
            ]
        )
        total_loans = active_loans.count()

        loan_status = 'No Active Loans' if not active_loans.exists() else 'Has Active Loans'

        loan_totals = active_loans.aggregate(
            total_principal=Sum('amount_approved'),
            total_balance=Sum('balance'),
            total_principal_balance=Sum('principal_balance'),
            total_interest_balance=Sum('interest_balance'),
        )

        # Get marital status display value
        marital_status = 'N/A'
        if biodata and biodata.marital_status:
            try:
                marital_status = MARITAL_STATUS.get(int(biodata.marital_status), 'N/A')
            except (ValueError, TypeError):
                marital_status = 'N/A'

        data.append({
            'Name': biodata.name if biodata else member.name or 'N/A',
            'Account No.': next((acc.acc_number for acc in member_accounts), 'N/A'),
            'Gender': biodata.gender if biodata else 'N/A',
            'NIN': biodata.nin if biodata else 'N/A',
            'DOB': biodata.dob if biodata and biodata.dob else None,
            'Marital Status': marital_status,
            'Contact': biodata.contact or 'N/A',
            'Other Contact': biodata.other_contact or 'N/A',
            'Email': biodata.email or 'N/A',
            'Date Joined': biodata.date_created or 'N/A',
            'Nationality': biodata.country.name if biodata else 'N/A',
            'Total No.of Loans(Running)': total_loans,
            'Account Balance': account_balance,
            'Account Type': account_types,
            'Total Shares of member': member.shares or 0,
            'Loan Status': loan_status,
            'Total Principal': loan_totals['total_principal'] or 0,
            'Total Loan Balance': loan_totals['total_balance'] or 0,
            'Total Principal Balance': loan_totals['total_principal_balance'] or 0,
            'Total Interest Balance': loan_totals['total_interest_balance'] or 0,
        })

    # Create DataFrame
    df = pd.DataFrame(data)

    # Calculate totals
    totals = {
        'Name': 'TOTALS',
        'Account No.': '',
        'DOB': '',
        'Marital Status': '',
        'Email': '',
        'Nationality': '',
        'Gender': '',
        'Account Balance': df['Account Balance'].sum(),
        'Account Type': '',
        'Total No.of Loans(Running)': df['Total No.of Loans(Running)'].sum(),
        'Total Shares of member': df['Total Shares of member'].sum(),
        'Loan Status': f"Total Active Loans: {len(df[df['Loan Status'] == 'Has Active Loans'])}",
        'Total Principal': df['Total Principal'].sum(),
        'Total Loan Balance': df['Total Loan Balance'].sum(),
        'Total Principal Balance': df['Total Principal Balance'].sum(),
        'Total Interest Balance': df['Total Interest Balance'].sum()
    }

    # Append totals row to DataFrame
    df = pd.concat([df, pd.DataFrame([totals])], ignore_index=True)

    # Format dates
    # date_columns = ['DOB', 'Date Joined']
    # df[date_columns] = df[date_columns].apply(lambda x: x.dt.strftime('%Y-%m-%d'))

    # Format numeric columns
    numeric_columns = ['Account Balance', 'Total Shares of member', 'Total Principal', 'Total Loan Balance',
                       'Total Principal Balance', 'Total Interest Balance']
    df[numeric_columns] = df[numeric_columns].fillna(0).round(2)

    # Create Excel file
    output = BytesIO()
    with pd.ExcelWriter(output, engine='openpyxl') as writer:
        df.to_excel(writer, sheet_name='Group Members', index=False, startrow=1)

        # Get the workbook and worksheet
        workbook = writer.book
        worksheet = writer.sheets['Group Members']

        # Add title row
        group_title = f"Group Members List - {group.name}"
        title_font = Font(size=14, bold=True)
        title_cell = worksheet.cell(row=1, column=1)
        title_cell.value = group_title
        title_cell.alignment = Alignment(horizontal="center")

        # Merge cells for title
        num_columns = len(df.columns)
        worksheet.merge_cells(start_row=1, start_column=1, end_row=1, end_column=num_columns)

        # Auto-adjust columns width
        for idx, column in enumerate(df.columns, start=1):  # start=1 because Excel columns start at 1
            max_length = 0
            column = str(column)

            # Get maximum length of column values
            series = df[column].astype(str)
            max_length = max(
                series.str.len().max(),  # length of values
                len(column)  # length of column name
            )

            # Add a little extra space
            adjusted_width = (max_length + 2)

            # Set column width
            column_letter = get_column_letter(idx)  # A is idx 1
            worksheet.column_dimensions[column_letter].width = min(max_length, 40)  # cap at 40

        # Style header row
        header_row = worksheet.row_dimensions[2]  # Row 2 is now the header row
        for cell in worksheet[2]:  # Row 2 is the header row
            cell.font = Font(bold=True)

        # Style totals row
        totals_row = len(df) + 2  # +2 because of title and header rows
        for cell in worksheet[totals_row]:
            cell.font = Font(bold=True)
            if cell.column_letter in ['J', 'L', 'N', 'O', 'P', 'Q']:  # Columns with numeric totals
                cell.number_format = '#,##0.00'

    # Create response
    output.seek(0)
    response = HttpResponse(
        output.read(),
        content_type='application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
    )
    response['Content-Disposition'] = f'attachment; filename=Group_Members_{group.name}.xlsx'

    return response

class GroupConcat(Func):
    function = 'GROUP_CONCAT'
    template = '%(function)s(DISTINCT %(expressions)s)'
    output_field = CharField()


def export_member_summary(request):
    try:
        business_context = get_business_loans_context_data(request)
        business_filter = business_context['business_filter']

        business = biz_data(request.user)
        business_id = business.id

        the_branch = biz_staff_branch(request.user)

        # Get business share price
        settings = OtherSettings.objects.filter(business_id__in=business_context['data_context'])
        share_price = settings.aggregate(total=Sum('share_price'))['total'] or Decimal('0')
        share_price = share_price / len(business_context['data_context'])

        # Get all members for the business
        # members = Member.objects.filter(biodata__business_id=business_id).select_related('biodata')
        account_numbers_subquery = AccountBroker.objects.filter(
            members=OuterRef('pk')
        ).annotate(
            account_numbers=GroupConcat('the_account__acc_number')
        ).values('account_numbers')[:1]

        members_data = (
            Member.objects.filter(branch__business_id__in=business_context['data_context'], is_active=True)
            .values(
                'id', 'biodata__name', 'name', 'shares'  # Fields to group by
            )
            .annotate(
                total_savings_from_acc=Coalesce(
                    Sum('owners__the_account__balance', distinct=True, output_field=FloatField()),
                    Value(0.0),
                ),
                total_loan_balance=Coalesce(
                    Sum('applicant__balance', filter=Q(applicant__loan_status=3), output_field=FloatField()),
                    Value(0.0),
                ),
                total_principal_balance=Coalesce(
                    Sum('applicant__principal_balance', filter=Q(applicant__loan_status=3),
                        output_field=FloatField()),
                    Value(0.0),
                ),
                total_interest_balance=Coalesce(
                    Sum('applicant__interest_balance', filter=Q(applicant__loan_status=3),
                        output_field=FloatField()),
                    Value(0.0),
                ),
                total_shares=Coalesce(F('shares'), Value(0.0), output_field=FloatField()),
                total_share_value=Coalesce(
                    F('shares') * Value(share_price), Value(0.0), output_field=FloatField()
                ),
                mem_name=Case(
                    When(is_group=True, then=F('name')),  # Use 'name' if is_group=True
                    default=F('biodata__name'),  # Otherwise, use 'biodata__name'
                    output_field=CharField(),
                ),
                account_numbers=Subquery(account_numbers_subquery),
            )
        ).values(
            'total_savings_from_acc',
            'total_loan_balance',
            'total_principal_balance',
            'total_interest_balance',
            'total_shares',
            'total_share_value',
            'mem_name',
            'account_numbers'
        )

        # Initialize lists to store data
        # data = []



        # Add to data list
        # data.append({
        #     'Name': member.biodata.name or member.name,
        #     'Savings Balance': savings_balance,
        #     'Principal Balance': principal_balance,
        #     'Interest Balance': interest_balance,
        #     'Total Shares': total_shares,
        #     'Shares Amount': shares_amount
        # })

        # Create DataFrame
        df = pd.DataFrame(members_data)
        df = df.rename(columns={
            'mem_name': 'Name',
            'account_numbers': 'Account number',
            'total_savings_from_acc': 'Savings Balance',
            'total_principal_balance': 'Principal Balance',
            'total_interest_balance': 'Interest Balance',
            'total_loan_balance': 'Loan Balance',
            'total_shares': 'Total Shares',
            'total_share_value': 'Shares Amount'

        })
        # Reorganize columns
        df = df[['Name', 'Account number', 'Savings Balance', 'Principal Balance', 'Interest Balance', 'Loan Balance', 'Total Shares', 'Shares Amount']]

        # Calculate totals
        totals = pd.Series({
            'Name': 'TOTALS',
            'Account number': '',
            'Savings Balance': df['Savings Balance'].sum(),
            'Principal Balance': df['Principal Balance'].sum(),
            'Interest Balance': df['Interest Balance'].sum(),
            'Loan Balance': df['Loan Balance'].sum(),
            'Total Shares': df['Total Shares'].sum(),
            'Shares Amount': df['Shares Amount'].sum()
        })

        # Append totals to DataFrame
        df = pd.concat([df, pd.DataFrame([totals])], ignore_index=True)

        # Format numbers
        # numeric_columns = [
        #     'Savings Balance', 'Principal Balance', 'Interest Balance',
        #     'Total Shares', 'Shares Amount'
        # ]
        #
        # for col in numeric_columns:
        #     df[col] = df[col].apply(lambda x: '{:,.2f}'.format(float(x)))

        # Create a new workbook and select the active sheet
        wb = Workbook()
        ws = wb.active
        ws.title = 'Members Summary Report'

        # Write headers
        headers = list(df.columns)
        for col, header in enumerate(headers, 1):
            cell = ws.cell(row=1, column=col, value=header)
            # Add header formatting
            cell.font = Font(bold=True)
            cell.fill = PatternFill(start_color='F0F0F0', end_color='F0F0F0', fill_type='solid')

        # Write data
        for row_idx, row in enumerate(df.values, 2):
            for col_idx, value in enumerate(row, 1):
                ws.cell(row=row_idx, column=col_idx, value=value)

        # Adjust column widths
        for col_idx, column in enumerate(df.columns, 1):
            max_length = max(
                df[column].astype(str).apply(len).max(),
                len(str(column))
            )
            ws.column_dimensions[get_column_letter(col_idx)].width = max_length + 2

        # Create response
        excel_buffer = BytesIO()
        wb.save(excel_buffer)
        excel_buffer.seek(0)

        # Create the HttpResponse object
        response = HttpResponse(
            excel_buffer.getvalue(),
            content_type='application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
        )

        response['Content-Disposition'] = f'attachment; filename="Members_Summary_Report_{datetime.datetime.now().strftime("%Y%m%d_%H%M%S")}.xlsx"'

        return response

    except Exception as exception:
        raise exception



def export_daily_member_report(request):
    try:
        business_context = get_business_loans_context_data(request)
        business_filter = business_context['business_filter']
        # Get today's date
        today = timezone.now().date()

        date_type = request.GET.get("date_type", None)

        single_date = request.GET.get("single_date", None)

        start_date = request.GET.get("start_date", None)

        end_date = request.GET.get("end_date", None)

        # base transactions query
        query = Q(Q(transaction_type='Savings') | Q(transaction_type='savings'))

        if single_date:
            query &= Q(tx_date=single_date)
        else:
            # default to today
            query &= Q(tx_date=today)

        if start_date:
            query &= Q(tx_date__gte=start_date)
        if end_date:
            query &= Q(tx_date__lte=end_date)

        # Get total savings for
        today_savings = Transactions.objects.filter(branch__business_id__in=business_context['data_context']).filter(query).annotate(
            account_number=Case(
                When(account_cr__category__name="Members", then=F("account_cr__member__acc_number"))
            ),
            amount=F('reporting_amount')
        ).values('account_number', 'amount')

        # base share transactions query
        share_transactions_query = Q()

        if single_date:
            share_transactions_query &= Q(date=single_date)
        else:
            # default to today
            share_transactions_query &= Q(date=today)

        if start_date:
            share_transactions_query &= Q(date__gte=start_date)
        if end_date:
            share_transactions_query &= Q(date__lte=end_date)

        # Get total shares bought today
        today_shares = SharesTransactions.objects.filter(branch__business_id__in=business_context['data_context']).filter(share_transactions_query).annotate(
            name=F('buyer__biodata__name'),
        ).values('name', 'shares')

        # Get new members data
        # base member query
        member_query = Q()

        if single_date:
            member_query &= Q(date_joined=single_date)
        else:
            # default to today
            member_query &= Q(date_joined=today)

        if start_date:
            member_query &= Q(date_joined__gte=start_date)
        if end_date:
            member_query &= Q(date_joined__lte=end_date)

        new_members = Member.objects.filter(branch__business_id__in=business_context['data_context']).filter(member_query).select_related('biodata').values(
            'biodata__name',
            'contact',
            'date_joined'
        )

        # Create DataFrames
        savings_df = pd.DataFrame(list(today_savings))
        shares_df = pd.DataFrame(list(today_shares))
        members_df = pd.DataFrame(list(new_members))

        # Create Excel file in memory
        output = BytesIO()

        # Create Excel writer
        with pd.ExcelWriter(output, engine='xlsxwriter') as writer:
            workbook = writer.book

            # Format for totals
            total_format = workbook.add_format({
                'bold': True,
                'border': 1,
                'bg_color': '#D3D3D3'
            })

            # Savings Sheet
            if not savings_df.empty:
                savings_df.to_excel(writer, sheet_name='Savings', index=False)
                worksheet = writer.sheets['Savings']

                # Add total row
                total_row = len(savings_df) + 1
                worksheet.write(total_row, 0, 'Total', total_format)
                worksheet.write_formula(
                    total_row, 1,
                    f'=SUM(B2:B{total_row})',
                    total_format
                )

            # Shares Sheet
            if not shares_df.empty:
                shares_df.to_excel(writer, sheet_name='Shares', index=False)
                worksheet = writer.sheets['Shares']

                # Add total row
                total_row = len(shares_df) + 1
                worksheet.write(total_row, 0, 'Total', total_format)
                worksheet.write_formula(
                    total_row, 1,
                    f'=SUM(B2:B{total_row})',
                    total_format,
                )
                worksheet.write_formula(
                    total_row, 2,
                    f'=SUM(C2:C{total_row})',
                    total_format,
                )

            # Members Sheet
            if not members_df.empty:
                members_df.to_excel(writer, sheet_name='New Members', index=False)
                worksheet = writer.sheets['New Members']

                # Add total count
                total_row = len(members_df) + 1
                worksheet.write(total_row, 0, f'Total New Members: {len(members_df)}', total_format)

            # Set column widths
            for sheet in writer.sheets.values():
                sheet.set_column(0, 0, 20)  # Column A
                sheet.set_column(1, 1, 15)  # Column B
                sheet.set_column(2, 2, 15)  # Column C

        # Prepare response
        output.seek(0)
        response = HttpResponse(
            output.read(),
            content_type='application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
        )
        response['Content-Disposition'] = f'attachment; filename=daily_report_{today}.xlsx'
        return response

    except Exception as exception:
        # Handle errors appropriately
        raise exception
        # return HttpResponse(f"An error occurred: {str(e)}", status=500)