import datetime
import random
import traceback

import pandas as pd
from django.db import transaction
from django.db.models import Q, Sum, Case, When, F, FloatField, ExpressionWrapper
from django.utils import timezone
from django.utils.translation import gettext as _

from django.core.exceptions import ValidationError

from django import forms
from django_countries.fields import CountryField
from django.shortcuts import get_object_or_404

from accounts.models import Business, Biodata, Staff, Branch
from commons.logger import logger
from loans.models import ApplicationAccountOrLoanType, LoanTypes, Loans
from transactions.models import Account, AccountCategory, Transactions, SharesTransactions
from transactions.reports import this_branch
from utils.general import random_with_N_digits, proper_dial
from .constants import CREATE, ACCOUNT, SMSCHARGE, UPDATE
from .models import Member, AccountBroker, MemberAccount, TransactionCharge, GroupMember, \
    OtherSettings, SaccoSmsSettings, BusinessShares, CurrencySetting, NotiSettings, ActivityLog, \
    SMSCost, AccountTypes
from .utils import biz_staff, biz_data, biz_staff_branch, checkAndSendMessage, biz_id, sendTransEmail, gen_account, \
    branch_id, check_account, check_charges


class AddMemberForm(forms.Form):
    error_messages = {
        'account_exists': _('Account already exists'),
    }
    title = forms.CharField(required=False)
    name = forms.CharField(required=False)
    gender = forms.CharField(required=False)
    marital_status = forms.CharField(required=False)
    upload_date = forms.DateField(required=False)
    dob = forms.DateField(required=False,
                          widget=forms.SelectDateWidget(
                              attrs={'class': 'form-control',
                                     'style': 'width:33.33333333333333%; ' 'display:inline' '-block;'},
                              years=range(1930, datetime.date.today().year + 1)))

    nin = forms.CharField(required=False)
    contact = forms.CharField(required=False)
    other_contact = forms.CharField(required=False)
    email = forms.EmailField(required=False)
    nok = forms.CharField(required=False)
    nok_contacts = forms.CharField(required=False)
    account_type = forms.CharField(required=False)
    member_type = forms.CharField(required=False)
    share_select = forms.CharField(required=False)
    initial_shares = forms.CharField(required=False)
    the_account = forms.CharField(required=False)
    code = CountryField().formfield(required=False)
    code1 = CountryField().formfield(required=False)
    code2 = CountryField().formfield(required=False)
    mm = forms.CharField(required=False)
    country = CountryField().formfield(required=False)
    # initial = forms.IntegerField(required=False)
    location = forms.CharField(widget=forms.TextInput, required=False)
    account_number = forms.CharField(widget=forms.TextInput, required=False)

    def __init__(self, *args, **kwargs):
        self.user = kwargs.pop('user', None)
        self.finance_year = kwargs.pop('finance_year', None)
        self.initial_depo = kwargs.pop('initial', None)
        super(AddMemberForm, self).__init__(*args, **kwargs)

    # def clean_account_number(self):

    @transaction.atomic
    def save(self, commit=True):
        cleaned_data = super().clean()
        if commit:
            user = self.user
            initial = self.initial_depo
            financial_year = self.finance_year
            biz = biz_data(user)
            title = cleaned_data.get('title')
            upload_date = cleaned_data.get('upload_date')
            name = cleaned_data.get('name')
            gender = cleaned_data.get('gender')
            marital_status = cleaned_data.get('marital_status')
            dob = cleaned_data.get('dob')
            member_type = cleaned_data.get('member_type')
            share_select = cleaned_data.get('share_select')
            initial_shares = cleaned_data.get('initial_shares')
            nin = cleaned_data.get('nin')
            contact = cleaned_data.get('contact')
            other_contact = cleaned_data.get('other_contact')
            mm = cleaned_data.get('mm')
            email = cleaned_data.get('email')
            nok = cleaned_data.get('nok')
            nok_contacts = cleaned_data.get('nok_contacts')
            account_type = cleaned_data.get('account_type')
            the_account = cleaned_data.get('the_account')
            location = cleaned_data.get('location')
            code = cleaned_data.get('code')
            code1 = cleaned_data.get('code1')
            code2 = cleaned_data.get('code2')
            country = cleaned_data.get('country')
            account_number = cleaned_data.get('account_number')
            bizz_id = biz_id(user)
            this_biz = Business.objects.filter(id=bizz_id).first()
            rec_no = datetime.datetime.now().strftime("%y%m%d%H%M%S")
            ref_no = datetime.datetime.now().strftime("%y%m%d%H%M%S")
            # print(f'the date is {upload_date}')

            if contact:
                primary_c = proper_dial(contact, code)
            else:
                primary_c = contact

            if other_contact:
                other_c = proper_dial(other_contact, code1)
            else:
                other_c = other_contact

            if mm:
                mm_c = proper_dial(mm, code2)
            else:
                mm_c = mm

            try:

                try:
                    bio = Biodata.objects.create(
                        name=name,
                        gender=gender,
                        marital_status=marital_status,
                        dob=dob,
                        nin=nin,
                        contact=primary_c,
                        other_contact=other_c,
                        mobile_money=mm_c,
                        title_id=title,
                        country=country,
                        email=email,
                        nok=nok,
                        location=location,
                        nok_contacts=nok_contacts,
                        created_by=biz_staff(user),
                        business=biz_data(user),
                        date_created=upload_date if upload_date else datetime.date.today()
                    )
                except Exception as e:
                    logger.error(str(e))
                    raise ValidationError(_(str(e)))

                setting_status = OtherSettings.objects.filter(business=biz_data(user)).first()
                share_cost = 0

                if member_type == '2':  # If this is a new member
                    if setting_status.set_ordinary:  # If this is a new member & all member are ordinary on reg
                        member = Member.objects.create(biodata=bio, created_by=biz_staff(user), member_type='o',
                                                       branch=biz_staff_branch(user),
                                                       date_joined=upload_date if upload_date else datetime.date.today())
                    else:  # If this is new but all member must be shareholders on reg
                        if share_select == '1':  # this means is a shareholder
                            share_cost = int(initial_shares) * int(setting_status.share_price)
                            member = Member.objects.create(biodata=bio, created_by=biz_staff(user), member_type='s',
                                                           shares=initial_shares, branch=biz_staff_branch(user))
                        else:  # this means is an ordinary
                            member = Member.objects.create(biodata=bio, created_by=biz_staff(user), member_type='o',
                                                           branch=biz_staff_branch(user))
                elif member_type == '1':  # If this is an existing member
                    if share_select == '1':  # If this is an existing member and a shareholder
                        member = Member.objects.create(biodata=bio, created_by=biz_staff(user), member_type='s',
                                                       shares=initial_shares, branch=biz_staff_branch(user))

                    else:  # If this is an existing member but ordinary
                        member = Member.objects.create(biodata=bio, created_by=biz_staff(user), member_type='o',
                                                       branch=biz_staff_branch(user))

                minimum = 10000000
                maximum = 99999999
                if biz.auto_acc_no:
                    if account_number is None or account_number == '':
                        gen_numbers = random.randint(minimum, maximum)
                        account_number = gen_account(user)
                # print(acc_number)
                account = MemberAccount.objects.create(account_type_id=account_type, acc_number=account_number)
                AccountBroker.objects.create(members=member, the_account=account, business=biz_data(user))
                cat, created = AccountCategory.objects.get_or_create(business=biz_data(user), name="Members",
                                                                     cat_type='liability', dr_cr='cr')
                equity_cat, created = AccountCategory.objects.get_or_create(business=biz_data(user),
                                                                            name="Shares", dr_cr='cr',
                                                                            cat_type='liability')
                # chart of accounts
                member_acc = Account.objects.create(category=cat, business=biz_data(user), member=account,
                                                    added_by=biz_staff(user))
                # fg = member_acc.member.members_account.biodata.name
                if the_account:
                    acc_involved = Account.objects.filter(id=int(the_account), business=biz_data(user)).first()
                else:
                    acc_involved = ''
                opening_acct = Account.objects.get(name='Opening Reserves', business=biz_data(user))
                equity, created = Account.objects.get_or_create(name='Share Capital', business=biz_data(user),
                                                                category=equity_cat)

                charges_acc = Account.objects.get(category__name="Registration Income", business=biz_data(user))
                depo_dict = []
                gen_dict = []

                gen_cgs = ApplicationAccountOrLoanType.objects.filter(account_type_id=account_type,
                                                                      general_charge__application='r')
                gen_charges = 0
                if gen_cgs.exists():
                    gen_dataframe = pd.DataFrame(gen_cgs.values())

                    for gen in gen_cgs:
                        amount = gen.general_charge.amount
                        if gen.general_charge.is_percentage:
                            amount = initial * (gen.general_charge / 100)
                        gen_dict.append(
                            {
                                'id': gen.id,
                                'amount': amount,
                                'rev_status': gen.general_charge.is_revenue
                            }
                        )
                    gen_charges_df = pd.DataFrame(gen_dict)
                    gen_frame = pd.merge(gen_dataframe, gen_charges_df, on='id')

                    # summing general charges dataframe
                    gen_charges = gen_frame['amount'].sum()

                    if member_type == '2':  # If this is a new member
                        for ch in gen_cgs:
                            if ch.general_charge.is_revenue == 'YES':
                                Transactions.objects.create(
                                    branch=biz_staff_branch(user),
                                    transaction_type='Registration',
                                    account_dr=member_acc,
                                    account_cr=charges_acc,
                                    narration="{}".format(ch.general_charge.charge),
                                    reporting_amount=ch.general_charge.amount,
                                    added_by=biz_staff(user),
                                    financial_year=financial_year,
                                    receipt=rec_no + 'c',
                                    reference=ref_no + 'c'
                                )
                            else:
                                charge = ch.general_charge.charge
                                if Account.objects.filter(name=charge).exists():
                                    charge_account = charge
                                else:
                                    reserves = AccountCategory.objects.filter(name='Reserves').first()
                                    charge_account = Account.objects.create(name=ch.general_charge.charge,
                                                                            category_id=reserves.id,
                                                                            business=biz_data(user))
                                Transactions.objects.create(
                                    branch=biz_staff_branch(user),
                                    transaction_type='Registration',
                                    account_dr=member_acc,
                                    account_cr=charge_account,
                                    narration="{}".format(ch.general_charge.charge),
                                    reporting_amount=ch.general_charge.amount,
                                    added_by=biz_staff(user),
                                    financial_year=financial_year,
                                    receipt=rec_no + 'c',
                                    reference=ref_no + 'c'
                                )

                dep_cgs = TransactionCharge.objects.filter(account_type_id=account_type,
                                                           charge_type='d',
                                                           status=True)
                depo_charges = 0
                if dep_cgs.exists():
                    depo_dataframe = pd.DataFrame(dep_cgs.values())
                    deposit_frame = []
                    # deposit charges
                    for d in dep_cgs:
                        if d.account_type.deposit_charge_vary:
                            if d.start <= initial <= d.end:
                                amount = d.charge
                                if d.is_charge_percentage:
                                    amount = initial * (d.charge / 100)
                                depo_dict.append(
                                    {'id': d.id, 'amount': amount, 'account_type_name': d.account_type.name})
                            else:
                                amount = 0
                        else:
                            amount = d.charge
                            if d.is_charge_percentage:
                                amount = initial * (d.charge / 100)
                            depo_dict.append({'id': d.id, 'amount': amount, 'account_type_name': d.account_type.name})
                        depo_data_charges = pd.DataFrame(depo_dict)

                        if not depo_dataframe.empty and not depo_data_charges.empty:
                            deposit_frame = pd.merge(depo_dataframe, depo_data_charges, on='id')
                            if d.account_type.deposit_charge:
                                if d.account_type.deposit_charge_vary:
                                    dfs = deposit_frame[
                                        (deposit_frame['start'] <= initial) & (initial <= deposit_frame['end'])]
                                else:
                                    dfs = deposit_frame
                            else:
                                dfs = pd.DataFrame(columns=['id', 'start', 'end', 'charge', 'account_type_id',
                                                            'charge_type', 'status', 'is_charge_percentage',
                                                            'created_date',
                                                            'deletion_date', 'amount', 'account_type_name'])
                            depo_charges = deposit_frame['amount'].sum()
                            if member_type == '2':
                                if not dfs.empty:
                                    for index, row in dfs.iterrows():
                                        Transactions.objects.create(
                                            branch=biz_staff_branch(user),
                                            transaction_type='Charge',
                                            account_dr=member_acc,
                                            account_cr=charges_acc,
                                            narration="Charge on deposit of " + str(initial),
                                            reporting_amount=row["amount"],
                                            added_by=biz_staff(user),
                                            financial_year=financial_year,
                                            receipt=rec_no + 'c',
                                            reference=ref_no + 'c'
                                        )
                                    break
                        else:
                            depo_charges = 0
                else:
                    depo_charges = 0

                total_charges = 0

                # new member type
                if member_type == '2':  # new member type
                    if not setting_status.set_ordinary:
                        narrative = f'Sale of {initial_shares} shares to {member.biodata.name} at {setting_status.share_price} each.'
                        business_shares = BusinessShares.objects.filter(business=biz_data(user)).first()
                        shares_record = SharesTransactions.objects.create(
                            buyer_id=member.id,
                            shares=initial_shares,
                            date=datetime.datetime.today(),
                            narration=narrative,
                            branch=this_branch(user)
                        )
                        # query buyer shares transactions
                        buyer_shares = SharesTransactions.objects.filter(Q(buyer=member) | Q(seller=member))
                        # print("results = %s" % buyer_shares.count())
                        total_bought = 0
                        total_sold = 0
                        for b in buyer_shares:
                            if b.buyer == member:
                                total_bought += b.shares
                            if b.seller == member:
                                total_sold += b.shares

                        current_shares = total_bought - total_sold

                        # update buyer shares balance
                        member.shares = current_shares
                        member.save()

                        # query seller shares transactions
                        seller_shares = SharesTransactions.objects.filter(seller=None)
                        total_bought = 0
                        total_sold = 0
                        for s in seller_shares:
                            total_sold += s.shares

                        # update business shares sold
                        business_shares.sold = total_sold
                        business_shares.save()

                        # Record a finance transaction
                        # shares account
                        shares_account = Account.objects.filter(business=biz_data(user),
                                                                category__name='Shares').first()
                        # payment account
                        payment_account = Account.objects.filter(business=biz_data(user), id=the_account).first()

                        share_cost = int(setting_status.share_price) * int(initial_shares)
                        total_charges = depo_charges + gen_charges + share_cost
                        Transactions.objects.create(
                            branch=biz_staff_branch(user),
                            transaction_type='Sale of shares',
                            account_dr_id=member_acc.id,
                            account_cr=shares_account,
                            narration=narrative,
                            reporting_amount=share_cost,
                            added_by=biz_staff(user),
                            shares=shares_record,
                            financial_year=financial_year,
                            receipt=rec_no + 's',
                            reference=ref_no + 's'
                        )
                    else:
                        total_charges = depo_charges + gen_charges

                Transactions.objects.create(
                    branch=biz_staff_branch(user),
                    transaction_type="Deposit" if member_type == '2' else 'Opening balance',
                    account_dr=acc_involved if member_type == '2' else opening_acct,
                    account_cr=member_acc,
                    narration="Initial deposit on registration" if member_type == '2' else 'Opening balance',
                    reporting_amount=initial,
                    added_by=biz_staff(user),
                    financial_year=financial_year,
                    receipt=rec_no,
                    reference=ref_no
                )
                member_debits = list(Transactions.objects.filter(account_dr=member_acc).aggregate(
                    total=Sum('reporting_amount')).values())[0]
                if member_debits is None:
                    member_debits = 0
                # print(f'my member debits {member_debits}')
                member_credits = list(Transactions.objects.filter(account_cr=member_acc).aggregate(
                    total=Sum('reporting_amount')).values())[0]
                # print(f'my member credits {member_credits}')
                account.balance = member_credits - member_debits
                account.save()

                if member_type == '2':  # If this is a new member
                    # ==== SEND NOTIFICATIONS ======
                    template_file_link = 'sacco/emails/transactional.html'
                    currency = CurrencySetting.objects.filter(business=biz_data(user)).first()
                    notify_type = NotiSettings.objects.filter(business=biz_data(user)).first()
                    smsSettingsObj = SaccoSmsSettings.objects.get(when_to_send='on new member registration',
                                                                  business=biz_data(user))
                    message = f'Dear Client,\nThis is to inform you that your account in {this_biz.name} has been opened with A/C No: {account.acc_number} & deposit of {currency.currency} {initial} '
                    html_body = f'<p>Dear {bio.name},</p> <p>This is to inform you that your account in {this_biz.name} has been opened with A/C No: {account.acc_number} and an initial deposit of {currency.currency} {initial}. <br>We hope that you have a good experience with our services and <b>{this_biz.name}</b> will do it’s best to please you. We will be looked forward to serving you in adherence to our quality standards.</p><p>Thanking you and with profound regards.</p>'
                    if smsSettingsObj.status:
                        if notify_type.notification_type == 1:  # Send by SMS only
                            if bio.contact:
                                checkAndSendMessage(user, message, bio.contact, bio.name, account=member_acc,
                                                    rec_no=rec_no + 'c')
                        elif notify_type.notification_type == 2:  # Send by Email only
                            if bio.email:
                                sendTransEmail(template_file_link, html_body, this_biz.name,
                                               'Account Opening Successful', email)

                        elif notify_type.notification_type == 3:  # Send by both SMS and Email
                            if bio.contact:
                                checkAndSendMessage(user, message, bio.contact, bio.name,
                                                    account=member_acc)

                            if bio.email:
                                sendTransEmail(template_file_link, html_body, this_biz.name,
                                               'Account Opening Successful', email)
            except Exception as e:
                logger.error(str(e))
        return member


class EditMemberForm(forms.Form):
    title = forms.CharField(required=False)
    name = forms.CharField(required=False)
    gender = forms.CharField(required=False)
    marital_status = forms.CharField(required=False)
    dob = forms.DateField(required=False, widget=forms.SelectDateWidget(
        attrs={'class': 'form-control', 'style': 'width:33.33333333333333%;''display:inline' '-block;'},
        years=range(1920, datetime.date.today().year + 1)), initial=timezone.now())

    nin = forms.CharField(required=False)
    contact = forms.CharField(required=False)
    other_contact = forms.CharField(required=False)
    email = forms.EmailField(required=False)
    nok = forms.CharField(required=False)
    nok_contacts = forms.CharField(required=False)
    account_type = forms.CharField(required=False)
    code = CountryField().formfield(required=False)
    code1 = CountryField().formfield(required=False)
    code2 = CountryField().formfield(required=False)
    mm = forms.CharField(required=False)
    country = CountryField().formfield(required=False)
    location = forms.CharField(widget=forms.TextInput, required=False)


class EditStaffForm(forms.Form):
    name = forms.CharField(required=False)
    gender = forms.CharField(required=False)
    dob = forms.DateField(
        widget=forms.SelectDateWidget(
            attrs={'class': 'form-control', 'style': 'width:33.33333333333333%; ''display:inline''-block;'},
            years=range(1920, datetime.date.today().year + 1)), initial=timezone.now())
    nin = forms.CharField(required=False)
    contact = forms.CharField(required=False)
    code = forms.CharField(required=False)

    def __init__(self, *args, **kwargs):
        self.user = kwargs.pop('user', None)
        super(EditStaffForm, self).__init__(*args, **kwargs)

    def clean(self):
        user = self.user
        staff = get_object_or_404(Staff, id=user.staff.id)
        cleaned_data = super().clean()
        name = cleaned_data['name']
        gender = cleaned_data['gender']
        dob = cleaned_data['dob']
        nin = cleaned_data['nin']
        contact = cleaned_data['contact']
        code = cleaned_data['code']

        staff.biodata.name = name
        staff.biodata.gender = gender
        staff.biodata.dob = dob
        staff.biodata.nin = nin
        try:
            staff.biodata.contact = proper_dial(contact, code)
        except Exception as e:
            raise ValidationError(_('Invalid Phone Number'))
        staff.save_related()

        return cleaned_data

class AddStaffForm(forms.Form):
    name = forms.CharField()
    gender = forms.CharField()
    marital_status = forms.CharField()
    dob = forms.DateField(
        widget=forms.SelectDateWidget(
            attrs={'class': 'form-control', 'style': 'width:33.33333333333333%; ' 'display:inline' '-block;'},
            years=range(1920, datetime.date.today().year + 1)), initial=timezone.now())
    code = forms.CharField(required=False)
    code1 = forms.CharField(required=False)
    code2 = forms.CharField(required=False)
    contact = forms.CharField(required=False)
    other_contact = forms.CharField(required=False)
    email = forms.EmailField(required=False)
    nok = forms.CharField(required=False)
    nok_contacts = forms.CharField(required=False)
    role = forms.CharField(required=False)
    employee_no = forms.CharField(required=True)
    location = forms.CharField(widget=forms.TextInput)
    can_access_central = forms.BooleanField(required=False)
    businesses = forms.ModelMultipleChoiceField(
        required=False,
        queryset=Business.objects.none(),  # Default to no queryset,
        widget=forms.SelectMultiple(attrs={
            'class': 'form-control select2',
            'data-placeholder': 'Select branch', # remember these are businesses
            'data-toggle': "select2",
            'name':"businesses",
            'id':"businesses",
            'multiple': "multiple"
        }))

    # def __init__(self, *args, **kwargs):
    #     self.user = kwargs.pop('user', None)
    #     super(AddStaffForm, self).__init__(*args, **kwargs)
    #
    #     if self.user:
    #         business_choices = [
    #             b for b in Business.objects.filter().values_list('id', flat=True)
    #         ]
    #         self.fields['businesses'].choices = business_choices

    def __init__(self, *args, **kwargs):
        # Extract the additional argument
        self.user = kwargs.pop('user', None)
        self.instance_type = kwargs.pop('instance_type', None)
        self.business_id = kwargs.pop('business_id', None)

        super(AddStaffForm, self).__init__(*args, **kwargs)

        if self.instance_type:
            self.fields['businesses'].queryset = Business.objects.filter(instance_type=self.instance_type)

    def clean(self):
        user = self.user
        cleaned_data = super().clean()
        name = cleaned_data.get('name')
        code = cleaned_data.get('code')
        code1 = cleaned_data.get('code1')
        code2 = cleaned_data.get('code2')
        gender = cleaned_data.get('gender')
        marital_status = cleaned_data.get('marital_status')
        dob = cleaned_data.get('dob')
        nin = cleaned_data.get('nin')
        contact = cleaned_data.get('contact')
        other_contact = cleaned_data.get('other_contact')
        email = cleaned_data.get('email')
        nok = cleaned_data.get('nok')
        nok_contacts = cleaned_data.get('nok_contacts')
        location = cleaned_data.get('location')
        role = cleaned_data.get('role')
        employee_no = cleaned_data.get('employee_no')
        can_access_central = cleaned_data.get('can_access_central')
        # print('CLEANED BUSINESSES', cleaned_data.get('businesses'))
        businesses = cleaned_data.get('businesses')

        if contact:
            primary_c = proper_dial(contact, code)
        else:
            primary_c = contact

        if other_contact:
            other_c = proper_dial(other_contact, code1)
        else:
            other_c = other_contact

        if nok_contacts:
            nok_c = proper_dial(nok_contacts, code2)
        else:
            nok_c = nok_contacts
        try:
            #check if the staff with a given number already exists
            check_emp_no = Biodata.objects.filter(employee_no=employee_no.strip(), business_id=self.business_id).exists()
            if check_emp_no:
                raise ValidationError(_('Employee with that employee number already exists'))

            bio = Biodata.objects.create(
                name=name,
                gender=gender,
                marital_status=marital_status,
                dob=dob,
                nin=nin,
                contact=primary_c,
                other_contact=other_c,
                email=email,
                nok=nok,
                location=location,
                nok_contacts=nok_c,
                created_by=biz_staff(user),
                business_id=self.business_id,
                employee_no=employee_no
            )
            # branches = Branch.objects.filter(business=biz_data(user)).count()
            # if branches > 1:

            # get the branch for the current business
            current_branch = Branch.objects.filter(business_id=self.business_id).first()

            this_staff = Staff.objects.create(
                biodata=bio,
                branch=current_branch,
                role_id=role,
                can_access_central=can_access_central,
            )

            # print('businesses-- FORM', businesses)

            # Add business relationships
            this_staff.businesses.clear()

            for business in businesses:
                # print(f'adding: {business}')
                this_staff.businesses.add(business)

            this_staff.save()

            log_title = 'Staff added'
            message = f"{user.staff.biodata.name} added a new staff {this_staff.biodata.name}"
            ActivityLog.objects.create(actor=user, title=log_title,
                                       action_type=CREATE,
                                       remarks=message,
                                       branch=branch_id(user))
        except Exception as e:
            raise ValidationError(_(str(e)))
        return cleaned_data


class AddAccountForm(forms.Form):
    error_messages = {
        'account_exists': _('Account Type already exists'),
    }
    account_type = forms.CharField()
    the_account = forms.IntegerField()
    is_new = forms.CharField()
    account_number = forms.CharField(widget=forms.TextInput, required=False)
    initial = forms.FloatField()

    def __init__(self, *args, **kwargs):
        self.user = kwargs.pop('user', None)
        self.member = kwargs.pop('member', None)
        self.finance_year = kwargs.pop('finance_year', None)
        self.initial_depo = kwargs.pop('initial', None)
        self.business_context = kwargs.pop('business_context', None)
        self.branch = kwargs.pop('branch', None)
        super(AddAccountForm, self).__init__(*args, **kwargs)

    def the_business(self):
        biz = biz_data(self.user)
        return biz

    def clean_account_number(self):
        cleaned_data = self.cleaned_data
        account_number = cleaned_data.get('account_number')
        if self.the_business().auto_acc_no is True:
            if account_number == '' or account_number is None:
                account_number = gen_account(self.user)
        else:
            if account_number != '' or account_number is not None:
                if check_account(account_number, self.user) is True:
                    # print('NUM CHECK')
                    raise ValidationError(_('Account already exists'), code='AccountNumber')
        if AccountBroker.objects.filter(the_account__acc_number=account_number, business=biz_data(self.user)).exists():
            # print('ACC BROKER')
            raise ValidationError(_('Account Type already exists'), code='AccountExists')
        return account_number

    def clean_initial(self):
        cleaned_data = self.cleaned_data
        account_type = cleaned_data.get('account_type')
        member_type = cleaned_data.get('member_type')
        initial = cleaned_data.get('initial')
        share_select = cleaned_data.get('share_select')
        initial_shares = cleaned_data.get('initial_shares')
        initial_before = float(str(initial).replace(',', '')) #new
        initial = float(initial_before)
        the_account_type = AccountTypes.objects.filter(id=account_type).first()
        min_balance = the_account_type.min_balance
        charge = int(check_charges(account_type, member_type, initial, self.user, initial_shares, share_select, self.branch))
        
        if initial and account_type:
            # print('fine')
            if initial >= charge + min_balance:
                return initial
            else:
                raise ValidationError(
                    _(f'Insufficient amount to cater for charges of {charge} and minimum balance of {min_balance}'),
                    code='Amount')
        return initial

    def save(self, commit=True):
        cleaned_data = self.cleaned_data
        user = self.user
        is_new = cleaned_data.get('is_new')
        initial = cleaned_data.get('initial')
        account_type = cleaned_data.get('account_type')
        the_account = cleaned_data.get('the_account')
        account_number = cleaned_data.get('account_number')

        rec_no = datetime.datetime.now().strftime("%y%m%d%H%M%S")
        ref_no = datetime.datetime.now().strftime("%y%m%d%H%M%S")

        try:
            with transaction.atomic():
                transaction_type = 'Opening balance'
                account = MemberAccount.objects.create(account_type_id=account_type, acc_number=account_number)
                # logger.error(f'Account created')
                member = Member.objects.get(id=self.member)
                broker = AccountBroker.objects.create(members=member, the_account=account, business_id=self.business_context[0])
                # print(f'Broker created')
                # logger.error(f'Broker created')
                cat, created = AccountCategory.objects.get_or_create(business_id=self.business_context[0], name="Members",
                                                                     cat_type='liability', dr_cr='cr')
                # logger.error(f'Cat {str(cat)}')
                equity_cat, created = AccountCategory.objects.get_or_create(business_id=self.business_context[0],
                                                                            name="Shares", dr_cr='cr',
                                                                            cat_type='liability')
                # logger.error(f'Equity cat {str(equity_cat)}')
                # chart of accounts
                member_acc = Account.objects.create(category=cat, business_id=self.business_context[0], member=account,
                                                    added_by=biz_staff(user))
                # print(f'Member acc {str(member_acc)} created')
                # logger.error(f'Member acc {str(member_acc)} created')

                if the_account:
                    acc_involved = Account.objects.filter(id=the_account, business_id=self.business_context[0]).first()

                opening_acct = Account.objects.get(name='Opening Reserves', business_id=self.business_context[0])

                account_dr = opening_acct

                # logger.error(f'ACCOUNT CREATION DONE')

                if is_new == 'YES':  # If this is a new member
                    transaction_type = 'Deposit'
                    account_dr = acc_involved

                    charges_acc = Account.objects.get(category__name="Registration Income", business_id=self.business_context[0])
                    gen_cgs = ApplicationAccountOrLoanType.objects.filter(account_type_id=account_type,
                                                                          general_charge__application='r',
                                                                          general_charge__status=True)

                    general_charges = gen_cgs.annotate(
                        amount=Case(
                            When(general_charge__is_percentage=True,
                                 then=ExpressionWrapper(initial * (F('general_charge__amount') / 100),
                                                        output_field=FloatField())),
                            default=F('general_charge__amount'),
                            output_field=FloatField()
                        )
                    ).annotate(rev_status=F('general_charge__is_revenue'))

                    # gen_charges = general_charges.values().aggregate(total=Sum('amount', default=0.00))
                    if general_charges.exists():

                        for ch in gen_cgs:
                            if ch.general_charge.is_revenue == 'YES':
                                Transactions.objects.create(
                                    branch=self.branch,
                                    transaction_type='Registration',
                                    account_dr=member_acc,
                                    account_cr=charges_acc,
                                    narration="{}".format(ch.general_charge.charge),
                                    reporting_amount=ch.general_charge.amount,
                                    added_by=biz_staff(user),
                                    receipt=rec_no + 'c',
                                    reference=ref_no + 'c'
                                )
                            else:
                                charge = ch.general_charge.charge
                                if Account.objects.filter(name=charge).exists():
                                    charge_account = charge
                                else:
                                    reserves = AccountCategory.objects.filter(name='Reserves').first()
                                    charge_account, created = Account.objects.create(name=ch.general_charge.charge,
                                                                                     category_id=reserves.id,
                                                                                     business_id=self.business_context[0])
                                Transactions.objects.create(
                                    branch=self.branch,
                                    transaction_type='Registration',
                                    account_dr=member_acc,
                                    account_cr=charge_account,
                                    narration="{}".format(ch.general_charge.charge),
                                    reporting_amount=ch.general_charge.amount,
                                    added_by=biz_staff(user),
                                    receipt=rec_no + 'c',
                                    reference=ref_no + 'c'
                                )

                    # logger.error(f'ACCOUNT CHARGESSSSSS DONE')

                    # Get deposit charges that match the specified criteria
                    dep_cgs = TransactionCharge.objects.filter(account_type_id=account_type, charge_type='d',
                                                               status=True)
                    # Calculate deposit charges total
                    depo_charges = 0
                    if dep_cgs.exists():
                        for d in dep_cgs:
                            if d.account_type.deposit_charge_vary:
                                # print('my dstart', d.start)
                                # print('my end', initial)
                                if d.start <= initial <= d.end:
                                    amount = initial * (d.charge / 100) if d.is_charge_percentage else d.charge
                            else:
                                amount = 0 if d.account_type.deposit_charge_vary else (
                                    initial * (d.charge / 100) if d.is_charge_percentage else d.charge)

                            depo_charges += amount

                            if is_new == 'YES' and amount > 0:
                                Transactions.objects.create(
                                    branch=self.branch,
                                    transaction_type='Charge',
                                    account_dr=member_acc,
                                    account_cr=charges_acc,
                                    narration=f"Charge on deposit of {initial}",
                                    reporting_amount=amount,
                                    added_by=biz_staff(user),
                                    receipt=f"{rec_no}c",
                                    reference=f"{ref_no}c"
                                )
                    # logger.info(f'DEPOSIT CHARGE DONE')

                Transactions.objects.create(
                    branch=self.branch,
                    transaction_type=transaction_type,
                    account_dr=account_dr,
                    account_cr=member_acc,
                    narration="Initial deposit on registration" if is_new == 'YES' else 'Opening balance',
                    reporting_amount=initial,
                    added_by=biz_staff(user),
                    receipt=rec_no,
                    reference=ref_no
                )

                # logger.info(f'ACTUAL DEPOSIT DONE')

                member_debits = Transactions.objects.filter(account_dr=member_acc).aggregate(
                    total=Sum('reporting_amount', default=0.0))['total']
                if member_debits is None:
                    member_debits = 0

                member_credits = Transactions.objects.filter(account_cr=member_acc).aggregate(
                    total=Sum('reporting_amount', default=0.0))['total']
                if member_credits is None:
                    member_credits = 0

                logger.error(f'Mem cr: {str(member_credits)} and mem_debits: {str(member_debits)}')
                account.balance = member_credits - member_debits
                account.save()

                # logger.error(f'ACCOUNT BALANCE UPDATE DONE')

                if is_new == 'YES':  # If this is a new member
                    # ==== SEND NOTIFICATIONS ======
                    template_file_link = 'sacco/emails/transactional.html'
                    currency = CurrencySetting.objects.filter(business_id=self.business_context[0]).first()
                    notify_type = NotiSettings.objects.filter(business_id=self.business_context[0]).first()
                    smsSettingsObj = SaccoSmsSettings.objects.get(when_to_send='on new member registration',
                                                                  business=biz_data(user))
                    message = f'Dear Client,\nThis is to inform you that your account in {biz_data(user).name} has been opened with A/C No: {account.acc_number} & deposit of {currency.currency} {initial} '
                    html_body = f'<p>Dear {member.name},</p> <p>This is to inform you that your account in {biz_data(user).name} has been opened with A/C No: {account.acc_number} and an initial deposit of {currency.currency} {initial}. <br>We hope that you have a good experience with our services and <b>{biz_data(user).name}</b> will do it’s best to please you. We will be looked forward to serving you in adherence to our quality standards.</p><p>Thanking you and with profound regards.</p>'
                    if smsSettingsObj.status:
                        if notify_type.notification_type == 1:  # Send by SMS only
                            if member.contact:
                                checkAndSendMessage(user, message, member.contact, member.name, account=member_acc,
                                                    rec_no=rec_no + 'c', this_account=account)
                        elif notify_type.notification_type == 2:  # Send by Email only
                            if member.email:
                                sendTransEmail(template_file_link, html_body, biz_data(user).name,
                                               'Account Opening Successful', member.email)

                        elif notify_type.notification_type == 3:  # Send by both SMS and Email
                            if member.contact:
                                checkAndSendMessage(user, message, member.contact, member.name,
                                                    account=member_acc, this_account=account)

                            if member.email:
                                sendTransEmail(template_file_link, html_body, biz_data(user).name,
                                               'Account Opening Successful', member.email)

                # logger.info(f'EVERYTHING IS DONE')
                return member
        except Exception as e:
            # logger.error(traceback.print_exc())
            # logger.error(f'Error in reg forms: {e}')
            # print(traceback.print_exc())
            # print('exception', str(e))
            raise e

class AddGroupAccountForm(forms.Form):
    error_messages = {
        'account_exists': _('Account Type already exists'),
    }
    account_type = forms.CharField()
    # the_account = forms.IntegerField()
    # is_new = forms.CharField()
    account_number = forms.CharField(widget=forms.TextInput, required=False)
    # initial = forms.IntegerField()

    def __init__(self, *args, **kwargs):
        self.user = kwargs.pop('user', None)
        self.member = kwargs.pop('member', None)
        self.business = kwargs.pop('business', None)
        super(AddGroupAccountForm, self).__init__(*args, **kwargs)

    def the_business(self):
        biz = biz_data(self.user)
        return biz

    def clean_account_type(self):
        cleaned_data = self.cleaned_data
        account_type = cleaned_data.get('account_type')
        v = AccountBroker.objects.filter(the_account__account_type_id=account_type, members_id=self.member, business=biz_data(self.user))
        if v.exists():
            raise ValidationError(_('Account Type already exists'), code='AccountType')
        return account_type

    def clean_account_number(self):
        cleaned_data = self.cleaned_data
        account_number = cleaned_data.get('account_number')
        if self.the_business().auto_acc_no is True:
            if account_number == '' or account_number is None:
                account_number = gen_account(self.user)
        else:
            if account_number != '' or account_number is not None:
                if check_account(account_number, self.user) is True:
                    raise ValidationError(_('Account already exists'), code='AccountNumber')

        return account_number


    # def clean_initial(self):
    #     user = self.user
    #     cleaned_data = self.cleaned_data
    #     account_type = cleaned_data.get('account_type')
    #     is_new = cleaned_data.get('is_new')
    #     initial = cleaned_data.get('initial')
    #     share_select = cleaned_data.get('share_select')
    #     initial_shares = cleaned_data.get('initial_shares')
    #     # initial_before = float(initial.replace(',', ''))
    #     # initial = float(initial_before)
    #     if account_type:
    #         the_account_type = AccountTypes.objects.filter(id=account_type).first()
    #         min_balance = the_account_type.min_balance
    #         charge = int(check_charges(account_type, is_new, initial, user, initial_shares, share_select, branch=biz_staff_branch(user)))
    #         if initial and account_type:
    #             # print('fine')
    #             if initial >= charge + min_balance:
    #                 return initial
    #             else:
    #                 raise ValidationError(
    #                     _(f'Insufficient amount to cater for charges of {charge} and minimum balance of {min_balance}'),
    #                     code='Amount')
    #     raise ValidationError(_('Account setup issue'), code='SetupIssue')

    def save(self, commit=True):
        cleaned_data = self.cleaned_data
        user = self.user
        businessId = self.business[0]
        # is_new = cleaned_data.get('is_new')
        account_type = cleaned_data.get('account_type')
        # initial = cleaned_data.get('initial')
        # the_account = cleaned_data.get('the_account')
        account_number = cleaned_data.get('account_number')

        rec_no = datetime.datetime.now().strftime("%y%m%d%H%M%S")
        ref_no = datetime.datetime.now().strftime("%y%m%d%H%M%S")

        try:
            with transaction.atomic():
                transaction_type = 'Opening balance'
                account = MemberAccount.objects.create(account_type_id=account_type, acc_number=account_number, balance=0)
                logger.error(f'Account created')
                member = Member.objects.get(id=self.member)
                broker = AccountBroker.objects.create(members=member, the_account=account, business_id=businessId)
                logger.error(f'Broker created')
                cat, created = AccountCategory.objects.get_or_create(business_id=businessId, name="Members",
                                                                     cat_type='liability', dr_cr='cr')
                logger.error(f'Cat {str(cat)}')
                equity_cat, created = AccountCategory.objects.get_or_create(business_id=businessId,
                                                                            name="Shares", dr_cr='cr',
                                                                            cat_type='liability')
                logger.error(f'Equity cat {str(equity_cat)}')
                # chart of accounts
                member_acc = Account.objects.create(category=cat, business_id=businessId, member=account,
                                                    added_by=biz_staff(user))
                logger.error(f'Member acc {str(member_acc)} created')

                # if the_account:
                #     acc_involved = Account.objects.filter(id=the_account, business=biz_data(user)).first()
                #
                # opening_acct = Account.objects.get(name='Opening Reserves', business=biz_data(user))

                # account_dr = opening_acct

                # logger.error(f'ACCOUNT CREATION DONE')

                # if is_new == 'YES':  # If this is a new member
                #     transaction_type = 'Deposit'
                #     account_dr = acc_involved
                #     charges_acc = Account.objects.get(category__name="Registration Income", business=biz_data(user))
                #     gen_cgs = ApplicationAccountOrLoanType.objects.filter(account_type_id=account_type,
                #                                                           general_charge__application='r',
                #                                                           general_charge__status=True)
                #     general_charges = gen_cgs.annotate(
                #         amount=Case(
                #             When(general_charge__is_percentage=True,
                #                  then=ExpressionWrapper(initial * (F('general_charge__amount') / 100),
                #                                         output_field=FloatField())),
                #             default=F('general_charge__amount'),
                #             output_field=FloatField()
                #         )
                #     ).annotate(rev_status=F('general_charge__is_revenue'))
                #
                #     # gen_charges = general_charges.values().aggregate(total=Sum('amount', default=0.00))
                #     if general_charges.exists():
                #         for ch in gen_cgs:
                #             if ch.general_charge.is_revenue == 'YES':
                #                 Transactions.objects.create(
                #                     branch=biz_staff_branch(user),
                #                     transaction_type='Registration',
                #                     account_dr=member_acc,
                #                     account_cr=charges_acc,
                #                     narration="{}".format(ch.general_charge.charge),
                #                     reporting_amount=ch.general_charge.amount,
                #                     added_by=biz_staff(user),
                #                     receipt=rec_no + 'c',
                #                     reference=ref_no + 'c'
                #                 )
                #             else:
                #                 charge = ch.general_charge.charge
                #                 if Account.objects.filter(name=charge).exists():
                #                     charge_account = charge
                #                 else:
                #                     reserves = AccountCategory.objects.filter(name='Reserves').first()
                #                     charge_account, created = Account.objects.create(name=ch.general_charge.charge,
                #                                                                      category_id=reserves.id,
                #                                                                      business=biz_data(user))
                #                 Transactions.objects.create(
                #                     branch=biz_staff_branch(user),
                #                     transaction_type='Registration',
                #                     account_dr=member_acc,
                #                     account_cr=charge_account,
                #                     narration="{}".format(ch.general_charge.charge),
                #                     reporting_amount=ch.general_charge.amount,
                #                     added_by=biz_staff(user),
                #                     receipt=rec_no + 'c',
                #                     reference=ref_no + 'c'
                #                 )
                #
                #     logger.error(f'ACCOUNT CHARGESSSSSS DONE')
                #
                #     # Get deposit charges that match the specified criteria
                #     dep_cgs = TransactionCharge.objects.filter(account_type_id=account_type, charge_type='d',
                #                                                status=True)
                #     # Calculate deposit charges total
                #     depo_charges = 0
                #     if dep_cgs.exists():
                #         for d in dep_cgs:
                #             if d.account_type.deposit_charge_vary:
                #                 if d.start <= initial <= d.end:
                #                     amount = initial * (d.charge / 100) if d.is_charge_percentage else d.charge
                #             else:
                #                 amount = 0 if d.account_type.deposit_charge_vary else (
                #                     initial * (d.charge / 100) if d.is_charge_percentage else d.charge)
                #
                #             depo_charges += amount
                #
                #             if is_new == 'YES' and amount > 0:
                #                 Transactions.objects.create(
                #                     branch=biz_staff_branch(user),
                #                     transaction_type='Charge',
                #                     account_dr=member_acc,
                #                     account_cr=charges_acc,
                #                     narration=f"Charge on deposit of {initial}",
                #                     reporting_amount=amount,
                #                     added_by=biz_staff(user),
                #                     receipt=f"{rec_no}c",
                #                     reference=f"{ref_no}c"
                #                 )
                #     logger.info(f'DEPOSIT CHARGE DONE')
                #
                # Transactions.objects.create(
                #     branch=biz_staff_branch(user),
                #     transaction_type=transaction_type,
                #     account_dr=account_dr,
                #     account_cr=member_acc,
                #     narration="Initial deposit on registration" if is_new == 'YES' else 'Opening balance',
                #     reporting_amount=initial,
                #     added_by=biz_staff(user),
                #     receipt=rec_no,
                #     reference=ref_no
                # )
                #
                # logger.info(f'ACTUAL DEPOSIT DONE')
                # member_debits = Transactions.objects.filter(account_dr=member_acc).aggregate(
                #     total=Sum('reporting_amount', default=0.0))['total']
                # if member_debits is None:
                #     member_debits = 0
                #
                # member_credits = Transactions.objects.filter(account_cr=member_acc).aggregate(
                #     total=Sum('reporting_amount', default=0.0))['total']
                # if member_credits is None:
                #     member_credits = 0
                #
                # logger.error(f'Mem cr: {str(member_credits)} and mem_debits: {str(member_debits)}')
                # account.balance = member_credits - member_debits
                # account.save()
                #
                # logger.error(f'ACCOUNT BALANCE UPDATE DONE')

                # if is_new == 'YES':  # If this is a new member
                #     # ==== SEND NOTIFICATIONS ======
                #     template_file_link = 'sacco/emails/transactional.html'
                #     currency = CurrencySetting.objects.filter(business=biz_data(user)).first()
                #     notify_type = NotiSettings.objects.filter(business=biz_data(user)).first()
                #     smsSettingsObj = SaccoSmsSettings.objects.get(when_to_send='on new member registration',
                #                                                   business=biz_data(user))
                #     message = f'Dear Client,\nThis is to inform you that your account in {biz_data(user).name} has been opened with A/C No: {account.acc_number} & deposit of {currency.currency} {initial} '
                #     html_body = f'<p>Dear {member.name},</p> <p>This is to inform you that your account in {biz_data(user).name} has been opened with A/C No: {account.acc_number} and an initial deposit of {currency.currency} {initial}. <br>We hope that you have a good experience with our services and <b>{biz_data(user).name}</b> will do it’s best to please you. We will be looked forward to serving you in adherence to our quality standards.</p><p>Thanking you and with profound regards.</p>'
                #     if smsSettingsObj.status:
                #         if notify_type.notification_type == 1:  # Send by SMS only
                #             if member.contact:
                #                 checkAndSendMessage(user, message, member.contact, member.name, account=member_acc,
                #                                     rec_no=rec_no + 'c', this_account=account)
                #         elif notify_type.notification_type == 2:  # Send by Email only
                #             if member.email:
                #                 sendTransEmail(template_file_link, html_body, biz_data(user).name,
                #                                'Account Opening Successful', member.email)
                #
                #         elif notify_type.notification_type == 3:  # Send by both SMS and Email
                #             if member.contact:
                #                 checkAndSendMessage(user, message, member.contact, member.name,
                #                                     account=member_acc, this_account=account)
                #
                #             if member.email:
                #                 sendTransEmail(template_file_link, html_body, biz_data(user).name,
                #                                'Account Opening Successful', member.email)
                #
                # logger.info(f'EVERYTHING IS DONE')
                return member
        except Exception as e:
            logger.error(traceback.print_exc())
            logger.error(f'Error in reg forms: {e}')
            # print(traceback.print_exc())
            # print('exception', str(e))
            raise e


class AddAccountTypeForm(forms.Form):
    name = forms.CharField(max_length=225)
    account_type = forms.CharField(max_length=1)
    minimum_balance = forms.CharField(max_length=255)
    maturity = forms.IntegerField()
    dormancy_period = forms.IntegerField()
    deposit_charge_vary = forms.CharField(max_length=1)
    withdraw_charge_vary = forms.CharField(max_length=1)
    transfer_charge_vary = forms.CharField(max_length=1)
    charge_on_deposit_switch = forms.BooleanField(required=False)
    charge_on_withdraw_switch = forms.BooleanField(required=False)
    charge_on_transfer_switch = forms.BooleanField(required=False)


class FinanceYearForm(forms.Form):
    name = forms.CharField(max_length=225)
    start_date = forms.DateField()
    end_date = forms.DateField()


class ChangeStaffPassword(forms.Form):
    new_password1 = forms.CharField()
    new_password2 = forms.CharField()

    error_messages = {
        'password_mismatch': _('The two password fields didn’t match.'),
        'staff_no_user': _('This staff is not a user')
    }

    def __init__(self, *args, **kwargs):
        self.user = kwargs.pop('user', None)
        self.staff = kwargs.pop('staff', None)
        super(ChangeStaffPassword, self).__init__(*args, **kwargs)

    def clean_new_password2(self):
        password1 = self.cleaned_data.get('new_password1')
        password2 = self.cleaned_data.get('new_password2')
        if password1 and password2:
            if password1 != password2:
                raise ValidationError(
                    self.error_messages['password_mismatch'],
                    code='password_mismatch',
                )
        # password_validation.validate_password(password2, self.staff.user)
        return password2

    def save(self, commit=True):
        password = self.cleaned_data["new_password1"]
        try:
            stf = Staff.objects.get(id=self.staff.id)
            user = stf.user
            print('USER', user)
            user.set_password(password)
            if commit:
                user.save()
            return stf
        except Exception as e:
            raise ValidationError(_(str(e)))


class EditStaffProfileForm(forms.Form):
    name = forms.CharField(required=False)
    gender = forms.CharField(required=False)
    marital_status = forms.CharField(required=False)
    dob = forms.DateField(
        widget=forms.SelectDateWidget(
            attrs={'class': 'form-control', 'style': 'width:33.33333333333333%; ' 'display:inline' '-block;'},
            years=range(1920, datetime.date.today().year + 1)), initial=timezone.now())

    nin = forms.CharField(required=False)
    contact = forms.CharField(required=False)
    other_contact = forms.CharField(required=False)
    email = forms.EmailField(required=False)
    nok = forms.CharField(required=False)
    nok_contacts = forms.CharField(required=False)
    account_type = forms.CharField(required=False)
    code = CountryField().formfield(required=False)
    code1 = CountryField().formfield(required=False)
    code2 = CountryField().formfield(required=False)
    mm = forms.CharField(required=False)
    country = CountryField().formfield(required=False)
    title = forms.CharField(required=False)
    location = forms.CharField(widget=forms.TextInput, required=False)
    employee_no= forms.CharField(required=True)
    can_access_central = forms.BooleanField(required=False)
    businesses = forms.ModelMultipleChoiceField(
        required=False,
        queryset=Business.objects.none(),  # Default to no queryset
        widget=forms.SelectMultiple(attrs={
            'class': 'form-control select2',
            'data-placeholder': 'Select businesses'
        }))

    def __init__(self, *args, **kwargs):
        # Extract the additional argument
        self.instance_type = kwargs.pop('instance_type', None)

        super(EditStaffProfileForm, self).__init__(*args, **kwargs)

        # Now use branch_id to dynamically get payment modes
        if self.instance_type:
            self.fields['businesses'].queryset = Business.objects.filter(instance_type=self.instance_type)


class WithdrawForm(forms.Form):
    date_created = forms.DateField(required=True)
    account_id = forms.IntegerField(required=True)
    amount = forms.CharField(required=True)
    the_account = forms.IntegerField(required=True)
    narration = forms.CharField(required=False)
    dep_ref = forms.CharField(required=False)

    def __init__(self, *args, **kwargs):
        self.user = kwargs.pop('user', None)
        self.member = kwargs.pop('member', None)
        self.finance_year = kwargs.pop('finance_year', None)
        self.business_context =kwargs.pop('business_context', None)

        super(WithdrawForm, self).__init__(*args, **kwargs)

    @staticmethod
    def calculate_withdraw_charges(with_cgs, amount, user, member_acc, this_account, date_created, rec_no, ref_no, business_context):
        # print('with_cgs', with_cgs.values())
        income_acc = Account.objects.filter(category__name="Withdraw Income", business_id=business_context['data_context'][0]).first()
        withdraw_charges = 0
        transactions = []
        # print(depo_cgs.)
        for withdraw in with_cgs:
            if withdraw.account_type.withdraw_charge_vary:
                if withdraw.start <= amount <= withdraw.end:
                    charge_amount = withdraw.charge if not withdraw.is_charge_percentage else amount * (withdraw.charge / 100)
                    withdraw_charges += charge_amount
                    transactions.append(
                        Transactions(
                            branch=business_context['branch'],
                            transaction_type='Charge on withdraw',
                            account_dr=member_acc,
                            account_cr=income_acc,
                            narration=f'Charge on withdraw of {amount} on {this_account.acc_number}',
                            reporting_amount=charge_amount,
                            added_by=biz_staff(user),
                            tx_date=date_created,
                            receipt=rec_no,
                            reference=ref_no
                        )
                    )

            else:
                charge_amount = withdraw.charge if not withdraw.is_charge_percentage else amount * (withdraw.charge / 100)
                withdraw_charges += charge_amount
                transactions.append(
                    Transactions(
                        branch=business_context['branch'],
                        transaction_type='Charge on withdraw',
                        account_dr=member_acc,
                        account_cr=income_acc,
                        narration=f'Charge on withdraw of {amount} on {this_account.acc_number}',
                        reporting_amount=charge_amount,
                        added_by=biz_staff(user),
                        tx_date=date_created,
                        receipt=rec_no,
                        reference=ref_no
                    )
                )

        Transactions.objects.bulk_create(transactions)

        return withdraw_charges

    @staticmethod
    def create_withdraw_transaction(user, date_created, amount, narration, rec_no, ref_no, this_account,
                                   member_acc, acc_involved, business_context):
        msg = f'Withdraw of {amount} to {this_account.acc_number}' if not narration else narration

        return Transactions.objects.create(
            branch=business_context['branch'],
            transaction_type='Withdraw',
            account_dr=member_acc,
            account_cr=acc_involved,
            narration=msg,
            reporting_amount=amount,
            added_by=biz_staff(user),
            tx_date=date_created,
            receipt=rec_no,
            reference=ref_no
        )

    @staticmethod
    def update_account_balance(this_account, member_acc):
        member_debit = Transactions.objects.filter(account_dr=member_acc).aggregate(total=Sum('reporting_amount'))[
                           'total'] or 0
        member_credit = Transactions.objects.filter(account_cr=member_acc).aggregate(total=Sum('reporting_amount'))[
                            'total'] or 0
        this_account.balance = member_credit - member_debit
        this_account.save()

    @staticmethod
    def send_notification(zemember, notify_type, this_biz, trans, amount, depo_charges, user, template_file_link, member_acc, this_account, business_context):
        currency = CurrencySetting.objects.filter(business_id=business_context['data_context'][0]).first()
        smsSettingsObj = SaccoSmsSettings.objects.get(when_to_send='On deposit transaction', business_id=business_context['data_context'][0])
        subject = "Withdraw request processed successfully"
        email = zemember.email
        name = zemember.name
        contact = zemember.contact
        if not zemember.is_group:
            email = zemember.biodata.email
            name = zemember.biodata.name
            contact = zemember.biodata.contact

        business = this_biz.name
        message = f'Dear {name}, a withdraw of {currency.currency} {amount} has been made on your account. Fee {currency.currency} {depo_charges}. Account Balance is '
        html_body = f'<p>Dear {name},</p> Your Withdraw request with the following details has been processed successfully.<br/>Transaction Number: {trans.txno}<br/>Amount: {currency.currency}{amount} <br/>Description: {trans.narration} <br/>Date: {trans.tx_date} <p>Please call our contact centre on +{this_biz.contact} in case you require further details.</p>'

        if smsSettingsObj.status:
            if notify_type.notification_type == 1:  # Send by SMS only
                if contact:
                    checkAndSendMessage(user, message, contact, name, account=member_acc, this_account=this_account)
            elif notify_type.notification_type == 2:  # Send by Email only
                if zemember.biodata.email:
                    sendTransEmail(template_file_link, html_body, business, subject, email)

            elif notify_type.notification_type == 3:  # Send by both SMS and Email
                if contact:
                    # print('send message')
                    checkAndSendMessage(user, message, contact, name, account=member_acc, this_account=this_account)

                if email:
                    sendTransEmail(template_file_link, html_body, business, subject, email)

    @transaction.atomic
    def save(self, commit=True):
        cleaned_data = super().clean()
        if commit:
            user = self.user
            zemember = self.member
            date_created = cleaned_data.get('date_created', datetime.date.today())
            amount = float(cleaned_data.get('amount').replace(',', ''))
            with_ref = cleaned_data.get('with_ref')
            narration = cleaned_data.get('narration', None)

            rec_no = datetime.datetime.now().strftime("%y%m%d%H%M%S")
            ref_no = with_ref or rec_no

            this_biz = Business.objects.filter(id=biz_id(user)).first()
            notify_type = NotiSettings.objects.filter(business=this_biz).first()
            template_file_link = 'sacco/emails/transactional.html'

            if not date_created:
                date_created = datetime.datetime.now().date()

            this_account = MemberAccount.objects.get(id=cleaned_data.get('account_id'))
            # acc_broker = AccountBroker.objects.filter(the_account_id=this_account.id).first()
            member_acc = Account.objects.filter(member=this_account).first()

            acc_involved = Account.objects.get(id=cleaned_data.get('the_account'), business=biz_data(user))

            withdraw_cgs = TransactionCharge.objects.filter(account_type_id=this_account.account_type,
                                                        charge_type='w', status=True)
            
            total_withdraw_charge = withdraw_cgs.aggregate(total=Sum("charge"))["total"]
            if total_withdraw_charge is None:
                total_withdraw_charge = 0
            #determine unwithdrawable amt
            # Needs to be improved
            the_active_loans = Loans.objects.filter(account=this_account, loan_status=3).exclude(savings_rem_balance=None)
            saving_not_withdrawable = 0
            if the_active_loans.exists():
                saving_not_withdrawable = the_active_loans.aggregate(total=Sum("savings_rem_balance"))["total"]
                if saving_not_withdrawable is None:
                    saving_not_withdrawable = 0

            if (total_withdraw_charge + amount + saving_not_withdrawable) > this_account.balance:
                raise ValidationError(_('insuffient balance'))

            depo_charges = self.calculate_withdraw_charges(withdraw_cgs, amount, user, member_acc, this_account, date_created, rec_no, ref_no, self.business_context)

            trans = self.create_withdraw_transaction(user, date_created, amount, narration, rec_no, ref_no,
                                                     this_account, member_acc, acc_involved, self.business_context)

            self.update_account_balance(this_account, member_acc)

            self.send_notification(zemember, notify_type, this_biz, trans, amount, depo_charges, user, template_file_link, member_acc, this_account, self.business_context)

        return cleaned_data

class DepositForm(forms.Form):
    date_created = forms.DateField(required=True)
    account_id = forms.IntegerField(required=True)
    amount = forms.CharField(required=True)
    the_account = forms.IntegerField(required=True)
    narration = forms.CharField(required=False)
    dep_ref = forms.CharField(required=False)
    paid_by = forms.CharField(required=False)

    def __init__(self, *args, **kwargs):
        self.user = kwargs.pop('user', None)
        self.member = kwargs.pop('member', None)
        self.finance_year = kwargs.pop('finance_year', None)
        self.the_branch = kwargs.pop('branch', None)
        super(DepositForm, self).__init__(*args, **kwargs)

    @staticmethod
    def calculate_deposit_charges(depo_cgs, amount, user, member_acc, this_account, date_created, rec_no, ref_no, paid_by, branch):
        income_acc = Account.objects.filter(category__name="Deposit Income", business=branch.business).first()
        depo_charges = 0
        transactions = []
        # print(depo_cgs.)
        for d in depo_cgs:
            if d.account_type.deposit_charge_vary:
                if d.start <= amount <= d.end:

                    charge_amount = d.charge if not d.is_charge_percentage else amount * (d.charge / 100)
                    depo_charges += charge_amount
                    transactions.append(
                        Transactions(
                            branch=branch,
                            transaction_type='Charge on deposit',
                            account_dr=member_acc,
                            account_cr=income_acc,
                            narration=f'Charge on deposit of {amount} on {this_account.acc_number}',
                            reporting_amount=charge_amount,
                            added_by=biz_staff(user),
                            tx_date=date_created,
                            receipt=rec_no,
                            reference=ref_no,
                            paid_by=paid_by
                        )
                    )

            else:
                charge_amount = d.charge if not d.is_charge_percentage else amount * (d.charge / 100)
                depo_charges += charge_amount
                # print('am herer', amount)
                transactions.append(
                    Transactions(
                        branch=branch,
                        transaction_type='Charge on deposit',
                        account_dr=member_acc,
                        account_cr=income_acc,
                        narration=f'Charge on deposit of {amount} on {this_account.acc_number}',
                        reporting_amount=charge_amount,
                        added_by=biz_staff(user),
                        tx_date=date_created,
                        receipt=rec_no,
                        reference=ref_no,
                        paid_by=paid_by
                    )
                )

            # if charge_amount > 0:

        Transactions.objects.bulk_create(transactions)

        return depo_charges

    @staticmethod
    def create_deposit_transaction(user, finance_year, date_created, amount, narration, rec_no, ref_no, this_account,
                                   member_acc, acc_involved, paid_by, branch):
        msg = f'Deposit of {amount} to {this_account.acc_number}' if not narration else narration

        return Transactions.objects.create(
            branch=branch,
            transaction_type='Deposit',
            account_dr=acc_involved,
            account_cr=member_acc,
            narration=msg,
            reporting_amount=amount,
            added_by=biz_staff(user),
            financial_year=finance_year,
            tx_date=date_created,
            receipt=rec_no,
            reference=ref_no,
            paid_by=paid_by
        )

    @staticmethod
    def update_account_balance(this_account, member_acc):
        member_debit = Transactions.objects.filter(account_dr=member_acc).aggregate(total=Sum('reporting_amount'))[
                           'total'] or 0
        member_credit = Transactions.objects.filter(account_cr=member_acc).aggregate(total=Sum('reporting_amount'))[
                            'total'] or 0
        this_account.balance = member_credit - member_debit
        this_account.save()

    @staticmethod
    def send_notification(zemember, notify_type, this_biz, trans, amount, dep_with, user, template_file_link, member_acc, this_account, branch):

        # print('de balance', balance)
        currency = CurrencySetting.objects.filter(business=branch.business).first()
        smsSettingsObj = SaccoSmsSettings.objects.get(when_to_send='On deposit transaction', business=branch.business)
        subject = "Deposit request processed successfully"
        email = zemember.email
        name = zemember.name
        contact = zemember.contact
        if not zemember.is_group:
            email = zemember.biodata.email
            name = zemember.biodata.name
            contact = zemember.biodata.contact

        business = this_biz.name
        message = f'Dear {name}, a deposit of {currency.currency} {amount} has been made on your account. Fee {currency.currency} {dep_with}. Account Balance is '
        html_body = f'<p>Dear {name},</p>  Your Deposit request with the following details has been processed successfully.<br/>Transaction Number: {trans.txno}<br/>Amount: {currency.currency}{amount} <br/>Description: {trans.narration} <br/>Date: {trans.tx_date} <p>Please call our contact centre on +{this_biz.contact} in case you require further details.</p>'

        if smsSettingsObj.status:
            if notify_type.notification_type == 1:  # Send by SMS only
                if contact:
                    checkAndSendMessage(user, message, contact, name, account=member_acc, this_account=this_account)
            elif notify_type.notification_type == 2:  # Send by Email only
                if email:
                    sendTransEmail(template_file_link, html_body, business, subject, email)

            elif notify_type.notification_type == 3:  # Send by both SMS and Email
                if contact:
                    # print('send message')
                    checkAndSendMessage(user, message, contact, name, account=member_acc, this_account=this_account)

                if email:
                    sendTransEmail(template_file_link, html_body, business, subject, email)

    @transaction.atomic
    def save(self, commit=True):
        user = self.user
        zemember = self.member
        cleaned_data = super().clean()
        date_created = cleaned_data.get('date_created', datetime.date.today())
        amount = float(cleaned_data.get('amount').replace(',', ''))
        dep_ref = cleaned_data.get('dep_ref')
        narration = cleaned_data.get('narration', None)
        paid_by = cleaned_data.get('paid_by', None)

        rec_no = datetime.datetime.now().strftime("%y%m%d%H%M%S")
        ref_no = dep_ref or rec_no

        this_biz = Business.objects.filter(id=biz_id(user)).first()
        notify_type = NotiSettings.objects.filter(business=this_biz).first()
        template_file_link = 'sacco/emails/transactional.html'

        if not date_created:
            date_created = datetime.datetime.now().date()

        this_account = MemberAccount.objects.get(id=cleaned_data.get('account_id'))
        # acc_broker = AccountBroker.objects.filter(the_account_id=this_account.id).first()
        member_acc = Account.objects.filter(member=this_account).first()

        acc_involved = Account.objects.get(id=cleaned_data.get('the_account'), business=self.the_branch.business)

        depo_cgs = TransactionCharge.objects.filter(account_type_id=this_account.account_type,
                                                    charge_type='d', status=True)

        depo_charges = self.calculate_deposit_charges(depo_cgs, amount, user, member_acc, this_account, date_created, rec_no, ref_no, paid_by, self.the_branch)

        trans = self.create_deposit_transaction(user, self.finance_year, date_created, amount, narration, rec_no,
                                                ref_no,
                                                this_account, member_acc, acc_involved, paid_by, self.the_branch)

        self.update_account_balance(this_account, member_acc)

        self.send_notification(zemember, notify_type, this_biz, trans, amount, depo_charges, user, template_file_link, member_acc, this_account, self.the_branch)

        return cleaned_data


class AddGroupForm(forms.Form):
    name = forms.CharField(required=False)
    dob = forms.DateField(
        widget=forms.SelectDateWidget(
            attrs={'class': 'form-control', 'style': 'width:33.33333333333333%; ' 'display:inline' '-block;'},
            years=range(1920, datetime.date.today().year + 1)), initial=timezone.now())

    contact = forms.CharField(required=False)
    other_contact = forms.CharField(required=False)
    email = forms.EmailField(required=False)
    account_type = forms.CharField(required=False)
    desc = forms.CharField(required=False)
    code = CountryField().formfield(required=False)
    code1 = CountryField().formfield(required=False)
    initial = forms.IntegerField(required=False)
    address = forms.CharField(widget=forms.TextInput, required=False)
    account_number = forms.CharField(widget=forms.TextInput, required=False)

    def __init__(self, *args, **kwargs):
        self.user = kwargs.pop('user', None)
        super(AddGroupForm, self).__init__(*args, **kwargs)

    @transaction.atomic
    def save(self):
        user = self.user
        bizness = biz_data(user)
        cleaned_data = super().clean()
        name = cleaned_data.get('name')
        dob = cleaned_data.get('dob')
        contact = cleaned_data.get('contact')
        other_contact = cleaned_data.get('other_contact')
        email = cleaned_data.get('email')
        account_type = cleaned_data.get('account_type')
        initial = cleaned_data.get('initial')
        address = cleaned_data.get('address')
        desc = cleaned_data.get('desc')
        code = cleaned_data.get('code')
        code1 = cleaned_data.get('code1')
        account_number = cleaned_data.get('account_number')
        if contact:
            primary_c = proper_dial(contact, code)
        else:
            primary_c = contact

        if other_contact:
            other_c = proper_dial(other_contact, code1)
        else:
            other_c = other_contact

        try:
            bio = Biodata.objects.create(
                name=name,
                dob=dob,
                contact=primary_c,
                other_contact=other_c,
                email=email,
                location=address,
                created_by=biz_staff(user),
                business=biz_data(user)
            )
        except Exception as e:
            raise ValidationError(_(str(e)))
        if bizness.share_status:
            member = Member.objects.create(biodata=bio, created_by=biz_staff(user),
                                           member_type='s', is_group=True, desc=desc, branch=biz_staff_branch(user))
        else:
            member = Member.objects.create(biodata=bio, created_by=biz_staff(user),
                                           member_type='o', is_group=True, desc=desc, branch=biz_staff_branch(user), )

        acc_number = account_number
        if account_number is None or account_number == '':
            acc_number = random_with_N_digits(8)

        while True:
            if AccountBroker.objects.filter(the_account__acc_number=acc_number, business=biz_data(user)).exists():
                acc_number = random_with_N_digits(8)
            account = MemberAccount.objects.create(balance=initial, account_type_id=account_type, acc_number=acc_number)
            mb = AccountBroker(members=member, the_account=account, business=biz_data(user))
            mb.save()
            cat, created = AccountCategory.objects.get_or_create(business=biz_data(user), name="Groups")
            # chart of accounts
            member_acc = Account.objects.create(category=cat, business=biz_data(user), member=account,
                                                added_by=biz_staff(user))

            cash_acc, created = Account.objects.get_or_create(category__name="Cash", business=biz_data(user))

            charges_acc, created = Account.objects.get_or_create(category__name="Registration Income",
                                                                 business=biz_data(user))

            charges = ApplicationAccountOrLoanType.objects.filter(general_charge__business=biz_data(user),
                                                                  general_charge__application='r',
                                                                  account_type_id=account_type)
            charge_trans = []
            Transactions.objects.create(
                branch=biz_staff_branch(user),
                transaction_type="Deposit",
                account_dr=cash_acc,
                account_cr=member_acc,
                narration="Initial deposit on registration",
                reporting_amount=initial,
                added_by=biz_staff(user)
            )
            # loop through charges to see charges that apply in member reg
            for ch in charges:
                charge_trans.append(
                    Transactions.objects.create(
                        branch=biz_staff_branch(user),
                        transaction_type=ch.general_charge.charge,
                        account_dr=member_acc,
                        account_cr=charges_acc,
                        narration="Registration Charge",
                        reporting_amount=ch.general_charge.amount,
                        added_by=biz_staff(user)
                    )
                )
            message = f"{user} created a group account {acc_number}"
            ActivityLog.objects.create(actor=user,
                                       action_type=CREATE,
                                       title='Group account create',
                                       branch=branch_id(user),
                                       remarks=message)
            return cleaned_data


class EditGroupForm(forms.Form):
    name = forms.CharField(required=False)
    dob = forms.DateField(
        widget=forms.SelectDateWidget(
            attrs={'class': 'form-control', 'style': 'width:33.33333333333333%; ' 'display:inline' '-block;'},
            years=range(1920, datetime.date.today().year + 1)), initial=timezone.now())

    contact = forms.CharField(required=False)
    other_contact = forms.CharField(required=False)
    email = forms.EmailField(required=False)
    account_type = forms.CharField(required=False)
    desc = forms.CharField(required=False)
    code = CountryField().formfield(required=False)
    code1 = CountryField().formfield(required=False)
    location = forms.CharField(widget=forms.TextInput, required=False)
    account_number = forms.CharField(widget=forms.TextInput, required=False)


class AddGroupMemberForm(forms.Form):
    name = forms.CharField(required=False)
    gender = forms.CharField(required=False)
    marital_status = forms.CharField(required=False)
    dob = forms.DateField(
        widget=forms.SelectDateWidget(
            attrs={'class': 'form-control', 'style': 'width:33.33333333333333%; ' 'display:inline' '-block;'},
            years=range(1920, datetime.date.today().year + 1)), initial=timezone.now())

    nin = forms.CharField(required=False)
    contact = forms.CharField(required=False)
    email = forms.EmailField(required=False)
    nok = forms.CharField(required=False)
    nok_contacts = forms.CharField(required=False)
    code = forms.CharField(required=True)
    code1 = forms.CharField(required=False)
    location = forms.CharField(widget=forms.TextInput, required=False)

    def __init__(self, *args, **kwargs):
        self.user = kwargs.pop('user', None)
        self.group = kwargs.pop('group', None)
        super(AddGroupMemberForm, self).__init__(*args, **kwargs)

    def clean_contact(self):
        cleaned_data = self.cleaned_data
        contact = cleaned_data.get('contact')
        code = cleaned_data.get('code')
        if contact:
            primary = proper_dial(contact, code)
            if not primary:
                primary = contact
            return primary
        return contact

    def clean_nok(self):
        cleaned_data = super().clean()
        nok_contacts = cleaned_data.get('nok_contacts')
        code1 = cleaned_data.get('code1')

        if nok_contacts:
            nok_c = proper_dial(nok_contacts, code1)
        else:
            nok_c = nok_contacts
        return nok_c

    @transaction.atomic
    def save(self, commit=True):
        user = self.user
        group = self.group
        cleaned_data = super().clean()
        name = cleaned_data.get('name')
        gender = cleaned_data.get('gender')
        marital_status = cleaned_data.get('marital_status')
        dob = cleaned_data.get('dob')
        nin = cleaned_data.get('nin')
        email = cleaned_data.get('email')
        nok = cleaned_data.get('nok')
        location = cleaned_data.get('location')
        nok_contacts = cleaned_data.get('nok_contacts')
        contact = cleaned_data.get('contact')
        # account_type = cleaned_data.get('account_type')

        if commit:
            try:
                bio = Biodata.objects.create(
                    name=name,
                    gender=gender,
                    marital_status=marital_status,
                    dob=dob,
                    nin=nin,
                    contact=contact,
                    email=email,
                    nok=nok,
                    location=location,
                    nok_contacts=nok_contacts,
                    created_by=biz_staff(user),
                    business=biz_data(user)
                )
                # print(bio.name)
                the_member = Member.objects.create(biodata=bio, date_joined=timezone.now().date(), branch_id=branch_id(user))
                # print(the_member.biodata.name)
                GroupMember.objects.create(member=the_member, group=group, added_by=biz_staff(user))
                # acc_number = gen_account(user)
                # account = MemberAccount.objects.create(account_type_id=account_type, acc_number=acc_number)
                # AccountBroker.objects.create(members=the_member, the_account=account, business=biz_data(user))
            except Exception as e:
                raise ValidationError(_(str(e)))


class EditGroupMemberForm(forms.Form):
    name = forms.CharField(required=False)
    gender = forms.CharField(required=False)
    marital_status = forms.CharField(required=False)
    dob = forms.DateField(
        widget=forms.SelectDateWidget(
            attrs={'class': 'form-control', 'style': 'width:33.33333333333333%; ' 'display:inline' '-block;'},
            years=range(1920, datetime.date.today().year + 1)), initial=timezone.now())

    nin = forms.CharField(required=False)
    contact = forms.CharField(required=False)
    other_contact = forms.CharField(required=False)
    email = forms.EmailField(required=False)
    nok = forms.CharField(required=False)
    nok_contacts = forms.CharField(required=False)
    code = CountryField().formfield(required=False)
    code1 = CountryField().formfield(required=False)
    code2 = CountryField().formfield(required=False)
    location = forms.CharField(widget=forms.TextInput, required=False)

    def __init__(self, *args, **kwargs):
        self.user = kwargs.pop('user', None)
        self.group = kwargs.pop('group', None)
        self.member = kwargs.pop('member', None)
        super(EditGroupMemberForm, self).__init__(*args, **kwargs)

    def clean(self):
        # user = self.user
        this_member = self.member
        cleaned_data = super().clean()
        name = cleaned_data.get('name')
        gender = cleaned_data.get('gender')
        marital_status = cleaned_data.get('marital_status')
        dob = cleaned_data.get('dob')
        nin = cleaned_data.get('nin')
        contact = cleaned_data.get('contact')
        email = cleaned_data.get('email')
        nok = cleaned_data.get('nok')
        nok_contacts = cleaned_data.get('nok_contacts')
        other_contact = cleaned_data.get('other_contact')
        location = cleaned_data.get('location')
        code = cleaned_data.get('code')
        code1 = cleaned_data.get('code1')
        code2 = cleaned_data.get('code2')
        # print(code1)
        if contact:
            primary_c = proper_dial(contact, code)
        else:
            primary_c = contact

        if nok_contacts:
            nok_c = proper_dial(nok_contacts, code1)
        else:
            nok_c = nok_contacts

        if other_contact:
            other_c = proper_dial(other_contact, code2)
        else:
            other_c = other_contact

        try:
            member_ = GroupMember.objects.get(id=this_member)
            member = member_.member
            member.biodata.name = name
            member.biodata.email = email
            member.biodata.gender = gender
            member.biodata.marital_status = marital_status
            member.biodata.dob = dob
            member.biodata.nin = nin
            member.biodata.contact = primary_c
            member.biodata.other_contact = other_c
            member.biodata.nok = nok
            member.biodata.nok_contacts = nok_c
            member.biodata.location = location
            member.save_related()

        except Exception as e:
            print(traceback.print_exc())
            raise ValidationError(_(str(e)))


class IndDepositForm(forms.Form):
    date_created = forms.DateField(required=True)
    unique = forms.IntegerField(required=True)
    amount = forms.CharField(required=True)
    the_account = forms.IntegerField(required=True)
    dep_narration = forms.CharField(required=False)
    dep_ref = forms.CharField(required=False)

    def __init__(self, *args, **kwargs):
        self.user = kwargs.pop('user', None)
        # self.member = kwargs.pop('member', None)
        self.finance_year = kwargs.pop('finance_year', None)
        super(IndDepositForm, self).__init__(*args, **kwargs)

    @transaction.atomic
    def clean(self):
        user = self.user
        financial_year = self.finance_year
        cleaned_data = super().clean()
        date_created = cleaned_data.get('date_created')
        account = cleaned_data.get('unique')
        depo_amount = cleaned_data.get('amount')
        rec_no = datetime.datetime.now().strftime("%y%m%d%H%M%S")
        the_account = cleaned_data.get('the_account')
        dep_narration = cleaned_data.get('dep_narration')
        ref_no = cleaned_data.get('dep_ref') or rec_no
        this_biz = Business.objects.filter(id=biz_id(user)).first()
        notify_type = NotiSettings.objects.filter(business=biz_data(user)).first()
        template_file_link = 'sacco/emails/transactional.html'
        currency = CurrencySetting.objects.filter(business=biz_data(user)).first()

        if not date_created:
            date_created = datetime.datetime.now().date()

        from_initial = depo_amount.replace(',', '')
        # print(from_initial)
        initial = int(float(from_initial))

        income_acc = Account.objects.filter(category__name="Deposit Income", business=biz_data(user)).first()

        this_account = MemberAccount.objects.get(id=account)

        # chart of accounts
        member_acct = AccountBroker.objects.filter(the_account_id=account).first()
        member_acc = Account.objects.filter(member_id=member_acct.the_account_id).first()

        acc_involved = Account.objects.get(id=int(the_account), business=biz_data(user))
        depo_dict = []
        depo_cgs = TransactionCharge.objects.filter(account_type_id=member_acct.the_account.account_type,
                                                    charge_type='d', status=True)
        varying = []
        flat = []
        # df = pd.DataFrame()
        for x in depo_cgs:
            if x.account_type.deposit_charge_vary:
                varying.append(x)
            else:
                flat.append(x)

        depo_charges = 0
        if depo_cgs.exists():
            depo_dataframe = pd.DataFrame(depo_cgs.values())

            for d in depo_cgs:
                if d.account_type.deposit_charge_vary:
                    # print('ze')
                    if d.start <= int(initial) <= d.end:
                        amount = d.charge
                        if d.is_charge_percentage:
                            amount = int(initial) * (d.charge / 100)
                        depo_dict.append({'id': d.id, 'amount': amount, 'account_type_name': d.account_type.name})
                        # print(depo_dict)
                else:
                    amount = d.charge
                    if d.is_charge_percentage:
                        amount = initial * (d.charge / 100)
                    # print(d.id)
                    depo_dict.append({'id': d.id, 'amount': amount, 'account_type_name': d.account_type.name})
                    # print('---')
                # print('final:  {}'.format(depo_dict))
                depo_data_charges = pd.DataFrame(depo_dict)
                # print('my {}'.format(depo_data_charges))
                if not depo_dataframe.empty and not depo_data_charges.empty:
                    # print(depo_dataframe)
                    # print('se {}'.format(depo_data_charges))
                    deposit_frame = pd.merge(depo_dataframe, depo_data_charges, on='id')
                    # print('charge {}'.format(depo_charges))
                    if d.account_type.deposit_charge:
                        if d.account_type.deposit_charge_vary:
                            dfs = deposit_frame[(deposit_frame['start'] <= initial) & (initial <= deposit_frame['end'])]
                        else:
                            dfs = deposit_frame
                    else:
                        dfs = pd.DataFrame(columns=['id', 'start', 'end', 'charge', 'account_type_id',
                                                    'charge_type', 'status', 'is_charge_percentage', 'created_date',
                                                    'deletion_date', 'amount', 'account_type_name'])

                    depo_charges = deposit_frame['amount'].sum()
                    # print(f'my cake {dfs}')
                    if not dfs.empty:
                        for index, row in dfs.iterrows():
                            Transactions.objects.create(
                                branch=biz_staff_branch(user),
                                transaction_type='Charge on deposit',
                                account_dr=member_acc,
                                account_cr=income_acc,
                                narration='Charge on deposit of ' + str(initial) + ' {}'.format(
                                    row["account_type_name"]),
                                reporting_amount=row["amount"],
                                added_by=biz_staff(user),
                                financial_year=financial_year,
                                tx_date=date_created,
                                receipt=rec_no,
                                reference=ref_no
                            )
                        break
                else:
                    depo_charges = 0

        trans = Transactions.objects.create(
            branch=biz_staff_branch(user),
            transaction_type='Deposit',
            account_dr=acc_involved,
            account_cr=member_acc,
            narration="Deposit on {}".format(this_account.acc_number) if not dep_narration else dep_narration,
            reporting_amount=initial,
            added_by=biz_staff(user),
            financial_year=financial_year,
            tx_date=date_created,
            receipt=rec_no,
            reference=ref_no
        )

        # Depositor's account -ve transactions
        member_debits = \
            list(Transactions.objects.filter(account_dr=member_acc).aggregate(total=Sum('reporting_amount')).values())[
                0]
        if member_debits is None:
            member_debits = 0

        # Depositor's account +ve transactions
        member_credits = \
            list(Transactions.objects.filter(account_cr=member_acc).aggregate(total=Sum('reporting_amount')).values())[
                0]

        # Depositor's account all transactions difference
        this_account.balance = member_credits - member_debits
        this_account.save()
        # print('new balo {}'.format(this_account.balance))
        the_member_ = MemberAccount.objects.get(id=this_account.id)
        # print(the_member_.balance)
        smsSettingsObj = SaccoSmsSettings.objects.get(when_to_send='On deposit transaction', business=biz_data(user))
        # print('notify_type', notify_type.notification_type)
        subject = "Deposit request processed successfully"
        business = this_biz.name
        email = member_acct.members.biodata.email
        message = f'Dear {this_biz.short_name} member, a deposit of {currency.currency} {initial} has been made on your account. Fee {currency.currency} {depo_charges}. Account Balance is {the_member_.balance}. TXN ID {trans.txno}. Thank you'
        html_body = f'<p>Dear <b>{member_acct.members.biodata.name}</b>,</p>  Your Deposit request with the following details has been processed successfully.<br/>Transaction Number: {trans.txno}<br/>Amount: {currency.currency} {initial} <br/>Description: {trans.narration} <br/>Date: {trans.tx_date} <p>Please call our contact centre on +{this_biz.contact} in case you require further details.</p>'
        if smsSettingsObj.status:
            if notify_type.notification_type == 1:  # Send by SMS only
                if member_acct.members.biodata.contact:
                    checkAndSendMessage(user, message, member_acct.members.biodata.contact,
                                        member_acct.members.biodata.name, account=member_acc)
            elif notify_type.notification_type == 2:  # Send by Email only
                if member_acct.members.biodata.email:
                    sendTransEmail(template_file_link, html_body, business, subject, email)

            elif notify_type.notification_type == 3:  # Send by both SMS and Email
                if member_acct.members.biodata.contact:
                    checkAndSendMessage(user, message, member_acct.members.biodata.contact,
                                        member_acct.members.biodata.name, account=member_acct)

                if member_acct.members.biodata.email:
                    sendTransEmail(template_file_link, html_body, business, subject, email)
        message = f"{user.staff.biodata.name} has deposited {depo_amount} to account {this_account.acc_number}"
        ActivityLog.objects.create(actor=user,
                                   action_type=ACCOUNT,
                                   title='Deposit',
                                   branch=branch_id(user),
                                   remarks=message)
        return cleaned_data


class IndWithForm(forms.Form):
    date_created = forms.DateField(required=True)
    with_unique = forms.IntegerField(required=True)
    amount = forms.CharField(required=True)
    narration = forms.CharField(required=False)
    the_account = forms.IntegerField(required=False)
    with_narration = forms.CharField(required=False)
    with_ref = forms.CharField(required=False)

    def __init__(self, *args, **kwargs):
        self.user = kwargs.pop('user', None)
        self.finance_year = kwargs.pop('finance_year', None)
        super(IndWithForm, self).__init__(*args, **kwargs)

    @transaction.atomic
    def clean(self):
        user = self.user
        financial_year = self.finance_year
        cleaned_data = super().clean()
        date_created = cleaned_data.get('date_created')
        account = cleaned_data.get('with_unique')
        depo_amount = cleaned_data.get('amount')
        the_account = cleaned_data.get('the_account')
        rec_no = datetime.datetime.now().strftime("%y%m%d%H%M%S")
        with_narration = cleaned_data.get('with_narration')
        ref_no = cleaned_data.get('with_ref') or rec_no
        this_biz = Business.objects.filter(id=biz_id(user)).first()
        notify_type = NotiSettings.objects.filter(business=biz_data(user)).first()
        template_file_link = 'sacco/emails/transactional.html'
        currency = CurrencySetting.objects.filter(business=biz_data(user)).first()
        # print(the_account)

        if not date_created:
            date_created = datetime.datetime.now().date()

        from_initial = depo_amount.replace(',', '')
        # print(from_initial)
        initial = int(float(from_initial))
        income_acc = Account.objects.filter(category__name="Withdraw Income", business=biz_data(user)).first()
        this_account = MemberAccount.objects.get(id=account)
        # print(this_account.acc_number)

        # chart of accounts
        member_acct = AccountBroker.objects.filter(the_account_id=account).first()
        member_acc = Account.objects.filter(member_id=member_acct.the_account_id).first()

        acc_involved = Account.objects.get(id=int(the_account), business=biz_data(user))

        depo_dict = []

        depo_cgs = TransactionCharge.objects.filter(account_type_id=member_acct.the_account.account_type,
                                                    charge_type='w', status=True)
        varying = []
        flat = []
        # df = pd.DataFrame()
        for x in depo_cgs:
            if x.account_type.deposit_charge_vary:
                varying.append(x)
            else:
                flat.append(x)

        depo_charges = 0
        if depo_cgs.exists():
            depo_dataframe = pd.DataFrame(depo_cgs.values())
            # print('stage1')
            # print(depo_cgs)
            # withdraw charges
            for d in depo_cgs:
                if d.account_type.withdraw_charge_vary:
                    # print('ze')
                    if d.start <= int(initial) <= d.end:
                        amount = d.charge
                        if d.is_charge_percentage:
                            amount = int(initial) * (d.charge / 100)
                        depo_dict.append({'id': d.id, 'amount': amount, 'account_type_name': d.account_type.name})
                        # print(depo_dict)
                else:
                    amount = d.charge
                    if d.is_charge_percentage:
                        amount = initial * (d.charge / 100)
                    # print(d.id)
                    depo_dict.append({'id': d.id, 'amount': amount, 'account_type_name': d.account_type.name})
                #     print('---')
                # print('final:  {}'.format(depo_dict))
                depo_data_charges = pd.DataFrame(depo_dict)
                # print('my {}'.format(depo_data_charges))
                if not depo_dataframe.empty and not depo_data_charges.empty:
                    # print(depo_dataframe)
                    # print('se {}'.format(depo_data_charges))
                    deposit_frame = pd.merge(depo_dataframe, depo_data_charges, on='id')
                    # print('charge {}'.format(depo_charges))
                    if d.account_type.withdraw_charge:
                        if d.account_type.withdraw_charge_vary:
                            dfs = deposit_frame[(deposit_frame['start'] <= initial) & (initial <= deposit_frame['end'])]
                        else:
                            dfs = deposit_frame
                    else:
                        dfs = pd.DataFrame(columns=['id', 'start', 'end', 'charge', 'account_type_id',
                                                    'charge_type', 'status', 'is_charge_percentage', 'created_date',
                                                    'deletion_date', 'amount', 'account_type_name'])
                    # print(f'final frame {dfs}')
                    depo_charges = dfs['amount'].sum()
                    if not dfs.empty:
                        for index, row in dfs.iterrows():
                            Transactions.objects.create(
                                branch=biz_staff_branch(user),
                                transaction_type='Charge on withdraw',
                                account_dr=member_acc,
                                account_cr=income_acc,
                                narration='Charge on withdraw of ' + str(initial) + ' {}'.format(
                                    row["account_type_name"]),
                                reporting_amount=row["amount"],
                                added_by=biz_staff(user),
                                financial_year=financial_year,
                                tx_date=date_created,
                                receipt=rec_no,
                                reference=ref_no
                            )
                        break
                else:
                    depo_charges = 0

        trans = Transactions.objects.create(
            branch=biz_staff_branch(user),
            transaction_type='Withdraw',
            account_dr=member_acc,
            account_cr=acc_involved,
            narration="Withdraw from {}".format(this_account.acc_number) if not with_narration else with_narration,
            reporting_amount=initial,
            added_by=biz_staff(user),
            financial_year=financial_year,
            tx_date=date_created,
            receipt=rec_no,
            reference=ref_no
        )

        member_debits = list(Transactions.objects.filter(account_dr=member_acc).aggregate(
            total=Sum('reporting_amount')).values())[0]
        if member_debits is None:
            member_debits = 0
        member_credits = list(Transactions.objects.filter(account_cr=member_acc).aggregate(
            total=Sum('reporting_amount')).values())[0]
        this_account.balance = member_credits - member_debits
        this_account.save()

        the_member_ = MemberAccount.objects.get(id=this_account.id)
        # print(the_member_.balance)

        smsSettingsObj = SaccoSmsSettings.objects.get(when_to_send='On withdraw transaction',
                                                      business=biz_data(user))
        # print('notify_type', notify_type.notification_type)
        subject = "Withdraw request processed successfully"
        business = this_biz.name
        email = member_acct.members.biodata.email
        message = f'Dear {this_biz.short_name} member, a withdraw of {currency.currency} {initial} has been made on your account. Fee {currency.currency} {depo_charges}. Bal {the_member_.balance}. TXN ID {trans.txno}.'
        html_body = f'<p>Dear {member_acct.members.biodata.name},</p>  Your withdraw request with the following details has been processed successfully.<br/>Transaction Number: {trans.txno}<br/>Amount: {currency.currency}{initial} <br/>Description: {trans.narration} <br/>Date: {trans.tx_date} <p>Please call our contact centre on +{this_biz.contact} in case you require further details.</p>'
        if smsSettingsObj.status:
            if notify_type.notification_type == 1:  # Send by SMS only
                if member_acct.members.biodata.contact:
                    checkAndSendMessage(user, message, member_acct.members.biodata.contact,
                                        member_acct.members.biodata.name, account=member_acc)
            elif notify_type.notification_type == 2:  # Send by Email only
                if member_acct.members.biodata.email:
                    send_mail = sendTransEmail(template_file_link, html_body, business, subject, email)

            elif notify_type.notification_type == 3:  # Send by both SMS and Email
                if member_acct.members.biodata.contact:
                    checkAndSendMessage(user, message, member_acct.members.biodata.contact,
                                        member_acct.members.biodata.name, account=member_acc)

                if member_acct.members.biodata.email:
                    sendTransEmail(template_file_link, html_body, business, subject, email)
            message = f"{user.staff.biodata.name} has withdrawn {depo_amount} from account {this_account.acc_number}"
            ActivityLog.objects.create(actor=user,
                                       action_type=ACCOUNT,
                                       title='Withdraw',
                                       branch=branch_id(user),
                                       remarks=message)
        return cleaned_data


class SMSPlanForm(forms.Form):
    amount = forms.FloatField()
    phone = forms.CharField()

    def clean_phone(self):
        cleaned_data = self.cleaned_data
        phone = cleaned_data['phone']
        if phone.startswith('0'):
            phone = phone.replace('0', '+256', 1)
        elif phone.startswith('256'):
            phone = phone.replace('256', '+256', 1)
        return phone


class SmsSettingForm(forms.Form):
    option = forms.ChoiceField(widget=forms.Select, choices=SMSCHARGE)
    amount = forms.FloatField()

    def __init__(self, *args, **kwargs):
        self.user = kwargs.pop('user', None)
        super(SmsSettingForm, self).__init__(*args, **kwargs)

    def save(self):
        cleaned_data = self.cleaned_data
        option = cleaned_data['option']
        amount = cleaned_data['amount']
        notifs = NotiSettings.objects.filter(business=biz_data(self.user)).first()
        notifs.faces_charge = option
        notifs.save()
        sms_cost, created = SMSCost.objects.get_or_create(business=biz_data(self.user))
        sms_cost.cost = amount
        sms_cost.save()
        return cleaned_data


class EditChargeForm(forms.Form):
    name = forms.CharField(),
    is_revenue = forms.CharField(),
    is_fine = forms.CharField(),
    application = forms.CharField(),
    amount = forms.FloatField(),
    is_percentage = forms.CharField(),
    execution_period_number = forms.IntegerField(min_value=1)
    loan_type = forms.ChoiceField(widget=forms.MultipleChoiceField())

    def __init__(self, *args, **kwargs):
        self.user = kwargs.pop('user', None)
        self.charge = kwargs.pop('charge', None)
        super(EditChargeForm, self).__init__(*args, **kwargs)

    def clean_is_fine(self):
        cleaned_data = self.cleaned_data
        is_fine = cleaned_data['is_fine']
        if is_fine == 'f':
            return True
        return False

    def clean_is_percentage(self):
        cleaned_data = self.cleaned_data
        is_percentage = cleaned_data['is_percentage']
        if is_percentage == 'p':
            return True
        return False

    def save(self, commit=True):
        charge = self.charge
        cleaned_data = self.cleaned_data
        if commit:
            if "execution_period_number" in cleaned_data and "execution_period" in cleaned_data:
                theExecPeriodNum = int(cleaned_data['execution_period_number'])
                theExecPeriod = cleaned_data['execution_period']
            else:
                theExecPeriodNum = None
                theExecPeriod = None
            charge.charge = cleaned_data['name']
            charge.is_revenue = cleaned_data['is_revenue']
            charge.application = cleaned_data['application']
            charge.amount = cleaned_data['amount']
            charge.is_percentage = cleaned_data['is_percentage']
            charge.execution_period = theExecPeriod
            charge.execution_period_number = theExecPeriodNum
            charge.is_fine = cleaned_data['is_fine']
            charge.save()
            ApplicationAccountOrLoanType.objects.filter(general_charge=charge).delete()

            if cleaned_data['application'] == 'l':
                # get all the id of loan types and save to database
                allLoanTypes = cleaned_data['loan_type']
                for id_loan in allLoanTypes:
                    try:
                        loanTypeObj = LoanTypes.objects.get(id=int(id_loan))
                        ApplicationAccountOrLoanType.objects.create(general_charge=charge, loan_type=loanTypeObj)
                    except ApplicationAccountOrLoanType.DoesNotExist as e:
                        logger.error(str(e))

            elif cleaned_data['application'] == 'r':
                allAccountTypes = cleaned_data['account_type']
                for id_loan in allAccountTypes:
                    try:
                        accountTypeObj = AccountTypes.objects.get(id=int(id_loan))
                        ApplicationAccountOrLoanType.objects.create(
                            general_charge=charge, account_type=accountTypeObj)
                    except AccountTypes.DoesNotExist as e:
                        logger.error(str(e))

            elif cleaned_data['application'] == 'o':
                if cleaned_data['other_type'] == 'l':
                    allLoanTypes = cleaned_data['loan_type']
                    # print(allLoanTypes)
                    for id_loan in allLoanTypes:
                        try:
                            loanTypeObj = LoanTypes.objects.get(id=int(id_loan))
                            ApplicationAccountOrLoanType.objects.create(general_charge=charge, loan_type=loanTypeObj)
                        except ApplicationAccountOrLoanType.DoesNotExist as e:
                            logger.error(str(e))
                else:
                    allAccountTypes = cleaned_data['account_type']
                    for id_loan in allAccountTypes:
                        try:
                            accountTypeObj = AccountTypes.objects.get(id=int(id_loan))
                            ApplicationAccountOrLoanType.objects.create(general_charge=charge,
                                                                        account_type=accountTypeObj)
                        except AccountTypes.DoesNotExist as e:
                            logger.error(str(e))

            log_title = 'General charge update'
            message = f"{self.user.staff.biodata.name} added a general charge {charge.charge}"
            ActivityLog.objects.create(actor=self.user, title=log_title,
                                       action_type=UPDATE,
                                       remarks=message,
                                       branch=branch_id(self.user))
        return charge
