import datetime
import json

import pandas as pd
from dateutil.parser import parse
from django.db import transaction, models
from django.db.models import Sum, Q
from django_rq import job
from django.utils.translation import gettext as _

from loans.models import ApplicationAccountOrLoanType
from sacco.models import Member, Biodata, AccountBroker, MemberAccount, TransactionCharge, BusinessShares, \
    SaccoSmsSettings, OtherSettings
from sacco.utils import biz_staff, biz_data, biz_staff_branch, checkAndSendMessage
from transactions.models import AccountCategory, Account, Transactions, SharesTransactions
from utils.general import random_with_N_digits, ug_dial


@job
def upload_member(file, user, biz, account_type, financial_year, balo_date):
    upload = pd.read_excel(file, comment='#', dtype={"Account Number": str})
    global employee_no
    upload.fillna('', inplace=True)
    upload.columns.astype(str)
    upload.columns = upload.columns.str.title()
    from_frame = upload.to_json(orient='records', indent=4, date_format='iso')
    json_obj = json.loads(from_frame)
    open_reserve = Account.objects.get(name='Opening Reserves', business=biz_data(user))
    business_shares = BusinessShares.objects.filter(business=biz_data(user)).first()
    with transaction.atomic():
        for obj in json_obj:
            if 'Employee Number' in obj:
                full_names, contact, other_contact, shares, raw_savings, join_date, gender, \
                acc_no, employee_no = str(obj['Full Name']).strip(), str(obj['Contact']).strip(), \
                                      str(obj['Other Contact']).strip(), obj['Shares'], str(obj['Savings Balance']), \
                                      obj['Join Date'], str(obj['Gender']), str(obj['Account Number']).strip(), str(
                    obj['Employee Number'])
            else:
                full_names, contact, other_contact, shares, raw_savings, join_date, gender, \
                acc_no = str(obj['Full Name']).strip(), str(obj['Contact']).strip(), str(
                    obj['Other Contact']).strip(), obj[
                             'Shares'], str(obj['Savings Balance']), \
                         obj['Join Date'], str(obj['Gender']), str(obj['Account Number'])
            if len(gender) > 1:
                gender = gender[0]

            if join_date:
                try:
                    python_date = datetime.datetime.strptime(join_date, '%Y-%m-%dT%H:%M:%S.%fZ')
                    join_day = python_date.date()
                except Exception as e:
                    join_day = None
                    print(str(e))
            else:
                join_day = None

            if acc_no:
                if acc_no.endswith('.0'):
                    acc_no = acc_no.replace('.0', '')

            if raw_savings:
                my_raw_savings = raw_savings.replace(',', '')
                conv_raw = float(my_raw_savings)
                savings = int(conv_raw)

            if full_names:
                myc = int(float(contact))
                if contact:
                    contact = ug_dial(str(myc))
                else:
                    contact = None

                if other_contact:
                    myo = int(float(contact))
                    other_contact = ug_dial(str(myo))
                else:
                    other_contact = None

                if employee_no.endswith('.0'):
                    employee_no = employee_no.replace('.0', '')

                if gender:
                    gender = gender.strip().upper()

                if biz.auto_acc_no:
                    print('okay')
                    if acc_no:
                        acc_number = acc_no
                    else:
                        while MemberAccount.objects.filter(acc_number=acc_number,
                                                           member_account__business=biz_data(user)).exists():
                            acc_number = random_with_N_digits(8)
                else:
                    acc_number = acc_no
                print(acc_number)
                if acc_number:
                    if not MemberAccount.objects.filter(acc_number=acc_number,
                                                        member_account__business=biz_data(user)).exists():

                        bio = Biodata.objects.create(name=full_names.title(), contact=contact,
                                                     other_contact=other_contact, gender=gender,
                                                     created_by=biz_staff(user),
                                                     business=biz_data(user), employee_no=employee_no)
                        if shares:
                            member = Member.objects.create(biodata=bio, created_by=biz_staff(user), shares=int(shares),
                                                           branch=biz_staff_branch(user), date_joined=join_day,
                                                           member_type='s')
                            # Shares transactions here
                            SharesTransactions.objects.create(
                                buyer_id=member.id,
                                shares=member.shares,
                                date=datetime.datetime.today(),
                                narration='Opening shares',
                                branch=biz_staff_branch(user)
                            )
                        else:
                            member = Member.objects.create(biodata=bio, created_by=biz_staff(user),
                                                           branch=biz_staff_branch(user), date_joined=join_day,
                                                           member_type='o')

                        account = MemberAccount.objects.create(acc_number=acc_number, account_type_id=account_type,
                                                               balance=savings)
                        AccountBroker.objects.create(members=member, the_account=account, business=biz_data(user))
                        cat = AccountCategory.objects.filter(business=biz_data(user), name="Members").first()
                        # chart of accounts
                        member_acc = Account.objects.create(category=cat, business=biz_data(user), member=account,
                                                            added_by=biz_staff(user))
                        Transactions.objects.create(
                            branch=biz_staff_branch(user),
                            transaction_type='Opening reserve',
                            account_dr=open_reserve,
                            account_cr=member_acc,
                            narration='Opening balance',
                            reporting_amount=savings,
                            added_by=biz_staff(user),
                            financial_year=financial_year,
                            tx_date=balo_date
                        )
                # query seller shares transactions
                seller_shares = SharesTransactions.objects.filter(seller__isnull=True, branch__business=biz_data(user))
                total_sold = 0
                for s in seller_shares:
                    total_sold += s.shares

                # update business shares sold
                business_shares.sold = total_sold
                business_shares.save()

    return 'upload_member'


def upload_new_members(file, user, biz, account_type, financial_year, balo_date, the_account):
    upload = pd.read_excel(file, comment='#', dtype={"Account Number": str})
    global employee_no
    upload.fillna('', inplace=True)
    upload.columns.astype(str)
    upload.columns = upload.columns.str.title()
    from_frame = upload.to_json(orient='records', indent=4, date_format='iso')
    json_obj = json.loads(from_frame)
    acc_involved = Account.objects.get(id=int(the_account), business=biz_data(user))
    charges_acc = Account.objects.get(category__name="Registration Income",
                                      business=biz_data(user))
    business_shares = BusinessShares.objects.filter(business=biz_data(user)).first()
    depo_dict = []
    gen_dict = []
    gen_cgs = ApplicationAccountOrLoanType.objects.filter(account_type_id=account_type,
                                                          general_charge__application='r')
    gen_charges = 0
    depo_charges = 0
    with transaction.atomic():
        for obj in json_obj:
            if 'Employee Number' in obj:
                full_names, contact, other_contact, shares, raw_savings, join_date, gender, \
                acc_no, employee_no = str(obj['Full Name']).strip(), str(obj['Contact']).strip(), \
                                      str(obj['Other Contact']).strip(), obj['Shares'], str(obj['Savings Balance']), \
                                      obj['Join Date'], str(obj['Gender']), str(obj['Account Number']).strip(), str(
                    obj['Employee Number'])
            else:
                full_names, contact, other_contact, shares, raw_savings, join_date, gender, \
                acc_no = str(obj['Full Name']).strip(), str(obj['Contact']).strip(), str(
                    obj['Other Contact']).strip(), obj[
                             'Shares'], str(obj['Savings Balance']), \
                         obj['Join Date'], str(obj['Gender']), str(obj['Account Number'])
            if len(gender) > 1:
                gender = gender[0]

            if join_date:
                try:
                    python_date = parse(join_date)
                    join_day = python_date.date()
                except Exception as e:
                    join_day = None
                    print(str(e))
            else:
                join_day = None

            if acc_no:
                if acc_no.endswith('.0'):
                    acc_no = acc_no.replace('.0', '')

            if raw_savings:
                my_raw_savings = raw_savings.replace(',', '')
                conv_raw = float(my_raw_savings)
                savings = int(conv_raw)

            if full_names:
                myc = int(float(contact))
                if contact:
                    contact = ug_dial(str(myc))
                else:
                    contact = None

                if other_contact:
                    myo = int(float(contact))
                    other_contact = ug_dial(str(myo))
                else:
                    other_contact = None

                if employee_no.endswith('.0'):
                    employee_no = employee_no.replace('.0', '')

                if gender:
                    gender = gender.strip().upper()

                acc_number = acc_no

                if biz.auto_acc_no:
                    if acc_no:
                        print('yeah')
                        acc_number = acc_no
                    else:
                        print('part2')
                        while MemberAccount.objects.filter(acc_number=acc_number,
                                                           member_account__business=biz_data(user)).exists():
                            print('chk')
                            acc_number = random_with_N_digits(8)
                        else:
                            acc_number = random_with_N_digits(8)

                if not MemberAccount.objects.filter(acc_number=acc_number,
                                                    member_account__business=biz_data(user)).exists():
                    bio = Biodata.objects.create(name=full_names.title(), contact=contact,
                                                 other_contact=other_contact, gender=gender,
                                                 created_by=biz_staff(user),
                                                 business=biz_data(user), employee_no=employee_no)
                    if shares:
                        setting_status = OtherSettings.objects.filter(business=biz_data(user)).first()
                        member = Member.objects.create(biodata=bio, created_by=biz_staff(user), shares=int(shares),
                                                       branch=biz_staff_branch(user), date_joined=join_day,
                                                       member_type='s')
                        narrative = f'Sale of {member.shares} shares to {member.biodata.name} at {setting_status.share_price} each.'
                        # Shares transactions here
                        shares_record = SharesTransactions.objects.create(
                            buyer_id=member.id,
                            shares=member.shares,
                            date=datetime.datetime.today(),
                            narration='Opening shares',
                            branch=biz_staff_branch(user)
                        )
                        buyer_shares = SharesTransactions.objects.filter(Q(buyer=member) | Q(seller=member))
                        # print("results = %s" % buyer_shares.count())
                        total_bought = 0
                        total_sold = 0
                        for b in buyer_shares:
                            if b.buyer == member:
                                total_bought += b.shares
                            if b.seller == member:
                                total_sold += b.shares

                        current_shares = total_bought - total_sold

                        # update buyer shares balance
                        member.shares = current_shares
                        member.save()

                        # query seller shares transactions
                        seller_shares = SharesTransactions.objects.filter(seller__isnull=True)
                        total_bought = 0
                        total_sold = 0
                        for s in seller_shares:
                            total_sold += s.shares

                        # update business shares sold
                        business_shares.sold = total_sold
                        business_shares.save()

                        # Record a finance transaction
                        # shares account
                        shares_account = Account.objects.filter(business=biz_data(user),
                                                                category__name='Shares').first()
                        share_cost = int(setting_status.share_price) * int(shares)

                        Transactions.objects.create(
                            branch=biz_staff_branch(user),
                            transaction_type='Sale of shares',
                            account_dr=shares_account,
                            account_cr_id=the_account,
                            narration=narrative,
                            reporting_amount=share_cost,
                            added_by=biz_staff(user),
                            shares=shares_record,
                            financial_year=financial_year
                        )
                    else:
                        member = Member.objects.create(biodata=bio, created_by=biz_staff(user),
                                                       branch=biz_staff_branch(user), date_joined=join_day,
                                                       member_type='o')
                    account = MemberAccount.objects.create(acc_number=acc_number, account_type_id=account_type,
                                                           balance=savings)
                    AccountBroker.objects.create(members=member, the_account=account, business=biz_data(user))
                    cat = AccountCategory.objects.filter(business=biz_data(user), name="Members").first()
                    # chart of accounts
                    member_acc = Account.objects.create(category=cat, business=biz_data(user), member=account,
                                                        added_by=biz_staff(user))
                    if gen_cgs.exists():
                        gen_dataframe = pd.DataFrame(gen_cgs.values())

                        for gen in gen_cgs:
                            amount = gen.general_charge.amount
                            if gen.general_charge.is_percentage:
                                amount = savings * (gen.general_charge / 100)
                            gen_dict.append(
                                {
                                    'id': gen.id,
                                    'amount': amount
                                }
                            )
                        gen_charges_df = pd.DataFrame(gen_dict)
                        gen_frame = pd.merge(gen_dataframe, gen_charges_df, on='id')

                        # summing general charges dataframe
                        gen_charges = gen_frame['amount'].sum()
                        for ch in gen_cgs:
                            Transactions.objects.create(
                                branch=biz_staff_branch(user),
                                transaction_type='Registration',
                                account_dr=member_acc,
                                account_cr=charges_acc,
                                narration="{}".format(ch.general_charge.charge),
                                reporting_amount=ch.general_charge.amount,
                                added_by=biz_staff(user),
                                financial_year=financial_year
                            )

                    dep_cgs = TransactionCharge.objects.filter(account_type_id=account_type,
                                                               charge_type='d',
                                                               status=True)
                    if dep_cgs.exists():
                        print(savings)
                        depo_dataframe = pd.DataFrame(dep_cgs.values())
                        # deposit charges
                        for d in dep_cgs:
                            if d.account_type.deposit_charge_vary:
                                if d.start <= float(savings) <= d.end:
                                    amount = d.charge
                                    if d.is_charge_percentage:
                                        amount = savings * (d.charge / 100)
                                    depo_dict.append(
                                        {'id': d.id, 'amount': amount, 'account_type_name': d.account_type.name})
                                    print(f'the depo {depo_dict}')
                            else:
                                amount = d.charge
                                if d.is_charge_percentage:
                                    amount = savings * (d.charge / 100)
                                depo_dict.append(
                                    {'id': d.id, 'amount': amount, 'account_type_name': d.account_type.name})
                            depo_data_charges = pd.DataFrame(depo_dict)
                            if not depo_dataframe.empty and not depo_data_charges.empty:
                                deposit_frame = pd.merge(depo_dataframe, depo_data_charges, on='id')
                                if d.account_type.deposit_charge_vary:
                                    dfs = deposit_frame[(deposit_frame['start'] <= savings) & (savings <= deposit_frame['end'])]
                                else:
                                    dfs = deposit_frame
                                print(f'my {dfs}')
                                depo_charges = deposit_frame['amount'].sum()
                                print(f'my charges {depo_charges}')

                                if not dfs.empty:
                                    for index, row in dfs.iterrows():
                                        print('hei')
                                        Transactions.objects.create(
                                            branch=biz_staff_branch(user),
                                            transaction_type='Deposit',
                                            account_dr=member_acc,
                                            account_cr=charges_acc,
                                            narration="Charge on deposit of {}".format(savings),
                                            reporting_amount=row["amount"],
                                            added_by=biz_staff(user),
                                            financial_year=financial_year
                                        )
                                break
                    Transactions.objects.create(
                        branch=biz_staff_branch(user),
                        transaction_type="Deposit",
                        account_dr=acc_involved,
                        account_cr=member_acc,
                        narration="Initial deposit on registration",
                        reporting_amount=savings,
                        added_by=biz_staff(user),
                        financial_year=financial_year,
                        tx_date=balo_date
                    )
                    if not shares:
                        total_charges = depo_charges + gen_charges
                    else:
                        total_charges = depo_charges + gen_charges + share_cost
                    member_debits = list(Transactions.objects.filter(account_dr=member_acc).aggregate(
                        total=Sum('reporting_amount')).values())[0]
                    if member_debits is None:
                        member_debits = 0
                    print(f'my member debits {member_debits}')
                    member_credits = list(Transactions.objects.filter(account_cr=member_acc).aggregate(
                        total=Sum('reporting_amount')).values())[0]
                    print(f'my member credits {member_credits}')
                    account.balance = member_credits - member_debits
                    account.save()
                    smsSettingsObj = SaccoSmsSettings.objects.get(when_to_send='on new member registration',
                                                                  business=biz_data(user))
                    message = f'From: {biz.name}\n You have deposited {savings} ugx and incurred total charges of {total_charges}UGX\n Your new balance is {account.balance}'
                    if smsSettingsObj.status:
                        if bio.contact:
                            theResponse = checkAndSendMessage(user, message, contact, bio.name)
                            print(theResponse)
    return 'upload_member'


@job
def bulk_member_deposit(user, acounts_list, depo, the_account, financial_year):
    error_messages = {
        'account_exists': _('Insufficient balance'),
    }
    acc_involved = Account.objects.get(id=int(the_account), business=biz_data(user))
    income_acc = Account.objects.filter(category__name="Deposit Income", business=biz_data(user)).first()
    from_initial = depo.replace(',', '')
    initial = int(float(from_initial))

    with transaction.atomic():

        for acc in acounts_list:
            try:
                this_account = MemberAccount.objects.get(id=acc)
                member_acc = Account.objects.filter(member=this_account, business=biz_data(user)).first()
                member_acct = AccountBroker.objects.filter(the_account=this_account).first()
                depo_dict = []

                depo_cgs = TransactionCharge.objects.filter(account_type_id=member_acct.the_account.account_type,
                                                            charge_type='d', status=True)
                depo_charges = 0
                if depo_cgs.exists():
                    depo_dataframe = pd.DataFrame(depo_cgs.values())

                    # deposit charges
                    for d in depo_cgs:
                        if d.account_type.deposit_charge_vary:
                            if d.start <= int(initial) <= d.end:
                                amount = d.charge
                                if d.is_charge_percentage:
                                    amount = int(initial) * (d.charge / 100)
                                depo_dict.append(
                                    {'id': d.id, 'amount': amount, 'account_type_name': d.account_type.name})
                        else:
                            amount = d.charge
                            if d.is_charge_percentage:
                                amount = initial * (d.charge / 100)
                            depo_dict.append({'id': d.id, 'amount': amount, 'account_type_name': d.account_type.name})
                        depo_data_charges = pd.DataFrame(depo_dict)
                        if not depo_dataframe.empty and not depo_data_charges.empty:
                            deposit_frame = pd.merge(depo_dataframe, depo_data_charges, on='id')
                            if d.account_type.deposit_charge:
                                if d.account_type.deposit_charge_vary:
                                    dfs = deposit_frame[
                                        (deposit_frame['start'] <= initial) & (initial <= deposit_frame['end'])]
                                else:
                                    dfs = deposit_frame
                            else:
                                dfs = pd.DataFrame(columns=['id', 'start', 'end', 'charge', 'account_type_id',
                                                            'charge_type', 'status', 'is_charge_percentage',
                                                            'created_date',
                                                            'deletion_date', 'amount', 'account_type_name'])
                            depo_charges = dfs['amount'].sum()
                            if not dfs.empty:
                                for index, row in dfs.iterrows():
                                    Transactions.objects.create(
                                        branch=biz_staff_branch(user),
                                        transaction_type='Charge on deposit',
                                        account_dr=member_acc,
                                        account_cr=income_acc,
                                        narration='Deposit Charge on {}'.format(row["account_type_name"]),
                                        reporting_amount=row["amount"],
                                        added_by=biz_staff(user),
                                        financial_year=financial_year,
                                    )
                                break
                        else:
                            depo_charges = 0

                    print('my deposit charges{}'.format(depo_charges))

                Transactions.objects.create(
                    branch=biz_staff_branch(user),
                    transaction_type='Deposit',
                    account_dr=acc_involved,
                    account_cr=member_acc,
                    narration="Deposit on {}".format(this_account.acc_number),
                    reporting_amount=initial,
                    added_by=biz_staff(user),
                    financial_year=financial_year,
                )

                member_debits = list(Transactions.objects.filter(account_dr=member_acc).aggregate(
                    total=Sum('reporting_amount')).values())[0]
                if member_debits is None:
                    member_debits = 0
                member_credits = list(Transactions.objects.filter(account_cr=member_acc).aggregate(
                    total=Sum('reporting_amount')).values())[0]
                # print(f'my member credits {member_credits}')
                this_account.balance = member_credits - member_debits
                this_account.save()
                the_member_ = MemberAccount.objects.get(id=this_account.id)
                smsSettingsObj = SaccoSmsSettings.objects.get(when_to_send='on new member registration',
                                                              business=biz_data(user))
                message = f'You have made a deposit of {initial} ugx and incurred total charges of {depo_charges}UGX\n Your new balance is {the_member_.balance}'
                if smsSettingsObj.status:
                    if member_acct.members.biodata.contact:
                        checkAndSendMessage(user, message, member_acct.members.biodata.contact,
                                            member_acct.members.biodata.name)
                else:
                    # email
                    print('send using email')

            except Exception as e:
                print(str(e))


def recalculate_shares(branch):

    shares = []
    # print('Recalculate shares:', branch)
    all_members = Member.objects.filter(branch_id=branch).prefetch_related('shares_buyer', 'shares_seller')
    # print('TOTAL MEMBERS',all_members.count(), '-------------')
    new_total_shares = 0
    for member in all_members:
        # print('current shares', member.shares)
        # member_id = member.id

        sold_shares = member.shares_seller.aggregate(total_sold=Sum('shares'))
        # sold_shares = SharesTransactions.objects.filter(seller=member_id).aggregate(total_sold=Sum('shares'))
        shares_sold = sold_shares['total_sold'] or 0
        bought_shares = member.shares_buyer.aggregate(total_bought=Sum('shares'))
        # bought_shares = SharesTransactions.objects.filter(buyer=member_id).aggregate(total_bought=Sum('shares'))
        shares_bought = bought_shares['total_bought'] or 0

        share_balance = float(shares_bought) - float(shares_sold)
        member.shares = share_balance
        shares.append(member)
        new_total_shares = new_total_shares + share_balance
    Member.objects.bulk_update(shares, ['shares'], batch_size=1000)
    the_branch = Branch.objects.get(id=branch)
    BusinessShares.objects.filter(business=the_branch.business).update(sold=new_total_shares)



def recalculate_savings(business):
    # print('Recalculate savings:', business)
    members = []
    all_member_ledgers = Account.objects.filter(business_id=business, member_id__gt=0).prefetch_related('account_cr', 'account_dr')
    for member_ledger in all_member_ledgers:
        obj = MemberAccount.objects.get(id=member_ledger.member_id)
        new_balance = Transactions.objects.filter(
            Q(account_dr=member_ledger) | Q(account_cr=member_ledger)).aggregate(
            bal=Sum(
                Case(
                    When(account_cr=member_ledger, then=F('reporting_amount')),
                    default=0, output_field=FloatField()))
                - Sum(
                Case(
                    When(account_dr=member_ledger, then=F('reporting_amount')),
                    default=0, output_field=FloatField()))
        )['bal']
        obj.balance = new_balance or 0
        members.append(obj)
    MemberAccount.objects.bulk_update(members, ['balance'], batch_size=1000)


def recalculate_savings(business):
    members = []
    all_member_ledgers = Account.objects.filter(business_id=business, member_id__gt=0).prefetch_related('account_cr',
                                                                                                        'account_dr')
    member_ledger_balances = all_member_ledgers.values('id').annotate(
        credit_sum=Sum(
            Case(When(account_cr=F('account_cr'), then=F('account_cr__reporting_amount')), default=0,
                 output_field=FloatField())
        ),
        debit_sum=Sum(
            Case(When(account_dr=F('account_dr'), then=F('account_dr__reporting_amount')), default=0,
                 output_field=FloatField())
        ),
    ).values('id', 'credit_sum', 'debit_sum')
    # print(member_ledger_balances)
    ledger_balance = list(member_ledger_balances)
    for ml in ledger_balance:
        member = MemberAccount.objects.get(account=ml['id'])
        new_balance = ml['credit_sum'] - ml['debit_sum']
        member.balance = new_balance
        members.append(member)

    # Bulk update the members' balances
    MemberAccount.objects.bulk_update(members, ['balance'], batch_size=1000)




class DepositForm(forms.Form):
    date_created = forms.DateField(required=True)
    account_id = forms.IntegerField(required=True)
    amount = forms.CharField(required=True)
    the_account = forms.IntegerField(required=True)
    narration = forms.CharField(required=False)
    dep_ref = forms.CharField(required=False)

    def __init__(self, *args, **kwargs):
        self.user = kwargs.pop('user', None)
        self.member = kwargs.pop('member', None)
        self.finance_year = kwargs.pop('finance_year', None)
        super(DepositForm, self).__init__(*args, **kwargs)

    @transaction.atomic
    def save(self, commit=True):
        user = self.user
        zemember = self.member
        financial_year = self.finance_year
        cleaned_data = super().clean()
        date_created = cleaned_data.get('date_created')
        account = cleaned_data.get('account_id')
        raw_initial = cleaned_data.get('amount')
        the_account = cleaned_data.get('the_account')
        narration = cleaned_data.get('narration')
        initial_before = float(raw_initial.replace(',', ''))
        initial = float(initial_before)
        rec_no = datetime.datetime.now().strftime("%y%m%d%H%M%S")
        ref_no = cleaned_data.get('dep_ref') or rec_no
        bizz_id = biz_id(user)
        this_biz = Business.objects.filter(id=bizz_id).first()
        notify_type = NotiSettings.objects.filter(business=this_biz).first()
        template_file_link = 'sacco/emails/transactional.html'

        if not date_created:
            date_created = datetime.datetime.now().date()
        income_acc = Account.objects.filter(category__name="Deposit Income", business=biz_data(user)).first()

        this_account = MemberAccount.objects.get(id=account)

        member_acct = AccountBroker.objects.filter(the_account_id=account).first()
        member_acc = Account.objects.filter(member_id=member_acct.the_account_id).first()

        acc_involved = Account.objects.get(id=int(the_account), business=biz_data(user))
        depo_dict = []
        depo_cgs = TransactionCharge.objects.filter(account_type_id=member_acct.the_account.account_type,
                                                    charge_type='d', status=True)
        # df = pd.DataFrame()
        depo_charges = 0
        if depo_cgs.exists():
            depo_dataframe = pd.DataFrame(depo_cgs.values())

            # deposit charges
            for d in depo_cgs:
                if d.account_type.deposit_charge_vary:
                    if d.start <= int(initial) <= d.end:
                        amount = d.charge
                        if d.is_charge_percentage:
                            amount = int(initial) * (d.charge / 100)
                        depo_dict.append({'id': d.id, 'amount': amount, 'account_type_name': d.account_type.name})
                        # print(depo_dict)
                else:
                    amount = d.charge
                    if d.is_charge_percentage:
                        amount = initial * (d.charge / 100)
                    depo_dict.append({'id': d.id, 'amount': amount, 'account_type_name': d.account_type.name})
                depo_data_charges = pd.DataFrame(depo_dict)
                # print('my {}'.format(depo_data_charges))
                if not depo_dataframe.empty and not depo_data_charges.empty:
                    # print(depo_dataframe)
                    # print('ok')
                    # print('se {}'.format(depo_data_charges))
                    deposit_frame = pd.merge(depo_dataframe, depo_data_charges, on='id')
                    # print('charge {}'.format(depo_charges))
                    if d.account_type.deposit_charge:
                        if d.account_type.deposit_charge_vary:
                            dfs = deposit_frame[(deposit_frame['start'] <= initial) & (initial <= deposit_frame['end'])]
                        else:
                            dfs = deposit_frame
                    else:
                        dfs = pd.DataFrame(columns=['id', 'start', 'end', 'charge', 'account_type_id',
                                                    'charge_type', 'status', 'is_charge_percentage', 'created_date',
                                                    'deletion_date', 'amount', 'account_type_name'])
                    depo_charges = deposit_frame['amount'].sum()
                    # print(f'my {dfs}')
                    if not dfs.empty:
                        for index, row in dfs.iterrows():
                            Transactions.objects.create(
                                branch=biz_staff_branch(user),
                                transaction_type='Charge on deposit',
                                account_dr=member_acc,
                                account_cr=income_acc,
                                narration=f'Charge on deposit of {initial} on {this_account.acc_number}',
                                reporting_amount=row["amount"],
                                added_by=biz_staff(user),
                                financial_year=financial_year,
                                tx_date=date_created,
                                receipt=rec_no,
                                reference=ref_no
                            )
                        break
                else:
                    depo_charges = 0

        trans = Transactions.objects.create(
            branch=biz_staff_branch(user),
            transaction_type='Deposit',
            account_dr=acc_involved,
            account_cr=member_acc,
            narration=f'Deposit of {initial} to {this_account.acc_number}' if not narration else narration,
            reporting_amount=initial,
            added_by=biz_staff(user),
            financial_year=financial_year,
            tx_date=date_created,
            receipt=rec_no,
            reference=ref_no
        )

        member_debit = list(Transactions.objects.filter(account_dr=member_acc).aggregate(
            total=Sum('reporting_amount')).values())[0]

        if member_debit is None or member_debit < 1:
            member_debits = 0
        else:
            member_debits = member_debit

        # print(f'my member debits {member_debits}')
        member_credit = list(Transactions.objects.filter(account_cr=member_acc).aggregate(
            total=Sum('reporting_amount')).values())[0]
        if member_credit is None or member_credit < 1:
            member_credits = 0
        else:
            member_credits = member_credit

        this_account.balance = member_credits - member_debits
        this_account.save()
        # print('new balo {}'.format(this_account.balance))
        the_member_ = MemberAccount.objects.get(id=this_account.id)
        currency = CurrencySetting.objects.filter(business=this_biz).first()
        # print(the_member_.balance)
        smsSettingsObj = SaccoSmsSettings.objects.get(when_to_send='On deposit transaction', business=biz_data(user))
        # print('notify_type', notify_type.notification_type)
        subject = "Deposit request processed successfully"
        business = this_biz.name
        email = zemember.biodata.email
        message = f'Dear {this_biz.short_name} member, a deposit of {currency.currency} {initial} has been made on your account. Fee {currency.currency} {depo_charges}. Account Balance is {the_member_.balance}. TXN ID {trans.txno}. Thank you'
        html_body = f'<p>Dear {zemember.biodata.name},</p>  Your Deposit request with the following details has been processed successfully.<br/>Transaction Number: {trans.txno}<br/>Amount: {currency.currency}{initial} <br/>Description: {trans.narration} <br/>Date: {trans.tx_date} <p>Please call our contact centre on +{this_biz.contact} in case you require further details.</p>'
        if smsSettingsObj.status:
            if notify_type.notification_type == 1:  # Send by SMS only
                if zemember.biodata.contact:
                    checkAndSendMessage(user, message, zemember.biodata.contact, zemember.biodata.name, account=member_acc)
            elif notify_type.notification_type == 2:  # Send by Email only
                if zemember.biodata.email:
                    sendTransEmail(template_file_link, html_body, business, subject, email)

            elif notify_type.notification_type == 3:  # Send by both SMS and Email
                if zemember.biodata.contact:
                    # print('send message')
                    checkAndSendMessage(user, message, zemember.biodata.contact, zemember.biodata.name, account=member_acc)

                if zemember.biodata.email:
                    sendTransEmail(template_file_link, html_body, business, subject, email)

        return cleaned_data




class WithdrawForm(forms.Form):
    date_created = forms.DateField(required=True)
    account_id = forms.IntegerField(required=True)
    amount = forms.CharField(required=True)
    the_account = forms.IntegerField(required=True)
    narration = forms.CharField(required=False)
    with_ref = forms.CharField(required=False)

    def __init__(self, *args, **kwargs):
        self.user = kwargs.pop('user', None)
        self.member = kwargs.pop('member', None)
        self.finance_year = kwargs.pop('finance_year', None)
        super(WithdrawForm, self).__init__(*args, **kwargs)

    @transaction.atomic
    def clean(self):
        user = self.user
        zemember = self.member
        financial_year = self.finance_year
        cleaned_data = super().clean()
        date_created = cleaned_data.get('date_created')
        account = cleaned_data.get('account_id')
        raw_initial = cleaned_data.get('amount')
        payment_account = cleaned_data.get('the_account')
        narration = cleaned_data.get('narration')
        initial_before = float(raw_initial.replace(',', ''))
        initial = float(initial_before)
        rec_no = datetime.datetime.now().strftime("%y%m%d%H%M%S")
        ref_no = cleaned_data.get('with_ref') or rec_no
        this_biz = Business.objects.filter(id=biz_id(user)).first()
        notify_type = NotiSettings.objects.filter(business=this_biz).first()
        template_file_link = 'sacco/emails/transactional.html'
        currency = CurrencySetting.objects.filter(business=this_biz).first()

        if not date_created:
            date_created = datetime.datetime.now().date()
        income_acc = Account.objects.filter(category__name="Withdraw Income", business=biz_data(user)).first()
        this_account = MemberAccount.objects.get(id=account)

        # chart of accounts
        member_acct = AccountBroker.objects.filter(the_account_id=account).first()
        member_acc = Account.objects.filter(member_id=member_acct.the_account_id).first()

        paying_account = Account.objects.get(id=int(payment_account), business=biz_data(user))
        withdraw_dict = []
        withdraw_cgs = TransactionCharge.objects.filter(account_type_id=member_acct.the_account.account_type,
                                                        charge_type='w', status=True)

        with_charges = 0
        if withdraw_cgs.exists():
            withdraw_dataframe = pd.DataFrame(withdraw_cgs.values())
            # print(depo_cgs)
            # deposit charges
            for w in withdraw_cgs:
                if w.account_type.withdraw_charge_vary:
                    if w.start <= initial <= w.end:
                        amount = w.charge
                        if w.is_charge_percentage:
                            amount = initial * (w.charge / 100)
                        withdraw_dict.append({'id': w.id, 'amount': amount, 'account_type_name': w.account_type.name})
                        # print(depo_dict)
                    else:
                        amount = 0
                else:
                    amount = w.charge
                    # print(w)
                    if w.is_charge_percentage:
                        amount = initial * (w.charge / 100)
                    # print(w.id)
                    withdraw_dict.append({'id': w.id, 'amount': amount, 'account_type_name': w.account_type.name})
                    # print('---')
                withdraw_data_charges = pd.DataFrame(withdraw_dict)
                # print(f'x is {depo_data_charges}')
                if not withdraw_dataframe.empty and not withdraw_data_charges.empty:
                    withdraw_frame = pd.merge(withdraw_dataframe, withdraw_data_charges, on='id')
                    if w.account_type.withdraw_charge:
                        if w.account_type.withdraw_charge_vary:
                            # print('am varying')
                            dfs = withdraw_frame[
                                (withdraw_frame['start'] <= initial) & (initial <= withdraw_frame['end'])]
                        else:
                            # print('amount')
                            dfs = withdraw_frame
                    else:
                        dfs = pd.DataFrame(columns=['id', 'start', 'end', 'charge', 'account_type_id',
                                                    'charge_type', 'status', 'is_charge_percentage', 'created_date',
                                                    'deletion_date', 'amount', 'account_type_name'])
                    # print(f'my x is {dfs}')
                    with_charges = dfs['amount'].sum()
                    if not dfs.empty:
                        for index, row in dfs.iterrows():
                            Transactions.objects.create(
                                branch=biz_staff_branch(user),
                                transaction_type='Charge on withdraw',
                                account_dr=member_acc,
                                account_cr=income_acc,
                                narration=f'Charge on withdraw of {initial}',
                                reporting_amount=row["amount"],
                                added_by=biz_staff(user),
                                financial_year=financial_year,
                                tx_date=date_created,
                                receipt=rec_no,
                                reference=ref_no
                            )
                        break
                else:
                    with_charges = 0

        trans = Transactions.objects.create(
            branch=biz_staff_branch(user),
            transaction_type='Withdraw',
            account_dr=member_acc,
            account_cr=paying_account,
            narration=f'Withdraw of {initial} from {this_account.acc_number}' if not narration else narration,
            reporting_amount=initial,
            added_by=biz_staff(user),
            financial_year=financial_year,
            tx_date=date_created,
            receipt=rec_no,
            reference=ref_no
        )

        member_debits = \
            list(Transactions.objects.filter(account_dr=member_acc).aggregate(total=Sum('reporting_amount')).values())[
                0]
        if member_debits is None:
            member_debits = 0
        # print(f'my member debits {member_debits}')
        member_credit = \
            list(Transactions.objects.filter(account_cr=member_acc).aggregate(total=Sum('reporting_amount')).values())[
                0]

        if member_credit is None or member_credit < 1:
            member_credits = 0
        else:
            member_credits = member_credit

        this_account.balance = member_credits - member_debits
        this_account.save()

        the_member_ = MemberAccount.objects.get(id=this_account.id)
        # print(the_member_.balance)

        smsSettingsObj = SaccoSmsSettings.objects.get(when_to_send='On withdraw transaction', business=biz_data(user))
        # print('notify_type', notify_type.notification_type)
        subject = "Withdraw request processed successfully"
        business = this_biz.name
        email = zemember.biodata.email
        message = f'Dear {this_biz.short_name} member, a withdraw of {currency.currency} {initial} has been made on your account. Fee {currency.currency} {with_charges}. Bal {the_member_.balance}. TXN ID {trans.txno}'
        html_body = f'<p>Dear {zemember.biodata.name},</p>  Your withdraw request with the following details has been processed ' \
                    f'successfully.<br/>Transaction Number: {trans.txno}<br/>Amount: {currency.currency} {initial} <br/>' \
                    f'Description: {trans.narration} <br/>Date: {trans.tx_date} <p>Please call our contact centre on +{this_biz.contact} in case you ' \
                    f'require further details.</p> '
        if smsSettingsObj.status:
            if notify_type.notification_type == 1:  # Send by SMS only
                if zemember.biodata.contact:
                    checkAndSendMessage(user, message, zemember.biodata.contact, zemember.biodata.name,
                                        account=member_acc, rec_no=rec_no)
            elif notify_type.notification_type == 2:  # Send by Email only
                if zemember.biodata.email:
                    sendTransEmail(template_file_link, html_body, business, subject, email)

            elif notify_type.notification_type == 3:  # Send by both SMS and Email
                if zemember.biodata.contact:
                    checkAndSendMessage(user, message, zemember.biodata.contact, zemember.biodata.name,
                                        account=member_acc)

                if zemember.biodata.email:
                    sendTransEmail(template_file_link, html_body, business, subject, email)

        return cleaned_data






class MembersWithdraw(PermView):
    perm_name = 'view_withdraws'
    deco_role = 'can_withdraw'
    template_name = 'sacco/member_with.html'
    form_class = WithdrawForm

    def get_member(self, request):
        pkey = self.kwargs.get("pk")
        member = get_object_or_404(Member, id=pkey, is_active=True, biodata__business=businessdata(request))
        # print(member.biodata.business.name)
        return member

    def get(self, request, pk, *args, **kwargs):
        member = self.get_member(request)
        form = self.form_class
        text = 'withdraws'.title()
        title = f'{member.biodata.name} {text}'
        member_accounts = AccountBroker.objects.filter(members=member)
        accounts_ = Account.objects.filter(business=businessdata(request),
                                           category__name__in=['Mobile Money', 'Cash', 'Bank'])
        currency = CurrencySetting.objects.filter(business=businessdata(request)).first()
        branch = biz_staff_branch(request.user)
        if 'with_charges' in request.session:
            withdraw_charge = request.session.get('with_charges')
        else:
            withdraw_charge = 0

        if 'data' in request.session:
            del request.session['data']

        accts = []
        for account in member_accounts:
            accts.append(account.the_account)
        withdraws = Transactions.objects.filter(transaction_type='Withdraw', account_dr__member__in=accts)
        if 'filter_trans' in request.GET:
            from_date = request.GET['from_date']
            to_date = request.GET['to_date']
            formatted_from = parser.parse(from_date)
            formatted_to = parser.parse(to_date)
            if to_date > from_date:
                if from_date and to_date:
                    withdraws = Transactions.objects.filter(
                        transaction_type='Withdraw',
                        account_dr__member__in=accts,
                        tx_date__gte=from_date,
                        tx_date__lte=to_date
                    ).order_by('-date_added')
        # print(withdraws)
        return render(request, self.template_name, locals())

    @can_do_this
    def post(self, request, *args, **kwargs):
        return_url = request.META.get('HTTP_REFERER', '/')
        current_transaction = date.today()
        member = self.get_member(request)
        raw_withdraw = request.POST['amount']
        account = request.POST['account_id']
        # print('hi muna')
        initial_before = float(raw_withdraw.replace(',', ''))
        withdraw = float(initial_before)
        inwords = num2words(withdraw)
        this_account = MemberAccount.objects.get(id=account)
        member_acct = AccountBroker.objects.filter(the_account=this_account).first()
        account_type = member_acct.the_account.account_type
        charges_and_balance = check_withdraw_charges(account, int(withdraw))
        the_account_type = AccountTypes.objects.filter(id=account_type.id).first()
        min_balance = the_account_type.min_balance
        fy = FinancialYear.objects.filter(start_date__lte=current_transaction,
                                          end_date__gte=current_transaction, business=biz_data(self.request.user),
                                          status=True)
        try:
            if fy.exists():
                financial_year = fy.first()
                form = self.form_class(request.POST, user=request.user, finance_year=financial_year, member=member)
                total_trans = this_account.balance - min_balance
                allowed_with = int(withdraw) + charges_and_balance
                if this_account.account_type.is_fixed:
                    maturity_date = member_acct.created_date + relativedelta(months=+the_account_type.maturity)
                    if member_acct.created_date >= maturity_date:
                        if int(allowed_with) <= total_trans:
                            if form.is_valid():
                                c = {
                                    'amount': withdraw,
                                    'charges': charges_and_balance,
                                    'owner': member_acct.members.biodata.name,
                                    'acc': this_account.acc_number,
                                    'rec': 'Withdraw',
                                    'url': return_url
                                }
                                request.session['data'] = c
                                request.session['with_charges'] = charges_and_balance
                                messages.success(request, 'success', extra_tags='Withdraw successful')
                                message = f"{request.user.staff.biodata.name} has withdrawn {raw_withdraw} " \
                                          f"from {this_account.acc_number}"
                                ActivityLog.objects.create(actor=request.user,
                                                           action_type=ACCOUNT,
                                                           title='Account withdraw',
                                                           branch=branch_id(request.user),
                                                           remarks=message)
                                return JsonResponse(
                                    {'msg': 'success', 'extra': 'Withdraw successful'}, safe=True)
                            else:
                                form_error = form.errors.as_json()
                                err_dict = ast.literal_eval(form_error)
                                error = err_dict['__all__'][0]['message']
                                x = ast.literal_eval(error)
                                return JsonResponse({'msg': 'error', 'extra': x[0]}, safe=True)
                        else:
                            return JsonResponse(
                                {'msg': 'error', 'extra': f'Insufficient balance to cater for charges of'
                                                          f' {charges_and_balance} and maintain a'
                                                          f' minimum balance of {min_balance}'}, safe=True)
                    else:
                        return JsonResponse({'msg': 'error', 'extra': 'Account not yet mature'}, safe=True)
                else:
                    if int(allowed_with) <= total_trans:
                        if form.is_valid():
                            # form.save()
                            request.session['with_charges'] = charges_and_balance
                            c = {
                                'amount': withdraw,
                                'charges': charges_and_balance,
                                'owner': member_acct.members.biodata.name,
                                'acc': this_account.acc_number,
                                'rec': 'Withdraw',
                                'url': return_url
                            }
                            request.session['data'] = c
                            return JsonResponse(
                                {'msg': 'success', 'extra': 'Withdraw successful', 'charge': charges_and_balance,
                                 'amount': withdraw, 'acc_no': this_account.acc_number,
                                 'owner': member_acct.members.biodata.name, 'words': inwords.title()}, safe=True)
                        else:
                            form_error = form.errors.as_json()
                            err_dict = ast.literal_eval(form_error)
                            error = err_dict['__all__'][0]['message']
                            x = ast.literal_eval(error)
                            return JsonResponse({'msg': 'error', 'extra': x[0]}, safe=True)
                    else:
                        return JsonResponse({'msg': 'error', 'extra': f'Insufficient balance to cater for charges of'
                                                                      f' {charges_and_balance} and maintain a'
                                                                      f' minimum balance of {min_balance}'}, safe=True)
            else:
                return JsonResponse({'msg': 'error', 'extra': 'Finance year not set'}, safe=True)
        except Exception as e:
            return JsonResponse({'msg': 'error', 'extra': str(e)}, safe=True)






def upload_new_members(file, user, biz, account_type, financial_year, balo_date, the_account):
    upload = pd.read_excel(file, comment='#', dtype={"Account Number": str})
    employee_no = None
    savings = 0
    share_cost = 0
    upload.fillna('', inplace=True)
    upload.columns.astype(str)
    upload.columns = upload.columns.str.title()
    from_frame = upload.to_json(orient='records', indent=4, date_format='iso')
    json_obj = json.loads(from_frame)
    acc_involved = Account.objects.get(id=int(the_account), business=biz_data(user))
    charges_acc = Account.objects.get(category__name="Registration Income",
                                      business=biz_data(user))
    business_shares = BusinessShares.objects.filter(business=biz_data(user)).first()
    member_upload = MemberUpload.objects.create(business=biz_data(user))
    depo_dict = []
    gen_dict = []
    gen_cgs = ApplicationAccountOrLoanType.objects.filter(account_type_id=account_type,
                                                          general_charge__application='r')
    dep_cgs = TransactionCharge.objects.filter(account_type_id=account_type, charge_type='d',
                                               status=True)
    gen_charges = 0
    depo_charges = 0
    c = 0
    items = len(json_obj)
    print(f'my count {items}')
    for obj in json_obj:
        c += 1
        print(c)
        if 'Employee Number' in obj:
            full_names, contact, other_contact, email, shares, raw_savings, join_date, gender, acc_no, employee_no = str(obj['Full Name']).strip(), str(obj['Contact']).strip(), str(obj['Other Contact']).strip(), str(obj.get('Email Address', None)).strip(), obj['Shares'], str(obj['Savings Balance']), obj['Join Date'], str(obj['Gender']), str(obj['Account Number']).strip(), str(obj['Employee Number'])
        else:
            full_names, contact, other_contact, email, shares, raw_savings, join_date, gender, acc_no = str(obj['Full Name']).strip(), str(obj['Contact']).strip(), str(obj['Other Contact']).strip(), str(obj.get('Email Address', None)).strip(), obj['Shares'], str(obj['Savings Balance']), obj['Join Date'], str(obj['Gender']), str(obj['Account Number'])

        if len(gender) > 1:
            gender = gender[0]

        if join_date:
            try:
                python_date = parse(join_date)
                join_day = python_date.date()
            except Exception as e:
                join_day = None
                print(str(e))
        else:
            join_day = None

        if acc_no:
            if acc_no.endswith('.0'):
                acc_no = acc_no.replace('.0', '')

        if raw_savings:
            my_raw_savings = raw_savings.replace(',', '')
            conv_raw = float(my_raw_savings)
            savings = int(conv_raw)

        myc = int(float(contact))
        if contact:
            contact = ug_dial(str(myc))
        else:
            contact = None

        if other_contact:
            myo = int(float(contact))
            other_contact = ug_dial(str(myo))
        else:
            other_contact = None

        if employee_no.endswith('.0'):
            employee_no = employee_no.replace('.0', '')

        if not email:
            email = None

        if gender:
            gender = gender.strip().upper()

        if biz.auto_acc_no:
            new_number = random_with_N_digits(8)
            if acc_no is None:
                acc_no = gen_account(user)

        if full_names:
            if not MemberAccount.objects.filter(acc_number=acc_no, member_account__business=biz_data(user)).exists():
                bio = Biodata.objects.create(name=full_names.title(), contact=contact,
                                             other_contact=other_contact, gender=gender,
                                             created_by=biz_staff(user), email=email,
                                             business=biz_data(user), employee_no=employee_no)
                if shares:
                    setting_status = OtherSettings.objects.filter(business=biz_data(user)).first()
                    member = Member.objects.create(biodata=bio, created_by=biz_staff(user), shares=int(shares),
                                                   branch=biz_staff_branch(user), date_joined=join_day,
                                                   member_type='s', member_upload=member_upload)
                    narrative = f'Sale of {member.shares} shares to {member.biodata.name} at {setting_status.share_price} each.'
                    # Shares transactions here
                    shares_record = SharesTransactions.objects.create(
                        buyer_id=member.id,
                        shares=member.shares,
                        date=datetime.datetime.today(),
                        narration='Opening shares',
                        branch=biz_staff_branch(user)
                    )
                    buyer_shares = SharesTransactions.objects.filter(Q(buyer=member) | Q(seller=member))
                    # print("results = %s" % buyer_shares.count())
                    total_bought = 0
                    total_sold = 0
                    for b in buyer_shares:
                        if b.buyer == member:
                            total_bought += b.shares
                        if b.seller == member:
                            total_sold += b.shares

                    current_shares = total_bought - total_sold

                    # update buyer shares balance
                    member.shares = current_shares
                    member.save()

                    # query seller shares transactions
                    seller_shares = SharesTransactions.objects.filter(seller__isnull=True)
                    total_sold = 0
                    for s in seller_shares:
                        total_sold += s.shares

                    # update business shares sold
                    business_shares.sold = total_sold
                    business_shares.save()

                    # Record a finance transaction
                    # shares account
                    shares_account = Account.objects.filter(business=biz_data(user),
                                                            category__name='Shares').first()
                    share_cost = int(setting_status.share_price) * int(shares)

                    Transactions.objects.create(
                        branch=biz_staff_branch(user),
                        transaction_type='Sale of shares',
                        account_dr_id=the_account,
                        account_cr=shares_account,
                        narration=narrative,
                        reporting_amount=share_cost,
                        added_by=biz_staff(user),
                        shares=shares_record,
                        financial_year=financial_year,
                        tx_date=join_day
                    )
                else:
                    member = Member.objects.create(biodata=bio, created_by=biz_staff(user),
                                                   branch=biz_staff_branch(user), date_joined=join_day,
                                                   member_type='o', member_upload=member_upload)
                account = MemberAccount.objects.create(acc_number=acc_no, account_type_id=account_type,
                                                       balance=savings)
                AccountBroker.objects.create(members=member, the_account=account, business=biz_data(user))
                cat = AccountCategory.objects.filter(business=biz_data(user), name="Members").first()
                # chart of accounts
                member_acc = Account.objects.create(category=cat, business=biz_data(user), member=account,
                                                    added_by=biz_staff(user))
                if gen_cgs.exists():
                    print('yes')
                    gen_dataframe = pd.DataFrame(gen_cgs.values())
                    for gen in gen_cgs:
                        amount = gen.general_charge.amount
                        if gen.general_charge.is_percentage:
                            amount = savings * (gen.general_charge / 100)
                        gen_dict.append(
                            {
                                'id': gen.id,
                                'amount': amount
                            }
                        )
                    gen_charges_df = pd.DataFrame(gen_dict)
                    gen_frame = pd.merge(gen_dataframe, gen_charges_df, on='id')

                    # summing general charges dataframe
                    gen_charges = gen_frame['amount'].sum()
                    for ch in gen_cgs:
                        print(f'th {ch}')
                        print(ch.general_charge.amount)
                        print('-------')
                        Transactions.objects.create(
                            branch=biz_staff_branch(user),
                            transaction_type='Registration',
                            account_dr=member_acc,
                            account_cr=charges_acc,
                            narration="{}".format(ch.general_charge.charge),
                            reporting_amount=ch.general_charge.amount,
                            added_by=biz_staff(user),
                            financial_year=financial_year,
                            tx_date=join_day
                        )

                if dep_cgs.exists():
                    list_result = dep_cgs.values('account_type_id', 'account_type__deposit_charge', 'charge', 'start', 'end', 'status', 'id')
                    depo_dataframe = pd.DataFrame(list_result)

                    for d in dep_cgs:
                        if d.account_type.deposit_charge_vary:
                            if d.start <= int(savings) <= d.end:
                                amount = d.charge
                                if d.is_charge_percentage:
                                    amount = savings * (d.charge / 100)
                                depo_dict = {'id': d.id, 'amount': amount, 'account_type_name': d.account_type.name}
                        else:
                            amount = d.charge
                            if d.is_charge_percentage:
                                amount = savings * (d.charge / 100)
                            depo_dict = {'id': d.id, 'amount': amount, 'account_type_name': d.account_type.name}
                    depo_data_charges = pd.DataFrame(depo_dict, index=[0])
                    if depo_dict:
                        print('here')
                        deposit_frame = pd.merge(depo_dataframe, depo_data_charges, on='id')
                        print(f'my df is {deposit_frame}')
                        print('my status is {}'.format(deposit_frame["account_type__deposit_charge"].values[0]))
                        if deposit_frame["account_type__deposit_charge"].values[0]:
                            print('yi')
                            Transactions.objects.create(
                                branch=biz_staff_branch(user),
                                transaction_type='Charge',
                                account_dr=member_acc,
                                account_cr=charges_acc,
                                narration="Charge on deposit of {}".format(savings),
                                reporting_amount=deposit_frame["amount"].values[0],
                                added_by=biz_staff(user),
                                financial_year=financial_year,
                                tx_date=join_day
                            )

                Transactions.objects.create(
                    branch=biz_staff_branch(user),
                    transaction_type="Deposit",
                    account_dr=acc_involved,
                    account_cr=member_acc,
                    narration="Initial deposit on registration",
                    reporting_amount=savings,
                    added_by=biz_staff(user),
                    financial_year=financial_year,
                    tx_date=join_day
                )
                if not shares:
                    total_charges = depo_charges + gen_charges
                else:
                    total_charges = depo_charges + gen_charges + share_cost
                member_debits = list(Transactions.objects.filter(account_dr=member_acc).aggregate(
                    total=Sum('reporting_amount')).values())[0]
                if member_debits is None:
                    member_debits = 0
                member_credits = list(Transactions.objects.filter(account_cr=member_acc).aggregate(
                    total=Sum('reporting_amount')).values())[0]
                # print(f'my member credits {member_credits}')
                account.balance = member_credits - member_debits
                account.save()
                smsSettingsObj = SaccoSmsSettings.objects.get(when_to_send='on new member registration',
                                                              business=biz_data(user))
                message = f'From: {biz.name}\n You have deposited {savings} ugx and incurred total charges of {total_charges}UGX\n Your new balance is {account.balance}'
                if smsSettingsObj.status:
                    if bio.contact:
                        theResponse = checkAndSendMessage(user, message, contact, bio.name)
                        # print(theResponse)

    return 'upload_member'



def bulk_member_deposit(user, acounts_list, depo, the_account, financial_year):
    try:
        acc_involved = Account.objects.get(id=int(the_account), business=biz_data(user))
    except Account.DoesNotExist:
        acc_involved = Account.objects.filter(id=int(the_account), business=biz_data(user)).first()
    income_acc = Account.objects.filter(category__name="Deposit Income", business=biz_data(user)).first()
    from_initial = depo.replace(',', '')
    initial = int(float(from_initial))

    with transaction.atomic():

        for acc in acounts_list:
            try:
                this_account = MemberAccount.objects.get(id=acc)
                member_acc = Account.objects.filter(member=this_account, business=biz_data(user)).first()
                member_acct = AccountBroker.objects.filter(the_account=this_account).first()
                depo_dict = []

                depo_cgs = TransactionCharge.objects.filter(account_type_id=member_acct.the_account.account_type,
                                                            charge_type='d', status=True)
                depo_charges = 0
                if depo_cgs.exists():
                    depo_dataframe = pd.DataFrame(depo_cgs.values())

                    # deposit charges
                    for d in depo_cgs:
                        if d.account_type.deposit_charge_vary:
                            if d.start <= int(initial) <= d.end:
                                amount = d.charge
                                if d.is_charge_percentage:
                                    amount = int(initial) * (d.charge / 100)

                                depo_dict.append(
                                    {'id': d.id, 'amount': amount, 'account_type_name': d.account_type.name})
                        else:
                            amount = d.charge
                            if d.is_charge_percentage:
                                amount = initial * (d.charge / 100)

                            depo_dict.append({'id': d.id, 'amount': amount, 'account_type_name': d.account_type.name})
                        depo_data_charges = pd.DataFrame(depo_dict)
                        if not depo_dataframe.empty and not depo_data_charges.empty:
                            deposit_frame = pd.merge(depo_dataframe, depo_data_charges, on='id')
                            if d.account_type.deposit_charge:
                                if d.account_type.deposit_charge_vary:
                                    dfs = deposit_frame[
                                        (deposit_frame['start'] <= initial) & (initial <= deposit_frame['end'])]
                                else:
                                    dfs = deposit_frame
                            else:
                                dfs = pd.DataFrame(columns=['id', 'start', 'end', 'charge', 'account_type_id',
                                                            'charge_type', 'status', 'is_charge_percentage',
                                                            'created_date',
                                                            'deletion_date', 'amount', 'account_type_name'])
                            depo_charges = dfs['amount'].sum()
                            if not dfs.empty:
                                for index, row in dfs.iterrows():

                                    Transactions.objects.create(
                                        branch=biz_staff_branch(user),
                                        transaction_type='Charge on deposit',
                                        account_dr=member_acc,
                                        account_cr=income_acc,
                                        narration='Charge on deposit of '+str(initial)+' on {}'.format(row["account_type_name"]),
                                        reporting_amount=row["amount"],
                                        added_by=biz_staff(user),
                                        financial_year=financial_year,
                                    )
                                break
                        else:
                            depo_charges = 0

                    # print('my deposit charges{}'.format(depo_charges))

                Transactions.objects.create(
                    branch=biz_staff_branch(user),
                    transaction_type='Deposit',
                    account_dr=acc_involved,
                    account_cr=member_acc,
                    narration="Deposit on {}".format(this_account.acc_number),
                    reporting_amount=initial,
                    added_by=biz_staff(user),
                    financial_year=financial_year,
                )

                member_debits = list(Transactions.objects.filter(account_dr=member_acc).aggregate(
                    total=Sum('reporting_amount')).values())[0]
                if member_debits is None:
                    member_debits = 0
                member_credits = list(Transactions.objects.filter(account_cr=member_acc).aggregate(
                    total=Sum('reporting_amount')).values())[0]
                # print(f'my member credits {member_credits}')
                this_account.balance = member_credits - member_debits
                this_account.save()
                the_member_ = MemberAccount.objects.get(id=this_account.id)
                smsSettingsObj = SaccoSmsSettings.objects.get(when_to_send='on new member registration',
                                                              business=biz_data(user))
                message = f'You have made a deposit of {initial} ugx and incurred total charges of {depo_charges}UGX' \
                          f'\n Your new balance is {the_member_.balance} '
                if smsSettingsObj.status:
                    if member_acct.members.biodata.contact:
                        checkAndSendMessage(user, message, member_acct.members.biodata.contact,
                                            member_acct.members.biodata.name)
                else:
                    # email
                    print('send using email')

            except Exception as e:
                print(str(e))