import uuid

from django.core.mail import send_mail
from django.db import transaction
from django.db.models import Q
from django.shortcuts import render, get_object_or_404
from django.http import HttpResponseRedirect
from django.template.loader import render_to_string
from django.urls import reverse_lazy
from django.contrib import messages

# NUGSOFT INDEX REDIRECT HERE
from django.views import View
from accounts.models import User, Business, SubscriptionPackage, Branch, Biodata, Staff, Position, Permissions, Role
from accounts.permission_mixin import OnlyNugsoft
from mufukoadmin.forms import PackageCreateForm, ManageBusinessForm, LicenseExtendForm, MessagePackageForm

from sacco.models import LicenseKey, Sms, \
    MemberAccount, CurrencySetting, GroupSettings, LoanSettings, NotiSettings, OtherSettings, SaccoSmsSettings, \
    GeneratedPassword, Member, BusinessShares, AccountTypes, MessagePackage
from sacco.utils import sendMessage
from transactions.models import AuditTrail, Transactions, AccountCategory, Account

from transactions.views import AccountCategoryJson
from utils.general import random_with_N_digits


# GENERAL DASHBOARD
class AdminIndex(OnlyNugsoft, View):
    template_name = 'mufukoadmin/admin.html'

    def get(self, request, *args, **kwargs):
        title = 'General Admin Dashboard'
        print('REDIRECT TO NUGSOFT ADMIN')
        businesses = Business.objects.all()
        audit_logs = AuditTrail.objects.all()
        members = MemberAccount.objects.all()
        trans = Transactions.objects.all()
        # print(businesses)
        return render(request, self.template_name, locals())


# VIEWING ALL BUSINESSES
class AllBusinesses(OnlyNugsoft, View):
    template_name = 'mufukoadmin/all_businesses.html'

    def get(self, request, *args, **kwargs):
        title = 'All Businesses'
        businesses_all = Business.objects.all()
        return render(request, self.template_name, locals())


# VIEWING ALL ACTIVE BUSINESSES
class ActiveBusinesses(OnlyNugsoft, View):
    template_name = 'mufukoadmin/active.html'

    def get(self, request, *args, **kwargs):
        title = 'All Active Businesses'
        businesses_all = Business.objects.filter(status=1)
        return render(request, self.template_name, locals())


# VIEWING ALL INACTIVE BUSINESSES
class InactiveBusinesses(OnlyNugsoft, View):
    template_name = 'mufukoadmin/inactive.html'

    def get(self, request, *args, **kwargs):
        title = 'All Inactive Businesses'
        businesses_all = Business.objects.filter(~Q(status=1))
        return render(request, self.template_name, locals())


# BUSINESS CREATION STARTS -------------------------
class AddBusiness(OnlyNugsoft, View):
    template_name = 'mufukoadmin/add_business.html'
    created_email = 'mufukoadmin/emails/action_email.html'

    def get(self, request, *args, **kwargs):
        title = 'Add Business'
        packages = SubscriptionPackage.objects.filter(status=True)
        return render(request, self.template_name, locals())

    @transaction.atomic()
    def post(self, request, *args, **kwargs):
        domain = request.headers['Host']
        if request.method == 'POST':
            # ====== GENERATE USERNAME ==========
            username = request.POST['staff_email'].split("@")[0]
            last_business = Business.objects.last()
            # print("Number ", last_business.business_number)
            try:
                if last_business.business_number is None:
                    business_number = 100000000 + 1
                else:
                    business_number = int(last_business.business_number) + 1
            except AttributeError:
                business_number = None
            except:
                business_number = None

            # ====== CHECK FOR UNWANTED DUPLICATES ======
            if User.objects.filter(username=username).exists():
                messages.warning(request, 'Duplicate detected',
                                 extra_tags='Generated username is already taken. Interchange name'
                                            ' positions and try again')
                return HttpResponseRedirect(reverse_lazy('add_business'))

            if User.objects.filter(email=request.POST['staff_email']).exists():
                messages.warning(request, 'Duplicate detected',
                                 extra_tags='Staff Email provided is already used. Try another email')
                return HttpResponseRedirect(reverse_lazy('add_business'))

            # ====== RECORD BUSINESS DETAILS ============
            new_business = Business.objects.create(
                name=request.POST['name'],
                code=uuid.uuid4(),
                address=request.POST['address'],
                contact='256' + request.POST['contact'][-9:],
                short_name=request.POST['short_name'],
                clients=request.POST['clients'],
                email=request.POST['email'],
                logo=request.FILES.get('logo', None),
                certificate=request.FILES.get('certificate', None),
                type=request.POST['type'],
                website=request.POST['website'],
                launch_date=request.POST['launch_date'],
                certificate_no=request.POST['certificate_no'],
                package_id=request.POST['package'],
                auto_acc_no=request.POST['generate'],
                # share_status=request.POST['shareholder'],
                business_number=business_number
            )

            # ====== GENERATE THE LICENSE FOR THE BUSINESS =====
            new_license = LicenseKey.objects.create(
                business_id=new_business.id,
                expiry_date=request.POST['license']
            )

            # ====== RECORD FIRST BRANCH OF THIS BUSINESS ============
            new_branch = Branch.objects.create(
                name='Main Branch',
                code=uuid.uuid4(),
                address=request.POST['address'],
                contact=request.POST['contact'],
                status=1,
                is_main=True,
                business_id=new_business.id
            )

            # ====== ADD STARTING SMSs TO BUSINESS =====
            new_sms = Sms.objects.create(
                branch_id=new_branch.id,
                quantity=150
            )

            # ====== RECORD BUSINESS FIRST STAFF POSITION ============
            new_position = Position.objects.create(title=request.POST['position'], business_id=new_business.id)

            # ====== RECORD BUSINESS FIRST BIO DATA RECORD ============
            new_bio_data = Biodata.objects.create(
                name=request.POST['staff'],
                gender=request.POST['gender'],
                nin=request.POST['nin'],
                email=request.POST['staff_email'],
                contact='256' + request.POST['staff_phone'][-9:],
                location=request.POST['staff_address'],
                business_id=new_business.id
            )

            # ====== RECORD FIRST USER ROLE ============
            new_role = Role.objects.create(
                title=request.POST['position'],
                business_id=new_business.id
            )

            # Assign All the permissions to this role
            all_permissions = Permissions.objects.all()
            new_position.permissions.set(all_permissions)

            # ====== RECORD BUSINESS FIRST STAFF ============
            new_staff = Staff.objects.create(
                biodata_id=new_bio_data.id,
                branch_id=new_branch.id,
                position_id=new_position.id,
                role_id=new_role.id,
                is_contact=True
            )

            # ====== GENERATE A TEMPORARY PASSWORD ======
            temp_pass = GeneratedPassword.objects.create(
                staff_id=new_staff.id,
                password=str(random_with_N_digits(8)),
                can_login=True
            )

            # ====== RECORD BUSINESS FIRST SYSTEM USER ============
            new_user = User.objects.create(
                email=request.POST['staff_email'],
                username=username,
                staff_id=new_staff.id,
                is_admin=False,
                is_superuser=False
            )
            new_user.set_password(temp_pass.password)
            new_user.save()

            # ====== RECORD CURRENCY SETTINGS ============
            currency_set = CurrencySetting.objects.create(
                business_id=new_business.id,
                currency='UGX'
            )

            if new_business.type == "1" or new_business.type == 1:
                # ====== RECORD STARTING SAVINGS PRODUCT ============
                product_set = AccountTypes.objects.create(
                    business_id=new_business.id,
                    name='General Savings Account'
                )

            # ====== RECORD GROUP SETTINGS ============
            group_set = GroupSettings.objects.create(
                business_id=new_business.id,
            )

            # ====== RECORD LOAN SETTINGS ============
            loan_set = LoanSettings.objects.create(
                business_id=new_business.id,
            )

            # ====== RECORD NOTIFICATION SETTINGS ============
            notify_set = NotiSettings.objects.create(
                business_id=new_business.id,
            )

            # ====== RECORD OTHER SETTINGS ============
            other_set = OtherSettings.objects.create(
                business_id=new_business.id,
                set_ordinary=request.POST['shareholder']
            )

            # ====== RECORD SHARES SETTINGS ============
            share_set = BusinessShares.objects.create(
                business_id=new_business.id,
            )

            # ====== RECORD FINANCIAL ACCOUNT CATEGORIES ============
            jsons = AccountCategoryJson()
            for c in jsons:
                # create category
                category = AccountCategory.objects.create(
                    name=c['fields']['name'],
                    dr_cr=c['fields']['dr_cr'],
                    cat_type=c['fields']['cat_type'],
                    business_id=new_business.id
                )

                # ====== RECORD FINANCIAL ACCOUNT TYPES ============
                for a in c['fields']['default_accounts']:
                    account = Account.objects.create(
                        name=a['name'],
                        display_name=a['name'],
                        category=category,
                        business_id=new_business.id,
                    )

            # ====== RECORD SMS SETTINGS ============
            all_sms_set = [{"when_to_send": "On new member registration", "section": "reg"},
                           {"when_to_send": "On withdraw transaction", "section": "withdraw"},
                           {"when_to_send": "On transfer transaction", "section": "transfer"},
                           {"when_to_send": "On deposit transaction", "section": "deposit"},
                           {"when_to_send": "On loan application", "section": "applied"},
                           {"when_to_send": "On loan appraisal", "section": "appraised"},
                           {"when_to_send": "On loan approval", "section": "approved"},
                           {"when_to_send": "On loan disbursement", "section": "disbursed"},
                           {"when_to_send": "On loan closing", "section": "closed"},
                           {"when_to_send": "On loan rejecting", "section": "rejected"},
                           {"when_to_send": "On loan waiving off", "section": "waived_off"},
                           {"when_to_send": "On loan Writing off", "section": "written_off"},
                           {"when_to_send": "On opening another account", "section": "other_reg"},
                           {"when_to_send": "committee Action on the loan", "section": "committee"},
                           {"when_to_send": "Loan due date warning", "section": "warning"},
                           {"when_to_send": "On loan payment", "section": "loan_pay"},
                           {"when_to_send": "On loan closing", "section": "loan_close"},
                           {"when_to_send": "On membership change", "section": "membership"},
                           {"when_to_send": "On sell of shares", "section": "shares"}]

            for setting in all_sms_set:
                sms_set = SaccoSmsSettings.objects.create(
                    when_to_send=setting['when_to_send'],
                    section=setting['section'],
                    status=True,
                    business_id=new_business.id
                )

            # ======= send email here ===========
            mail_content = "<p>Hello <b>" + new_bio_data.name + "</b>, <p>Welcome to Mfuko Plus, We are excited to have you join us!</p></p> <p>An account for <b>" + new_business.name + "</b> has been created successfully on Mfuko Plus. <br> To get started with your new account, follow the link: " + domain + ", login with your registered email: <b>" + new_user.email + "</b> and a temporary password of <b>" + temp_pass.password + "</b>.</p><p>If you get stuck along the way or have any questions, don't hesitate to reach out!</p><p>P.S. If you're new to these systems, check out our video tutorials on our <a href='https://www.youtube.com/channel/UC0cS5PICXbGsmyxw4q13UvA'>YouTube channel</a> that takes you through all the basics you will need to know.</p>"
            subject = "Welcome to Mfuko Plus"
            new_business_mail = {
                'site_name': 'Mfuko Plus',
                'subject': subject,
                'body': mail_content,
                'protocol': 'http',
            }
            email_html = render_to_string(self.created_email, new_business_mail)

            send_mail(subject, '', '',
                      [new_business.email, new_bio_data.email, 'macdanson2@gmail.com', 'info@mfuko.net'],
                      html_message=email_html,
                      fail_silently=False)

            # =========== SEND SMS HERE ====================
            message = 'Hello, an account for ' + new_business.name + ' has been created on Mfuko Plus. Your login email is ' + new_user.email + ' and temporary password is ' + temp_pass.password + ', follow the link ' + domain + '. For more call us +256777844758'
            receiver = new_bio_data.contact
            sms_sending = sendMessage(receiver, message)

            messages.success(request, 'success',
                             new_business.name.capitalize() + ' registered successfully and details have been sent to the provided emails and phone numbers'.capitalize())
            return HttpResponseRedirect(reverse_lazy('add_business'))
        return render(request, self.template_name, locals())


# MANAGING BUSINESS STARTS -------------------------
class ManageBusiness(OnlyNugsoft, View):
    template_name = 'mufukoadmin/all_businesses.html'
    action_email_html = 'mufukoadmin/emails/action_email.html'
    form_class = ManageBusinessForm

    def get(self, request, pk=None, *args, **kwargs):
        return HttpResponseRedirect(reverse_lazy('businesses'))

    def post(self, request, pk=None, *args, **kwargs):
        domain = request.headers['Host']
        if request.method == 'POST':
            business = get_object_or_404(Business, id=pk)
            form = self.form_class(request.POST)
            if form.is_valid():
                status_cleaned = form.cleaned_data['status']
                reason = request.POST['reason']
                business.status = status_cleaned
                business.save()

                if status_cleaned == 1:
                    action = "activated"
                    salute = "Cheers"
                elif status_cleaned == 2:
                    action = "deactivated"
                    salute = "Regrets"
                else:
                    action = "suspended"
                    salute = "Regrets"

                # send email with this reason
                subject = "SERVICE SUSPENSION NOTICE"
                c = {
                    "short_name": business.short_name,
                    'salute': salute,
                    'site_name': 'Mfuko Plus',
                    'body': '<p>Hello ' + business.__str__().capitalize() + ',</p>  Your account at Mfuko Plus: ' + domain + ' has been <b>' + action.__str__() + '</b> reason: <b><em>' + reason + '</em></b>',
                }
                email_html = render_to_string(self.action_email_html, c)

                send_mail(subject, '', '', [business.email, 'macdanson2@gmail.com'], html_message=email_html,
                          fail_silently=False)

                messages.success(request, 'success',
                                 business.__str__().capitalize() + ' status updated successfully'.capitalize())
            return HttpResponseRedirect(reverse_lazy('businesses'))


# PACKAGE CREATION STARTS -------------------------
class ManagePackages(OnlyNugsoft, View):
    template_name = 'mufukoadmin/packages.html'
    form_class = PackageCreateForm

    def get(self, request, *args, **kwargs):
        title = 'Mfuko Plus Packages'
        packages_all = SubscriptionPackage.objects.filter(status=True)
        return render(request, self.template_name, locals())

    def post(self, request, *args, **kwargs):
        if request.method == 'POST':
            new_name = request.POST['name']
            new_limit = request.POST['limit']
            # ====== CHECK FOR PACKAGE DUPLICATES ======
            # ---- check by name and active ---
            if SubscriptionPackage.objects.filter(name=new_name, status=True).exists():
                messages.warning(request, 'Duplicate detected',
                                 extra_tags='Package name ' + new_name + ' already exists')
                return HttpResponseRedirect(reverse_lazy('packages'))
            elif SubscriptionPackage.objects.filter(name=new_name, status=False).exists():
                SubscriptionPackage.objects.filter(name=new_name).update(status=True, limit=new_limit)
                messages.warning(request, 'success',
                                 extra_tags='Package name ' + new_name + ' package restored successfully')
                return HttpResponseRedirect(reverse_lazy('packages'))

            # ---- check by limit and active ---
            if SubscriptionPackage.objects.filter(limit=new_limit, status=True).exists():
                messages.warning(request, 'Duplicate detected',
                                 extra_tags='Package with a limit of ' + new_limit + ' already exists')
                return HttpResponseRedirect(reverse_lazy('packages'))
            elif SubscriptionPackage.objects.filter(limit=new_limit, status=False).exists():
                SubscriptionPackage.objects.filter(limit=new_limit).update(status=True, name=new_name)
                messages.warning(request, 'success',
                                 extra_tags='Package name ' + new_name + ' package added successfully')
                return HttpResponseRedirect(reverse_lazy('packages'))

            # ====== RECORD A NEW PACKAGE ============
            new_package = SubscriptionPackage.objects.create(name=new_name, limit=new_limit)

            messages.success(request, 'success',
                             new_package.name.capitalize() + ' package added successfully'.capitalize())
            return HttpResponseRedirect(reverse_lazy('packages'))
        return render(request, self.template_name, locals())


# PACKAGE UPDATING STARTS -------------------------
class UpdatePackages(OnlyNugsoft, View):
    template_name = 'mufukoadmin/packages.html'
    form_class = PackageCreateForm

    def get(self, request, pk=None, *args, **kwargs):
        package = get_object_or_404(SubscriptionPackage, id=pk)

        return HttpResponseRedirect(reverse_lazy('packages'))

    def post(self, request, pk=None, *args, **kwargs):
        if request.method == 'POST':
            package = get_object_or_404(SubscriptionPackage, id=pk)
            form = self.form_class(request.POST)
            if form.is_valid():
                name_cleaned = form.cleaned_data['name']
                limit_cleaned = form.cleaned_data['limit']
                package.name = name_cleaned
                package.limit = limit_cleaned
                package.save()
                messages.success(request, 'success',
                                 name_cleaned.capitalize() + ' package updated successfully'.capitalize())
            return HttpResponseRedirect(reverse_lazy('packages'))


# PACKAGE DELETING STARTS -------------------------
class DeletePackages(OnlyNugsoft, View):
    template_name = 'mufukoadmin/packages.html'

    def get(self, request, pk=None, *args, **kwargs):
        package = get_object_or_404(SubscriptionPackage, id=pk)
        business_count = Business.objects.filter(package_id=pk).count()
        if business_count > 0:
            SubscriptionPackage.objects.filter(id=pk).update(status=False)
        else:
            package.delete()

        messages.success(request, 'success', extra_tags='Package removed successfully')
        return HttpResponseRedirect(reverse_lazy('packages'))


# LICENSE STARTS -------------------------
class LicenseKeys(OnlyNugsoft, View):
    template_name = 'mufukoadmin/licenses.html'

    def get(self, request, *args, **kwargs):
        title = 'Mfuko Plus Licenses'
        licenses_all = LicenseKey.objects.filter(status=True)
        return render(request, self.template_name, locals())


# EXTENDING LICENSE STARTS -------------------------
class ExtendLicense(OnlyNugsoft, View):
    template_name = 'mufukoadmin/licenses.html'
    form_class = LicenseExtendForm

    def get(self, request, pk=None, *args, **kwargs):
        license_key = get_object_or_404(LicenseKey, id=pk)

        return HttpResponseRedirect(reverse_lazy('licenses'))

    def post(self, request, pk=None, *args, **kwargs):
        if request.method == 'POST':
            license_key = get_object_or_404(LicenseKey, id=pk)
            form = self.form_class(request.POST)
            if form.is_valid():
                expiry_date = form.cleaned_data['expiry_date']
                license_key.expiry_date = expiry_date
                license_key.save()
                messages.success(request, 'success', 'License Key extended successfully'.capitalize())
            return HttpResponseRedirect(reverse_lazy('licenses'))


# VIEW BUSINESS PROFILE -------------------------
class BusinessProfile(OnlyNugsoft, View):
    template_name = 'mufukoadmin/business.html'

    def get_business(self, request):
        pkey = self.kwargs.get("pk")
        business = get_object_or_404(Business, id=pkey)
        return business

    def get(self, request, *args, **kwargs):
        business = self.get_business(request)
        title = business.__str__() + " Profile"
        clients = Member.objects.filter(biodata__business=business, is_active=True, is_group=False)
        groups = Member.objects.filter(biodata__business=business, is_active=True, is_group=True)
        staffs = Staff.objects.filter(biodata__business=business, is_active=True)
        users = User.objects.filter(staff__biodata__business=business, is_active=True)
        sms = Sms.objects.filter(branch__business=business)
        branches = Branch.objects.filter(business=business, status=1)
        contacts = Staff.objects.filter(biodata__business=business, is_contact=True, is_active=True)

        return render(request, self.template_name, locals())


# MESSAGE PACKAGE CREATION STARTS -------------------------
class SMSPackages(OnlyNugsoft, View):
    template_name = 'mufukoadmin/sms_packages.html'
    form_class = MessagePackageForm

    def get(self, request, *args, **kwargs):
        title = 'Mfuko SMS Packages'
        sms_packages = MessagePackage.objects.filter(status=True)
        return render(request, self.template_name, locals())

    def post(self, request, *args, **kwargs):
        if request.method == 'POST':
            new_name = request.POST['name']
            price = request.POST['price']
            sms = request.POST['sms']
            unit_cost = int(price) / int(sms)
            # ====== CHECK FOR PACKAGE DUPLICATES ======
            # ---- check by name and active ---
            if MessagePackage.objects.filter(name=new_name, status=True).exists():
                messages.warning(request, 'Duplicate detected', extra_tags='Package name ' + new_name + ' already exists')
                return HttpResponseRedirect(reverse_lazy('sms_packages'))
            elif MessagePackage.objects.filter(name=new_name, status=False).exists():
                MessagePackage.objects.filter(name=new_name).update(status=True, price=price, sms=sms, unit_cost=unit_cost)
                messages.warning(request, 'success', extra_tags='Package name ' + new_name + ' package restored successfully')
                return HttpResponseRedirect(reverse_lazy('sms_packages'))

            # ---- check by price ---
            if MessagePackage.objects.filter(price=price, status=True).exists():
                messages.warning(request, 'Duplicate detected',
                                 extra_tags='Package with a price of ' + price + ' already exists')
                return HttpResponseRedirect(reverse_lazy('sms_packages'))

            # ====== RECORD A NEW PACKAGE ============
            unit_cost = int(price) / int(sms)
            new_package = MessagePackage.objects.create(name=new_name, price=price, sms=sms, unit_cost=unit_cost)

            messages.success(request, 'success', new_package.name.capitalize() + ' package added successfully'.capitalize())
            return HttpResponseRedirect(reverse_lazy('sms_packages'))
        return render(request, self.template_name, locals())


# PACKAGE UPDATING STARTS -------------------------
class UpdateSmsPackages(OnlyNugsoft, View):
    template_name = 'mufukoadmin/sms_packages.html'
    form_class = MessagePackageForm

    def get(self, request, pk=None, *args, **kwargs):
        get_object_or_404(MessagePackage, id=pk)

        return HttpResponseRedirect(reverse_lazy('sms_packages'))

    def post(self, request, pk=None, *args, **kwargs):
        if request.method == 'POST':
            package = get_object_or_404(MessagePackage, id=pk)
            form = self.form_class(request.POST)
            if form.is_valid():
                name_cleaned = form.cleaned_data['name']
                price_cleaned = form.cleaned_data['price']
                sms_cleaned = form.cleaned_data['sms']
                unit_cost = int(price_cleaned) / int(sms_cleaned)

                package.name = name_cleaned
                package.price = price_cleaned
                package.sms = sms_cleaned
                package.unit_cost = unit_cost
                package.save()
                messages.success(request, 'success', name_cleaned.capitalize() + ' package updated successfully'.capitalize())
            return HttpResponseRedirect(reverse_lazy('sms_packages'))


# PACKAGE DELETING STARTS -------------------------
class DeleteSmsPackages(OnlyNugsoft, View):
    template_name = 'mufukoadmin/sms_packages.html'

    def get(self, request, pk=None, *args, **kwargs):
        package = get_object_or_404(MessagePackage, id=pk)
        business_count = Business.objects.filter(package_id=pk).count()
        MessagePackage.objects.filter(id=pk).update(status=False)

        messages.success(request, 'success', extra_tags='Package removed successfully')
        return HttpResponseRedirect(reverse_lazy('sms_packages'))
