from __future__ import annotations

import re
import sys
import traceback
from datetime import *
from math import ceil, floor
from typing import Any, List

from dateutil.parser import parse
from dateutil.relativedelta import relativedelta
from django.contrib import messages
from django.contrib.auth.decorators import login_required
from django.core.paginator import Paginator, PageNotAnInteger, EmptyPage
from django.db.models import Q, F, FloatField, OuterRef, Subquery
# from django.db.models import Year, Month, Day
from django.db.models.functions import Coalesce
from django.db.models.functions import Extract
from django.http import HttpResponseRedirect, HttpResponse, Http404
from django.shortcuts import get_object_or_404
from django.shortcuts import render
from django.urls import reverse_lazy
from django.utils.decorators import method_decorator
from django.views import View
from django.views.generic import TemplateView, DetailView

from accounts.models import User, Business, Staff
from accounts.permission_mixin import BusinessUserMixin
from loans.forms.loan_application import EditRunningLoanForm
from loans.models import *
from loans.models import (
    LoanTopups,
    ApplicationAccountOrLoanType,
    LoanTypes,
    Loans,
    Loansecurities,
)
from loans.other_utils.amotization import reducing_amortization, flat_amortization, flat_amortization_aging_report, \
    reducing_amortization_aging_report
from loans.utils.amortisation import loan_payment_details, months_with_ids, get_principal
from loans.utils.bounced_loans_utils import (
    get_bounced_loans_submitted_values,
    check_bounced_loans_fields_xlsx,
    reverse_transactions_and_close_loan,
)
from loans.utils.loan_details import update_loan_balance_field, update_loan_summary_details, \
    calculate_amount_due_PAR, get_aging_report_data_v2, calculate_amount_due_v2, \
    adjust_schedule_date, get_business_loans_context_data
from loans.utils.loan_number_creation import get_new_loan_number
from loans.utils.loan_uploads_utils import (
    get_loan_principal_balance,
    close_old,
    get_top_up_upload_submitted_values,
    create_loan_upload_record,
    create_top_up_loan,
    check_loan_top_up_fields_xlsx,
    trim_lower_case_dictionary_keys, get_loan_details_payments,
)
from loans.utils.others import find_end_date
from sacco.decorators import can_do_this
from sacco.models import AccountBroker, SaccoSmsSettings
from sacco.models import CurrencySetting
from sacco.models import GroupMember
from sacco.models import LoanSettings, ActivityLog
from sacco.utils import *
from sacco.utils import (
    businessdata,
    branchdata,
    userdata,
    checkAndSendMessage,
    sendTransEmail,
)
from sacco.views import PermView
from transactions.models import Account, Transactions, StaffLedgerAssociated
# Create your views here.
from transactions.reports import today, this_branch


@method_decorator(login_required, name="dispatch")
class LoanSettingsView(View):
    template_name = "loans/loan_settings.html"

    def get(self, request, *args, **kwargs):
        business_context = get_business_loans_context_data(request)
        if business_context['branch'] is None:
            messages.error(
                request, "error",
                extra_tags="Action not allowed in the central view. Please logout and login into the respective branch"
            )
            return HttpResponseRedirect(request.META.get("HTTP_REFERER"))
        title = "Loan settings"
        try:
            loan_settings = LoanSettings.objects.filter(
                business=business_context['branch'].business
            ).first()
        except Exception as ex:
            pass
        return render(request, self.template_name, locals())

    def post(self, request, *args, **kwargs):
        business_context = get_business_loans_context_data(request)
        min_guarantors = request.POST["min_guarantors"]
        top_up_limit = request.POST["top_up_limit"]
        non_member = request.POST["non_member"]
        decimals = request.POST["decimals"]
        documents = request.POST["documents"]
        fines = request.POST["fines"]
        min_approvals = request.POST["min_approvals"]
        use_shares = request.POST["use_shares"]
        use_public_holidays = request.POST["use_public_holidays"]

        if non_member == "1":
            non_members = True
        else:
            non_members = False

        if fines == "1":
            fine = True
        else:
            fine = False

        if documents == "1":
            document = True
        else:
            document = False

        # ======== SAVE SETTINGS ============
        try:
            update_loan_settings = LoanSettings.objects.filter(
                business=business_context['branch'].business
            ).first()
            update_loan_settings.minimum_guarantors = min_guarantors
            update_loan_settings.decimals = decimals
            update_loan_settings.top_up_limit = top_up_limit
            update_loan_settings.min_approvals = min_approvals
            update_loan_settings.use_shares = use_shares
            update_loan_settings.allow_others = non_members
            update_loan_settings.upload_minutes = document
            update_loan_settings.fines_on = fine
            update_loan_settings.use_public_holidays =True if use_public_holidays == 'yes' else False
            update_loan_settings.save()
            messages.success(
                request, "success", extra_tags="Loan settings updated successfully"
            )
        except:
            LoanSettings.objects.create(
                minimum_guarantors=min_guarantors,
                decimals=decimals,
                top_up_limit=top_up_limit,
                allow_others=non_members,
                upload_minutes=document,
                fines_on=fine,
                min_approvals=min_approvals,
                use_shares=use_shares,
                use_public_holidays=use_public_holidays,
                business=business_context['branch'].business,
            )
            messages.success(
                request, "success", extra_tags="Loan type created successfully"
            )
        return HttpResponseRedirect(reverse_lazy("loan_settings"))


# applied loans
@method_decorator(login_required, name="dispatch")
class AppliedLoansView(View):
    template_name = "loans/applied_loans.html"

    def get(self, request, *args, **kwargs):
        title = "Loan Applications"

        business_context = get_business_loans_context_data(request)
        business_filter = business_context['business_filter']

        create_or_view_all_loans_for_officers = False
        if request.user.staff.position.permissions.filter(item_name='create_or_view_all_loans_for_officers').exists():
            create_or_view_all_loans_for_officers = True

        if create_or_view_all_loans_for_officers:
            loan_applications = Loans.objects.filter(
                loan_status__lte=2, branch__business_id__in=business_context['data_context']
            )
        else:
            loan_applications = Loans.objects.filter(
                loan_status__lte=2, branch__business_id__in=business_context['data_context'], officer=request.user.staff
            )

        # filters - loan number, name status
        loan_status_filter = request.GET.get('loan_status_filter', '')
        member_name = request.GET.get('member_name', '')
        loan_number = request.GET.get('loan_number', '')

        if loan_status_filter:
            loan_applications = loan_applications.filter(loan_status=int(loan_status_filter))
        if member_name:
            loan_applications = loan_applications.filter(applicant__biodata__name__icontains=member_name)
        if loan_number:
            loan_applications = loan_applications.filter(loan_number__icontains=loan_number)

        # Pagination
        page = request.GET.get('page', 1)
        paginator = Paginator(loan_applications, 10)

        try:
            loan_applications = paginator.page(page)
        except PageNotAnInteger:
            loan_applications = paginator.page(1)
        except EmptyPage:
            loan_applications = paginator.page(paginator.num_pages)

        return render(request, self.template_name, locals())


# running loans
@method_decorator(login_required, name="dispatch")
class RunningLoans(View):
    template_name = "loans/running_loans.html"

    def get(self, request, *args, **kwargs):
        title = "Running Loans"

        business_context = get_business_loans_context_data(request)
        business_filter = business_context['business_filter']

        loan_settings = LoanSettings.objects.filter(
            business=businessdata(request)
        ).first()

        decimals = loan_settings.decimals

        loan_officers = Staff.objects.filter(branch__business_id__in=business_context['data_context'])

        create_or_view_all_loans_for_officers = False

        if request.user.staff.position.permissions.filter(item_name='create_or_view_all_loans_for_officers').exists():
            create_or_view_all_loans_for_officers = True

        if create_or_view_all_loans_for_officers:
            running_loans = Loans.objects.filter(branch__business_id__in=business_context['data_context'], loan_status=3).annotate(
                applicant_group=Subquery(
                    GroupMember.objects.filter(
                        member=OuterRef('applicant')
                    ).values('group__name')[:1])
            )

        else:
            running_loans = Loans.objects.filter(
                branch__business_id__in=business_context['data_context'], loan_status=3, officer=request.user.staff
            ).annotate(
                applicant_group=Subquery(
                    GroupMember.objects.filter(
                        member=OuterRef('applicant')
                    ).values('group__name')[:1])
            )

        # Filtering
        loan_officer = request.GET.get('loan_officer', '')
        start_date = request.GET.get('start_date', '')
        end_date = request.GET.get('end_date', '')

        if loan_officer:
            running_loans = running_loans.filter(officer_id=loan_officer)
        if start_date:
            running_loans = running_loans.filter(schedule_start__gte=start_date)
        if end_date:
            running_loans = running_loans.filter(schedule_start__lte=end_date)

        # Pagination
        page = request.GET.get('page', 1)
        paginator = Paginator(running_loans, 10)

        try:
            running_loans = paginator.page(page)
        except PageNotAnInteger:
            running_loans = paginator.page(1)
        except EmptyPage:
            running_loans = paginator.page(paginator.num_pages)

        return render(request, self.template_name, locals())

    def post(self, request, *args, **kwargs):
        try:
            title = "Running Loans"

            loan_officer = request.POST.get("loan_officer") or None

            start_date = request.POST.get("start_date") or None

            end_date = request.POST.get("end_date") or None

            if start_date and end_date and start_date > end_date:
                messages.error(request, "error", extra_tags="Start date cannot be after end date")
                return render(request, self.template_name, locals())

            loan_settings = LoanSettings.objects.filter(
                business=businessdata(request)
            ).first()

            decimals = loan_settings.decimals

            loan_officers = Staff.objects.filter(branch=this_branch(request.user))

            create_or_view_all_loans_for_officers = False

            if request.user.staff.position.permissions.filter(
                    item_name='create_or_view_all_loans_for_officers').exists():
                create_or_view_all_loans_for_officers = True

            if create_or_view_all_loans_for_officers:
                running_loans = Loans.objects.filter(
                    branch__business=businessdata(request), loan_status=3,
                ).annotate(
                    applicant_group=Subquery(
                        GroupMember.objects.filter(
                            member=OuterRef('applicant')
                        ).values('group__name')[:1])
                )

                # Add filters conditionally to avoid unnecessary conditions
                if loan_officer:
                    running_loans = running_loans.filter(officer=loan_officer)

                if start_date:
                    running_loans = running_loans.filter(created_date__gte=start_date)

                if end_date:
                    running_loans = running_loans.filter(created_date__lte=end_date)

            else:
                running_loans = Loans.objects.filter(
                    branch__business=businessdata(request), loan_status=3, officer=loan_officer
                ).annotate(
                    applicant_group=Subquery(
                        GroupMember.objects.filter(
                            member=OuterRef('applicant')
                        ).values('group__name')[:1])
                )

                if start_date:
                    running_loans = running_loans.filter(created_date__gte=start_date)

                if end_date:
                    running_loans = running_loans.filter(created_date__lte=end_date)

            return render(request, self.template_name, locals())

        except Exception as exception:
            messages.error(request, "error", extra_tags="An error has occured")
            return render(request, self.template_name, locals())


@method_decorator(login_required, name="dispatch")
class TopupReport(View):
    template_name = "loans/loan_topups.html"

    def get(self, request, *args, **kwargs):
        business_context = get_business_loans_context_data(request)
        business_filter = business_context['business_filter']

        if business_context['the_current_branch_name'] == 'all branches':
            title = "Loan topups for all branches"
        else:
            the_current_branch_name = business_context['the_current_branch_name']
            title = f"Loan topups for {the_current_branch_name}"
        # title = "Loan topups"
        settings = LoanSettings.objects.filter(business=businessdata(request)).first()

        loan_topups = LoanTopups.objects.filter(
            branch__business_id__in=business_context['data_context'], approved=True, topup_loan__isnull=False
        )
        context = {
            'title': title,
            'loan_topups': loan_topups
        }

        return render(request, self.template_name, context)


@method_decorator(login_required, name="dispatch")
class OtherLoans(View):
    template_name = "loans/other_loans.html"

    def get(self, request, pk, *args, **kwargs):
        if pk == 5:
            title = "Rejected Loans"

        if pk == 6:
            title = "Waived off Loans"

        if pk == 7:
            title = "Written off Loans"

        if pk == 8:
            title = "Topped up Loans"

        loanApplications = Loans.objects.filter(
            branch__business=businessdata(request), loan_status=pk
        )
        return render(request, self.template_name, locals())


@method_decorator(login_required, name="dispatch")
class ClearedLoans(View):
    template_name = "loans/cleared_loans.html"

    def get(self, request, *args, **kwargs):
        business_context = get_business_loans_context_data(request)
        business_filter = business_context['business_filter']

        if business_context['the_current_branch_name'] == 'all branches':
            title = "Cleared Applications for all branches"
        else:
            the_current_branch_name = business_context['the_current_branch_name']
            title = f"Cleared Applications for {the_current_branch_name}"
        # title = "Cleared Applications"
        settings = LoanSettings.objects.filter(business=businessdata(request)).first()
        decimals = settings.decimals
        cleared_and_closed_loans = Loans.objects.filter(
            branch__business_id__in=business_context['data_context'], loan_status=4
        )
        return render(request, self.template_name, locals())


# ==========NEW LOAN APPLICATION============================
@method_decorator(login_required, name="dispatch")
class ApplyLoan(View):
    template_name = "loans/apply_loan.html"

    def get(self, request, *args, **kwargs):
        business_context = get_business_loans_context_data(request)

        if business_context['branch'] is None:
            messages.error(
                request, "error",
                extra_tags="Action not allowed in the central view. Please logout and login into the respective branch"
            )
            return HttpResponseRedirect(request.META.get("HTTP_REFERER"))

        if business_context['the_current_branch_name'] == 'all branches':
            title = "Loan Application for all branches"
        else:
            the_current_branch_name = business_context['the_current_branch_name']
            title = f"Loan Application for {the_current_branch_name}"
        # title = "Loan Application"
        loan_types = LoanTypes.objects.filter(
            business=business_context['branch'].business, is_active=True
        )
        members = AccountBroker.objects.filter(
            members__biodata__business=business_context['branch'].business
        )
        officers = Staff.objects.filter(branch=business_context['branch'], is_active=True)
        create_or_view_all_loans_for_officers = False
        if request.user.staff.position.permissions.filter(item_name='create_or_view_all_loans_for_officers').exists():
            create_or_view_all_loans_for_officers = True

        the_staff = request.user.staff

        return render(request, self.template_name, locals())

    # =================RECEIVE FORM APPLICATION DATA==============================
    def post(self, request, *args, **kwargs):
        business_context = get_business_loans_context_data(request)
        loan_type = LoanTypes.objects.filter(id=request.POST["loan_type"]).first()
        applicant = AccountBroker.objects.filter(
            the_account_id=request.POST["account"]
        ).first()
        amount = request.POST["amount"]
        amount = amount.replace(",", "")
        application_date = request.POST["adate"]
        interest = request.POST.get("interest", None)
        duration = request.POST["duration"]
        rate_type = request.POST["rate_type"]
        payment_method = request.POST["payment_method"]
        officer = request.POST.get('officer')
        disregard_mandatory_saving = request.POST.get('disregard_mandatory')
        if disregard_mandatory_saving == 'yes':
            disregard_mandatory_saving = True
        else:
            disregard_mandatory_saving = False
        # Check if officer_id is not available or is an empty string
        if not officer:
            officer = request.user.staff.id
        else:
            officer = officer
        guarantorship_type = request.POST.get("guarantorship_type", None)
        # print('guarantorship_type',guarantorship_type)
        if guarantorship_type == 'group':
            guarantorship_type = True
        else:
            guarantorship_type = False

        # determine the mandatory savings balance, percentage and guarantor
        # default group percentage
        loan_percentage_required = 15
        # if group guarantorship
        the_group_guarantor = None
        if guarantorship_type:
            # 15% for first loan and 20% for after that
            check_all_loans = Loans.objects.filter(account=request.POST["account"], branch=business_context['branch'])

            if check_all_loans.count() > 0:
                loan_percentage_required = 20
            grp_mm = GroupMember.objects.filter(member=applicant.members)
            if grp_mm.exists():
                the_group_guarantor = grp_mm.first().group
            else:
                the_group_guarantor = None
        else:
            loan_percentage_required = 20

        # determine the maximum amt this person can get basing on shares
        total_shares_owned = applicant.members.shares
        base_loan_amt = 300000
        if total_shares_owned < 1:
            max_amt_for_loan = base_loan_amt
        else:
            max_amt_for_loan = float(total_shares_owned) * base_loan_amt

        if float(amount) > max_amt_for_loan:
            messages.error(
                request, "error", extra_tags=f"Member cannot take a loan of more than {max_amt_for_loan}"
            )
            return HttpResponseRedirect(reverse_lazy("apply_loan"))

        #         ============SAVE LOAN APPLICATION INTO THE DATABASE======================
        the_loan_created = Loans.objects.create(
            account_id=applicant.the_account.id,
            applicant_id=applicant.members.id,
            branch=business_context['branch'],
            amount_requested=amount,
            rate=interest,
            rate_type=rate_type,
            requested_on=application_date,
            loan_type_id=loan_type.id,
            interval=loan_type.interval,
            sub_intervals=loan_type.sub_intervals,
            requested_duration=duration,
            repayment_method=payment_method,
            loan_number=get_new_loan_number(),
            added_by_id=userdata(request),
            group_guaratorship=guarantorship_type,
            savings_percentage_balance=loan_percentage_required,
            group_guarantor=the_group_guarantor,
            officer_id=officer,
            disregard_manadatory_saving=disregard_mandatory_saving
        )
        # if guarantorship_type:
        #     guarantee = Loanguarantors.objects.create(
        #         loan=the_loan_created, biodata_id=the_loan_created.applicant.biodata, added_by_id=userdata(request)
        #     )

        messages.success(
            request, "success", extra_tags="Loan Application submitted successfully"
        )
        return HttpResponseRedirect(reverse_lazy("loan_applications"))


# =============================Loan types start==========================
class LoanTypesView(PermView):
    deco_role = "add_loan_product"
    perm_name = "edit_loan_product"
    template_name = "loans/loan_types.html"

    def get(self, request, *args, **kwargs):
        business_context = get_business_loans_context_data(request)
        if business_context['the_current_branch_name'] == 'all branches':
            title = "Loan Products for all branches"
        else:
            the_current_branch_name = business_context['the_current_branch_name']
            title = f"Loan Products for {the_current_branch_name}"
        # title = "Loan Products"

        loantypes = LoanTypes.objects.filter(
            business=business_context['branch'].business, is_active=True
        )

        return render(request, self.template_name, locals())

    # ====WHEN THE POST IS MADE==========================
    @can_do_this
    def post(self, request, *args, **kwargs):
        business_context = get_business_loans_context_data(request)
        form_type = request.POST["form_type"]
        if form_type == "1":
            name = request.POST["name"]
            maxamount = request.POST["maxamount"]
            interest = request.POST["interest"]
            excused = request.POST["excused"]
            formula = request.POST["formula"]
            warning_days = request.POST["warningd"]
            maximum_securities = request.POST["maximum_securities"]
            security_ratio = request.POST["security_ratio"]
            payInterval = request.POST["interval"]
            subSchedule = request.POST["subSchedule"]
            maximum_period = request.POST["maximum_period"]

            # =========== INSERTING LOAN TYPES INTO THE DATABASE ============
            LoanTypes.objects.create(
                name=name,
                max_loan=maxamount,
                interest_rate=interest,
                excuse_d=excused,
                warning_d=warning_days,
                maximum_securities=maximum_securities,
                security_ratio=security_ratio,
                business=business_context['branch'].business,
                added_by_id=userdata(request),
                interval=payInterval,
                formula=formula,
                sub_intervals=subSchedule,
                maximum_period=maximum_period,
            )

            messages.success(
                request, "success", extra_tags="Loan type created successfully"
            )

        # ======== EDITING LOAN TYPE ===============
        if form_type == "2":
            name = request.POST["name"]
            typeid = request.POST["typeid"]
            maxamount = request.POST["maxamount"]
            interest = request.POST["interest"]
            formula = request.POST["formula"]
            excused = request.POST["excused"]
            warning_days = request.POST["warningd"]
            maximum_securities = request.POST["maximum_securities"]
            security_ratio = request.POST["security_ratio"]
            pinterval = request.POST["interval"]
            maximum_period = request.POST["maximum_period"]

            # ===== INSERTING LOAN TYPES INTO THE DATABASE ========
            edittypes = LoanTypes.objects.filter(id=typeid)
            for edittype in edittypes:
                edittype.name = name
                edittype.max_loan = maxamount
                edittype.interest_rate = interest
                edittype.formula = formula
                edittype.excuse_d = excused
                edittype.warning_d = warning_days
                edittype.maximum_securities = maximum_securities
                edittype.security_ratio = security_ratio
                edittype.interval = pinterval
                edittype.maximum_period = maximum_period
                edittype.save()
                messages.success(
                    request, "success", extra_tags="Loan type Edited successfully"
                )

        # ========= DELETING LOAN TYPES INTO THE DATABASE ============
        if form_type == "3":
            typeid = request.POST["typeid"]
            deletetypes = LoanTypes.objects.filter(id=typeid)
            for deletetype in deletetypes:
                deletetype.is_active = False

                deletetype.save()
                messages.success(
                    request, "success", extra_tags="Loan type Deleted successfully"
                )

        # ========== ADD LOAN SETTINGS ================
        if form_type == "settings":
            typeid = request.POST["typeid"]
            messages.success(
                request, "success", extra_tags="Loan type Deleted successfully"
            )
        return HttpResponseRedirect(reverse_lazy("loan_types"))


# =============================Loan Details==========================
@method_decorator(login_required, name="dispatch")
class LoanDetails(View):
    template_name = "loans/loan_details.html"

    def get(self, request, pk, *args, **kwargs):
        business_context = get_business_loans_context_data(request)
        title = "Loan Details "
        loanid = pk
        loansid = int(loanid)
        
        nowdate = today.strftime("%Y-%m-%d")
        # this_business = Business.objects.filter(id=businessdata(request)).first()
        real_loan_details = Loans.objects.get(id=pk)

        all_staff = Staff.objects.filter(branch=business_context['branch'])

        can_access_committee_action = False
        if request.user.staff.position.permissions.filter(item_name='committee_action').exists():
            can_access_committee_action = True
        can_disburse_loan_money = False
        if request.user.staff.position.permissions.filter(item_name='can_disburse_loan_money').exists():
            can_disburse_loan_money = True

        can_handle_loan_repayment = False
        if request.user.staff.position.permissions.filter(item_name='can_handle_loan_repayment').exists():
            can_handle_loan_repayment = True

        can_delete_loan = False
        if request.user.staff.position.permissions.filter(item_name='delete_loan_at_all_stages').exists():
            can_delete_loan = True

        branch_to_use = real_loan_details.branch
        business_to_use = real_loan_details.branch.business

        # print('------------------------------------------------------')
        # print(branch_to_use)
        # print(business_to_use)
        # print('---------------------------------------------------------')

        try:
            detail = Loans.objects.raw(
                "SELECT l.*, b.the_account_id as accountid, b.members_id as memberid, a.acc_number, m.biodata_id, m.id, c.name AS member_name, year(l.approved_on) as yapp, month(l.approved_on) as appmonth, day(l.approved_on) as appday FROM account_broker b, member_accounts a, sacco_member m, biodata c, loans l where b.the_account_id = a.id and b.business_id = %s and b.members_id = m.id and m.biodata_id = c.id and l.account_id = a.id and l.branch_id = %s and l.id=%s"
                % (business_to_use.id, branch_to_use.id, pk)
            )[0]
            # approved_on_date = real_loan_details.approved_on
            # detail = Loans.objects.filter(
            #     id=real_loan_details.id,
            # ).annotate(
            #     accountid=F('account__broker__the_account_id'),
            #     memberid=F('account__broker__members_id'),
            #     acc_number=F('account__acc_number'),
            #     biodata_id=F('account__broker__members__biodata_id'),
            #     member_id=F('account__broker__members__id'),
            #     member_name=F('account__broker__members__biodata__name'),
            #     yapp='',
            #     appmonth='',
            #     appday=''
            # ).first()

        except IndexError as e:
            print(e)
            raise Http404()
        except AttributeError as _e:
            detail = None
        except (Exception,) as e:
            print(e)
            detail = None
        # print('detai')
        print('------------------------------------------------------------------------')

        applicant_savings = MemberAccount.objects.get(id=detail.accountid).balance or 0
        share_price = (
            OtherSettings.objects.filter(business=business_to_use)
            .first()
            .share_price
        )
        member_shares = Member.objects.get(id=detail.applicant_id).shares * share_price
        # print('Applicant Savings:', applicant_savings, 'Member Shares:', member_shares, "share_price:", share_price)
        laonacct = Account.objects.filter(
            name="Loan Receivables", business=business_to_use
        )[0]
        istopup = detail.is_topup

        formula = detail.loan_type.formula
        approved_amount = detail.amount_approved
        try:
            memberledger = Account.objects.filter(member_id=detail.accountid)[0]
        except Exception as mb:
            #     =======================CREATE MEMBER LEDGER==================================
            lcategory = AccountCategory.objects.filter(
                name="Members", business=business_to_use
            )[0]
            creatledger = Account.objects.create(
                member_id=detail.accountid,
                category_id=lcategory.id,
                business=business_to_use,
                added_by_id=userdata(request),
            )

            memberledger = creatledger

        if detail.is_topup == True and int(detail.loan_status) < 3:
            title = "Loan topup application details"

        elif int(detail.loan_status) < 3:
            title = "Loan application details"

        try:
            paid = Transactions.objects.raw(
                "SELECT SUM(reporting_amount) as tpaid,id from transactions where (transaction_type='Loan repayment' OR transaction_type='Loan interest') AND loan_id= %s "
                % (loanid)
            )[0]
            paidamount = paid.tpaid

        except:
            paidamount = 0

        #    ========================CLEARED INTERESTS============================================

        interestspaid = Transactions.objects.filter(
            loan_id=loanid,
            transaction_type="Loan interest",
            branch__business=business_to_use,
        )
        intpaid = interestspaid.count()

        loanr = Account.objects.filter(
            name="Loan receivables", business=business_to_use
        )[0]
        try:
            added_adjustment = Transactions.objects.filter(
                Q(account_dr=loanr.id, loan_id=loanid),
                Q(transaction_type="Principal Adjustment"),
            ).aggregate(adjustedup=Sum("reporting_amount"))
            adjustment_added = added_adjustment["adjustedup"]
            if adjustment_added is None:
                adjustment_added = 0

        except Exception as exp:
            print(str(exp))
            traceback.print_exc()
            adjustment_added = 0

        #     ========adjusted down =================

        try:
            subtracted_adjustment = Transactions.objects.filter(
                Q(account_cr=loanr.id, loan_id=loanid),
                Q(transaction_type="Principal Adjustment"),
            ).aggregate(adjusteddown=Sum("reporting_amount"))
            adjustment_down = subtracted_adjustment["adjusteddown"]
            if adjustment_down is None:
                adjustment_down = 0

        except Exception as exp:
            print(str(exp))
            adjustment_down = 0

        # print("UP IS ", adjustment_added, "DOWN IS ", adjustment_down)

        try:
            totaladjustments = Transactions.objects.filter(
                loan_id=loanid, transaction_type="Loan Adjustment"
            ).aggregate(adjustments=Sum("reporting_amount"))["adjustments"]

            adjustments = Transactions.objects.filter(
                loan_id=loanid, transaction_type="Loan Adjustment"
            )

            try:
                totaladjustments = totaladjustments + 0

            except:
                totaladjustments = 0

        except Exception as ex:
            print("ERROR IS ", str(ex))
            totaladjustments = 0

        # ===========================================================================================
        # print('ADJUSTMENT IS ',totaladjustments)
        try:
            loan_setings = LoanSettings.objects.filter(business=business_to_use)[
                0
            ]
            decimals = loan_setings.decimals
            nonemember = loan_setings.allow_others
            minguarantors = loan_setings.minimum_guarantors
            top_up_limit = loan_setings.top_up_limit
            min_approvals = loan_setings.min_approvals
            use_shares = loan_setings.use_shares
        except Exception as gex:
            decimals = 0
            minguarantors = 0
            nonemember = False
            top_up_limit = 0
            min_approvals = 1
            use_shares = 1

        securities = Loansecurities.objects.filter(loan=loanid)

        # ======================SEND TO THE VIEW CALCULATED CHARGES======================================

        totalcharges = 0

        if int(detail.loan_status) > 1:
            actualchargesx = detail.charges
            try:
                actualcharges = json.loads(actualchargesx)
                for chg in actualcharges:
                    chargeamount = chg["amount"]
                    totalcharges = totalcharges + chargeamount

            except:
                totalcharges = 0

        # ============ GET CHARGES IF THE CHARGES ARE STILL IN APPLICATION STAGE ====================
        else:
            charges = ApplicationAccountOrLoanType.objects.filter(
                loan_type=detail.loan_type,
                general_charge__status=1,
                general_charge__application="l",
            )

            actualcharges = []
            for charge in charges:
                ispercentage = charge.general_charge.is_percentage
                if ispercentage == 1:
                    if detail.loan_status < 2:
                        chargeamount = (
                                               charge.general_charge.amount * detail.amount_requested
                                       ) / 100
                    else:
                        chargeamount = (
                                               charge.general_charge.amount * detail.amount_approved
                                       ) / 100

                else:
                    chargeamount = charge.general_charge.amount

                actualcharges.append(
                    {"name": charge.general_charge.charge, "amount": chargeamount}
                )

                totalcharges = totalcharges + chargeamount

        if detail.is_topup == True:
            oldloan = LoanTopups.objects.filter(topup_loan_id=loanid).first()

            oldloanid = oldloan.closed_loan.id

            # =============================GET PRINCIPAL BALANCE=========================
            recevables_account = Account.objects.filter(
                name="Loan Receivables", business=business_to_use
            )[0]

            try:
                try:
                    #     ===========IF MEMBER HAS MADE SOME PAYMENT==================
                    if detail.loan_status > 2:
                        lcredits = Transactions.objects.filter(
                            Q(loan_id=oldloanid, account_cr=recevables_account.id),
                            ~Q(transaction_type="Closing Loan"),
                        ).aggregate(principlecredits=Sum("reporting_amount"))[
                            "principlecredits"
                        ]
                    else:
                        lcredits = Transactions.objects.filter(
                            loan_id=oldloanid, account_cr=recevables_account.id
                        ).aggregate(principlecredits=Sum("reporting_amount"))[
                            "principlecredits"
                        ]
                except:
                    lcredits = 0

                ldebits = Transactions.objects.filter(
                    loan_id=oldloanid, account_dr=recevables_account.id
                ).aggregate(principledebits=Sum("reporting_amount"))["principledebits"]

                if lcredits is None:
                    oldbalance = ldebits

                else:
                    oldbalance = float(ldebits) - float(lcredits)
                oldbalance = float(oldbalance)

            except Exception as ex:
                print("Error is ", str(ex))

                oldbalance = 0

            totalpold = Transactions.objects.filter(
                loan_id=oldloanid, transaction_type="Loan repayment"
            ).aggregate(totalpaidprincipalold=Sum("reporting_amount"))

            totalpaidprincipalold = totalpold["totalpaidprincipalold"]
            try:
                totalpaidprincipalold = float(totalpaidprincipalold)
            except:
                totalpaidprincipalold = 0

        if detail.loan_status < 2:
            if detail.charges_paid == False:
                if detail.is_topup == True:
                    disbusable = approved_amount - (totalcharges)
                else:
                    disbusable = approved_amount
            else:
                if detail.is_topup == True:
                    disbusable = approved_amount - (totalcharges)
                else:
                    disbusable = approved_amount - totalcharges
        else:
            if not detail.charges_paid:
                if detail.is_topup:
                    disbusable = approved_amount - (oldbalance)
                else:
                    disbusable = approved_amount
            else:
                if detail.is_topup == True:
                    disbusable = approved_amount - (oldbalance)
                else:
                    disbusable = approved_amount

        disbusable = round(disbusable, decimals)

        guarantors = Loanguarantors.objects.filter(loan=loanid)
        docs = LoanUpload.objects.filter(loan=loanid)
        loantypes = LoanTypes.objects.filter(
            business=business_to_use, is_active=True
        )

        if loan_setings.upload_minutes is True and docs.count() > 0:
            must_upload = 1
        elif loan_setings.upload_minutes is True and docs.count() < 1:
            must_upload = 0
        else:
            must_upload = 1

        # ==================GET INTEREST IN RELATION TO SETTINGS=====================
        # get the payment interval and if None add it the loan obj
        if detail.interval is not None:
            typeinterval = detail.interval
        else:
            typeinterval = detail.loan_type.interval
            detail.interval = typeinterval
            detail.save()

        interest = (detail.rate / 100) / 12

        if detail.loan_status > 2:
            # print('meee!')
            # repayments = Transactions.objects.raw(
            #     "SELECT sum(reporting_amount) as paidamount, id, receipt, date_added FROM transactions where loan_id= %s and (transaction_type='Loan repayment' or transaction_type='Loan interest' ) GROUP BY receipt ORDER BY tx_date ASC"
            #     % (loanid)
            # )
            repayments = Transactions.objects.filter(loan_id=loanid).filter(
                Q(transaction_type='Loan repayment') | Q(transaction_type='Loan interest') | Q(
                    transaction_type='Loan Fines'))

            if repayments.exists():
                # print('repayments', repayments)
                repayments_df = pd.DataFrame(list(repayments.values()))
                repayments_df['receipt'].fillna(value='no_receipt', inplace=True)
                repayments_df = repayments_df.groupby('receipt').apply(get_loan_details_payments).reset_index()
                repayments = repayments_df.to_dict(orient='records')
            else:
                repayments = []

        if detail.loan_status > 2:
            maccount = detail.accountid

            # ===================CHECK IF THE LOAN HAS A TOP UP APPLICATION ALREADY=========================
            try:
                checktopup = Loans.objects.filter(
                    account_id=maccount, loan_status__lt=3
                )[0]
                sometopup = 1
            except:
                sometopup = 0
            try:
                paidamount = round(paidamount, decimals)

            except:
                paidamount = 0

            duration = detail.approved_duration
            appdate = detail.schedule_start

            principal = detail.amount_approved
            # print('PRINCIPAL ONE', principal)

            loanacct = Account.objects.filter(
                name="Loan Receivables", business=business_to_use
            ).first()

            # ==================TOTAL PRINCIPAL DEBIT===============================

            totaldebit = Transactions.objects.filter(
                Q(loan_id=loanid, account_dr_id=loanacct.id),
                ~Q(transaction_type="Principal Adjustment"),
            ).aggregate(totalpdebit=Sum("reporting_amount"))

            # incase the is removal of an extra charge (this is a fix for the extra charge that was being added to loan amt ap
            # approved due to 'added to principal'
            extra_charge_removed = Transactions.objects.filter(
                loan_id=loanid, transaction_type="Remove extra loan charge"
            )
            if extra_charge_removed:
                # print("there was an extra charge removed")
                extra_charge_removed = extra_charge_removed.aggregate(
                    ex_charge=Sum("reporting_amount")
                )["ex_charge"]
            else:
                extra_charge_removed = 0

            # print(f'principal first:{principal}')
            # print(f'totalpdebit:{totaldebit["totalpdebit"]} add:{adjustment_down} adu{adjustment_added} ext:{extra_charge_removed} ta:{totaladjustments}')
            totaldebit = 0 if totaldebit['totalpdebit'] is None else totaldebit['totalpdebit']
            # print('ret', totaldebit)
            principal = (
                    float(totaldebit)
                    - float(adjustment_down - adjustment_added)
                    - extra_charge_removed
                    + totaladjustments
            )
            # print(f'principal second:{principal}')
            approved_amount = principal

            principal = float(principal)
            approved_amount = float(approved_amount)

            if formula == 5:
                interest = detail.rate / 100

            if formula == 1:  # per annum
                if typeinterval == 3:  # months
                    interest = (detail.rate / 100) / 12

                if typeinterval == 0:  # Days
                    interest = (detail.rate / 100) / 365

                if typeinterval == 1:  # weeks
                    interest = (detail.rate / 100) / 52

                if typeinterval == 2:  # Fortnight
                    interest = (detail.rate / 100) / 26

                if typeinterval == 4:
                    interest = detail.rate / 100

            if formula == 2:  # per month
                if typeinterval == 3:
                    interest = detail.rate / 100

                if typeinterval == 0:
                    interest = (detail.rate / 100) / 30

                if typeinterval == 1:  # week - schedule
                    interest = (detail.rate / 100) / 4

                if typeinterval == 2:
                    interest = (detail.rate / 100) / 2

                if typeinterval == 4:
                    interest = (detail.rate / 100) * 12

            if formula == 3:
                if typeinterval == 3:
                    interest = (detail.rate / 100) * 4

                if typeinterval == 0:
                    interest = (detail.rate / 100) * 7

                if typeinterval == 1:
                    interest = detail.rate / 100

                if typeinterval == 2:
                    interest = (detail.rate / 100) * 2

                if typeinterval == 4:
                    interest = (detail.rate / 100) * 52

            if formula == 4:
                if typeinterval == 3:
                    interest = (detail.rate / 100) * 30

                if typeinterval == 0:
                    interest = detail.rate / 100

                if typeinterval == 1:
                    interest = (detail.rate / 100) * 7

                if typeinterval == 2:
                    interest = (detail.rate / 100) * 14

                if typeinterval == 4:
                    interest = (detail.rate / 100) * 365

            # ===================GET LOAN REPAYMENTS========================================

            # =====================IF INTERVAL IS MONTHLY =========================
            if typeinterval == 3:
                to_add = relativedelta(months=1)
                try:
                    all_duration = relativedelta(months=duration)
                except Exception as exp:
                    print(str(exp))

            # =====================IF INTERVAL IS DAILY =========================

            if typeinterval == 0:
                to_add = relativedelta(days=1)
                all_duration = relativedelta(days=duration)

            # =====================IF INTERVAL IS WEEKLY =========================

            if typeinterval == 1:
                to_add = relativedelta(weeks=1)
                all_duration = relativedelta(weeks=duration)

            # =====================IF INTERVAL IS BI-WEEKLY =========================

            if typeinterval == 2:
                to_add = relativedelta(weeks=2)
                all_duration = relativedelta(weeks=duration * 2)

            # =====================IF INTERVAL IS YEARLY =========================

            if typeinterval == 4:
                to_add = relativedelta(years=1)
                all_duration = relativedelta(years=int(duration))

            # ================================DETERMINING THE FINES AND ON WHICH SCHEDULES==========================
            nowdate = datetime.datetime.today().date()

            # get loan fines

            loanfines = ApplicationAccountOrLoanType.objects.filter(
                loan_type_id=detail.loan_type.id,
                general_charge__application="o",
                general_charge__is_fine=True,
            )

            # loanfines=[]

            sfine = 0
            fine = 0

            recevables_account = Account.objects.filter(
                name="Loan Receivables", business=business_to_use
            )[0]

            try:
                credits = Transactions.objects.filter(
                    Q(loan_id=loanid, account_cr=recevables_account.id),
                    ~Q(transaction_type="Closing Loan"),
                ).aggregate(principlecredits=Sum("reporting_amount"))[
                    "principlecredits"
                ]

                debits = Transactions.objects.filter(
                    loan_id=loanid, account_dr=recevables_account.id
                ).aggregate(principledebits=Sum("reporting_amount"))["principledebits"]
                if credits is None:
                    principalbal = debits
                else:
                    principalbal = float(debits) - float(credits)
                # print('CREDITS:',credits, "debits:", debits)
                principalbal = float(principalbal)
                # print("PRINCIPAL FOUR", principal)
            except Exception as ex:
                print("Error is ", str(ex))
                principalbal = 0
            nextinterest = 0
            # =======================IF RATE TYPE IS FLAT============================
            cumrative_payments = 0
            if detail.rate_type == 1:
                days = []
                schedules = approved_amount / duration
                interestperschedue = approved_amount * interest
                print('###----',interestperschedue, schedules)
                payable = interestperschedue + schedules

                startyear = detail.yapp
                startmonth = detail.appmonth + 1
                startday = detail.appday

                # ========================================GET PAID SCHEDULES=========================

                paidschedules = paidamount / payable
                paidschedules = floor(paidschedules)

                # ====================================Balance for the next schedule=======================
                nextbal = paidamount - (payable * paidschedules)
                nextintallment = payable - nextbal

                # print('NeXT INSTALL ', nextintallment, 'bal ', nextbal)
                if 1 > nextintallment > 0:
                    nextintallment = payable + nextintallment
                    nextschedule = paidschedules + 2
                else:
                    nextschedule = paidschedules + 1

                edate = appdate + to_add
                # print(appdate)
                date1 = edate
                totalpayable = payable * duration

                totalinterest = totalpayable - principal

                totalinterest = round(totalinterest, decimals)

                totalpayable = round(totalpayable, decimals)

                rbalance = totalpayable

                date2 = date1 + all_duration
                m = 1

                try:
                    principalbalance = principal

                    sch_date = date1
                    # while date1 < date2:
                    while m <= duration:
                        pbalance = 0

                        if intpaid == m or intpaid > m:
                            actualinterest = 0
                        else:
                            actualinterest = interestperschedue

                        rbalance = rbalance - payable
                        rbalance = rbalance
                        if m == nextschedule:
                            isnext = 1
                            spaid = nextbal
                            sbalance = nextintallment

                        elif m < nextschedule:
                            isnext = 0
                            spaid = payable
                            sbalance = 0

                        else:
                            isnext = 2
                            spaid = 0
                            sbalance = payable

                        if date1 < nowdate and isnext > 0:
                            for loanfine in loanfines:
                                fineamount = loanfine.general_charge.amount
                                if loanfine.general_charge.is_percentage == True:
                                    sfines = (fineamount / 100) * payable
                                else:
                                    sfines = fineamount

                                sfine = float(sfine) + float(sfines)
                        else:
                            sfine = 0

                        sfine = round(sfine, decimals)

                        # print("S FINE", sfine)

                        # print("PRINCIPAL BALANCE BEFORE", principalbalance)

                        principalbalance = principalbalance - schedules

                        # print("PRINCIPAL BALANCE AFTER", principalbalance)

                        days.append(
                            [
                                date1.strftime("%Y-%m-%d"),
                                isnext,
                                round(spaid, decimals),
                                round(sbalance, decimals),
                                round(rbalance, decimals),
                                round(actualinterest, decimals),
                                sfine,
                                round(principalbalance, decimals),
                                round(payable, decimals),
                            ]
                        )

                        payable = round(payable, decimals)
                        schedules = round(schedules, decimals)
                        interestperschedue = round(interestperschedue, decimals)
                        if date1.weekday() == 5:
                            date1 = date1 + to_add + relativedelta(days=1)
                            date2 = date2 + relativedelta(days=1)

                        else:
                            date1 = date1 + to_add

                        sfine = 0

                        m = m + 1
                except Exception as erp:
                    print("ERROR %", str(erp))

                generalbalance: int | Any = (totalpayable - paidamount) - (
                        adjustment_down - adjustment_added
                )
                try:
                    if generalbalance == 0 or generalbalance < 1:
                        generalbalance = 0
                        detail.loan_status = 4
                except:
                    pass
                generalbalance = round(generalbalance, decimals)
                detail.balance = generalbalance
                detail.save()

            # =============================================IF THE RATE TYPE IS REDUCING================================
            if detail.rate_type == 2:
                in_date = ""
                loan_dict = {}
                data = []
                days = []
                # print('apt', appdate)
                # print('to_add', to_add)
                if appdate:
                    edate = appdate + to_add
                else:
                    edate = to_add
                date1 = edate
                principals = 0
                date2 = date1 + all_duration
                runningbal = approved_amount
                if interest > 0:
                    numilator = (
                            principal
                            * interest
                            * (
                                    (pow((1 + interest), duration))
                                    / (pow((1 + interest), duration) - 1)
                            )
                    )
                else:
                    numilator = principal / duration
                payable = numilator

                totalpayable = payable * duration

                totalpayable = totalpayable

                rprincipal = 0
                rbalance = principal
                # ========================================GET PAID SCHEDULES=========================

                try:
                    paidschedules = paidamount / payable
                except ZeroDivisionError:
                    paidschedules = 0
                paidschedules = floor(paidschedules)

                # print('PAID SCHEDULES ARE ', paidschedules, 'Paid Is ', paidamount, 'Payabale is ', payable)

                # ====================================Balance for the next schedule=======================
                nextbal = paidamount - (payable * paidschedules)
                nextintallment = payable + float(totaladjustments) - nextbal
                # print('NeXT INSTALL ',nextintallment, 'bal ',nextbal )
                if nextintallment < 1 and nextintallment > 0:
                    nextintallment = payable + nextintallment
                    nextschedule = paidschedules + 2
                else:
                    nextschedule = paidschedules + 1
                m = 1
                isnext = 0
                sfine = 0
                sch_date = date1

                if detail.sub_intervals == True:
                    if detail.interval == 4:
                        pricipalperschdule = principal / duration
                        paymentperiod = []
                        remainingprincipal = principal
                        rem_principal = principal

                        # gettotalinterest
                        totalinterest = 0
                        for a in range(0, int(duration)):
                            perinterest = rem_principal * interest
                            yearlypayable = rem_principal + perinterest
                            rem_principal = rem_principal - pricipalperschdule
                            totalinterest = totalinterest + perinterest

                        pbalance = principal
                        sch_date = date1
                        # while date1 < date2:
                        while m <= duration:
                            date3 = date1 - relativedelta(years=1)
                            runningbal = principal + totalinterest
                            for a in range(0, int(duration)):
                                generalint = remainingprincipal * interest
                                generalpayable = pricipalperschdule + generalint
                                interestperschedue = generalint / 12
                                interestperschedue = round(interestperschedue, 2)
                                payable = generalpayable / 12
                                rprincipal = payable - interestperschedue
                                # rprincipal = round(rprincipal, decimals)

                                paidschedules = paidamount / payable
                                paidschedules = floor(paidschedules)

                                for sched in range(0, 12):
                                    nextbal = paidamount - (payable * paidschedules)
                                    nextintallment = payable - nextbal
                                    if nextintallment < 1 and nextintallment > 0:
                                        nextintallment = payable + nextintallment
                                        nextschedule = paidschedules + 2
                                    else:
                                        nextschedule = paidschedules + 1

                                    sfine = 0
                                    sfines = 0
                                    if m == nextschedule:
                                        isnext = 1
                                        if nextbal < 1:
                                            spaid = 0
                                        else:
                                            spaid = round(nextbal, decimals)
                                        sbalance = round(
                                            nextintallment + totaladjustments, decimals
                                        )

                                    elif m < nextschedule:
                                        isnext = 0
                                        spaid = round(payable, decimals)
                                        sbalance = 0

                                    else:
                                        isnext = 2
                                        spaid = 0
                                        sbalance = round(payable, decimals)

                                    days.append(date1.strftime("%Y-%m-%d"))

                                    if date1 < nowdate and isnext > 0:
                                        for loanfine in loanfines:
                                            fineamount = loanfine.general_charge.amount
                                            if (
                                                    loanfine.general_charge.is_percentage
                                                    == True
                                            ):
                                                sfines = (fineamount / 100) * payable
                                            else:
                                                sfines = fineamount

                                            sfine = float(sfine) + float(sfines)
                                    else:
                                        sfine = 0

                                    sfine = round(sfine, decimals)

                                    runningbal = runningbal - payable
                                    rbalance = round(runningbal, decimals)

                                    if intpaid == m or intpaid > m:
                                        actualinterest = 0

                                    else:
                                        actualinterest = interestperschedue

                                    pbalance = pbalance - rprincipal
                                    if spaid > 1:
                                        cumrative_payments = cumrative_payments + spaid

                                    else:
                                        cumrative_payments = 0

                                    loan_dict[in_date] = date3.strftime("%Y-%m-%d")
                                    loan_dict[rprincipal] = rprincipal
                                    loan_dict[rbalance] = rbalance
                                    loan_dict[isnext] = isnext
                                    loan_dict[interestperschedue] = interestperschedue
                                    loan_dict[spaid] = spaid
                                    loan_dict[sbalance] = sbalance
                                    loan_dict[actualinterest] = actualinterest
                                    data.append(
                                        [
                                            round(rprincipal, decimals),
                                            loan_dict[in_date],
                                            loan_dict[round(rbalance, decimals)],
                                            loan_dict[
                                                round(interestperschedue, decimals)
                                            ],
                                            loan_dict[round(isnext, decimals)],
                                            loan_dict[round(spaid, decimals)],
                                            loan_dict[round(sbalance, decimals)],
                                            loan_dict[actualinterest],
                                            sfine,
                                            round(payable, decimals),
                                            round(pbalance, decimals),
                                            round(cumrative_payments, decimals),
                                        ]
                                    )

                                    if date1.weekday() != 5:
                                        date1 = date1 + to_add
                                    else:
                                        date1 = date1 + to_add + relativedelta(days=1)
                                        date2 = date2 + relativedelta(days=1)

                                    # date1 = sch_date + m * to_add
                                    date3 = sch_date + relativedelta(months=m)
                                    m = m + 1

                                    sfine = 0

                                remainingprincipal = (
                                        remainingprincipal - pricipalperschdule
                                )
                        generalbalance = (totalpayable + totaladjustments) - (
                                adjustment_down - adjustment_added
                        )

                        try:
                            if generalbalance == 0 or generalbalance < 1:
                                generalbalance = 0
                                detail.loan_status = 4

                        except:
                            pass
                        generalbalance = round(generalbalance, decimals)
                        detail.balance = generalbalance
                        detail.save()

                    # print('PRINCIPAL Five', principal)

                    if detail.interval == 3:
                        # print('I AM IN THE MONTH')
                        pricipalperschdule = principal / duration
                        paymentperiod = []
                        remainingprincipal = principal
                        rem_principal = principal
                        segments = ceil(duration / 12)
                        actualyears = floor(duration / 12)
                        halfyear = duration - (actualyears * 12)

                        totalinterest = 0
                        for a in range(0, int(actualyears)):
                            perinterest = rem_principal * interest * 12
                            # print('CUMLATIVE INT',perinterest )
                            yearlypayable = rem_principal + perinterest
                            rem_principal = rem_principal - pricipalperschdule * 12
                            totalinterest = totalinterest + perinterest
                            paymentperiod.append(12)
                        # ====FOR THE BALANCE OF MONTH==========================

                        if halfyear > 0:
                            perinterest = rem_principal * interest * halfyear
                            totalinterest = totalinterest + perinterest
                            paymentperiod.append(halfyear)
                        # print('PERIODS ', halfyear)
                        pbalance = principal
                        date3 = date1
                        # print('runningbal ', runningbal)
                        # print('totalinterest ', totalinterest)
                        runningbal = principal + totalinterest
                        index = 0
                        for a in paymentperiod:
                            if a < 12:
                                generalint = remainingprincipal * interest
                            else:
                                generalint = remainingprincipal * interest * (a / 12)
                            generalpayable = pricipalperschdule + generalint
                            interestperschedue = generalint
                            interestperschedue = interestperschedue
                            payable = generalpayable
                            rprincipal = payable - interestperschedue

                            paidschedules = paidamount / payable
                            paidschedules = floor(paidschedules)

                            for sched in range(0, int(a)):
                                nextbal = paidamount - (payable * paidschedules)
                                nextintallment = payable - nextbal

                                if nextintallment < 1 and nextintallment > 0:
                                    nextintallment = payable + nextintallment
                                    nextschedule = paidschedules + 2
                                else:
                                    nextschedule = paidschedules + 1

                                sfine = 0
                                sfines = 0
                                if m == nextschedule:
                                    isnext = 1
                                    if nextbal < 1:
                                        spaid = 0
                                    else:
                                        spaid = nextbal
                                    sbalance = nextintallment

                                elif m < nextschedule:
                                    isnext = 0
                                    spaid = payable
                                    sbalance = 0

                                else:
                                    isnext = 2
                                    spaid = 0
                                    sbalance = payable

                                days.append(date1.strftime("%Y-%m-%d"))

                                if date1 < nowdate and isnext > 0:
                                    for loanfine in loanfines:
                                        fineamount = loanfine.general_charge.amount
                                        if (
                                                loanfine.general_charge.is_percentage
                                                == True
                                        ):
                                            sfines = (fineamount / 100) * payable
                                        else:
                                            sfines = fineamount

                                        sfine = float(sfine) + float(sfines)
                                else:
                                    sfine = 0

                                sfine = round(sfine, decimals)

                                runningbal = runningbal - payable
                                rbalance = runningbal
                                if intpaid == m or intpaid > m:
                                    actualinterest = 0

                                else:
                                    actualinterest = interestperschedue

                                pbalance = pbalance - rprincipal
                                if spaid > 0:
                                    cumrative_payments = cumrative_payments + spaid

                                else:
                                    cumrative_payments = 0

                                loan_dict[in_date] = date3.strftime("%Y-%m-%d")
                                loan_dict[rprincipal] = rprincipal
                                loan_dict[rbalance] = rbalance
                                loan_dict[isnext] = isnext
                                loan_dict[interestperschedue] = interestperschedue
                                loan_dict[spaid] = spaid
                                loan_dict[sbalance] = sbalance
                                loan_dict[actualinterest] = actualinterest
                                if rbalance < 1:
                                    rbalance = 0

                                if rprincipal < 1:
                                    rprincipal = 0

                                if pbalance < 1:
                                    pbalance = 0

                                data.append(
                                    [
                                        round(rprincipal, decimals),
                                        loan_dict[in_date],
                                        round(rbalance, decimals),
                                        round(interestperschedue, decimals),
                                        loan_dict[round(isnext, decimals)],
                                        round(spaid, decimals),
                                        round(sbalance, decimals),
                                        round(actualinterest, decimals),
                                        sfine,
                                        round(payable, decimals),
                                        round(pbalance, decimals),
                                        round(cumrative_payments, decimals),
                                    ]
                                )
                                date3 = sch_date + relativedelta(months=m)
                                date1 = sch_date + m * to_add
                                m = m + 1
                                sfine = 0
                                # date1 = date1 + to_add
                                # date1 = sch_date + m * to_add
                                # date3 = date3 + relativedelta(months=1)

                            index = index + 1
                            remainingprincipal = (
                                    remainingprincipal - pricipalperschdule * 12
                            )

                else:
                    pbalance = principal
                    sch_date = date1

                    generalbalance = (totalpayable + totaladjustments - paidamount) - (
                            adjustment_down - adjustment_added
                    )
                    try:
                        if generalbalance == 0 or generalbalance < 1:
                            generalbalance = 0
                            detail.loan_status = 4

                    except:
                        pass
                    generalbalance = round(generalbalance, decimals)
                    detail.balance = generalbalance
                    detail.save()

                    expectedinterest = 0
                    # while date1 < date2:
                    while m <= duration:
                        sfine = 0
                        sfines = 0
                        if m == nextschedule or (
                                m == duration and paidschedules == duration
                        ):
                            isnext = 1

                            spaid = round(nextbal, decimals)
                            sbalance = round(nextintallment, decimals)

                        elif m < nextschedule:
                            isnext = 0
                            spaid = round(payable, decimals)
                            sbalance = 0

                        else:
                            isnext = 2
                            spaid = 0
                            sbalance = round(payable, decimals)

                        days.append(date1.strftime("%Y-%m-%d"))

                        # print(nowdate)
                        # print(date1)
                        if date1 < nowdate and isnext > 0:
                            for loanfine in loanfines:
                                fineamount = loanfine.general_charge.amount
                                if loanfine.general_charge.is_percentage == True:
                                    sfines = (fineamount / 100) * payable
                                else:
                                    sfines = fineamount

                                sfine = float(sfine) + float(sfines)
                        else:
                            sfine = 0

                        sfine = round(sfine, decimals)
                        print(sfine)

                        interestperschedue = rbalance * interest
                        interestperschedue = interestperschedue

                        expectedinterest = expectedinterest + interestperschedue
                        if interest > 0:
                            rbalance = principal * (
                                    (pow((1 + interest), duration) - pow((1 + interest), m))
                                    / (pow((1 + interest), duration) - 1)
                            )
                        else:
                            rbalance = rbalance - payable
                        if intpaid == m or intpaid > m:
                            actualinterest = 0

                        else:
                            actualinterest = interestperschedue

                        rprincipal = payable - interestperschedue
                        rprincipal = rprincipal

                        rbalance = rbalance

                        pbalance = pbalance - rprincipal
                        if spaid > 0:
                            cumrative_payments = cumrative_payments + spaid

                        else:
                            cumrative_payments = 0

                        loan_dict[in_date] = date1.strftime("%Y-%m-%d")
                        loan_dict[rprincipal] = rprincipal
                        loan_dict[rbalance] = rbalance
                        loan_dict[isnext] = isnext
                        loan_dict[interestperschedue] = interestperschedue
                        loan_dict[spaid] = spaid
                        loan_dict[sbalance] = sbalance
                        loan_dict[actualinterest] = actualinterest
                        data.append(
                            [
                                round(rprincipal, decimals),
                                loan_dict[in_date],
                                round(rbalance, decimals),
                                round(interestperschedue, decimals),
                                round(isnext, decimals),
                                round(spaid, decimals),
                                round(sbalance, decimals),
                                round(actualinterest, decimals),
                                sfine,
                                round(payable, decimals),
                                round(pbalance, decimals),
                                round(cumrative_payments, decimals),
                            ]
                        )
                        date1 = sch_date + m * to_add
                        m = m + 1
                        sfine = 0

                        # date1 = date1 + to_add

                        # //principal = rprincipal

                if detail.sub_intervals == True:
                    totalinterest = round(totalinterest, decimals)
                    totalpayable = approved_amount + totalinterest
                else:
                    totalinterest = round(totalpayable - principal, decimals)

                generalbalance = (totalpayable + totaladjustments - paidamount) - (
                        adjustment_down - adjustment_added
                )

                # print(
                #     "AT THIS POINT PRINCIPALA IS ",
                #     principal,
                #     " aproved is ",
                #     approved_amount,
                # )
                try:
                    if generalbalance == 0 or generalbalance < 1:
                        generalbalance = 0
                        detail.loan_status = 4

                except:
                    pass
                generalbalance = round(generalbalance, decimals)
                detail.balance = generalbalance
                detail.save()

                #
                totalpayable = round(totalpayable, decimals)

            nowdate = datetime.datetime.today()

        if detail.loan_status == 3:
            if nowdate.date() > date2:
                isoverdue = True

            else:
                isoverdue = False
        needed_value = (
                int(detail.loan_type.security_ratio) / 100 * int(detail.amount_requested)
        )
        total_security = list(
            Loansecurities.objects.filter(loan_id=loanid)
            .aggregate(total=Sum("value"))
            .values()
        )[0]

        if total_security is None:
            total_securities = 0
        else:
            total_securities = total_security

        if use_shares:
            total_security_value = total_securities + applicant_savings + member_shares
        else:
            total_security_value = total_securities + applicant_savings

        nsecurities = Loansecurities.objects.filter(loan_id=loanid).count()
        nguarantors = Loanguarantors.objects.filter(loan_id=loanid).count()
        maxi_securities = detail.loan_type.maximum_securities
        currency = CurrencySetting.objects.filter(
            business=business_to_use
        ).first()

        members_accts = Member.objects.filter(biodata__business=business_to_use)
        # print('Amount paid: ', paidamount, 'totalpayable', totalpayable)

        # =============TAKE BANKS======================
        # bankaccounts = Account.objects.filter(
        #     Q(business=businessdata(request)),
        #     Q(category__name="Bank")
        #     | Q(category__name="Mobile Money")
        #     | Q(category__name="Cash"),
        # )

        # determine if the staff is attached to any accounts
        check_associated_acc = StaffLedgerAssociated.objects.filter(staff=request.user.staff)
        if check_associated_acc.exists():
            acc_ids = list(check_associated_acc.values_list('account_asscociated__id', flat=True))
            bankaccounts = Account.objects.filter(id__in=acc_ids)
        else:
            bankaccounts = Account.objects.filter(
                Q(business=business_to_use),
                Q(category__name="Bank")
                | Q(category__name="Mobile Money")
                | Q(category__name="Cash"),
            )

        # determine if to show the close loan manually
        # if principal is paid off completely, status is disbursed
        real_loan_details = Loans.objects.get(id=pk)
        this_loan_fines = LoanFines.objects.filter(loan_id=loanid)
        query_fines = this_loan_fines.aggregate(Sum("amount"))
        total_fines = query_fines["amount__sum"] or 0
        # print('total_fines', total_fines)
        query_fines_paid = LoanFinePayments.objects.filter(loan_id=loanid).aggregate(
            Sum("amount")
        )
        total_fine_paid = query_fines_paid["amount__sum"] or 0
        # print('total_fine_paid', total_fine_paid)
        unpaid_fines = total_fines - total_fine_paid

        # generalbalance

        # percent_unpaid
        try:
            # print("GENERAL BAL B4 any", generalbalance)
            generalbalance = generalbalance + unpaid_fines
            percent_paid = paidamount / (totalpayable + unpaid_fines) * 100
            percent_unpaid = generalbalance / (totalpayable + unpaid_fines) * 100

            percent_paid = round(percent_paid, 2)
            percent_unpaid = round(percent_unpaid, 2)

        except:
            percent_paid = 0
            percent_unpaid = 0

        # namayingo
        # security
        # default group percentage
        loan_percentage_required = real_loan_details.savings_percentage_balance
        if loan_percentage_required is None:
            loan_percentage_required = 20

        # if group guarantorship
        the_group_guarantor = real_loan_details.group_guarantor
        # if real_loan_details.group_guaratorship:
        #     #15% for first loan and 20% for after that
        #     check_all_loans = Loans.objects.filter(account=real_loan_details.account, applicant=real_loan_details.applicant)
        #     if check_all_loans.count() > 0:
        #         loan_percentage_required = 20
        #     grp_mm = GroupMember.objects.filter(member=real_loan_details.applicant)
        #     if grp_mm.exists():
        #         the_group_guarantor = grp_mm.first().group
        # elif not real_loan_details.group_guaratorship:
        #     loan_percentage_required = 15
        savings_required = (loan_percentage_required / 100) * real_loan_details.amount_approved
        if savings_required == 0:
            savings_required = (loan_percentage_required / 100) * real_loan_details.amount_requested

        if real_loan_details.disregard_manadatory_saving:
            savings_required = 0
            loan_percentage_required = 0

        # if shares value is contributes -- then include it
        security_value = securities.aggregate(total=Sum("value"))['total']
        if security_value is None:
            security_value = 0
        remaining_required_security = float(needed_value) - float(applicant_savings) - security_value
        if remaining_required_security < 0:
            remaining_required_security = 0

        more_savings_required = savings_required - applicant_savings

        # update the summary details
        if real_loan_details.loan_status > 2:
            update_loan_summary_details(real_loan_details)

        # days - flat
        # data - reducing

        # Adjust the dates basing on the public holidays
        # get the loan settings
        the_loan_setting_public = LoanSettings.objects.get(business=real_loan_details.branch.business)
        if the_loan_setting_public.use_public_holidays and real_loan_details.loan_status in [3,4, 8, 7, 6]:
            if real_loan_details.rate_type == 1: # flat rate
                days = adjust_schedule_date(days, real_loan_details, 0)
            else: # reducing balance
                data = adjust_schedule_date(data, real_loan_details, 1)


        return render(request, self.template_name, locals())

    #     ================ CAPTURE LOAN SECURITY FROM THE FORM===========================

    def post(self, request, *args, **kwargs):
        business_context = get_business_loans_context_data(request)
        global loandetail

        form_type = request.POST["form_type"]
        loanid = request.POST.get("loanid", None)

        if business_context['branch'] is None:
            messages.error(
                request, "error",
                extra_tags="Action not allowed in the central view. Please logout and login into the respective branch"
            )
            return HttpResponseRedirect(request.META.get("HTTP_REFERER"))

        try:
            loandetail = Loans.objects.get(id=loanid)
        except Loans.DoesNotExist:
            loandetail = None
        # print("THIS IS THE TYPE " + form_type)
        if form_type == "1":
            name = request.POST["name"]
            value = request.POST["value"]
            value = float(value.replace(",", ""))
            proofs = self.request.FILES.get(
                "proof",
            )

            description = request.POST["description"]

            savesecurity = Loansecurities.objects.create(
                loan_id=loanid,
                name=name,
                value=value,
                file=proofs,
                description=description,
            )
            messages.success(
                request,
                "success",
                extra_tags=f"Loan Security of {savesecurity.name} valued at {savesecurity.value} added successfully",
            )

        if form_type == "add_fines":
            business_context = get_business_loans_context_data(request)
            loan_id = request.POST["loan_id"]
            narration = request.POST["fine_desc"]
            fine_amount = request.POST["fine_amount"]
            fine_date = request.POST["fine_date"]
            fine_date = datetime.datetime.strptime(fine_date, '%Y-%m-%d').date()
            fine_amount = float(fine_amount.replace(",", ""))
            if int(fine_amount) < 1:
                messages.error(
                    request,
                    "error",
                    extra_tags=f"Invalid Fine charged. {fine_amount} can not be processed as fine",
                )
                return HttpResponseRedirect(request.META.get("HTTP_REFERER"))

            LoanFines.objects.create(
                loan_id=loan_id,
                amount=fine_amount,
                added_by_id=userdata(request),
                narration=narration,
            )
            # Also add the fine to the fines receivable
            fine_receivables_acc = Account.objects.filter(
                name="fine receivables", business=business_context['branch'].business
            )
            if fine_receivables_acc.exists():
                fine_receivables_acc = fine_receivables_acc.first()
            else:
                fine_receivables_acc = Account.objects.create(
                    name="fine receivables",
                    business=business_context['branch'].business,
                    display_name="fine receivables",
                    category=AccountCategory.objects.filter(
                        name="Account Receivables", business=business_context['branch'].business
                    ).first(),
                )

            finesaccount = Account.objects.filter(
                name="Loan Fines", business=business_context['branch'].business
            )
            if finesaccount.exists():
                finesaccount = finesaccount.first()
            else:
                finescat = AccountCategory.objects.filter(
                    name="Loan incomes", business=business_context['branch'].business
                ).first()
                finesaccount = Account.objects.create(
                    name="Loan Fines",
                    business=business_context['branch'].business,
                    category_id=finescat.id,
                    added_by_id=userdata(request),
                )

            fines_transaction = Transactions.objects.create(
                reporting_amount=fine_amount,
                narration=f"Loan fine of {fine_amount}",
                account_dr=fine_receivables_acc,
                account_cr_id=finesaccount.id,
                loan_id=loan_id,
                tx_date=fine_date,
                transaction_type="Fines",
                added_by_id=userdata(request),
                branch=business_context['branch'],
            )

            messages.success(
                request,
                "success",
                extra_tags=f"Loan fine of {fine_amount} for {narration} added successfully",
            )

        if form_type == "addfile":
            name = request.POST["name"]

            file = request.FILES["file"]
            description = request.POST["description"]

            file_uploaded = LoanUpload.objects.create(
                loan_id=loanid, file_name=name, file=file, added_by_id=userdata(request)
            )
            messages.success(
                request,
                "success",
                extra_tags=f"{file_uploaded.file_name} uploaded successfully",
            )

        if form_type == "editsecurity":
            name = request.POST["name"]
            value = request.POST["value"]
            description = request.POST["description"]
            securityid = request.POST["securityid"]
            loanid = request.POST["loanid"]
            editsecurities = Loansecurities.objects.filter(id=securityid)
            for editsecurity in editsecurities:
                editsecurity.name = name
                editsecurity.value = value
                editsecurity.description = description
                editsecurity.save()
                messages.success(
                    request,
                    "success",
                    extra_tags=f"Loan Security of {editsecurity.name} Edited successfully with the value of {editsecurity.value}",
                )

        # ===========================DELETE SECURITY=========================================
        if form_type == "deletesecurity":
            securityid = request.POST["securityid"]
            loanid = request.POST["loanid"]
            deletesecurities = Loansecurities.objects.filter(id=securityid)
            for deletesecurity in deletesecurities:
                deletesecurity.delete()

                messages.success(
                    request,
                    "success",
                    extra_tags=f"Loan security of {deletesecurity.name} valued at {deletesecurity.value} has been successfully deleted",
                )

        # =============================DELETE GUARANTOR=========================

        if form_type == "deleteguarantor":
            guarantorid = request.POST["guarantorid"]
            loanid = request.POST["loanid"]
            deleteguarantors = Loanguarantors.objects.filter(id=guarantorid)
            for dg in deleteguarantors:
                dg.delete()

                messages.success(
                    request,
                    "success",
                    extra_tags=f"Loan guarantor {dg.biodata.name} successfully deleted",
                )

        #     =======IF THE POSTED IS FOR GUARANTOR===============================================================

        if form_type == "2":
            gtype = request.POST["gtype"]
            loanid = request.POST["loanid"]
            if int(gtype) == 2:
                name = request.POST["name"]
                gender = request.POST["gender"]
                dob = request.POST["dob"]
                code = request.POST["code"]
                contact = request.POST["contact"]
                code1 = request.POST["code1"]
                other_contact = request.POST["other_contact"]
                email = request.POST["email"]
                marital_status = request.POST["marital_status"]
                country = request.POST["country"]
                location = request.POST["location"]
                nin = request.POST["nin"]

                savebio = Biodata.objects.create(
                    name=name,
                    gender=gender,
                    dob=dob,
                    contact=contact,
                    other_contact=other_contact,
                    email=email,
                    marital_status=marital_status,
                    country=country,
                    location=location,
                    nin=nin,
                    business=business_context['branch'].business,
                )
                guarantee = Loanguarantors.objects.create(
                    loan_id=loanid, biodata_id=savebio.id, added_by_id=userdata(request)
                )
            else:
                guarantor = request.POST["account"]
                guarantee = Loanguarantors.objects.create(
                    loan_id=loanid, biodata_id=guarantor, added_by_id=userdata(request)
                )
            messages.success(
                request,
                "success",
                extra_tags=f"{guarantee.biodata.name} added successfully as a guarantor",
            )

        # ==============================EDIT LOAN APPLICATION ============================

        if form_type == "editloan":
            loan_type = request.POST["loan_type"]
            loanid = request.POST["loanid"]
            amount = request.POST["amount"]
            adate = request.POST["adate"]
            interest = request.POST["interest"]
            duration = request.POST["duration"]
            payment_method = request.POST["payment_method"]

            rate_type = request.POST["rate_type"]

            editloans = Loans.objects.filter(id=loanid)
            for editloan in editloans:
                editloan.loan_type_id = loan_type
                editloan.amount_requested = amount
                editloan.requested_on = adate
                editloan.rate = interest
                editloan.rate_type = rate_type
                editloan.requested_duration = duration
                editloan.repayment_method = payment_method
                editloan.save()

                messages.success(
                    request,
                    "success",
                    extra_tags="Loan application modification request processed successfully and changes have been accepted",
                )

        # ==========================        Delete loan application  ================================

        if form_type == "deleteloan":
            loanid = request.POST["loanid"]

            the_loan = Loans.objects.filter(id=loanid)
            if not the_loan.exists():
                messages.error(
                    request,
                    "error",
                    extra_tags="Loan nolonger exists",
                )
                return HttpResponseRedirect(reverse_lazy("loan_applications"))
            the_loan_number_to_del = the_loan.first().loan_number
            if the_loan.first().amount_approved != 0:
                the_amt_str = f'with approved amount of {str(the_loan.first().amount_approved)}'
            else:
                the_amt_str = f'with requested amount of {str(the_loan.first().amount_requested)}'
            # check if its a topup
            top_check = LoanTopups.objects.filter(topup_loan=loanid)
            is_top_up = False
            old_loan_id = None
            if not top_check.exists():
                # check if topped_up_loan is null
                if the_loan.first().topped_up_loan is not None:
                    is_top_up = True
                    old_loan_id = the_loan.first().topped_up_loan
            else:
                is_top_up = True
                old_loan_id = top_check.first().closed_loan_id

            if not is_top_up:
                # delete the transactions
                Transactions.objects.filter(loan_id=the_loan.first().id).delete()
                # delete the loan
                the_loan.delete()
            else:
                # delete closing transaction for old loan
                closing_transaction = Transactions.objects.filter(
                    loan_id=old_loan_id, transaction_type="Closing Loan"
                )
                if closing_transaction.exists():
                    closing_transaction.delete()
                    Loans.objects.filter(id=old_loan_id).update(loan_status=3)
                else:
                    # get the last two transactions
                    last_two_transactions = Transactions.objects.filter(
                        Q(transaction_type='Loan repayment') | Q(transaction_type='Loan interest'),
                        loan_id=old_loan_id).order_by('-id')[:2]
                    if last_two_transactions.count() == 2:
                        if last_two_transactions[0].transaction_type != last_two_transactions[1].transaction_type:
                            # Determine if to delete both or to remove one
                            if last_two_transactions[0].transaction_type == 'Loan repayment':
                                # then just remove the last one
                                Transactions.objects.filter(id=last_two_transactions[0].id).delete()
                            else:
                                # then remove both
                                # last_two_transactions.delete()
                                delete_ids = [
                                    last_two_transactions[0].id,
                                    last_two_transactions[1].id
                                ]
                                Transactions.objects.filter(id__in=delete_ids).delete()
                        else:
                            # delete the first one -- if they are the same - delete the last entered
                            Transactions.objects.filter(id=last_two_transactions[0].id).delete()
                    else:
                        # it will be one or no transactions
                        last_two_transactions.delete()
                    # change status to running for the old loan
                    Loans.objects.filter(id=old_loan_id).update(loan_status=3)
                # delete the transactions
                Transactions.objects.filter(loan_id=the_loan.first().id).delete()
                # delete the loan
                the_loan.delete()
                top_check.delete()

            # add the audit trail
            the_message = f'Deleted the loan with loan number:{str(the_loan_number_to_del)} {the_amt_str}'
            ActivityLog.objects.create(
                actor=request.user,
                title='Loan removal',
                action_type="DELETE",
                remarks=the_message,
                branch=business_context['branch'].id,
            )

            messages.success(
                request,
                "success",
                extra_tags="Loan application Deleting request processed successfully and loan application deleted",
            )
            return HttpResponseRedirect(reverse_lazy("loan_applications"))

        # ==========================        Delete loan application  ================================

        if form_type == "writeoff":
            loanid = request.POST["loanid"]
            writeoff = Loans.objects.filter(id=loanid)[0]
            approved_amount = writeoff.amount_approved
            try:
                alreadypaid = Transactions.objects.filter(
                    loan_id=loanid, transaction_type="Loan repayment"
                ).annotate(totalpaid=Sum("reporting_amount"))
                if alreadypaid.exists():
                    for apaid in alreadypaid:
                        totalpaid = float(apaid.totalpaid)
                        expense = approved_amount - totalpaid
                else:
                    expense = approved_amount

            except Exception as le:
                # print(str(le))
                expense = approved_amount  # print(expense)

            # expense = expense
            # print(expense)
            wdate = request.POST["wdate"]
            narration = request.POST["narration"]

            # ===========================CHECK IF DEBTS CATEGORY IS THERE==============================

            try:
                cbaddebts = AccountCategory.objects.filter(
                    name="Bad debts", business=business_context['branch'].business
                )[0]

            except Exception as badd:
                # print(str(badd))

                cbaddebts = AccountCategory.objects.create(
                    name="Bad debts",
                    cat_type="expenses",
                    dr_cr="dr",
                    business=business_context['branch'].business,
                )

            # ===========================CHECK IF DEBTS CATEGORY IS THERE==============================

            try:
                badloans = Account.objects.filter(
                    name="Bad loans written off",
                    category_id=cbaddebts.id,
                    business=business_context['branch'].business,
                )[0]

            except Exception as rex:
                # print(str(rex))
                badloans = Account.objects.create(
                    name="Bad loans written off",
                    category_id=cbaddebts.id,
                    added_by_id=userdata(request),
                    business=business_context['branch'].business,
                )

            debited = badloans.id
            credited = Account.objects.filter(
                name="Loan Receivables", business=business_context['branch'].business
            )[0]
            receipt = int(random.uniform(1000, 9999))
            savebaddebt = Transactions.objects.create(
                reporting_amount=expense,
                narration=narration,
                account_dr_id=debited,
                account_cr_id=credited.id,
                loan_id=loanid,
                tx_date=wdate,
                receipt=receipt,
                transaction_type=badloans.name,
                added_by_id=userdata(request),
                branch=business_context['branch'],
            )

            writeoff.loan_status = 7
            writeoff.save()

            messages.success(
                request,
                "success",
                extra_tags="Loan Writing-off request has been processed successfully",
            )
            return HttpResponseRedirect(reverse_lazy("running_loans"))

        if form_type == "adjust":
            loanid = request.POST["loanid"]
            wdate = request.POST["wdate"]
            amount = request.POST["amount"]
            type = request.POST["type"]
            add = request.POST["add"]
            narrative = request.POST["narration"]
            amount = float(amount.replace(",", ""))

            try:
                if int(type) == 1:
                    intrestaccount = Account.objects.filter(
                        name="Loan interests", business=business_context['branch'].business
                    ).first()

                    receivable = Account.objects.filter(
                        name="Loan interest receivables", business=business_context['branch'].business
                    )[0]

                    loanr = Account.objects.filter(
                        name="Loan receivables", business=business_context['branch'].business
                    )[0]

                    trans = Transactions.objects.create(
                        reporting_amount=amount,
                        narration=narrative,
                        account_dr_id=receivable.id,
                        account_cr_id=intrestaccount.id,
                        loan_id=loanid,
                        tx_date=wdate,
                        transaction_type="Loan Adjustment",
                        added_by_id=userdata(request),
                        branch=business_context['branch'],
                    )

                #     =========== if we adjust principal============================

                elif int(type) == 2:
                    checkrecievables = Account.objects.filter(
                        name="Loan Adjustments", business=business_context['branch'].business
                    ).exists()
                    if checkrecievables:
                        pass
                    else:
                        rcategeory = AccountCategory.objects.filter(
                            name="Bad Debts", business=business_context['branch'].business
                        )[0]
                        createleger = Account.objects.create(
                            name="Loan Adjustments",
                            business=business_context['branch'].business,
                            category_id=rcategeory.id,
                            added_by_id=userdata(request),
                        )

                    loanr = Account.objects.filter(
                        name="Loan receivables", business=business_context['branch'].business
                    )[0]
                    adjustacct = Account.objects.filter(
                        name="Loan Adjustments", business=business_context['branch'].business
                    )[0]

                    if int(add) == 1:
                        acdr = loanr.id
                        accr = adjustacct.id

                    elif int(add) == 2:
                        accr = loanr.id
                        acdr = adjustacct.id

                    trans = Transactions.objects.create(
                        reporting_amount=amount,
                        narration=narrative,
                        account_dr_id=acdr,
                        account_cr_id=accr,
                        loan_id=loanid,
                        tx_date=wdate,
                        transaction_type="Principal Adjustment",
                        added_by_id=userdata(request),
                        branch=business_context['branch'],
                    )

            except Exception as exp:
                print("The error could be ".exp)

            # messages.success(request, 'success', extra_tags='Loan adjustment has been processed successfully')
            # return HttpResponseRedirect(reverse_lazy('running_loans'))

        # =====================================================SUBMIT TO COMMITTEE===================================

        if form_type == "approval":
            loanid = request.POST["loanid"]
            submitted = Loans.objects.filter(id=loanid)
            for submit in submitted:
                submit.loan_status = 1
                submit.save()
                messages.success(
                    request,
                    "success",
                    extra_tags="Loan application submitted to the approving committee successfully",
                )

        # ============= MANUALLY CLOSE THE LOAN =========================
        if form_type == "close_loan":
            loanid = request.POST["loanid"]
            the_loan_obj = Loans.objects.filter(id=loanid)
            business_data = business_context['branch'].business.id

            with transaction.atomic():
                try:
                    if the_loan_obj:
                        the_loan = the_loan_obj.first()
                        # record transactions for balancing/ removing the interest if they were not paid
                        ln_interest_rec_acc = Account.objects.filter(
                            name="loan interest receivables", business=business_data
                        )
                        if ln_interest_rec_acc:
                            ln_interest_rec_acc = ln_interest_rec_acc.first()
                        else:
                            # print("Loan interest acc")
                            ln_interest_rec_acc = Account.objects.create(
                                name="loan interest receivables",
                                business=business_data,
                                display_name="loan interest receivables",
                                category=AccountCategory.objects.get(
                                    name="Account Receivables", business=business_data
                                ),
                            )
                        ln_interest_revenue_acc = Account.objects.filter(
                            name="Loan Interest", business=businessdata(request)
                        ).first()

                        loan_interest_receivable_debits = Transactions.objects.filter(
                            loan=the_loan, account_dr=ln_interest_rec_acc
                        )
                        print("total debits", loan_interest_receivable_debits)
                        if loan_interest_receivable_debits:
                            loan_interest_receivable_debits = (
                                loan_interest_receivable_debits.aggregate(
                                    total_debits=Sum("reporting_amount")
                                )["total_debits"]
                            )
                        else:
                            loan_interest_receivable_debits = 0

                        loan_interest_receivable_credits = Transactions.objects.filter(
                            loan=the_loan, account_cr=ln_interest_rec_acc
                        )
                        # print("total credits", loan_interest_receivable_credits)
                        if loan_interest_receivable_credits:
                            loan_interest_receivable_credits = (
                                loan_interest_receivable_credits.aggregate(
                                    total_credits=Sum("reporting_amount")
                                )["total_credits"]
                            )
                        else:
                            loan_interest_receivable_credits = 0

                        # print(
                        #     "loan_interest_receivable_debits",
                        #     loan_interest_receivable_debits,
                        #     loan_interest_receivable_credits,
                        # )
                        if (
                                loan_interest_receivable_debits
                                > loan_interest_receivable_credits
                        ):
                            Transactions.objects.create(
                                account_cr=ln_interest_rec_acc,
                                account_dr=ln_interest_revenue_acc,
                                loan=the_loan,
                                transaction_type="uncorrected interest",
                                narration="uncorrected interest",
                                tx_date=datetime.datetime.today().date(),
                                branch_id=business_context['branch'],
                                reporting_amount=(
                                        loan_interest_receivable_debits
                                        - loan_interest_receivable_credits
                                ),
                            )
                            # print("transaction created")

                        the_loan_obj.update(loan_status=4)

                        messages.success(
                            request,
                            "success",
                            extra_tags="Loan closing request processed successfully and has been closed",
                        )
                    else:
                        messages.error(
                            request, "error", extra_tags="Loan doesnot exist"
                        )
                except Exception as e:
                    print("Error", e)

                    messages.error(
                        request,
                        "error",
                        extra_tags="Please contact administrators for support",
                    )
                # print(
                #     "++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++"
                # )

        # =====================================================DISBURSE LOAN===================================

        if form_type == "disburse":
            loanid = request.POST["loanid"]
            amount_approved = request.POST["damount"]
            totalcharges = request.POST["totalcharges"]
            ddate = request.POST["d_date"]
            credited = request.POST["credited"]
            narration = request.POST["narration"]
            chargespaid = request.POST["chargespaid"]
            istopup = request.POST["istopup"]
            rec_no = datetime.datetime.now().strftime("%y%m%d%H%M%S")
            voucher = request.POST["voucher"] or rec_no
            chargesp = chargespaid
            disburse_to_mem_acc = request.POST["disburse_to"]

            if istopup is True or istopup == "True":
                oldloanid = request.POST["oldloanid"]
                oldbal = request.POST["oldbalance"]
                istopup = True
            # ================== IF THE CHARGES ARE PAID BEFORE DISBURSEMENT========================

            # ===== Check if the ledgers exist =====
            checkrecievables = Account.objects.filter(
                name="Loan Receivables", business=business_context['branch'].business
            ).exists()
            checkrecharges = Account.objects.filter(
                name="Loan Charges", business=business_context['branch'].business
            ).exists()
            checkfines = Account.objects.filter(
                name="Loan Fines", business=business_context['branch'].business
            ).exists()

            if checkrecharges:
                pass
            else:
                rcategeory2 = AccountCategory.objects.filter(
                    name="Loan incomes", business=business_context['branch'].business
                )[0]
                createleger2 = Account.objects.create(
                    name="Loan Charges",
                    business=business_context['branch'].business,
                    category_id=rcategeory2.id,
                    added_by_id=userdata(request),
                )

            if checkrecievables:
                pass
            else:
                rcategeory = AccountCategory.objects.filter(
                    name="Account Receivables", business=business_context['branch'].business
                )[0]
                createleger = Account.objects.create(
                    name="Loan Receivables",
                    business=business_context['branch'].business,
                    category_id=rcategeory.id,
                    added_by_id=userdata(request),
                )

            # =====================GET LEDGER ACCOUNTS============================
            laonacct = Account.objects.filter(
                name="Loan Receivables", business=business_context['branch'].business
            )[0]
            reserve = Account.objects.filter(
                name="Opening Reserves", business=business_context['branch'].business
            )[0]
            chargeacct = Account.objects.filter(
                name="Loan Charges", business=business_context['branch'].business
            )[0]

            # print('CHARGES Are ',chargespaid)
            if int(chargespaid) == 1:
                chargesp = 1
                deducted = False
                try:
                    chargesaccount = request.POST["chargesaccoun"]
                except:
                    messages.error(
                        request, "error", extra_tags="Charges account not selected"
                    )
                    return HttpResponseRedirect(request.META.get("HTTP_REFERER"))

            # ================IF CHARGES ARE TOPPED UP ON THE LOAN=================================
            elif int(chargespaid) == 2:
                chargesp = 2
                deducted = False
                chargesaccount = laonacct.id

            # =================== IF THE CHARGES RE DEDUCTED FROM THE LOAN=========================
            elif int(chargespaid) == 3:
                chargesp = 2
                deducted = True
                chargesaccount = laonacct.id
            # =================== =MemberAccount=========================
            elif int(chargespaid) == 4:
                memberledger = Account.objects.filter(member_id=loandetail.account_id)[
                    0
                ]
                chargesaccount = memberledger.id

                chargesp = 1
                deducted = False
            if chargesp == 1:
                chargesp = True
            else:
                chargesp = False

            submitted = Loans.objects.filter(id=loanid)
            for submit in submitted:
                # determine the mandatory savings
                mandatory_saving = (
                        float(submit.savings_percentage_balance) / 100 * float(submit.amount_approved)
                )
                if submit.disregard_manadatory_saving:
                    mandatory_saving = 0
                # check if mandatory saving is available
                applicant_savings_current = MemberAccount.objects.get(id=submit.account_id).balance or 0
                if applicant_savings_current < mandatory_saving:
                    messages.error(
                        request,
                        "Insufficient balance",
                        extra_tags=f"Member doesnot meet the required manadtory savings",
                    )
                    return HttpResponseRedirect(request.META.get("HTTP_REFERER"))

                submit.loan_status = 3
                submit.charges_paid = chargesp
                submit.schedule_start = ddate
                submit.voucher = voucher
                submit.savings_rem_balance = float(mandatory_saving)
                submit.save()

                # ==========================Save loan transaction=======================
            # determine the loan amount basing
            loan_report_amt = amount_approved
            if chargespaid == "2":
                loan_report_amt = submitted[0].amount_approved

            # normal transaction
            if disburse_to_mem_acc == 'no':
                saveloan = Transactions.objects.create(
                    reporting_amount=loan_report_amt,
                    narration=narration,
                    account_cr_id=credited,
                    account_dr_id=laonacct.id,
                    tx_date=ddate,
                    loan_id=loanid,
                    transaction_type="give loan",
                    branch=business_context['branch'],
                    added_by_id=userdata(request)
                )
            else:
                # get member acc
                loan_re_queried = Loans.objects.get(id=loanid)
                person_trans_acc = Account.objects.get(member=loan_re_queried.account)
                saveloan = Transactions.objects.create(
                    reporting_amount=loan_report_amt,
                    narration=narration,
                    # credit member acc
                    account_cr=person_trans_acc,
                    account_dr_id=laonacct.id,
                    tx_date=ddate,
                    loan_id=loanid,
                    transaction_type="give loan",
                    branch=business_context['branch'],
                    added_by_id=userdata(request)
                )

            if istopup is True:
                # ======================OLD BALANCE TRANSACTIONS ====================================
                # =========================CREDITING  IT ON THE NEW LOAN==================
                cdebitreserve = Transactions.objects.create(
                    reporting_amount=oldbal,
                    narration="Closing The load due to topup",
                    account_cr_id=laonacct.id,
                    account_dr_id=reserve.id,
                    tx_date=ddate,
                    loan_id=oldloanid,
                    transaction_type="Closing Loan",
                    branch=business_context['branch'],
                    added_by_id=userdata(request)
                )

                creditreserve = Transactions.objects.create(
                    reporting_amount=oldbal,
                    narration="Closing The load due to topup",
                    account_cr_id=reserve.id,
                    account_dr_id=laonacct.id,
                    tx_date=ddate,
                    loan_id=loanid,
                    transaction_type="give loan",
                    branch=business_context['branch'],
                    added_by_id=userdata(request)
                )

            actualchargesx = submit.charges

            # gen_cgs = ApplicationAccountOrLoanType.objects.filter(loan_type=submitted.first().loan_type,
            #                                                       general_charge__application='r')
            # print(gen_cgs)
            actualcharges = json.loads(actualchargesx)

            for chg in actualcharges:
                chargeamount = chg["amount"]
                chargename = chg["name"]
                narration = chargename
                chargedebit = chargesaccount

                # try:
                savecharge = Transactions.objects.create(
                    reporting_amount=chargeamount,
                    transaction_type="Loan charge",
                    narration=narration,
                    account_cr_id=chargeacct.id,
                    account_dr_id=chargedebit,
                    tx_date=ddate,
                    receipt=rec_no,
                    reference=voucher,
                    loan_id=loanid,
                    branch=business_context['branch'],
                    added_by_id=userdata(request)
                )

            notify_type = NotiSettings.objects.filter(
                business=biz_data(self.request.user)
            ).first()
            guarantors = Loanguarantors.objects.filter(loan=loanid)
            # print(guarantors.values())
            template_file_link = "sacco/emails/transactional.html"
            # bizz_id = biz_id(self.request.user)
            this_biz = business_context['branch'].business
            # get guarantors
            de_loan = (
                Loans.objects.filter(id=loanid).prefetch_related("guarantors").first()
            )
            gs = (
                de_loan.guarantors.annotate(name=F("biodata__name"))
                .annotate(contact=F("biodata__contact"))
                .annotate(email=F("biodata__email"))
                .values("name", "email", "contact")
            )
            # print(gs)
            for guarantorxx in gs:
                html_body = f"<p>Dear {guarantorxx['name']},</p> <p>This is to inform you that the loan requested by {de_loan.applicant.biodata.name} of {de_loan.amount_approved} has been disbursed. <br>We hope that you have a good experience with our services and <b>{this_biz.name}</b> will do it’s best to please you. We will be looked forward to serving you in adherence to our quality standards.</p><p>Thanking you and with profound regards.</p>"
                message = f"Dear {0},\nThis is to inform you that the loan requested by {de_loan.applicant.biodata.name} of {de_loan.amount_approved} has been disbursed"

                if guarantorxx["contact"]:
                    if notify_type.notification_type == 1:  # Send by SMS only
                        checkAndSendMessage(
                            self.request.user,
                            message,
                            guarantorxx["contact"],
                            guarantorxx["name"],
                        )

                elif notify_type.notification_type == 2:  # Send by Email only
                    if guarantorxx["email"]:
                        sendTransEmail(
                            template_file_link,
                            html_body,
                            this_biz.name,
                            "Account Opening Successful",
                            guarantorxx["email"],
                        )

                elif notify_type.notification_type == 3:  # Send by both SMS and Email
                    if guarantorxx["contact"]:
                        checkAndSendMessage(
                            self.request.user,
                            message,
                            guarantorxx["contact"],
                            guarantorxx["name"],
                        )

                    if guarantorxx["email"]:
                        sendTransEmail(
                            template_file_link,
                            html_body,
                            this_biz.name,
                            "Account Opening Successful",
                            guarantorxx["email"],
                        )

            messages.success(
                request,
                "success",
                extra_tags=f"Loan amount of {submit.amount_approved} has been disbursed successfully",
            )

            # =====================================================DISBURSE LOAN===================================
        if form_type == "repay":
            loanid = request.POST["loanid"]
            amount = request.POST["amount"]
            narrative = request.POST["narration"]
            amount = float(amount.replace(",", ""))
            ddate = request.POST["d_date"]
            debited_account = request.POST["debited"]
            memberledger = Account.objects.filter(member_id=loandetail.account_id)[0]

            if debited_account == "m":
                debited = memberledger.id
            else:
                debited = debited_account

            fineamount = request.POST["fineamount"]
            fineamount = float(fineamount.replace(",", ""))
            business = business_context['branch'].business
            user = User.objects.filter(staff_id=userdata(request)).first()
            applicant = AccountBroker.objects.filter(
                the_account=loandetail.account_id
            ).first()
            interestamount = request.POST["interestamount"]
            interestamount = float(interestamount.replace(",", ""))
            principalamount = amount - float(interestamount) - float(fineamount)
            principalbalance = float(request.POST["principalbalance"])
            credited = Account.objects.filter(
                name="Loan Receivables", business=business_context['branch'].business
            ).first()

            if principalamount > principalbalance:
                principaldifference = principalamount - principalbalance
                interestamount = interestamount + principaldifference
                principalamount = principalbalance

            if loandetail.repayment_method == 2 or debited_account == "m":
                try:
                    # ================CHECK LOAN BALANCE=================================
                    currentbalance = float(applicant.the_account.balance)

                    newbalance = currentbalance - amount
                    if currentbalance >= amount:
                        applicant.the_account.balance = newbalance
                        applicant.the_account.save()
                    else:
                        messages.error(
                            request,
                            "Insufficient balance",
                            extra_tags=f"Member has insufficient funds to pay {amount} as loan repayment",
                        )
                        return HttpResponseRedirect(request.META.get("HTTP_REFERER"))

                except Exception as bal:
                    traceback.print_exc()
                    messages.error(
                        request,
                        "Something went wrong",
                        extra_tags=f"The system could not process the loan repayment of {amount}",
                    )
                    return HttpResponseRedirect(
                        reverse_lazy("loan_details", args=loanid)
                    )

            receipt = datetime.datetime.now().strftime("%y%m%d%H%M%S")

            # ==========================Save loan transaction=======================
            # Check if the ledgers exist=====================
            checkrecievables = Account.objects.filter(
                name="Loan Receivables", business=business
            ).exists()
            checkinterest = Account.objects.filter(
                name="Loan interests", business=business
            ).exists()
            checkfines = Account.objects.filter(
                name="Loan Fines", business=business
            ).exists()
            checkinterestrecievables = Account.objects.filter(
                name="Loan interest receivables", business=business
            ).exists()

            rcategeory = AccountCategory.objects.filter(
                name="Account Receivables", business=business
            ).first()

            if checkfines:
                finesaccount = Account.objects.filter(
                    name="Loan Fines", business=business
                ).first()
            else:
                finescat = AccountCategory.objects.filter(
                    name="Loan incomes", business=business
                ).first()
                finesaccount = Account.objects.create(
                    name="Loan Fines",
                    business=business,
                    category_id=finescat.id,
                    added_by_id=userdata(request),
                )

            if float(fineamount) > 0:
                # if the custom fine is greater than 0 then pay that
                this_loan_fines = LoanFines.objects.filter(loan_id=loanid)
                query_fines = this_loan_fines.aggregate(Sum("amount"))
                total_fines = query_fines["amount__sum"] or 0
                # print('total_fines', total_fines)
                query_fines_paid = LoanFinePayments.objects.filter(
                    loan_id=loanid
                ).aggregate(Sum("amount"))
                total_fine_paid = query_fines_paid["amount__sum"] or 0
                # print('total_fine_paid', total_fine_paid)
                unpaid_fines = total_fines - total_fine_paid

                if unpaid_fines > 0:
                    LoanFinePayments.objects.create(
                        loan_id=loanid,
                        amount=float(fineamount),
                        added_by_id=userdata(request),
                    )
                    fines_transaction = Transactions.objects.create(
                        reporting_amount=fineamount,
                        narration=f"payment of {fineamount} as Loan Fines",
                        account_dr_id=debited,
                        account_cr_id=finesaccount.id,
                        loan_id=loanid,
                        tx_date=ddate,
                        receipt=receipt,
                        transaction_type="Loan Fines",
                        added_by_id=userdata(request),
                        branch=business_context['branch'],
                    )
                # else just create a the fines transaction
                else:
                    fines_transaction = Transactions.objects.create(
                        reporting_amount=fineamount,
                        narration=f"payment of {fineamount} as Loan Fines",
                        account_dr_id=debited,
                        account_cr_id=finesaccount.id,
                        loan_id=loanid,
                        tx_date=ddate,
                        receipt=receipt,
                        transaction_type="Loan Fines",
                        added_by_id=userdata(request),
                        branch=business_context['branch'],
                    )

            if checkinterest:
                pass
            else:
                rcategeory2 = AccountCategory.objects.filter(
                    name="Loan incomes", business=business
                ).first()
                createleger2 = Account.objects.create(
                    name="Loan interests",
                    business=business,
                    category_id=rcategeory2.id,
                    added_by_id=userdata(request),
                )

            if checkrecievables:
                pass
            else:
                Account.objects.create(
                    name="Loan Receivables",
                    business=business,
                    category_id=rcategeory.id,
                    added_by_id=userdata(request),
                )

            if checkinterestrecievables:
                pass
            else:
                createintrestaccount = Account.objects.create(
                    name="Loan interest receivables",
                    business=business_context['branch'].business,
                    category_id=rcategeory.id,
                    added_by_id=userdata(request),
                )

            intrestaccount = Account.objects.filter(
                name="Loan interest receivables", business=business
            ).first()

            repayment_trans = Transactions.objects.create(
                reporting_amount=principalamount,
                narration=f"Loan payment of {principalamount} as principal"
                if not narrative
                else narrative,
                account_dr_id=debited,
                account_cr_id=credited.id,
                loan_id=loanid,
                tx_date=ddate,
                receipt=receipt,
                transaction_type="Loan repayment",
                added_by_id=userdata(request),
                branch=business_context['branch'],
            )

            # ============ RECORD INTEREST =================
            if float(interestamount) > 0:
                interest_trans = Transactions.objects.create(
                    reporting_amount=interestamount,
                    narration=f"payment of {interestamount} as Loan interest",
                    account_dr_id=debited,
                    account_cr_id=intrestaccount.id,
                    loan_id=loanid,
                    tx_date=ddate,
                    receipt=receipt,
                    transaction_type="Loan interest",
                    added_by_id=userdata(request),
                    branch=business_context['branch'],
                )

                business_data = businessdata(request)
                ln_interest_rec_acc = Account.objects.filter(
                    name="loan interest receivables", business=business_context['branch'].business
                )
                if ln_interest_rec_acc:
                    ln_interest_rec_acc = ln_interest_rec_acc.first()
                else:
                    # print("Loan interest acc")
                    ln_interest_rec_acc = Account.objects.create(
                        name="loan interest receivables",
                        business=business_context['branch'].business,
                        display_name="loan interest receivables",
                        category=AccountCategory.objects.get(
                            name="Account Receivables", business=business_context['branch'].business
                        ),
                    )

                ln_interest_revenue_acc = Account.objects.filter(
                    name="Loan Interest", business=business_context['branch'].business
                ).first()

                Exp_interest = Transactions.objects.create(
                    reporting_amount=interestamount,
                    narration=f"Expected interest of {interestamount}",
                    account_dr=ln_interest_rec_acc,
                    account_cr=ln_interest_revenue_acc,
                    loan_id=loanid,
                    tx_date=ddate,
                    transaction_type="Expected interest",
                    added_by_id=userdata(request),
                    branch=business_context['branch'],
                )
                # print('Exp_interest', Exp_interest)

            # ========= UPDATE LOAN BALANCE ================================
            oldbalance = loandetail.balance
            newbalance_afterpayment = oldbalance - amount

            if newbalance_afterpayment < 0:
                newbalance_afterpayment = 0

            loandetail.balance = newbalance_afterpayment
            loandetail.save()

            messages.success(
                request,
                "success",
                extra_tags=f"Loan payment recorded successfully. Principal of {principalamount}, Interest of {interestamount} and Fines of {fineamount}",
            )
            payer_account = Account.objects.filter(member=loandetail.account).first()
            # ==== SEND NOTIFICATIONS ===============
            template_file_link = "sacco/emails/transactional.html"
            currency = CurrencySetting.objects.filter(
                business=business_context['branch'].business
            ).first()
            notify_type = NotiSettings.objects.filter(
                business=business_context['branch'].business
            ).first()
            smsSettingsObj = SaccoSmsSettings.objects.get(
                when_to_send="On loan payment", business=business_context['branch'].business
            )
            email = applicant.members.biodata.email
            mail_receiver = applicant.members.biodata.name
            message = f"From: {business.short_name}\n Dear customer, a payment of {currency.currency} {amount} to service your loan has been processed successfully. Principal Paid {currency.currency} {principalamount}. Interest Paid {interestamount}. TXN ID {repayment_trans.txno}"
            html_body = f"<p>Dear {mail_receiver},</p>Your loan repayment request with the following details has been processed successfully.<br/>Transaction Number: {repayment_trans.txno}<br/>Amount: {currency.currency}{amount} <br/>Description: {repayment_trans.narration} <br/>Principal Amount: {principalamount} <br/>Interest Amount: {interestamount} <br/>Date: {repayment_trans.tx_date} <p>Please call our contact centre on +{business.contact} in case you require further details.</p>"
            if smsSettingsObj.status:
                if notify_type.notification_type == 1:  # Send by SMS only
                    if applicant.members.biodata.contact:
                        if loandetail:
                            checkAndSendMessage(
                                user,
                                message,
                                applicant.members.biodata.contact,
                                applicant.members.biodata.name,
                                account=payer_account,
                            )
                elif notify_type.notification_type == 2:  # Send by Email only
                    if applicant.members.biodata.email:
                        sendTransEmail(
                            template_file_link,
                            html_body,
                            business.name,
                            "Successful Loan repayment",
                            email,
                        )

                elif notify_type.notification_type == 3:  # Send by both SMS and Email
                    if applicant.members.biodata.contact:
                        if loandetail:
                            checkAndSendMessage(
                                user,
                                message,
                                applicant.members.biodata.contact,
                                applicant.members.biodata.name,
                                account=payer_account,
                            )

                    if applicant.members.biodata.email:
                        sendTransEmail(
                            template_file_link,
                            html_body,
                            business.name,
                            "Successful Loan repayment",
                            email,
                        )

            return HttpResponseRedirect(
                f"/loans/loan-receipt/{repayment_trans.id}/?fine={str(fineamount)}"
            )

        # ============ COMMITTEE ACTION ===============
        if form_type == "topup":
            loanid = request.POST["loanid"]
            account = request.POST["account"]
            amount = request.POST["amount"]
            applicant = AccountBroker.objects.filter(the_account_id=account).first()
            amount = amount.replace(",", "")
            adate = request.POST["d_date"]
            interest = request.POST["interest"]
            duration = request.POST["duration"]
            rate_type = request.POST["rate_type"]
            payment_method = request.POST["payment_method"]
            loantype = request.POST["loan_type"]
            loan_type = LoanTypes.objects.filter(id=loantype)[0]

            #         ============SAVE LOAN APPLICATION INTO THE DATABASE======================
            loanapply = Loans.objects.create(
                account_id=account,
                branch=business_context['branch'],
                amount_requested=amount,
                rate=interest,
                rate_type=rate_type,
                requested_on=adate,
                applicant=applicant.members,
                loan_type_id=loan_type.id,
                interval=loan_type.interval,
                requested_duration=duration,
                repayment_method=payment_method,
                sub_intervals=loan_type.sub_intervals,
                is_topup=True,
                added_by_id=userdata(request),
            )

            LoanTopups.objects.create(
                closed_loan_id=loanid,
                topup_loan_id=loanapply.id,
                branch=business_context['branch'],
            )

            messages.success(
                request,
                "success",
                extra_tags=f"Loan top up application of {loanapply.amount_requested} submitted successfully",
            )
            return HttpResponseRedirect(reverse_lazy("loan_applications"))

            # ============ COMMITTEE ACTION ===============
        if form_type == "deleterepayment":
            loanid = request.POST["loanid"]
            receipt = request.POST["receipt"]
            tx_date = request.POST["tdate"]
            amount = request.POST["amount"]
            reason = request.POST["reason"]
            try:
                # remove the expected interest transaction
                interest_trans = Transactions.objects.filter(loan_id=loanid, receipt=receipt,
                                                             transaction_type='Loan interest')
                if interest_trans.exists():
                    for the_interest in interest_trans:
                        last_exp_interest = Transactions.objects.filter(loan_id=loanid,
                                                                        transaction_type='Expected interest',
                                                                        reporting_amount=the_interest.reporting_amount).order_by(
                            '-id')
                        if last_exp_interest.exists():
                            last_exp_interest = last_exp_interest.first()
                            Transactions.objects.filter(id=last_exp_interest.id).delete()
                trans = Transactions.objects.filter(loan_id=loanid, receipt=receipt)
                for tr in trans:
                    tr.delete()

                Loans.objects.filter(id=loanid).update(loan_status=3)

                message = f"{request.user.staff.biodata.name} deleted a loan repayment transaction  of {amount} Made on {tx_date} Amount {amount}]. Reason being {reason}"
                ActivityLog.objects.create(
                    actor=request.user,
                    title=request.user.staff.biodata.name,
                    action_type="DELETE",
                    remarks=message,
                    branch=business_context['branch'].id,
                )
            except Exception as ext:
                print("Error is ", str(ext))

            messages.success(
                request,
                "success",
                extra_tags=f"Loan repayment transaction deleted successfully deleted",
            )

            return HttpResponseRedirect(request.META.get("HTTP_REFERER"))

        if form_type == "com_action":
            loanid = request.POST["loanid"]
            comaction = request.POST["comaction"]

            actions = Loans.objects.filter(id=loanid)

            for caction in actions:
                if comaction == "1":
                    # ======================SEND TO THE VIEW CALCULATED CHARGES======================================

                    # =========================IF THE LOAN HAS A RUNNING LOAN TO TOPUp=========================
                    if caction.is_topup == True:
                        oldloan = Loans.objects.filter(
                            account_id=caction.account_id, loan_status=3
                        )[0]
                        oldloan.loan_status = 8
                        oldloan.save()

                        #     =================SAVE IN THE TOPUPS TABLE========================

                    charges = ApplicationAccountOrLoanType.objects.filter(
                        loan_type_id=caction.loan_type,
                        general_charge__status=1,
                        general_charge__application="l",
                    )
                    # ======================SEND TO THE VIEW CALCULATED CHARGES======================================
                    actualcharges = []
                    totalcharges = 00

                    disbusable = caction.amount_approved - totalcharges

                    amounts_approved = request.POST["a_approved"]

                    approved_rate = request.POST["approved_rate"]
                    approved_date = request.POST["approved_date"]
                    approved_duration = request.POST["approved_duration"]

                    for charge in charges:
                        ispercentage = charge.general_charge.is_percentage
                        if ispercentage == 1:
                            chargeamount = (
                                                   float(charge.general_charge.amount)
                                                   * float(amounts_approved)
                                           ) / 100

                        else:
                            chargeamount = charge.general_charge.amount

                        actualcharges.append(
                            {
                                "name": charge.general_charge.charge,
                                "amount": chargeamount,
                            }
                        )

                        totalcharges = totalcharges + chargeamount
                    caction.loan_status = 2
                    caction.charges = json.dumps(actualcharges)
                    caction.approved_duration = approved_duration
                    caction.rate = approved_rate
                    caction.amount_approved = amounts_approved
                    caction.approved_on = approved_date

                    messages.success(
                        request,
                        "success",
                        extra_tags=f"Loan application of {amounts_approved} has been approved successfully",
                    )

                # ===================== APPLICATION IS REJECTED =====================
                if comaction == "2":
                    caction.loan_status = 5
                    caction.save()

                    messages.success(
                        request,
                        "success",
                        extra_tags=f"Loan application of {actions.amount_requested} rejected successfully because the applicant failed to meet the minimum requirements",
                    )
                    return HttpResponseRedirect(reverse_lazy("loan_applications"))

                    # ===================== APPPLICATION IS BOUNCED=====================
                if comaction == "3":
                    caction.loan_status = 0

                    messages.success(
                        request,
                        "success",
                        extra_tags=f"Loan application of {actions.amount_requested} recalled for a fresh appraisal",
                    )
                    return HttpResponseRedirect(reverse_lazy("loan_applications"))

                caction.save()

        if form_type == "adj_del":
            tid = request.POST["del_adj"]
            transacton = Transactions.objects.filter(id=tid)
            for da in transacton:
                da.delete()

        return HttpResponseRedirect(request.META.get("HTTP_REFERER"))


@method_decorator(login_required, name="dispatch")
class UpdateLoanOfficer(View):
    def post(self, request, pk, *args, **kwargs):
        business_context = get_business_loans_context_data(request)
        if business_context['branch'] is None:
            messages.error(
                request, "error",
                extra_tags="Action not allowed in the central view. Please logout and login into the respective branch"
            )
        return HttpResponseRedirect(request.META.get("HTTP_REFERER"))
        new_loan_officer = request.POST["new_loan_officer"]
        the_staff = Staff.objects.filter(id=new_loan_officer)
        if the_staff.exists():
            # check if disabled
            if not the_staff.first().is_active:
                messages.error(
                    request,
                    "error",
                    extra_tags=f"Officer nolonger available",
                )
                return HttpResponseRedirect(request.META.get("HTTP_REFERER"))
            # upadte the loan officer
            Loans.objects.filter(id=pk).update(
                officer_id=new_loan_officer
            )
            messages.success(
                request,
                "success",
                extra_tags=f"Officer changed successfully",
            )
            return HttpResponseRedirect(request.META.get("HTTP_REFERER"))
        else:
            messages.error(
                request,
                "error",
                extra_tags=f"Officer nolonger available",
            )
            return HttpResponseRedirect(request.META.get("HTTP_REFERER"))


# ================ PRINT RECEIPT =======================
@method_decorator(login_required, name="dispatch")
class LoanReceipt(View):
    template_name = "loans/loan_receipt.html"

    def get(self, request, rec, *args, **kwargs):
        business_context = get_business_loans_context_data(request)
        title = "Loan Repayment Receipt"
        fineamount = request.GET.get("fine")
        currency = CurrencySetting.objects.filter(
            business=business_context['branch'].business
        ).first()
        the_business_id = business_context['branch'].business.id
        the_branch_id = business_context['branch'].id
        # loan_paid = Transactions.objects.filter(id=rec).first()
        loan_paid = get_object_or_404(Transactions, id=rec)
        loan_id = loan_paid.loan_id
        receipt = loan_paid.receipt
        payment_details = Transactions.objects.filter(receipt=receipt).exclude(transaction_type='Expected interest')
        detail = Loans.objects.raw(
            "SELECT l.*, b.the_account_id as accountid, b.members_id as memberid, a.acc_number, m.biodata_id, m.id, c.name AS member_name, year(l.approved_on) as yapp, month(l.approved_on) as appmonth, day(l.approved_on) as appday FROM account_broker b, member_accounts a, sacco_member m, biodata c, loans l where b.the_account_id = a.id and b.business_id = %s and b.members_id = m.id and m.biodata_id = c.id and l.account_id = a.id and l.branch_id = %s and l.id=%s"
            % (the_business_id, the_branch_id, loan_id)
        )[0]
        loan_detail = Transactions.objects.filter(
            receipt=receipt, loan_id=loan_id
        ).exclude(transaction_type='Expected interest').aggregate(Sum("reporting_amount"))
        total_paid = loan_detail["reporting_amount__sum"]

        this_loan_fines = LoanFines.objects.filter(loan_id=loan_id)
        query_fines = this_loan_fines.aggregate(Sum("amount"))
        total_fines = query_fines["amount__sum"] or 0
        # print('total_fines', total_fines)
        query_fines_paid = LoanFinePayments.objects.filter(loan_id=loan_id).aggregate(
            Sum("amount")
        )
        total_fine_paid = query_fines_paid["amount__sum"] or 0
        # print('total_fine_paid', total_fine_paid)
        # unpaid_fines = total_fines - total_fine_paid
        # print("GENERAL DETAIL +++++", detail.balance)
        # actual_loan_balance = detail.balance + total_fine_paid
        actual_loan_balance = detail.balance + float(fineamount)

        return render(request, self.template_name, locals())


# ===================CHECK IF IT ALREADY EXISTS=============================
class CheckType(View):
    def get(self, request, *args, **kwargs):
        business_context = get_business_loans_context_data(request)
        loantype = request.GET["type"]
        checktype = LoanTypes.objects.filter(
            name=loantype, business=business_context['branch'].business
        ).exists()
        if checktype:
            return HttpResponse(1)
        else:
            return HttpResponse(2)


# ================= LON TYPE DETAILS =========================
class Getdetail(View):
    def get(self, request, loantype, *args, **kwargs):
        # loantype = type
        checktype = LoanTypes.objects.filter(id=loantype)[0]
        value1 = checktype.interest_rate
        value2 = checktype.get_interval_display()
        value3 = checktype.interval
        value4 = checktype.maximum_period
        formulae = checktype.get_period

        return HttpResponse("%s|%s|%s|%s|%s" % (value1, value2, value3, value4, formulae))


# ======================LOAN CALCULATOR============================
@method_decorator(login_required, name="dispatch")
class LoanCalculator(View):
    template_name = "loans/loan_calculator.html"

    def get(self, request, *args, **kwargs):
        title = "Loan calculator"
        loantypes = LoanTypes.objects.filter(
            business=businessdata(request), is_active=True
        )
        return render(request, self.template_name, locals())

    # =================RECEIVE FORM APPLICATION DATA==============================
    def post(self, request, *args, **kwargs):
        loan_type = request.POST["loan_type"]

        loan_typedetails = LoanTypes.objects.filter(id=loan_type)[0]

        formula = loan_typedetails.formula
        title = "Loan calculator"
        loantypes = LoanTypes.objects.filter(
            business=businessdata(request), is_active=True
        )

        amount = request.POST["amount"]

        principal = float(amount.replace(",", ""))
        appdate = request.POST["adate"]
        appdate = datetime.datetime.strptime(appdate, "%Y-%m-%d")
        interestrate = request.POST["interest"]
        interestrate = float(interestrate)
        duration = int(request.POST["duration"])
        rate_type = request.POST["rate_type"]
        typeinterval = request.POST["typeinterval"]
        typeinterval = int(typeinterval)
        # print(rate_type)

        try:
            editsetings = LoanSettings.objects.filter(business=businessdata(request))[0]
            decimals = editsetings.decimals
            nonemember = editsetings.allow_others
        except IndexError as e:
            decimals = 0
            nonemember = 0

        #         ============AMOTIZE THE LOAN TO ESTIMATE THE PAYMENTS======================

        paidamount = 0

        if formula == 5:
            interest = interestrate / 100

        if formula == 1:
            if typeinterval == 3:
                interest = (interestrate / 100) / 12

            if typeinterval == 0:
                interest = (interestrate / 100) / 365

            if typeinterval == 1:
                interest = (interestrate / 100) / 52

            if typeinterval == 2:
                interest = (interestrate / 100) / 26

            if typeinterval == 4:
                interest = interestrate / 100

        if formula == 2:
            if typeinterval == 3:
                interest = interestrate / 100

            if typeinterval == 0:
                interest = (interestrate / 100) / 30

            if typeinterval == 1:
                interest = (interestrate / 100) / 4

            if typeinterval == 2:
                interest = (interestrate / 100) / 2

            if typeinterval == 4:
                interest = (interestrate / 100) * 12

        if formula == 3:
            if typeinterval == 3:
                interest = (interestrate / 100) * 4

            if typeinterval == 0:
                interest = (interestrate / 100) * 7

            if typeinterval == 1:
                interest = interestrate / 100

            if typeinterval == 2:
                interest = (interestrate / 100) * 2

            if typeinterval == 4:
                interest = (interestrate / 100) * 52

        if formula == 4:
            if typeinterval == 3:
                interest = (interestrate / 100) * 30

            if typeinterval == 0:
                interest = interestrate / 100

            if typeinterval == 1:
                interest = (interestrate / 100) * 7

            if typeinterval == 2:
                interest = (interestrate / 100) * 14

            if typeinterval == 4:
                interest = (interestrate / 100) * 256

        # ===================GET LOAN REPAYMENTS========================================

        # =====================IF INTERVAL IS MONTHLY =========================
        if typeinterval == 3:
            to_add = relativedelta(months=1)
            all_duration = relativedelta(months=duration)

        # =====================IF INTERVAL IS DAILY =========================

        if typeinterval == 0:
            to_add = relativedelta(days=1)
            all_duration = relativedelta(days=duration)

        # =====================IF INTERVAL IS WEEKLY =========================

        if typeinterval == 1:
            to_add = relativedelta(weeks=1)
            all_duration = relativedelta(weeks=duration)

        # =====================IF INTERVAL IS BI-WEEKLY =========================

        if typeinterval == 2:
            to_add = relativedelta(weeks=2)
            all_duration = relativedelta(weeks=duration * 2)

        # =====================IF INTERVAL IS YEARLY =========================

        if typeinterval == 4:
            to_add = relativedelta(years=1)
            all_duration = relativedelta(years=duration)

        # ================================IF THE INTEREST IS FLAT RATE==========================

        if rate_type == "1":
            in_date = ""

            loan_dict = {}
            days = []
            schedules = round(principal / duration, decimals)
            interestperschedue = round(principal * interest, decimals)
            payable = interestperschedue + schedules
            payable = round(payable, decimals)

            # ========================================GET PAID SCHEDULES=========================

            paidschedules = round(paidamount, 0) / round(payable, 0)
            paidschedules = floor(paidschedules)

            # ====================================Balance for the next schedule=======================
            nextbal = paidamount - (payable * paidschedules)

            nextintallment = payable - nextbal

            nextschedule = paidschedules + 1
            data = []

            edate = appdate + to_add
            # print(appdate)
            date1 = edate
            totalpayable = payable * duration

            rbalance = totalpayable

            date2 = date1 + all_duration
            m = 1
            sch_date = date1

            # while date1 < date2:
            while m <= duration:
                rbalance = rbalance - payable
                rbalance = round(rbalance, decimals)
                if m == nextschedule:
                    isnext = 1
                    spaid = round(nextbal, decimals)
                    sbalance = round(nextintallment, decimals)

                elif m < nextschedule:
                    isnext = 0
                    spaid = round(payable, decimals)
                    sbalance = 0

                else:
                    isnext = 2
                    spaid = 0
                    sbalance = round(payable, decimals)

                days.append(
                    [date1.strftime("%Y-%m-%d"), isnext, spaid, sbalance, rbalance]
                )
                date1 = sch_date + to_add

                m = m + 1

        # =============================================IF THE RATE TYPE IS REDUCING=====================================

        if rate_type == "2":
            # print('IT IS TYPE 2')
            in_date = ""
            loan_dict = {}
            data = []
            days = []
            edate = appdate + to_add
            # print(appdate)
            date1 = edate
            principals = 0
            date2 = date1 + all_duration
            runningbal = principal
            schedules = 9
            # interestperschedue = schedules * interest / 100;

            numilator = (
                    principal
                    * interest
                    * (
                            (pow((1 + interest), duration))
                            / (pow((1 + interest), duration) - 1)
                    )
            )
            payable = round(numilator, decimals)

            totalpayable = payable * duration

            rprincipal = 0
            rbalance = principal
            # ========================================GET PAID SCHEDULES=========================

            paidschedules = round(paidamount, 0) / round(payable, 0)
            paidschedules = floor(paidschedules)

            nextschedule = paidschedules + 1
            # ====================================Balance for the next schedule=======================
            nextbal = floor(paidamount - (payable * paidschedules))
            nextintallment = payable - nextbal

            m = 1

            sch_date = date1

            # while date1 < date2:
            while m <= duration:
                days.append(date1.strftime("%Y-%m-%d"))
                # date1 = date1 + to_add
                date1 = sch_date + m * to_add

                interestperschedue = rbalance * interest
                interestperschedue = round(interestperschedue, decimals)
                rprincipal = payable - interestperschedue
                rprincipal = round(rprincipal, decimals)
                rbalance = principal * (
                        (pow((1 + interest), duration) - pow((1 + interest), m))
                        / (pow((1 + interest), duration) - 1)
                )

                rbalance = round(rbalance, decimals)

                loan_dict[in_date] = date1.strftime("%Y-%m-%d")
                loan_dict[rprincipal] = rprincipal
                loan_dict[rbalance] = rbalance
                loan_dict[interestperschedue] = interestperschedue

                data.append(
                    [
                        loan_dict[rprincipal],
                        loan_dict[in_date],
                        loan_dict[rbalance],
                        loan_dict[interestperschedue],
                    ]
                )
                m = m + 1

        return render(request, self.template_name, locals())


# =================== CHECK IF THE MEMBER ALREADY HAS A LOAN =============================
class ChekLoan(View):
    def get(self, request, account, *args, **kwargs):
        business_context = get_business_loans_context_data(request)
        allow_adding_extra_loan = 'False'
        if request.user.staff.position.permissions.filter(item_name='create_extra_loan_application').exists():
            allow_adding_extra_loan = 'True'

        try:
            find_loans = Loans.objects.filter(
                account_id=account, loan_status__lte=3
            ).first()
            found_loan_status = find_loans.loan_status
            found_loan_id = find_loans.id
        except:
            found_loan_status = 10
            found_loan_id = 0
        # check if member belongs to a group
        # get member
        theBroker = AccountBroker.objects.filter(the_account_id=account, business=business_context['branch'].business)
        the_member = None
        if theBroker.exists():
            the_member = theBroker.first().members
        in_group = 0
        if the_member is not None:
            the_group = GroupMember.objects.filter(member=the_member)
            if the_group.exists():
                the_group = the_group.first().group.name
                in_group = the_group

        # determine the maximum amt this person can get basing on shares
        total_shares_owned = theBroker.first().members.shares
        # base_loan_amt = 300000
        # if total_shares_owned < 1:
        #     max_amt_for_loan = base_loan_amt
        # else:
        #     max_amt_for_loan = float(total_shares_owned) * base_loan_amt

        base_loan_amt = 300000
        max_amt_for_loan = float(total_shares_owned) * base_loan_amt

        # check if member belongs to a group and see if there are any members in arears
        the_grp_mem = GroupMember.objects.filter(member=the_member.id)
        arrears_amt = 0
        result_str = ""
        if the_grp_mem.exists():
            # get all members of that group
            the_grp = the_grp_mem.first().group
            members_ids = list(GroupMember.objects.filter(group=the_grp).values_list('member_id', flat=True))
            # get all loans and update the balances
            all_group_loans = Loans.objects.filter(applicant_id__in=members_ids)
            for ln in all_group_loans:
                update_loan_summary_details(ln)
            # all_group_loans = Loans.objects.filter(applicant_id__in=members_ids)
            all_group_loans = list(Loans.objects.filter(applicant_id__in=members_ids, loan_status=3).order_by('id').exclude(
                loan_summary_details='').values_list('loan_summary_details', flat=True))
            # len > 0
            if len(all_group_loans) > 0:
                df = pd.DataFrame(eval(item) for item in all_group_loans)
                df = df[df['is_error'] == 'no']
                df['arrears_amt'] = df['principal_in_arrears'] + df['interest_in_arrears']

                arrear_loans_ids = df["id"].tolist()
                # get all account numbers for the loans
                all_acc_numbers = list(Loans.objects.filter(id__in=arrear_loans_ids).order_by('id').values_list('account__acc_number', flat=True))

                # Create string of "loan_number:arrears_amt"
                loans_str = ", ".join(
                    f"{row['loan_number']}({row['name']} - {all_acc_numbers[idx]}) : {round(row['arrears_amt'], 2)}"
                    for idx, row in df.iterrows()
                )

                # Total arrears
                total_arrears = round(df['arrears_amt'].sum(), 2)
                if total_arrears > 0:
                    # Combine into one string
                    result_str = f"{loans_str} == Total Arrears: {total_arrears}"
                else:
                    result_str = ""
            else:
                result_str = ""

        # print('result_str', result_str)

        # arrears_amt = 0
        # print('arrears_amt', total_arrears)

        return HttpResponse("%s|%s|%s|%s|%s|%s|%s" % (
        found_loan_status, found_loan_id, in_group, allow_adding_extra_loan, max_amt_for_loan, total_shares_owned, result_str))


class MemberLoansView(BusinessUserMixin, View):
    template_name = "loans/member_loans.html"

    def get_member(self, request):
        business_context = get_business_loans_context_data(request)
        pkey = self.kwargs.get("pk")
        branch = business_context['branch']
        member = get_object_or_404(
            Member, id=pkey, is_active=True, branch_id=branch)
        return member

    def get(self, request, pk, *args, **kwargs):
        member = self.get_member(request)
        if member.is_group is False:
            title = f"{member.biodata.name} Loans"
        else:
            title = f"{member.name} Loans"
        member_accounts = member.accounts.all()
        accts = []
        for account in member_accounts:
            accts.append(account.id)

        memberloans = Loans.objects.filter(account_id__in=accts)

        return render(request, self.template_name, locals())


# ================================= UPLAOD LOANS==========================================
@method_decorator(login_required, name="dispatch")
class LoanUploadsView(View):
    template_name = "loans/loan_uploads.html"

    def get(self, request, *args, **kwargs):
        business_context = get_business_loans_context_data(request)

        if business_context['branch'] is None:
            messages.error(
                request, "error",
                extra_tags="Action not allowed in the central view. Please logout and login into the respective branch"
            )
            return HttpResponseRedirect(request.META.get("HTTP_REFERER"))


        if business_context['the_current_branch_name'] == 'all branches':
            title = "Loan Uploads for all branches"
        else:
            the_current_branch_name = business_context['the_current_branch_name']
            title = f"Loan Uploads for {the_current_branch_name}"
        # title = "Loan Uploads"

        bankaccounts = Account.objects.filter(
            Q(business=business_context['branch'].business),
            (
                    Q(category__name="Bank")
                    | Q(category__name="Mobile Money")
                    | Q(category__name="Cash")
                    | Q(category__name="Reserves")
            ),
        )
        uplaods = UploadLoans.objects.filter(branch=business_context['branch']).order_by(
            "-id"
        )
        loantypes = LoanTypes.objects.filter(
            business=business_context['branch'].business, is_active=True
        )

        return render(request, self.template_name, locals())

    # ====WHEN THE POST IS MADE==========================

    def post(self, request, *args, **kwargs):
        business_context = get_business_loans_context_data(request)
        formtype = request.POST["form_type"]
        if formtype == "delete":
            pk = request.POST["uploadid"]
            upload = UploadLoans.objects.filter(id=pk)[0]
            # if upload.loan_type is None:
            #     # Loan topup uploads
            #     all_loans = Loans.objects.filter(upload=upload)
            #     for ln in all_loans:
            #         # Sort the old loan
            #         reverse_old_loan_and_transactions(ln)
            #         reverse_new_loan_and_transactions(ln)
            # delete the upload
            upload.delete()

            if upload.loan_type is None:
                messages.success(
                    request, "success", extra_tags=f"loans deleted successfully"
                )
            else:
                messages.success(
                    request,
                    "success",
                    extra_tags=f"{upload.total_records} loans of {upload.loan_type.name} deleted successfully",
                )

        if formtype == "upload":
            upload_file = request.FILES["proof"]
            udatea = request.POST["udate"]

            udate = datetime.datetime.strptime(udatea, "%Y-%m-%d").date()
            # udate = datetime.strptime(udatea, '%Y-%m-%d').date()

            loantype = request.POST["loan_type"]
            loan_type = LoanTypes.objects.filter(id=loantype)[0]
            credited = request.POST["credited"]
            narration = request.POST["narration"]
            lstatus = request.POST["lstatus"]
            data = upload_file.read().decode("utf-8")
            rate_type = request.POST["rate_type"]
            charges_debitted_acc = request.POST["charges_debitted_acc"]

            failed_loans = []
            failed_loans_number = 0

            if "charge_application" in request.POST:
                charge_application = request.POST["charge_application"]
            else:
                charge_application = ""

            if "charge_application" in request.POST:
                apply_charges_from = request.POST["apply_charges_from"]
            else:
                apply_charges_from = ""

            n = 0

            # ============TRY TO GET THE BALANCE=========================

            rate_type = request.POST["rate_type"]
            typeinterval = loan_type.interval
            typeinterval = int(typeinterval)

            try:
                editsetings = LoanSettings.objects.filter(
                    business=business_context['branch'].business
                )[0]
                decimals = editsetings.decimals
                formula = editsetings.formula
            except:
                decimals = 0
                formula = 1

            #         ============AMOTIZE THE LOAN TO ESTIMATE THE PAYMENTS======================

            paidamount = 0

            # you can use the re library --> import re
            rows = re.split("\n", data)  # splits along new line

            nloans = 0
            # === RECORD THE UPLOAD=========================
            bupload = UploadLoans.objects.create(
                upload_date=udate,
                file=upload_file,
                loan_type_id=loantype,
                branch=business_context['branch'],
                added_by_id=userdata(request),
            )

            for index, row in enumerate(rows):
                n = n + 1

                cells = row.split(",")

                if n > 1:
                    try:
                        actualcharges = []
                        memberacct = cells[0].strip()
                        approvedon = parse(cells[1]).date()

                        # ===================ADD CHARGES TO THE LOAN============================================
                        if apply_charges_from == "system":
                            if int(lstatus) == 2:
                                charges = ApplicationAccountOrLoanType.objects.filter(
                                    loan_type=loantype,
                                    general_charge__status=1,
                                    general_charge__application="l",
                                )

                                for charge in charges:
                                    ispercentage = charge.general_charge.is_percentage
                                    if ispercentage == 1:
                                        chargeamount = (
                                                int(charge.general_charge.amount * amount)
                                                / 100
                                        )

                                    else:
                                        chargeamount = charge.general_charge.amount

                                    actualcharges.append(
                                        {
                                            "charge_id": charge.general_charge.id,
                                            "name": charge.general_charge.charge,
                                            "amount": chargeamount,
                                        }
                                    )

                        amount = cells[2]
                        amount = amount.replace(",", "")
                        amount = float(amount)
                        principal = amount
                        amt_requested = amount
                        # print(principal)
                        # ================= determine current principal ===============
                        # print('THE CELL 5', )

                        # print('charge_application',charge_application)
                        # print('apply_charges_from', apply_charges_from, 'lstatus', lstatus)

                        if lstatus == "2" and apply_charges_from == "system":
                            if charge_application == "added_to_principal":
                                for ch in actualcharges:
                                    principal = principal - float(ch["amount"])
                        elif lstatus == "2" and apply_charges_from == "excel":
                            if cells[5].strip() and cells[5] != "0":
                                charge_required = float(cells[5].replace(",", ""))
                                if charge_application == "added_to_principal":
                                    principal = principal - charge_required

                        # print('PRINCIPAL', principal)

                        # =================== end ==========================================

                        intrate = cells[3]
                        cpaid = True

                        try:
                            intrat = [
                                float(s) for s in re.findall(r"[0-9]+.[0-9]+", intrate)
                            ][0]
                            a = 1 + intrat
                        except:
                            intrat = int(re.findall(r"^\D*(\d+)", intrate)[0])
                            a = 2 + intrat

                        intrate = intrat
                        # print('intrate', intrate)

                        interestrate = intrate
                        duration = cells[4]
                        duration = [int(s) for s in re.findall(r"\b\d+\b", duration)][0]

                        if formula == 2:
                            interest = interestrate / 100

                        if formula == 1:
                            if typeinterval == 3:
                                interest = (interestrate / 100) / 12

                            if typeinterval == 0:
                                interest = (interestrate / 100) / 365

                            if typeinterval == 1:
                                # print('interestrate', interestrate)
                                interest = (interestrate / 100) / 52

                            if typeinterval == 2:
                                interest = (interestrate / 100) / 26

                            if typeinterval == 4:
                                interest = interestrate / 100

                        if int(rate_type) == 1:
                            interests = interest * principal
                            totalinterest = interests * duration
                            totalbalance = principal + totalinterest

                        else:
                            if interest:
                                numilator = (
                                        principal
                                        * interest
                                        * (
                                                (pow((1 + interest), duration))
                                                / (pow((1 + interest), duration) - 1)
                                        )
                                )
                            else:
                                numilator = principal
                            payable = round(numilator, decimals)
                            totalbalance = payable * duration

                        # getmemberid = MemberAccount.objects.filter(acc_number=memberacct,
                        #                                            account_type__business_id=businessdata(request))[0]
                        applicant = AccountBroker.objects.filter(
                            the_account__acc_number=memberacct,
                            business=business_context['branch'].business,
                        )

                        if not applicant.exists():
                            failed_loans.append({
                                'reason': 'Account number was not found',
                                'account number': memberacct
                            })
                            failed_loans_number += 1
                        else:
                            applicant = applicant.first()
                            if lstatus == "1":
                                use_loan_status = 0
                            else:
                                use_loan_status = 3

                            # check if the staff member exists
                            employee_number = cells[6]
                            check_staff = Biodata.objects.filter(business=business_context['branch'].business,
                                                                 employee_no=employee_number.strip())
                            if not check_staff.exists():
                                failed_loans.append({
                                    'reason': 'Provided employee number not found',
                                    'account number': memberacct
                                })
                                failed_loans_number += 1
                            else:
                                the_staff_officer = Staff.objects.filter(biodata__employee_no=employee_number.strip(),
                                                                         biodata__business=business_context['branch'].business)
                                recordloan = Loans.objects.create(
                                    account_id=applicant.the_account_id,
                                    applicant_id=applicant.members_id,
                                    amount_requested=amt_requested,
                                    amount_approved=amt_requested,
                                    rate=intrate,
                                    rate_type=rate_type,
                                    requested_on=approvedon,
                                    approved_on=approvedon,
                                    schedule_start=approvedon,
                                    requested_duration=duration,
                                    approved_duration=duration,
                                    loan_status=use_loan_status,
                                    branch=business_context['branch'],
                                    upload_id=bupload.id,
                                    charges_paid=cpaid,
                                    loan_type_id=loantype,
                                    sub_intervals=loan_type.sub_intervals,
                                    interval=loan_type.interval,
                                    # charges=json.dumps(
                                    #     actualcharges),
                                    balance=totalbalance,
                                    loan_number=get_new_loan_number(),
                                    officer=the_staff_officer.first(),
                                    added_by=request.user.staff
                                )

                                # update the summary details
                                the_actual_loan = Loans.objects.get(id=recordloan.id)
                                if the_actual_loan.loan_status == 3:
                                    update_loan_summary_details(the_actual_loan)

                                # ======= record the charges ==========
                                for charge in actualcharges:
                                    LoanCharges.objects.create(
                                        amount=float(charge["amount"]),
                                        charge_id=charge["charge_id"],
                                        loan=recordloan,
                                    )

                                # ========= =====
                                nloans = nloans + 1

                                actualchargesx = recordloan.charges
                                # print('THIS IS THE SAVE JASON %s' % actualchargesx)

                                #     RECORD TRANSACTION===================================

                                checkrecievables = Account.objects.filter(
                                    name="Loan Receivables", business=business_context['branch'].business
                                ).exists()
                                checkrecharges = Account.objects.filter(
                                    name="Loan Charges", business=business_context['branch'].business
                                ).exists()

                                if checkrecharges:
                                    pass
                                else:
                                    rcategeory2 = AccountCategory.objects.filter(
                                        name="Loan incomes", business=business_context['branch'].business
                                    )[0]
                                    checkrecharges = Account.objects.create(
                                        name="Loan Charges",
                                        business_id=business_context['branch'].business,
                                        category_id=rcategeory2.id,
                                        added_by_id=userdata(request),
                                    )
                                if checkrecievables:
                                    pass
                                else:
                                    rcategeory = AccountCategory.objects.filter(
                                        name="Account Receivables",
                                        business=business_context['branch'].business,
                                    )[0]
                                    createleger = Account.objects.create(
                                        name="Loan Receivables",
                                        business_id=business_context['branch'].business,
                                        category_id=rcategeory.id,
                                        added_by_id=userdata(request),
                                    )

                                # =====================GET LEDGER ACCOUNTS============================

                                # laonacct = Account.objects.filter(category__name='Account Receivables', business=businessdata(request))[0]
                                laonacct = Account.objects.filter(
                                    name="Loan Receivables", business=business_context['branch'].business
                                )[0]
                                chargeacct = Account.objects.filter(
                                    name="Loan Charges", business=business_context['branch'].business
                                )[0]

                                # print('THIS AT INSERTION brfore %s' % actualchargesx)
                                # actualcharges = json.loads(actualchargesx)

                                # only record charges and transactions if they are running loans
                                if lstatus == "2":
                                    if apply_charges_from == "system":
                                        for chg in actualcharges:
                                            chargeamount = float(chg["amount"])
                                            chargename = chg["name"]

                                            c_narration = chargename

                                            try:
                                                savecharge = Transactions.objects.create(
                                                    reporting_amount=chargeamount,
                                                    transaction_type=chargename,
                                                    narration=c_narration,
                                                    account_cr_id=chargeacct.id,
                                                    account_dr_id=charges_debitted_acc,
                                                    tx_date=udate,
                                                    loan_id=recordloan.id,
                                                    branch=business_context['branch'],
                                                    added_by=request.user.staff

                                                )
                                            except:
                                                pass
                                    else:
                                        if cells[5].strip() and cells[5] != "0":
                                            savecharge = Transactions.objects.create(
                                                reporting_amount=float(charge_required),
                                                transaction_type="loan charge",
                                                narration=f"loan charge",
                                                account_cr_id=chargeacct.id,
                                                # account_dr_id=laonacct.id,
                                                account_dr_id=charges_debitted_acc,
                                                tx_date=udate,
                                                loan_id=recordloan.id,
                                                branch=business_context['branch'],
                                                added_by=request.user.staff
                                            )

                                    saveloan = Transactions.objects.create(
                                        reporting_amount=principal,
                                        narration=narration,
                                        account_cr_id=credited,
                                        account_dr_id=laonacct.id,
                                        tx_date=udate,
                                        loan_id=recordloan.id,
                                        transaction_type="give loan",
                                        branch=business_context['branch'],
                                        added_by=request.user.staff
                                    )
                                # except Exception as exx:
                                #     print(str(exx))

                    except Exception as lex:
                        traceback.print_exc()
                        print("Not added :", memberacct)
                        if str(lex) != 'list index out of range':
                            failed_loans.append({
                                'reason': 'Please check the validity of fields provided in the row',
                                'account number': cells[0].strip(),
                                'actual_error': str(lex)
                            })
                            failed_loans_number += 1

                        print(str(lex))

                bupload.total_records = nloans
                bupload.failed_loans = json.dumps({
                    'failed_loans': failed_loans
                })
                bupload.total_failed_loans = failed_loans_number
                bupload.save()

            messages.success(
                request,
                "success",
                extra_tags=f"{bupload.total_records} loans of {bupload.loan_type.name} uploaded successfully",
            )

        # =================================EDITING LOAN TYPE===================================

        return HttpResponseRedirect(reverse_lazy("loan_uploads"))


@method_decorator(login_required, name="dispatch")
class LoanTopUploadsView(View):
    def post(self, request, *args, **kwargs):
        values_submitted = get_top_up_upload_submitted_values(request)
        business_context = get_business_loans_context_data(request)

        try:
            data = pd.read_excel(values_submitted["upload_file"])
            data = data.to_dict("records")
            # check if field are there
            if len(data) > 0:
                fields_missing = check_loan_top_up_fields_xlsx(
                    trim_lower_case_dictionary_keys(data[0]),
                    values_submitted["apply_charges_from"],
                )
                if len(fields_missing) > 0:
                    messages.error(
                        request,
                        "error",
                        extra_tags="Some required fields are missing. Please refer to sample template",
                    )
                    return HttpResponseRedirect(reverse_lazy("loan_uploads"))

                loans_added = []
                loans_failed = []
                # business_data = businessdata(request)
                business_data = business_context['branch'].business
                branch_data = business_context['branch']
                user_data = userdata(request)

                # create upload loan record
                loan_upload_obj = create_loan_upload_record(
                    values_submitted, len(data), branch_data, user_data
                )

                for excel_loan in data:
                    excel_loan = trim_lower_case_dictionary_keys(excel_loan)
                    principal_bal_status = get_loan_principal_balance(
                        excel_loan["loan number"], business_data
                    )
                    if principal_bal_status["status"] == "failed":
                        # print("WE HERE")
                        loans_failed.append(excel_loan["loan number"])
                    else:
                        try:
                            with transaction.atomic():
                                # print(
                                #     f'Interval:{str(principal_bal_status["loan"])} and formula: {principal_bal_status["loan"].loan_type.formula}'
                                # )
                                if principal_bal_status["loan"].rate_type == 2:
                                    amo = reducing_amortization(
                                        principal_bal_status["loan"].amount_approved,
                                        (principal_bal_status["loan"].rate / 100),
                                        principal_bal_status["loan"].approved_duration,
                                        loan=principal_bal_status["loan"],
                                        formula=principal_bal_status[
                                            "loan"
                                        ].loan_type.formula,
                                    )
                                else:
                                    amo = flat_amortization(
                                        principal_bal_status["loan"].amount_approved,
                                        (principal_bal_status["loan"].rate / 100),
                                        principal_bal_status["loan"].approved_duration,
                                        loan=principal_bal_status["loan"],
                                        formula=principal_bal_status[
                                            "loan"
                                        ].loan_type.formula,
                                    )

                                the_amo = pd.DataFrame(amo)
                                # print(the_amo)

                                total_interest = sum(item["Interest"] for item in amo)
                                # print('total_interest', total_interest)
                                interest_paid = (
                                    principal_bal_status["loan"]
                                    .loan_trans.filter(transaction_type="Loan interest")
                                    .aggregate(
                                        total=Coalesce(
                                            Sum("reporting_amount"),
                                            0.0,
                                            output_field=FloatField(),
                                        )
                                    )["total"]
                                )
                                # print('total_PAID', interest_paid)
                                interest_balance = total_interest - interest_paid
                                # interest_balance = float("{:.2f}".format(interest_balance))
                                # print('interEST_BALANCE', interest_paid)
                                # close_old
                                old_loan_amt_data = close_old(
                                    principal_bal_status,
                                    interest_balance,
                                    values_submitted,
                                    business_data,
                                    branch_data,
                                )
                                create_top_up_loan(
                                    old_loan_amt_data,
                                    excel_loan,
                                    branch_data,
                                    loan_upload_obj,
                                    user_data,
                                    values_submitted,
                                    business_data,
                                )
                                loans_added.append(excel_loan["loan number"])
                        except Exception as e:
                            traceback.print_exc(file=sys.stdout)
                            print("ERROR", e)

                            logger.error(str(e))

                            loans_failed.append(excel_loan["loan number"])
                UploadLoans.objects.filter(id=loan_upload_obj.id).update(
                    total_records=len(loans_added)
                )
                # if the field are not there

                if len(loans_added) == 0 and len(loans_failed) > 0:
                    messages.warning(
                        request,
                        "Warning",
                        extra_tags=f"{len(loans_added)} loans added successfully and {len(loans_failed)} loans failed to be added",
                    )
                else:
                    messages.success(
                        request,
                        "success",
                        extra_tags=f"{len(loans_added)} loans added successfully and {len(loans_failed)} loans failed to be added",
                    )
                return HttpResponseRedirect(reverse_lazy("loan_uploads"))
            else:
                # there is no data -- ADD MESSAGE
                messages.error(request, "error", extra_tags="Excel file is empty")
                return HttpResponseRedirect(reverse_lazy("loan_uploads"))

        except Exception as e:
            print("ERROR", e)
            logger.error(str(e))
            messages.error(
                request,
                "error",
                extra_tags="Excel upload failed please try again later",
            )
            return HttpResponseRedirect(reverse_lazy("loan_uploads"))


@method_decorator(login_required, name="dispatch")
class ResolveBouncedLoansView(View):
    def post(self, request, *args, **kwargs):
        values_submitted = get_bounced_loans_submitted_values(request)

        try:
            data = pd.read_excel(values_submitted["upload_file"])
            data = data.to_dict("records")
            # check if field are there
            if len(data) > 0:
                fields_missing = check_bounced_loans_fields_xlsx(
                    trim_lower_case_dictionary_keys(data[0])
                )
                if len(fields_missing) > 0:
                    messages.error(
                        request,
                        "error",
                        extra_tags="Some required fields are missing. Please refer to sample template",
                    )
                    return HttpResponseRedirect(reverse_lazy("running_loans"))

                loans_added = []
                loans_failed = []
                business_data = businessdata(request)
                business_data = Business.objects.get(id=business_data)
                branch_data = branchdata(request)
                branch_data = Branch.objects.get(id=branch_data)
                transactions_added = []

                for excel_loan in data:
                    excel_loan = trim_lower_case_dictionary_keys(excel_loan)
                    try:
                        with transaction.atomic():
                            loan_reversed = reverse_transactions_and_close_loan(
                                business_data, branch_data, excel_loan, values_submitted
                            )
                            if loan_reversed["status"] == "failed":
                                loans_failed.append(
                                    {
                                        "loan number": excel_loan["loan number"],
                                        "reason": loan_reversed["reason"],
                                    }
                                )
                            else:
                                transactions_added.append(loan_reversed["transaction"])
                                loans_added.append(excel_loan["loan number"])

                    except Exception as e:
                        logger.error(str(e))
                        # print('Error', e)
                        traceback.print_exc(file=sys.stdout)
                        loans_failed.append(
                            {
                                "loan number": excel_loan["loan number"],
                                "reason": "Loan number is wrong or contact administrators for assistance",
                            }
                        )

                BouncedLoansUpload.objects.create(
                    narration=values_submitted["narration"],
                    total_records_added=len(loans_added),
                    total_records_failed=len(loans_failed),
                    transactions_created=json.dumps(
                        {
                            "transactions_created": transactions_added,
                        }
                    ),
                    branch=branch_data,
                    file=values_submitted["upload_file"],
                    failed_loans=json.dumps({"failed_loans": loans_failed}),
                )

                if len(loans_added) == 0 and len(loans_failed) > 0:
                    messages.warning(
                        request,
                        "Warning",
                        extra_tags=f"{len(loans_added)} loans bounced successfully and {len(loans_failed)} loans failed to be bounced",
                    )
                else:
                    messages.success(
                        request,
                        "success",
                        extra_tags=f"{len(loans_added)} loans bounced successfully and {len(loans_failed)} loans failed to be bounced",
                    )
                return HttpResponseRedirect(reverse_lazy("running_loans"))
            else:
                # there is no data -- ADD MESSAGE
                messages.error(request, "error", extra_tags="Excel file is empty")
                return HttpResponseRedirect(reverse_lazy("running_loans"))

        except Exception as e:
            logger.error(str(e))
            # print('ERROR', e)
            messages.error(
                request,
                "error",
                extra_tags="Excel upload failed please try again later",
            )
            return HttpResponseRedirect(reverse_lazy("running_loans"))


# @method_decorator(login_required, name='dispatch')
# class MultipleLoanTopUploadsView(View):
#     def post(self, request, *args, **kwargs):
#         values_submitted = get_top_up_upload_submitted_values(request)
#
#         try:
#             data = pd.read_excel(values_submitted['upload_file'])
#             data = data.to_dict('records')
#             # check if field are there
#             if len(data) > 0:
#                 fields_missing = check_multiple_loan_top_up_fields_xlsx(trim_lower_case_dictionary_keys(data[0]), values_submitted['apply_charges_from'])
#                 if len(fields_missing) > 0:
#                     messages.error(request, 'error', extra_tags='Some required fields are missing. Please refer to sample template')
#                     return HttpResponseRedirect(reverse_lazy('loan_uploads'))
#
#                 loans_added = []
#                 loans_failed = []
#                 business_data = businessdata(request)
#                 branch_data=branchdata(request)
#                 user_data=userdata(request)
#
#                 #create upload loan record
#                 loan_upload_obj = create_loan_upload_record(values_submitted, len(data), branch_data, user_data)
#
#                 all_uploaded_loans = data
#
#                 while len(all_uploaded_loans) > 0:
#                     #get first loan
#                     first_loan_top_up = all_uploaded_loans[0]
#                     #get all similar loans and put in a list
#                     similar_loan_top_ups = [d for d in all_uploaded_loans if first_loan_top_up['initial loan number'] == d['initial loan number']]
#                     print('current NOT SORTED', similar_loan_top_ups)
#                     #organise them basing on date
#                     similar_loan_top_ups = sorted(similar_loan_top_ups, key=lambda x: x['date'])
#                     print('current WELL ARRANGED', similar_loan_top_ups)
#
#                     current_old_loan_number = similar_loan_top_ups[0]['initial loan number']
#                     original_old_loan_number = similar_loan_top_ups[0]['initial loan number']
#                     for excel_loan in similar_loan_top_ups:
#                         excel_loan['initial loan number'] = current_old_loan_number
#                         excel_loan['loan number'] = excel_loan.pop('initial loan number')
#                         excel_loan = trim_lower_case_dictionary_keys(excel_loan)
#                         print(excel_loan)
#                         principal_bal_status=get_loan_principal_balance(excel_loan['loan number'], business_data)
#                         print('principal_bal_status', principal_bal_status)
#                         if principal_bal_status['status'] == 'failed':
#                             loans_failed.append(excel_loan['loan number'])
#                         else:
#                             with transaction.atomic():
#                                 try:
#                                     #close_old
#                                     old_loan_amt_data = close_old(principal_bal_status, values_submitted, business_data, branch_data)
#                                     new_loan_created = create_top_up_loan_multiple(old_loan_amt_data, excel_loan, branch_data, loan_upload_obj, user_data, values_submitted, business_data)
#                                     current_old_loan_number = new_loan_created['new_loan_number']
#                                     loans_added.append(excel_loan['loan number'])
#                                 except Exception as e:
#                                     print('ERROR', e)
#                                     loans_failed.append(excel_loan['loan number'])
#
#                     #remove the loans from all loans
#                     all_uploaded_loans = [d for d in all_uploaded_loans if
#                                             original_old_loan_number != d['initial loan number']]
#                     print('LOANS COUNT', len(all_uploaded_loans))
#
#
#                 UploadLoans.objects.filter(id=loan_upload_obj.id).update(
#                     total_records=len(loans_added)
#                 )
#                 #if the field are not there
#
#
#                 if len(loans_added) == 0 and len(loans_failed) > 0:
#                     messages.warning(request, 'Warning',
#                                      extra_tags=f'{len(loans_added)} loans added successfully and {len(loans_failed)} loans failed to be added')
#                 else:
#                     messages.success(request, 'success',
#                                      extra_tags=f'{len(loans_added)} loans added successfully and {len(loans_failed)} loans failed to be added')
#                 return HttpResponseRedirect(reverse_lazy('loan_uploads'))
#             else:
#                 #there is no data -- ADD MESSAGE
#                 messages.error(request, 'error', extra_tags='Excel file is empty')
#                 return HttpResponseRedirect(reverse_lazy('loan_uploads'))
#
#         except Exception as e:
#             print('ERROR', e)
#             messages.error(request, 'error', extra_tags='Excel upload failed please try again later')


@method_decorator(login_required, name="dispatch")
class UploadDetails(View):
    template_name = "loans/upload_details.html"

    def get(self, request, pk, *args, **kwargs):
        business_context = get_business_loans_context_data(request)
        title = "Upload details "
        loanid = pk
        loansid = int(loanid)
        # detail = get_object_or_404(Loans, id=pk)
        # detail = Loans.objects.get(id=pk)
        nowdate = today.strftime("%Y-%m-%d")

        the_branch_id = business_context['branch'].id
        the_business_id = business_context['branch'].business.id

        loanApplications = Loans.objects.raw(
            "SELECT l. *, b.the_account_id as accountid, b.members_id as memberid, a.acc_number, m.biodata_id, m.id, c.name AS member_name FROM account_broker b, member_accounts a, sacco_member m, biodata c, loans l where b.the_account_id = a.id and b.business_id = %s and b.members_id = m.id and m.biodata_id = c.id and l.account_id = a.id and l.branch_id = %s and l.upload_id = %s"
            % (the_business_id, the_branch_id, pk)
        )
        # print(loanApplications)
        # return HttpResponseRedirect(reverse_lazy('upload_details'))
        return render(request, self.template_name, locals())


# ======================GET INTRESTS FOR LOANS
def getInterests(request, loanid, cr_interests, dr_interests, nowdate, mem_acct):
    title = "Loan Details "

    business_context = get_business_loans_context_data(request)
    the_branch_id = business_context['branch'].id
    the_business_id = business_context['branch'].business.id

    loansid = int(loanid)
    nowdate = today.strftime("%Y-%m-%d")

    principal_narration = "Loan repayment for " + mem_acct

    interest_narration = "Loan interest from " + mem_acct

    # this_business = Business.objects.filter(id=businessdata(request)).first()
    try:
        detail = Loans.objects.raw(
            "SELECT l.*, b.the_account_id as accountid, b.members_id as memberid, a.acc_number, m.biodata_id, m.id, c.name AS member_name, year(l.approved_on) as yapp, month(l.approved_on) as appmonth, day(l.approved_on) as appday FROM account_broker b, member_accounts a, sacco_member m, biodata c, loans l where b.the_account_id = a.id and b.business_id = %s and b.members_id = m.id and m.biodata_id = c.id and l.account_id = a.id and l.branch_id = %s and l.id=%s"
            % (the_business_id, the_branch_id, loanid)
        )[0]

        share_price = (
            OtherSettings.objects.filter(business_id=the_business_id)
            .first()
            .share_price
        )

        formula = detail.loan_type.formula
        approved_amount = detail.amount_approved
        try:
            memberledger = Account.objects.filter(member_id=detail.accountid)[0]
        except Exception as mb:
            #     =======================CREATE MEMBER LEDGER==================================
            lcategory = AccountCategory.objects.filter(
                name="Members", business_id=businessdata(request)
            )[0]
            creatledger = Account.objects.create(
                member_id=detail.accountid,
                category_id=lcategory.id,
                business_id=the_business_id,
                added_by_id=userdata(request),
            )

            memberledger = creatledger

        if detail.is_topup == True and int(detail.loan_status) < 3:
            title = "Loan topup application details"

        elif int(detail.loan_status) < 3:
            title = "Loan application details"

        try:
            paid = Transactions.objects.raw(
                "SELECT SUM(reporting_amount) as tpaid,id from transactions where (transaction_type='Loan repayment' OR transaction_type='Loan interest') AND loan_id= %s "
                % (loanid)
            )[0]
            paidamount = paid.tpaid

        except:
            paidamount = 0

        #    ========================CLEARED INTERESTS============================================

        interestspaid = Transactions.objects.filter(
            loan_id=loanid,
            transaction_type="Loan interest",
            branch__business_id=the_business_id,
        )
        intpaid = interestspaid.count()

        # except Exception as ex:
        #     intpaid = 0

        # ===========================================================================================
        loanr = Account.objects.filter(
            name="Loan receivables", business_id=the_business_id
        )[0]
        try:
            added_adjustment = Transactions.objects.filter(
                Q(account_dr=loanr.id, loan_id=loanid),
                Q(transaction_type="Principal Adjustment"),
            ).aggregate(adjustedup=Sum("reporting_amount"))
            adjustment_added = added_adjustment["adjustedup"]
            if adjustment_added is None:
                adjustment_added = 0

        except Exception as exp:
            print(str(exp))
            traceback.print_exc()
            adjustment_added = 0

        #     ========adjusted down =================
        try:
            subtracted_adjustment = Transactions.objects.filter(
                Q(account_cr=loanr.id, loan_id=loanid),
                Q(transaction_type="Principal Adjustment"),
            ).aggregate(adjusteddown=Sum("reporting_amount"))
            adjustment_down = subtracted_adjustment["adjusteddown"]
            if adjustment_down is None:
                adjustment_down = 0

        except Exception as exp:
            print(str(exp))
            adjustment_down = 0

        # print("UP IS ", adjustment_added, "DOWN IS ", adjustment_down)

        try:
            totaladjustments = Transactions.objects.filter(
                loan_id=loanid, transaction_type="Loan Adjustment"
            ).aggregate(adjustments=Sum("reporting_amount"))["adjustments"]

            adjustments = Transactions.objects.filter(
                loan_id=loanid, transaction_type="Loan Adjustment"
            )

            try:
                totaladjustments = totaladjustments + 0

            except:
                totaladjustments = 0

        except Exception as ex:
            print("ERROR IS ", str(ex))
            totaladjustments = 0

        # ===========================================================================================
        # print('ADJUSTMENT IS ', totaladjustments)
        try:
            loan_setings = LoanSettings.objects.filter(business=businessdata(request))[
                0
            ]
            decimals = loan_setings.decimals

        except Exception as gex:
            decimals = 0

        # ======================SEND TO THE VIEW CALCULATED CHARGES======================================

        totalcharges = 0

        if int(detail.loan_status) > 1:
            actualchargesx = detail.charges
            try:
                actualcharges = json.loads(actualchargesx)
                for chg in actualcharges:
                    chargeamount = chg["amount"]
                    totalcharges = totalcharges + chargeamount

            except:
                totalcharges = 0

        # ============ GET CHARGES IF THE CHARGES ARE STILL IN APPLICATION STAGE ====================

        if detail.is_topup == True:
            oldloan = LoanTopups.objects.filter(topup_loan_id=loanid).first()

            oldloanid = oldloan.closed_loan.id

            # =============================GET PRINCIPAL BALANCE=========================
            recevables_account = Account.objects.filter(
                name="Loan Receivables", business=the_business_id
            )[0]

            try:
                try:
                    #     ===========IF MEMBER HAS MADE SOME PAYMENT==================
                    if detail.loan_status > 2:
                        lcredits = Transactions.objects.filter(
                            Q(loan_id=oldloanid, account_cr=recevables_account.id),
                            ~Q(transaction_type="Closing Loan"),
                        ).aggregate(principlecredits=Sum("reporting_amount"))[
                            "principlecredits"
                        ]

                    else:
                        lcredits = Transactions.objects.filter(
                            loan_id=oldloanid, account_cr=recevables_account.id
                        ).aggregate(principlecredits=Sum("reporting_amount"))[
                            "principlecredits"
                        ]
                except:
                    lcredits = 0

                ldebits = Transactions.objects.filter(
                    loan_id=oldloanid, account_dr=recevables_account.id
                ).aggregate(principledebits=Sum("reporting_amount"))["principledebits"]

                if lcredits is None:
                    oldbalance = ldebits

                else:
                    oldbalance = float(ldebits) - float(lcredits)
                oldbalance = float(oldbalance)

            except Exception as ex:
                print("Error is ", str(ex))

                oldbalance = 0

            totalpold = Transactions.objects.filter(
                loan_id=oldloanid, transaction_type="Loan repayment"
            ).aggregate(totalpaidprincipalold=Sum("reporting_amount"))

            totalpaidprincipalold = totalpold["totalpaidprincipalold"]
            try:
                totalpaidprincipalold = float(totalpaidprincipalold)
            except:
                totalpaidprincipalold = 0

        if detail.loan_status > 2:
            if detail.charges_paid == False:
                if detail.is_topup == True:
                    disbusable = approved_amount - (oldbalance)
                else:
                    disbusable = approved_amount - totalcharges
            else:
                if detail.is_topup == True:
                    disbusable = approved_amount - (oldbalance)
                else:
                    disbusable = approved_amount

        # ==================GET INTEREST IN RELATION TO SETTINGS=====================
        if detail.interval is not None:
            typeinterval = detail.interval

        else:
            typeinterval = detail.loan_type.interval
            detail.interval = typeinterval
            detail.save()

        interest = (detail.rate / 100) / 12

        # get_query = Transactions.objects.filter(loan_id=loanid).values('receipt').distinct().annotate(
        #     paidamount=Sum("reporting_amount", default=0.0)).annotate(
        #     fine=Sum("reporting_amount", default=0.0)).annotate(test=Case(When
        #                                                               ))

        if detail.loan_status > 2:
            repayments = Transactions.objects.raw(
                "SELECT sum(reporting_amount) as paidamount, id, receipt FROM transactions where loan_id= %s and (transaction_type='Loan repayment' or transaction_type='Loan interest' ) GROUP BY receipt"
                % (loanid)
            )
        if detail.loan_status > 2:
            maccount = detail.accountid

            # ===================CHECK IF THE LOAN HAS A TOP UP APPLICATION ALREADY=========================

            try:
                paidamount = round(paidamount, decimals)

            except:
                paidamount = 0

            duration = detail.approved_duration
            appdate = detail.schedule_start

            loanacct = Account.objects.filter(
                name="Loan Receivables", business_id=the_business_id
            ).first()

            totaldebit = Transactions.objects.filter(
                loan_id=loanid, account_dr_id=loanacct.id
            ).aggregate(totalpdebit=Sum("reporting_amount"))
            principal = totaldebit["totalpdebit"]
            approved_amount = principal

            principal = float(principal)
            approved_amount = float(approved_amount)

            if formula == 5:
                interest = detail.rate / 100

            if formula == 1:
                if typeinterval == 3:
                    interest = (detail.rate / 100) / 12

                if typeinterval == 0:
                    interest = (detail.rate / 100) / 365

                if typeinterval == 1:
                    interest = (detail.rate / 100) / 52

                if typeinterval == 2:
                    interest = (detail.rate / 100) / 26

                if typeinterval == 4:
                    interest = detail.rate / 100

            if formula == 2:
                if typeinterval == 3:
                    interest = detail.rate / 100

                if typeinterval == 0:
                    interest = (detail.rate / 100) / 30

                if typeinterval == 1:
                    interest = (detail.rate / 100) / 4

                if typeinterval == 2:
                    interest = (detail.rate / 100) / 2

                if typeinterval == 4:
                    interest = (detail.rate / 100) * 12

            if formula == 3:
                if typeinterval == 3:
                    interest = (detail.rate / 100) * 4

                if typeinterval == 0:
                    interest = (detail.rate / 100) * 7

                if typeinterval == 1:
                    interest = detail.rate / 100

                if typeinterval == 2:
                    interest = (detail.rate / 100) * 2

                if typeinterval == 4:
                    interest = (detail.rate / 100) * 52

            if formula == 4:
                if typeinterval == 3:
                    interest = (detail.rate / 100) * 30

                if typeinterval == 0:
                    interest = detail.rate / 100

                if typeinterval == 1:
                    interest = (detail.rate / 100) * 7

                if typeinterval == 2:
                    interest = (detail.rate / 100) * 14

                if typeinterval == 4:
                    interest = (detail.rate / 100) * 365

            # ===================GET LOAN REPAYMENTS========================================

            # =====================IF INTERVAL IS MONTHLY =========================
            if typeinterval == 3:
                to_add = relativedelta(months=1)
                try:
                    all_duration = relativedelta(months=duration)
                except Exception as exp:
                    print(str(exp))

            # =====================IF INTERVAL IS DAILY =========================

            if typeinterval == 0:
                to_add = relativedelta(days=1)
                all_duration = relativedelta(days=duration)

            # =====================IF INTERVAL IS WEEKLY =========================

            if typeinterval == 1:
                to_add = relativedelta(weeks=1)
                all_duration = relativedelta(weeks=duration)

            # =====================IF INTERVAL IS BI-WEEKLY =========================

            if typeinterval == 2:
                to_add = relativedelta(weeks=2)
                all_duration = relativedelta(weeks=duration * 2)

            # =====================IF INTERVAL IS YEARLY =========================

            if typeinterval == 4:
                to_add = relativedelta(years=1)
                all_duration = relativedelta(years=duration)

            # ================================DETERMINING THE FINES AND ON WHICH SCHEDULES==========================
            nowdate = datetime.datetime.today().date()

            recevables_account = Account.objects.filter(
                name="Loan Receivables", business_id=the_business_id
            )[0]

            # =======================IF RATE TYPE IS FLAT============================
            cumrative_payments = 0
            if detail.rate_type == 1:
                days = []
                schedules = approved_amount / duration

                interestperschedue = approved_amount * interest
                payable = interestperschedue + schedules

                # ========================================GET PAID SCHEDULES=========================
                paidschedules = paidamount / payable
                paidschedules = floor(paidschedules)

                # ====================================Balance for the next schedule=======================
                nextbal = paidamount - (payable * paidschedules)
                nextintallment = payable - nextbal

                if 1 > nextintallment > 0:
                    nextintallment = payable + nextintallment
                    nextschedule = paidschedules + 2
                else:
                    nextschedule = paidschedules + 1

                edate = appdate + to_add

                date1 = edate
                totalpayable = payable * duration

                totalinterest = totalpayable - principal

                totalpayable = round(totalpayable, decimals)

                rbalance = totalpayable

                date2 = date1 + all_duration
                m = 1

                try:
                    principalbalance = principal

                    sch_date = date1
                    # while date1 < date2:
                    while m <= duration:
                        pbalance = 0

                        if intpaid == m or intpaid > m:
                            actualinterest = 0
                        else:
                            actualinterest = interestperschedue

                        rbalance = rbalance - payable
                        rbalance = rbalance
                        if m == nextschedule:
                            isnext = 1
                            spaid = nextbal
                            sbalance = nextintallment

                        elif m < nextschedule:
                            isnext = 0
                            spaid = payable
                            sbalance = 0

                        else:
                            isnext = 2
                            spaid = 0
                            sbalance = payable

                        principalbalance = principalbalance - schedules

                        interestperschedue = round(interestperschedue, decimals)

                        # !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
                        if date1 < nowdate + relativedelta(months=1):
                            reciept = date1.strftime("%Y%m%d") + str(loanid)
                            txdate = date1 - relativedelta(days=30)
                            # print("I am in Flat rate")
                            # ============== CHECK IF THE RECORD IS ALREADY SAVED==================
                            try:
                                check = Transactions.objects.filter(
                                    loan_id=loanid, receipt=reciept
                                )[0]
                            except:
                                interestperschedue = round(interestperschedue, 2)
                                Transactions.objects.create(
                                    reporting_amount=interestperschedue,
                                    narration=interest_narration,
                                    account_dr_id=dr_interests,
                                    account_cr_id=cr_interests,
                                    loan_id=loanid,
                                    tx_date=txdate,
                                    receipt=reciept,
                                    transaction_type="interest on loan",
                                    added_by_id=userdata(request),
                                    branch_id=the_branch_id
                                )

                        if date1.weekday() == 5:
                            date1 = date1 + to_add + relativedelta(days=1)
                            date2 = date2 + relativedelta(days=1)

                        else:
                            date1 = date1 + to_add
                        date1 = date1 + relativedelta(months=1)

                        m = m + 1
                except Exception as erp:
                    print("ERROR %", str(erp))

                generalbalance = (totalpayable + totaladjustments - paidamount) - (
                        adjustment_down - adjustment_added
                )
                try:
                    if generalbalance == 0 or generalbalance < 1:
                        generalbalance = 0
                        detail.loan_status = 4

                except:
                    pass
                generalbalance = round(generalbalance, decimals)
                detail.balance = generalbalance
                detail.save()

            # =============================================IF THE RATE TYPE IS REDUCING================================
            if detail.rate_type == 2:
                in_date = ""
                loan_dict = {}
                data = []
                days = []

                edate = appdate + to_add
                date1 = edate

                date2 = date1 + all_duration

                if interest > 0:
                    numilator = (
                            principal
                            * interest
                            * (
                                    (pow((1 + interest), duration))
                                    / (pow((1 + interest), duration) - 1)
                            )
                    )
                else:
                    numilator = principal / duration
                payable = numilator

                totalpayable = payable * duration

                totalpayable = totalpayable

                rbalance = principal
                # ========================================GET PAID SCHEDULES=========================

                paidschedules = paidamount / payable
                paidschedules = floor(paidschedules)

                # ====================================Balance for the next schedule=======================
                nextbal = paidamount - (payable * paidschedules)
                nextintallment = payable - nextbal

                if nextintallment < 1 and nextintallment > 0:
                    nextintallment = payable + nextintallment
                    nextschedule = paidschedules + 2
                else:
                    nextschedule = paidschedules + 1
                m = 1

                sch_date = date1

                if detail.sub_intervals == True:
                    if detail.interval == 4:
                        pricipalperschdule = principal / duration

                        remainingprincipal = principal
                        rem_principal = principal

                        totalinterest = 0
                        for a in range(0, int(duration)):
                            perinterest = rem_principal * interest

                            rem_principal = rem_principal - pricipalperschdule
                            totalinterest = totalinterest + perinterest

                        pbalance = principal
                        sch_date = date1
                        # while date1 < date2:
                        while m <= duration:
                            date3 = date1 - relativedelta(years=1)
                            runningbal = principal + totalinterest
                            for a in range(0, int(duration)):
                                generalint = remainingprincipal * interest
                                generalpayable = pricipalperschdule + generalint
                                interestperschedue = generalint / 12
                                interestperschedue = round(interestperschedue, 2)
                                payable = generalpayable / 12
                                rprincipal = payable - interestperschedue

                                paidschedules = paidamount / payable
                                paidschedules = floor(paidschedules)

                                for sched in range(0, 12):
                                    nextbal = paidamount - (payable * paidschedules)
                                    nextintallment = payable - nextbal
                                    if nextintallment < 1 and nextintallment > 0:
                                        nextintallment = payable + nextintallment
                                        nextschedule = paidschedules + 2
                                    else:
                                        nextschedule = paidschedules + 1

                                    sfine = 0
                                    sfines = 0
                                    if m == nextschedule:
                                        isnext = 1
                                        if nextbal < 1:
                                            spaid = 0
                                        else:
                                            spaid = round(nextbal, decimals)
                                        sbalance = round(nextintallment, decimals)

                                    elif m < nextschedule:
                                        isnext = 0
                                        spaid = round(payable, decimals)
                                        sbalance = 0

                                    else:
                                        isnext = 2
                                        spaid = 0
                                        sbalance = round(payable, decimals)

                                    days.append(date1.strftime("%Y-%m-%d"))

                                    runningbal = runningbal - payable
                                    rbalance = round(runningbal, decimals)

                                    if intpaid == m or intpaid > m:
                                        actualinterest = 0

                                    else:
                                        actualinterest = interestperschedue

                                    pbalance = pbalance - rprincipal
                                    if spaid > 1:
                                        cumrative_payments = cumrative_payments + spaid

                                    else:
                                        cumrative_payments = 0

                                    if date1.weekday() != 5:
                                        date1 = date1 + to_add
                                    else:
                                        date1 = date1 + to_add + relativedelta(days=1)
                                        date2 = date2 + relativedelta(days=1)

                                    # date1 = sch_date + m * to_add
                                    date3 = sch_date + relativedelta(months=m)
                                    m = m + 1

                                    sfine = 0

                                remainingprincipal = (
                                        remainingprincipal - pricipalperschdule
                                )
                        generalbalance = (totalpayable) - paidamount
                        try:
                            if generalbalance == 0 or generalbalance < 1:
                                generalbalance = 0
                                detail.loan_status = 4

                        except:
                            pass
                        generalbalance = round(generalbalance, decimals)
                        detail.balance = generalbalance
                        detail.save()

                    if detail.interval == 3:
                        print("I AM IN THE MONTH")
                        pricipalperschdule = principal / duration
                        paymentperiod = []
                        remainingprincipal = principal
                        rem_principal = principal
                        segments = ceil(duration / 12)
                        actualyears = floor(duration / 12)
                        halfyear = duration - (actualyears * 12)

                        totalinterest = 0
                        for a in range(0, int(actualyears)):
                            perinterest = rem_principal * interest * 12
                            # print('CUMLATIVE INT',perinterest )
                            yearlypayable = rem_principal + perinterest
                            rem_principal = rem_principal - pricipalperschdule * 12
                            totalinterest = totalinterest + perinterest
                            paymentperiod.append(12)
                        # ====FOR THE BALANCE OF MONTH==========================

                        if halfyear > 0:
                            perinterest = rem_principal * interest * halfyear
                            totalinterest = totalinterest + perinterest
                            paymentperiod.append(halfyear)
                        # print('PERIODS ', halfyear)
                        pbalance = principal
                        date3 = date1
                        print("runningbal ", runningbal)
                        print("totalinterest ", totalinterest)
                        runningbal = principal + totalinterest
                        index = 0
                        for a in paymentperiod:
                            if a < 12:
                                generalint = remainingprincipal * interest
                            else:
                                generalint = remainingprincipal * interest * (a / 12)
                            generalpayable = pricipalperschdule + generalint
                            interestperschedue = generalint
                            interestperschedue = interestperschedue
                            payable = generalpayable
                            rprincipal = payable - interestperschedue

                            paidschedules = paidamount / payable
                            paidschedules = floor(paidschedules)

                            for sched in range(0, int(a)):
                                nextbal = paidamount - (payable * paidschedules)
                                nextintallment = payable - nextbal

                                if nextintallment < 1 and nextintallment > 0:
                                    nextintallment = payable + nextintallment
                                    nextschedule = paidschedules + 2
                                else:
                                    nextschedule = paidschedules + 1

                                sfine = 0
                                sfines = 0
                                if m == nextschedule:
                                    isnext = 1
                                    if nextbal < 1:
                                        spaid = 0
                                    else:
                                        spaid = nextbal
                                    sbalance = nextintallment

                                elif m < nextschedule:
                                    isnext = 0
                                    spaid = payable
                                    sbalance = 0

                                else:
                                    isnext = 2
                                    spaid = 0
                                    sbalance = payable

                                days.append(date1.strftime("%Y-%m-%d"))

                                runningbal = runningbal - payable
                                rbalance = runningbal
                                if intpaid == m or intpaid > m:
                                    actualinterest = 0

                                else:
                                    actualinterest = interestperschedue

                                pbalance = pbalance - rprincipal
                                if spaid > 0:
                                    cumrative_payments = cumrative_payments + spaid

                                else:
                                    cumrative_payments = 0

                                loan_dict[in_date] = date3.strftime("%Y-%m-%d")
                                loan_dict[rprincipal] = rprincipal
                                loan_dict[rbalance] = rbalance
                                loan_dict[isnext] = isnext
                                loan_dict[interestperschedue] = interestperschedue
                                loan_dict[spaid] = spaid
                                loan_dict[sbalance] = sbalance
                                loan_dict[actualinterest] = actualinterest
                                if rbalance < 1:
                                    rbalance = 0

                                if rprincipal < 1:
                                    rprincipal = 0

                                if pbalance < 1:
                                    pbalance = 0

                                data.append(
                                    [
                                        round(rprincipal, decimals),
                                        loan_dict[in_date],
                                        round(rbalance, decimals),
                                        round(interestperschedue, decimals),
                                        loan_dict[round(isnext, decimals)],
                                        round(spaid, decimals),
                                        round(sbalance, decimals),
                                        round(actualinterest, decimals),
                                        sfine,
                                        round(payable, decimals),
                                        round(pbalance, decimals),
                                        round(cumrative_payments, decimals),
                                    ]
                                )
                                date3 = sch_date + relativedelta(months=m)
                                date1 = sch_date + m * to_add
                                m = m + 1
                                sfine = 0
                                # date1 = date1 + to_add
                                # date1 = sch_date + m * to_add
                                # date3 = date3 + relativedelta(months=1)

                            index = index + 1
                            remainingprincipal = (
                                    remainingprincipal - pricipalperschdule * 12
                            )

                else:
                    pbalance = principal
                    sch_date = date1
                    generalbalance = (totalpayable + totaladjustments - paidamount) - (
                            adjustment_down - adjustment_added
                    )

                    try:
                        if generalbalance == 0 or generalbalance < 1:
                            generalbalance = 0
                            detail.loan_status = 4

                    except:
                        pass
                    generalbalance = round(generalbalance, decimals)
                    detail.balance = generalbalance
                    detail.save()

                    expectedinterest = 0
                    # while date1 < date2:
                    while m <= duration:
                        if m == nextschedule:
                            spaid = round(nextbal, decimals)

                        elif m < nextschedule:
                            spaid = round(payable, decimals)
                        else:
                            spaid = 0

                        days.append(date1.strftime("%Y-%m-%d"))

                        interestperschedue = rbalance * interest
                        interestperschedue = interestperschedue

                        expectedinterest = expectedinterest + interestperschedue
                        if interest > 0:
                            rbalance = principal * (
                                    (pow((1 + interest), duration) - pow((1 + interest), m))
                                    / (pow((1 + interest), duration) - 1)
                            )
                        else:
                            rbalance = rbalance - payable
                        if intpaid == m or intpaid > m:
                            actualinterest = 0

                        else:
                            actualinterest = interestperschedue

                        rprincipal = payable - interestperschedue
                        rprincipal = rprincipal

                        rbalance = rbalance

                        m = m + 1
                        if date1 < nowdate + relativedelta(months=1):
                            reciept = date1.strftime("%Y%m%d") + str(loanid)
                            txdate = date1 - relativedelta(days=30)

                            # ============== CHECK IF THE RECORD IS ALREADY SAVED==================
                            try:
                                check = Transactions.objects.filter(
                                    loan_id=loanid, receipt=reciept
                                )[0]

                            except:
                                interestperschedue = round(interestperschedue, 2)
                                Transactions.objects.create(
                                    reporting_amount=interestperschedue,
                                    narration=interest_narration,
                                    account_dr_id=dr_interests,
                                    account_cr_id=cr_interests,
                                    loan_id=loanid,
                                    tx_date=txdate,
                                    receipt=reciept,
                                    transaction_type="interest on loan",
                                    added_by_id=userdata(request),
                                    branch_id=branchdata(request),
                                )

                        date1 = date1 + relativedelta(months=1)

    except Exception as lex:
        traceback.print_exc()
        print(" HERE NOW ", loanid, " /", str(lex))


@method_decorator(login_required, name="dispatch")
class LoanInterest(View):
    template_name = "loans/loan_interest.html"

    def get(self, request, *args, **kwargs):
        title = "Loan Interest"
        loans = Loans.objects.filter(branch_id=branchdata(request), loan_status__gt=2)
        # ================INCOME CREDIT ACCOUNT=======================================================

        intrestaccount = Account.objects.filter(
            name="Loan interest", business=businessdata(request)
        ).first()
        cr_interests = intrestaccount.id

        laonacct = Account.objects.filter(
            name="Loan interest receivables", business=businessdata(request)
        )[0]
        dr_interests = laonacct.id

        nowdate = today.strftime("%Y-%m-%d")
        for ln in loans:
            loanid = ln.id
            mem_acct = ln.account.acc_number
            # print('loanid ',loanid, ' DR ',dr_interests, ' cr ', cr_interests, mem_acct)
            getInterests(request, loanid, cr_interests, dr_interests, nowdate, mem_acct)

        messages.success(
            request, "success", extra_tags="Interests Updated successfully"
        )

        # =================================EDITING LOAN TYPE===================================
        return render(request, self.template_name)


class BaseLoanBalanceView:
    def get_loan(
            self, loans, account, interest_account, date_input, loan_balance
    ) -> []:
        # print(interest_account.name)

        for loan in loans:
            rate = loan.rate / 100
            # print('rate', rate)
            if loan.rate_type == 2:
                amo = reducing_amortization(
                    loan.amount_approved,
                    rate,
                    loan.approved_duration,
                    loan=loan,
                    formula=loan.loan_type.formula,
                )
            else:
                amo = flat_amortization(
                    loan.amount_approved,
                    rate,
                    loan.approved_duration,
                    loan=loan,
                    formula=loan.loan_type.formula,
                )
            total_interest = sum(item["Interest"] for item in amo)
            # print('amo', total_interest)

            interest_paid = loan.loan_trans.filter(
                transaction_type="Loan interest"
            ).aggregate(
                total=Coalesce(Sum("reporting_amount"), 0.0, output_field=FloatField())
            )[
                "total"
            ]

            expected = loan.loan_trans.filter(transaction_type="give loan").aggregate(
                paid=Coalesce(Sum("reporting_amount"), 0.0, output_field=FloatField())
            )["paid"]

            principle_paid = loan.loan_trans.filter(
                transaction_type="Loan repayment"
            ).aggregate(
                paid=Coalesce(Sum("reporting_amount"), 0.0, output_field=FloatField())
            )[
                "paid"
            ]

            if date_input:
                interest = loan.loan_trans.filter(
                    transaction_type="interest on loan", tx_date__lte=date_input
                ).aggregate(
                    total=Coalesce(
                        Sum("reporting_amount"), 0.0, output_field=FloatField()
                    )
                )[
                    "total"
                ]

                interest_paid = loan.loan_trans.filter(
                    transaction_type="Loan interest", tx_date__lte=date_input
                ).aggregate(
                    total=Coalesce(
                        Sum("reporting_amount"), 0.0, output_field=FloatField()
                    )
                )[
                    "total"
                ]

                expected = loan.loan_trans.filter(
                    transaction_type="give loan", tx_date__lte=date_input
                ).aggregate(
                    paid=Coalesce(
                        Sum("reporting_amount"), 0.0, output_field=FloatField()
                    )
                )[
                    "paid"
                ]

                principle_paid = loan.loan_trans.filter(
                    transaction_type="Loan repayment", tx_date__lte=date_input
                ).aggregate(
                    paid=Coalesce(
                        Sum("reporting_amount"), 0.0, output_field=FloatField()
                    )
                )[
                    "paid"
                ]
                # total_paid = principle_paid + interest_paid

                print("expected", interest)
                print("paid", interest_paid)

            total_loan_debit = Transactions.objects.filter(
                Q(loan=loan, account_dr=account)).aggregate(total_debit=Sum("reporting_amount"))['total_debit']
            if total_loan_debit is None:
                total_loan_debit = 0.0
            loan_principle = total_loan_debit
            principle_balance = loan_principle - principle_paid
            interest_balance = total_interest - interest_paid
            balance_total = principle_balance + interest_balance

            loan_fine_bal = loan.fine_balance
            if loan.fine_balance is None:
                loan_fine_bal = 0.0

            # if principle_balance >= 1:
            loan_balance.append(
                {
                    "name": loan.account.names,
                    "account": loan.account.acc_number,
                    "principle_balance": "{:,.2f}".format(principle_balance),
                    "interest_balance": "{:,.2f}".format(interest_balance),
                    "fine_balance": "{:,.2f}".format(loan_fine_bal),
                    "balance_total": "{:,.2f}".format(
                        balance_total + loan_fine_bal
                    ),
                    "loan_principle": "{:,.2f}".format(loan_principle),
                }
            )
        return loan_balance


@method_decorator(login_required, name="dispatch")
class LoanBalancesView(BaseLoanBalanceView, TemplateView):
    template_name = "loans/balances.html"

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        date_input = self.request.GET.get("date")
        current_date = datetime.date.today()
        user = self.request.user
        # branch = biz_staff_branch(user)
        business = biz_data(user)
        loans = (
            Loans.objects.filter(branch__business=business, loan_status__gte=3)
            .annotate(fines=Sum("loan_fines_custom__amount", default=0.0))
            .annotate(fine_payment=Sum("loan_fine_pay", default=0.0))
            .annotate(fine_balance=F("fines") - F("fine_payment"))
            .prefetch_related("account")
            .prefetch_related("loan_trans")
            .prefetch_related("loan_trans", "loan_fines_custom", "loan_trans")
            .prefetch_related("loan_fines_custom")
            .prefetch_related("loan_trans")
        )
        # print('n'*100)
        # print(loans.query)

        account = Account.objects.filter(
            name="Loan Receivables", business=business
        ).first()
        interest_account = Account.objects.filter(
            name="Loan interest receivables", business=business
        ).first()
        loan_balance = []
        context["title"] = "Loan Balances"
        context["loans"] = self.get_loan(
            loans, account, interest_account, date_input, loan_balance
        )
        context["today"] = current_date
        return context


class ExportLoansBalancesExcel(BaseLoanBalanceView, View):
    def get(self, request, **kwargs):
        date_input = self.request.GET.get("date")
        user = self.request.user
        business = biz_data(user)
        loans = (
            Loans.objects.filter(branch__business=business, loan_status__gte=3)
            .annotate(fines=Sum("loan_fines_custom__amount", default=0.0))
            .annotate(fine_payment=Sum("loan_fine_pay", default=0.0))
            .annotate(fine_balance=F("fines") - F("fine_payment"))
            .prefetch_related("account", "loan_trans")
        )
        # print(loans.values('fines'))
        account = Account.objects.filter(
            name="Loan Receivables", business=business
        ).first()
        interest_account = Account.objects.filter(
            name="Loan interest receivables", business=business
        ).first()
        loan_balance = []
        dataset = self.get_loan(
            loans, account, interest_account, date_input, loan_balance
        )
        df = pd.DataFrame(dataset)
        df.columns = [
            "Name",
            "Account",
            "Principle balance",
            "Interest balance",
            "Fine balance",
            "Total balance",
            "Loan Principle",
        ]
        response = HttpResponse(
            content_type="application/vnd.openxmlformats-officedocument.spreadsheetml.sheet"
        )
        response[
            "Content-Disposition"
        ] = 'attachment; filename="Loans_balance_report_{}.xlsx"'.format(
            str(business.name).replace(" ", "_")
        )
        df.to_excel(response, index=False)
        return response


@method_decorator(login_required, name="dispatch")
class LoanReports(TemplateView):
    template_name = "loans/loan_reports.html"

    def filter_loans(self) -> {}:
        filter_item = {}
        today_ = datetime.datetime.today()
        from_date = self.request.GET.get("from_date", None)
        to_date = self.request.GET.get("to_date", None)
        if from_date and to_date:
            filter_item.update(
                {"schedule_start__gte": from_date, "schedule_start__lte": to_date}
            )
        else:
            filter_item["schedule_start"] = today_
        return filter_item

    def filter_transactions(self) -> {}:
        filters = {}
        period = self.request.GET.get("date", None)
        today_ = datetime.datetime.today()
        user = self.request.user
        branch = biz_staff_branch(user)
        filters["branch"] = branch
        if period:
            filters["tx_date__lte"] = period
        else:
            filters["tx_date__lte"] = today_
        return filters

    def range_filters(self) -> dict:
        from_date = self.request.GET.get("from_date", None)
        to_date = self.request.GET.get("to_date", None)
        filters = {}
        user = self.request.user
        today_ = datetime.datetime.today()
        filters["branch"] = biz_staff_branch(user)
        if from_date and to_date:
            filters.update({"tx_date__gte": from_date, "tx_date__lte": to_date})
        else:
            filters["tx_date"] = today_
        return filters

    def show_loans(self, loans, account, interest_account) -> list:
        loans_list = []
        cleaned_loans = (
            Loans.objects.filter(
                id__in=list(loans.values_list("loan_id", flat=True)), loan_status=3
            )
            .prefetch_related("account")
            .prefetch_related("loan_trans")
        )
        for loan in cleaned_loans:
            debits_ = (
                loan.loan_trans.filter(account_dr=account)
                .filter(**self.filter_transactions())
                .aggregate(
                    debits=Coalesce(
                        Sum("reporting_amount"), 0.0, output_field=FloatField()
                    )
                )["debits"]
            )
            credits_ = (
                loan.loan_trans.filter(account_cr=account)
                .filter(**self.filter_transactions())
                .aggregate(
                    credits=Coalesce(
                        Sum("reporting_amount"), 0.0, output_field=FloatField()
                    )
                )["credits"]
            )

            debits_int_ = (
                loan.loan_trans.filter(account_dr=interest_account)
                .filter(**self.filter_transactions())
                .aggregate(
                    total=Coalesce(
                        Sum("reporting_amount"), 0.0, output_field=FloatField()
                    )
                )["total"]
            )
            credits_int_ = (
                loan.loan_trans.filter(account_cr=interest_account)
                .filter(**self.filter_transactions())
                .aggregate(
                    total=Coalesce(
                        Sum("reporting_amount"), 0.0, output_field=FloatField()
                    )
                )["total"]
            )

            loans_list.append(
                {
                    "name": loan.account.names,
                    "account": loan.account.acc_number,
                    "principle": "{:0,.2f}".format(float(loan.amount_approved)),
                    "interest": "{:0,.2f}".format(float(credits_int_)),
                    "balance": "{:0,.2f}".format(float(debits_ - credits_)),
                    "disbursed": "{:0,.2f}".format(float(debits_ - credits_)),
                    "duration": loan.approved_duration,
                }
            )

        return loans_list

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        user = self.request.user
        business = biz_data(user)

        account = Account.objects.filter(
            name="Loan Receivables", business=business
        ).first()
        interest_account = Account.objects.filter(
            name="Loan interest receivables", business=business
        ).first()
        amount_out = Transactions.objects.filter(account_dr=account)
        disbursed_queryset = amount_out.filter(**self.filter_transactions())
        loans = disbursed_queryset.values("loan")
        disbursed = disbursed_queryset.aggregate(
            credits=Coalesce(Sum("reporting_amount"), 0.0, output_field=FloatField())
        )["credits"]
        amount_loaned = amount_out.filter(**self.range_filters()).aggregate(
            debits=Coalesce(Sum("reporting_amount"), 0.0, output_field=FloatField())
        )["debits"]
        receivables = Transactions.objects.filter(account_cr=account)
        recovered = receivables.filter(**self.filter_transactions()).aggregate(
            debits=Coalesce(Sum("reporting_amount"), 0.0, output_field=FloatField())
        )["debits"]
        amount_received = receivables.filter(**self.range_filters()).aggregate(
            debits=Coalesce(Sum("reporting_amount"), 0.0, output_field=FloatField())
        )["debits"]
        money_at_risk = disbursed - recovered

        credits_ = (
            Transactions.objects.filter(account_cr=interest_account)
            .filter(**self.filter_transactions())
            .aggregate(
                credits=Coalesce(
                    Sum("reporting_amount"), 0.0, output_field=FloatField()
                )
            )["credits"]
        )

        debits_ = (
            Transactions.objects.filter(account_dr=interest_account)
            .filter(**self.filter_transactions())
            .aggregate(
                debits=Coalesce(Sum("reporting_amount"), 0.0, output_field=FloatField())
            )["debits"]
        )

        context["title"] = "Loan Reports"
        context["today"] = datetime.datetime.today().date()
        context["disbursed"] = "{:0,.2f}".format(float(disbursed))
        context["recovered"] = "{:0,.2f}".format(float(recovered))
        context["interest"] = "{:0,.2f}".format(float(credits_))
        context["at_risk"] = "{:0,.2f}".format(float(money_at_risk))
        context["amount_received"] = "{:0,.2f}".format(float(amount_received))
        context["amount_loaned"] = "{:0,.2f}".format(float(amount_loaned))
        context["loans"] = self.show_loans(loans, account, interest_account)
        # print(self.show_loans(loans, account, interest_account))
        # print(context['loans'])
        return context


# @method_decorator(login_required, name='dispatch')
# class ExpUploadsView(View):


# ++++++++++++++++++++++   START OF EXPORT LOANS REPORT  +++++++++++++++++++++++++
@method_decorator(login_required, name="dispatch")
class ExportLoanReports(View):
    def filter_loans(self) -> {}:
        filter_item = {}
        today_ = datetime.datetime.today()
        from_date = self.request.GET.get("from_date", None)
        to_date = self.request.GET.get("to_date", None)
        if from_date and to_date:
            filter_item.update(
                {"schedule_start__gte": from_date, "schedule_start__lte": to_date}
            )
        else:
            filter_item["schedule_start"] = today_

        return filter_item

    def filter_transactions(self) -> {}:
        filters = {}
        period = self.request.GET.get("date", None)
        today_ = datetime.datetime.today()
        user = self.request.user
        branch = biz_staff_branch(user)
        filters["branch"] = branch
        if period:
            filters["tx_date__lte"] = period
        else:
            filters["tx_date__lte"] = today_
        return filters

    def range_filters(self) -> dict:
        from_date = self.request.GET.get("from_date", None)
        to_date = self.request.GET.get("to_date", None)
        filters = {}
        user = self.request.user
        today_ = datetime.datetime.today()
        filters["branch"] = biz_staff_branch(user)
        if from_date and to_date:
            filters.update({"tx_date__gte": from_date, "tx_date__lte": to_date})
        else:
            filters["tx_date"] = today_
        return filters

    def show_loans(self, loans, account, interest_account) -> list:
        loans_list = []
        cleaned_loans = (
            Loans.objects.filter(
                id__in=list(loans.values_list("loan_id", flat=True)), loan_status=3
            )
            .prefetch_related("account")
            .prefetch_related("loan_trans")
        )
        for loan in cleaned_loans:
            debits_ = (
                loan.loan_trans.filter(account_dr=account)
                .filter(**self.filter_transactions())
                .aggregate(
                    debits=Coalesce(
                        Sum("reporting_amount"), 0.0, output_field=FloatField()
                    )
                )["debits"]
            )
            credits_ = (
                loan.loan_trans.filter(account_cr=account)
                .filter(**self.filter_transactions())
                .aggregate(
                    credits=Coalesce(
                        Sum("reporting_amount"), 0.0, output_field=FloatField()
                    )
                )["credits"]
            )

            debits_int_ = (
                loan.loan_trans.filter(account_dr=interest_account)
                .filter(**self.filter_transactions())
                .aggregate(
                    total=Coalesce(
                        Sum("reporting_amount"), 0.0, output_field=FloatField()
                    )
                )["total"]
            )
            credits_int_ = (
                loan.loan_trans.filter(account_cr=interest_account)
                .filter(**self.filter_transactions())
                .aggregate(
                    total=Coalesce(
                        Sum("reporting_amount"), 0.0, output_field=FloatField()
                    )
                )["total"]
            )

            loans_list.append(
                {
                    "name": loan.account.names,
                    "account": loan.account.acc_number,
                    "principle": "{:0,.2f}".format(float(loan.amount_approved)),
                    "interest": "{:0,.2f}".format(float(credits_int_)),
                    "balance": "{:0,.2f}".format(float(debits_ - credits_)),
                    "disbursed": "{:0,.2f}".format(float(debits_ - credits_)),
                    "duration": loan.approved_duration,
                }
            )

        return loans_list

    def get(self, request, **kwargs):
        # context = super().get_context_data(**kwargs)
        user = self.request.user
        business = biz_data(user)

        account = Account.objects.filter(
            name="Loan Receivables", business=business
        ).first()
        interest_account = Account.objects.filter(
            name="Loan interest receivables", business=business
        ).first()
        amount_out = Transactions.objects.filter(account_dr=account)
        disbursed_queryset = amount_out.filter(**self.filter_transactions())
        loans = disbursed_queryset.values("loan")
        disbursed = disbursed_queryset.aggregate(
            credits=Coalesce(Sum("reporting_amount"), 0.0, output_field=FloatField())
        )["credits"]
        amount_loaned = amount_out.filter(**self.range_filters()).aggregate(
            debits=Coalesce(Sum("reporting_amount"), 0.0, output_field=FloatField())
        )["debits"]
        receivables = Transactions.objects.filter(account_cr=account)
        recovered = receivables.filter(**self.filter_transactions()).aggregate(
            debits=Coalesce(Sum("reporting_amount"), 0.0, output_field=FloatField())
        )["debits"]
        amount_received = receivables.filter(**self.range_filters()).aggregate(
            debits=Coalesce(Sum("reporting_amount"), 0.0, output_field=FloatField())
        )["debits"]
        money_at_risk = disbursed - recovered

        credits_ = (
            Transactions.objects.filter(account_cr=interest_account)
            .filter(**self.filter_transactions())
            .aggregate(
                credits=Coalesce(
                    Sum("reporting_amount"), 0.0, output_field=FloatField()
                )
            )["credits"]
        )
        debits_ = (
            Transactions.objects.filter(account_dr=interest_account)
            .filter(**self.filter_transactions())
            .aggregate(
                debits=Coalesce(Sum("reporting_amount"), 0.0, output_field=FloatField())
            )["debits"]
        )
        # context['title'] = 'Loan Reports'
        # context['today'] = datetime.datetime.today().date()
        # context['disbursed'] = "{:0,.2f}".format(float(disbursed))
        # context['recovered'] = "{:0,.2f}".format(float(recovered))
        # context['interest'] = "{:0,.2f}".format(float(credits_))
        # context['at_risk'] = "{:0,.2f}".format(float(money_at_risk))
        # context['amount_received'] = "{:0,.2f}".format(float(amount_received))
        # context['amount_loaned'] = "{:0,.2f}".format(float(amount_loaned))
        # context['loans'] = self.show_loans(loans, account, interest_account)
        the_loans = self.show_loans(loans, account, interest_account)
        print(the_loans)
        df = pd.DataFrame(the_loans)
        response = HttpResponse(content_type="application/ms-excel")
        response["Content-Disposition"] = 'attachment; filename="sample_excel.xlsx"'

        # Create an Excel writer using Pandas
        writer = pd.ExcelWriter(response, engine="xlsxwriter")
        df.to_excel(writer, sheet_name="Sheet1", index=False)
        writer.save()

        return response


# ++++++++++++++++++++++   END OF EXPORT LOANS REPORT  +++++++++++++++++++++++++


@method_decorator(login_required, name="dispatch")
class LoanRepaymentUploadsView(View):
    template_name = "loans/loan_repayments_uploads.html"

    def get(self, request, *args, **kwargs):
        business_context = get_business_loans_context_data(request)
        if len(business_context['data_context'][0]) != 1:
            messages.error(
                request, "error",
                extra_tags="Action not allowed in the central view. Please logout and login into the respective branch"
            )
            return HttpResponseRedirect(request.META.get("HTTP_REFERER"))

        title = "Loan repayment uploads"
        loan_repayments_uploads = LoanRepaymentUpload.objects.filter(
            branch=business_context['branch']
        )
        http = "http://" if not request.is_secure() else "https://"
        baseUrl = request.headers["HOST"]
        baseUrl = http + baseUrl

        return render(request, self.template_name, locals())


@method_decorator(login_required, name="dispatch")
class ReverseLoanRepaymentsUploadView(View):
    def post(self, request, pk, *args, **kwargs):
        upload = LoanRepaymentUpload.objects.filter(id=pk)
        if upload.exists():
            upload_obj = upload.first()
            transactions_created = json.loads(upload_obj.transactions_created)[
                "transactions_created"
            ]
            loans_closed = json.loads(upload_obj.transactions_created)["loans_closed"]
            for trans in transactions_created:
                old_transaction = Transactions.objects.filter(id=trans)
                if old_transaction.exists():
                    old_transaction.delete()
            # open all closed loans
            for ln in loans_closed:
                Loans.objects.filter(id=ln).update(loan_status=3)

            upload.delete()
        messages.success(
            request, "success", extra_tags="Repayments successfully reversed"
        )
        return HttpResponseRedirect(reverse_lazy("loan_repayment_uploads"))


@method_decorator(login_required, name="dispatch")
class ChangeLoanDurationView(View):
    def post(self, request, pk, *args, **kwargs):
        the_duration = int(float(request.POST["current_duration"]))
        loan_obj = Loans.objects.filter(id=pk)
        if loan_obj.exists():
            loan_obj.update(approved_duration=the_duration)
        else:
            messages.error(
                request,
                "error",
                extra_tags="Failed to change duration, Loan nolonger exists",
            )
        messages.success(request, "success", extra_tags="Duration updated successfully")
        return HttpResponseRedirect(reverse_lazy("loan_details", args=[pk]))


@method_decorator(login_required, name="dispatch")
class OtherLoanReports(View):
    template_name = "loans/other_loans_report.html"

    def get_dates(self, loan, interval):
        start_date = loan.approved_on
        if loan.schedule_start is not None:
            start_date = loan.schedule_start
        if loan.interval == IntervalChoices.DAYS:
            new_date = (
                start_date + datetime.timedelta(days=interval) if start_date else None
            )
            # Print the new date
            return new_date
        elif loan.interval == IntervalChoices.WEEKS:
            new_date = start_date + datetime.timedelta(weeks=interval)
            return new_date
        elif loan.interval == IntervalChoices.MONTHS:
            new_date = start_date + relativedelta(months=interval)
            return new_date
        elif loan.interval == IntervalChoices.YEARS:
            new_date = start_date + relativedelta(years=interval)
            return new_date

    def amotize(self, loan):
        rate = loan.rate / 100
        amortization = loan_payment_details(
            loan.rate_type,
            rate,
            loan.amount_approved,
            loan.approved_duration,
            loan.interval,
            loan.loan_type.formula,
            acc=loan.account,
            loan=loan,
        )
        schedules = amortization.gen_schedules()
        payable = schedules["installment"]
        interest_payable = schedules["interest_paid"]
        return {"payable": payable, "sum_interest": interest_payable}

    def get_principal(self, loan) -> float:
        loanacct = Account.objects.filter(
            name="Loan Receivables", business=businessdata(self.request)
        ).first()
        adjustment_added = loan.loan_trans.filter(
            Q(account_dr=loanacct.id), Q(transaction_type="Principal Adjustment")
        ).aggregate(
            total=Coalesce(Sum("reporting_amount", output_field=FloatField()), 0.00)
        )[
            "total"
        ]
        adjustment_down = loan.loan_trans.filter(
            Q(account_cr=loanacct.id), Q(transaction_type="Principal Adjustment")
        ).aggregate(
            total=Coalesce(Sum("reporting_amount", output_field=FloatField()), 0.00)
        )[
            "total"
        ]

        # adjustments = loan.loan_trans.objects.filter(transaction_type='Loan Adjustment')
        totaldebit = loan.loan_trans.filter(
            Q(account_dr_id=loanacct.id), ~Q(transaction_type="Principal Adjustment")
        ).aggregate(
            total=Coalesce(Sum("reporting_amount", output_field=FloatField()), 0.00)
        )[
            "total"
        ]
        # incase the is removal of an extra charge (this is a fix for the extra charge that was being added to loan amt ap
        # approved due to 'added to principal'
        extra_charge_removed = loan.loan_trans.filter(
            transaction_type="Remove extra loan charge"
        ).aggregate(
            ex_charge=Coalesce(Sum("reporting_amount", output_field=FloatField()), 0.00)
        )[
            "ex_charge"
        ]

        principal = (
                float(totaldebit)
                - float(adjustment_down - adjustment_added)
                - extra_charge_removed
        )

        return principal

    def paid_amount(self, loan) -> float:
        paid = loan.loan_trans.filter(
            Q(transaction_type="Loan repayment") | Q(transaction_type="Loan interest")
        ).aggregate(
            paid=Coalesce(Sum("reporting_amount"), 0.0, output_field=FloatField())
        )[
            "paid"
        ]
        return paid

    def principal_paid(self, loan) -> float:
        paid = loan.loan_trans.filter(transaction_type="Loan repayment").aggregate(
            paid=Coalesce(Sum("reporting_amount"), 0.0, output_field=FloatField())
        )["paid"]
        return paid

    def interest_paid(self, loan) -> float:
        paid = loan.loan_trans.filter(transaction_type="Loan interest").aggregate(
            paid=Coalesce(Sum("reporting_amount"), 0.0, output_field=FloatField())
        )["paid"]
        return paid

    def get_running_loans(self, loans) -> List[dict]:
        payment_schedule = []
        current_date = datetime.datetime.today().date().month
        this_year = datetime.datetime.today().date().year
        filtered_month = self.request.GET.get("month", None)
        if filtered_month:
            current_date = int(filtered_month)
        for loan in loans:
            # print(loan.loan_trans.all())
            paid = self.paid_amount(loan)
            interest_paid = self.interest_paid(loan)

            # pp = loan.amount_approved
            pp = self.get_principal(loan)
            # pp = loan.loan_trans.filter()
            start_date = loan.schedule_start
            rate = loan.rate / 100

            amo = loan_payment_details(
                loan.rate_type,
                rate,
                loan.amount_approved,
                loan.approved_duration,
                loan.interval,
                loan.loan_type.formula,
                acc=loan.account,
                loan=loan,
            )
            for i, deloan in enumerate(amo.gen_schedules().to_dict("records")):
                interest = deloan["interest_paid"]
                payable_per_schedule = deloan["installment"]
                paid_here = min(paid, payable_per_schedule)
                paid_int_here = min(interest_paid, interest)
                interest_balance = interest - paid_int_here
                paid -= paid_here
                interest_paid -= paid_int_here

                schedule_balance = payable_per_schedule - paid_here
                payment_date = find_end_date(
                    loan.interval, start_date=start_date
                ).gen_end_date(i + 1)
                if schedule_balance > 0:
                    print("pay", type(payment_date.month))
                    print("current", type(current_date))
                    if (
                            payment_date.month == current_date
                            and payment_date.year == this_year
                    ):
                        payment_schedule.append(
                            {
                                "id": loan.id,
                                "applicant": loan.applicant.biodata.name,
                                "payment_date": payment_date,
                                "principle": deloan["principal_paid"],
                                "interest": interest,
                                "installment": payable_per_schedule,
                                "paid": paid_here,
                                "schedule_balance": schedule_balance,
                                "requested_duration": loan.requested_duration,
                                "principle_balance": deloan["remaining_balance"],
                                "interest_balance": interest_balance,
                                "interval": loan.get_interval_display(),
                                "account": loan.account.acc_number,
                            }
                        )
        return payment_schedule

    def paginate(self, loans):
        page = self.request.GET.get("page", 1)

        paginator = Paginator(loans, 30)
        try:
            running_loans = paginator.page(page)
        except PageNotAnInteger:
            running_loans = paginator.page(1)
        except EmptyPage:
            running_loans = paginator.page(paginator.num_pages)
        return running_loans

    def get(self, request, *args, **kwargs):
        current_date = datetime.datetime.today().date()
        title = "Running Loans"
        loan_settings = LoanSettings.objects.filter(
            business=businessdata(request)
        ).first()
        decimals = loan_settings.decimals
        filter = {"branch__business": businessdata(request), "loan_status": 3}
        query = self.request.GET.get("q", None)
        if query:
            filter["applicant__biodata__name__icontains"] = query
        running_loans = Loans.objects.filter(**filter).prefetch_related("loan_trans")
        paginated_loans = self.paginate(self.get_running_loans(running_loans))
        context = {
            "running_loans": paginated_loans,
            "title": title,
            "settings": loan_settings,
            "decimals": decimals,
            "months": months_with_ids,
            "this_month": current_date.month,
        }

        return render(request, self.template_name, context)


@method_decorator(login_required, name="dispatch")
class EditRunningLoanView(View):
    template_name = "loans/edit_running_loan.html"

    def get(self, request, *args, **kwargs):
        pk = self.kwargs["id"]
        obj = Loans.objects.filter(id=pk, branch_id=branchdata(self.request)).first()
        if obj is None:
            raise Http404("No running Loan with ID")
        return HttpResponseRedirect(reverse_lazy("loan_details", args=[pk]))

    @can_do_this
    def post(self, request, *args, **kwargs):
        pk = self.kwargs["id"]
        obj = Loans.objects.filter(id=pk, branch_id=branchdata(self.request)).first()
        staff = biz_staff(self.request.user)
        if obj is None:
            raise Http404("No running Loan with ID")
        form = EditRunningLoanForm(data=self.request.POST, staff=staff, loan=obj)
        form.save()
        messages.success(self.request, "success", extra_tags="Successfully updated")
        return HttpResponseRedirect(reverse_lazy("loan_details", args=[pk]))


@method_decorator(login_required, name="dispatch")
class OtherLoanReportsExcel(View):
    template_name = "loans/other_loans_report.html"

    def amotize(self, loan):
        rate = loan.rate / 100
        amortization = loan_payment_details(
            loan.rate_type,
            rate,
            loan.amount_approved,
            loan.approved_duration,
            loan.interval,
            loan.loan_type.formula,
            acc=loan.account,
            loan=loan,
        )
        schedules = amortization.gen_schedules()
        payable = schedules["installment"]
        interest_payable = schedules["interest_paid"]
        return {"payable": payable, "sum_interest": interest_payable}

    def paid_amount(self, loan) -> float:
        paid = loan.loan_trans.filter(
            Q(transaction_type="Loan repayment") | Q(transaction_type="Loan interest")
        ).aggregate(
            paid=Coalesce(Sum("reporting_amount"), 0.0, output_field=FloatField())
        )[
            "paid"
        ]
        return paid

    def principal_paid(self, loan) -> float:
        paid = loan.loan_trans.filter(transaction_type="Loan repayment").aggregate(
            paid=Coalesce(Sum("reporting_amount"), 0.0, output_field=FloatField())
        )["paid"]
        return paid

    def interest_paid(self, loan) -> float:
        paid = loan.loan_trans.filter(transaction_type="Loan interest").aggregate(
            paid=Coalesce(Sum("reporting_amount"), 0.0, output_field=FloatField())
        )["paid"]
        return paid

    def get_running_loans(self, loans) -> List[dict]:
        payment_schedule = []
        current_date = datetime.datetime.today().date().month
        this_year = datetime.datetime.today().date().year
        filtered_month = self.request.GET.get("month", None)
        if filtered_month:
            current_date = int(filtered_month)
        for loan in loans:
            # print(loan.loan_trans.all())
            paid = self.paid_amount(loan)
            interest_paid = self.interest_paid(loan)

            start_date = loan.schedule_start
            rate = loan.rate / 100

            amo = loan_payment_details(
                loan.rate_type,
                rate,
                loan.amount_approved,
                loan.approved_duration,
                loan.interval,
                loan.loan_type.formula,
                acc=loan.account,
                loan=loan,
            )
            for i, deloan in enumerate(amo.gen_schedules().to_dict("records")):
                interest = deloan["interest_paid"]
                payable_per_schedule = deloan["installment"]
                paid_here = min(paid, payable_per_schedule)
                paid_int_here = min(interest_paid, interest)
                interest_balance = interest - paid_int_here
                paid -= paid_here
                interest_paid -= paid_int_here

                schedule_balance = payable_per_schedule - paid_here
                payment_date = find_end_date(
                    loan.interval, start_date=start_date
                ).gen_end_date(i + 1)
                if schedule_balance > 0:
                    if (
                            payment_date.month == current_date
                            and payment_date.year == this_year
                    ):
                        payment_schedule.append(
                            {
                                "Applicant": loan.applicant.biodata.name,
                                "AccountNumber": loan.account.acc_number,
                                "PaymentDate": payment_date,
                                "Principle": deloan["principal_paid"],
                                "Interest": interest,
                                "Installment": payable_per_schedule,
                                "AmountPaid": paid_here,
                                "MonthlyScheduleBalance": schedule_balance,
                                "MonthlyPrincipleBalance": deloan["remaining_balance"],
                                "MonthlyInterestBalance": interest_balance,
                            }
                        )
        return payment_schedule

    def get(self, request, *args, **kwargs):
        loan_settings = LoanSettings.objects.filter(
            business=businessdata(request)
        ).first()
        filter = {"branch__business": businessdata(request), "loan_status": 3}
        query = self.request.GET.get("q", None)
        if query:
            filter["applicant__biodata__name__icontains"] = query
        running_loans = Loans.objects.filter(**filter).prefetch_related("loan_trans")
        the_loans = self.get_running_loans(running_loans)
        df = pd.DataFrame(the_loans)
        response = HttpResponse(
            content_type="application/vnd.openxmlformats-officedocument.spreadsheetml.sheet"
        )
        response[
            "Content-Disposition"
        ] = f'attachment; filename="{loan_settings.business.name}.xlsx"'
        # df['created_at'] = df['created_at'].dt.tz_localize(None)
        # df['updated_at'] = df['updated_at'].dt.tz_localize(None)
        df.to_excel(response, index=False)
        return response


class LoanTypeDetail(DetailView):
    template_name = "loans/loan_type_detail.html"
    context_object_name = "loan_type"

    def get_object(self, queryset=None):
        pk = self.kwargs["pk"]
        obj = LoanTypes.objects.prefetch_related("application_account_loan_type").get(
            id=pk
        )
        return obj

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        obj = self.get_object()
        ac = ""
        ad = ""
        # x = Transactions.objects.filter(Q(account_dr=ac) | Q(account_cr=ad))
        context["title"] = obj.name
        context["charges"] = obj.application_account_loan_type.filter(
            general_charge__status=1
        )
        return context


@method_decorator(login_required, name='dispatch')
class AgingReports(TemplateView):
    template_name = 'loans/loan_aging_report.html'

    def paid_amount(self, obj) -> float:
        loan = Loans.objects.get(id=obj.id)
        paid = loan.loan_trans.filter(
            Q(transaction_type='Loan repayment') | Q(transaction_type='Loan interest')
        ).aggregate(
            paid=Coalesce(Sum('reporting_amount'), 0.0, output_field=FloatField()))['paid']
        return paid

    def display_other_columns(self, group):
        return group[['client', 'Current', '1-30', '31-60', '61-90', '90+', 'amount_outstanding']]

    def get_aging_loans(self, loans) -> List[dict]:
        payment_schedule = []
        current_today = datetime.datetime.today().date()
        for loan in loans:
            # print(loan.loan_trans.all())
            paid = self.paid_amount(loan)

            # pp = loan.amount_approved
            pp = get_principal(loan, self.request)
            # pp = loan.loan_trans.filter()
            start_date = loan.schedule_start
            rate = loan.rate / 100
            if loan.rate_type == 2:
                amo = reducing_amortization_aging_report(pp, rate, loan.approved_duration,
                                                         loan=loan.id, formula=loan.formula)
            else:
                amo = flat_amortization_aging_report(pp, rate, loan.approved_duration,
                                                     loan=loan.id, formula=loan.formula)
            for i, deloan in enumerate(amo):
                payable_per_schedule = deloan['Payment']
                paid_here = min(paid, payable_per_schedule)
                paid -= paid_here
                schedule_balance = payable_per_schedule - paid_here
                payment_date = find_end_date(loan.interval, start_date=start_date).gen_end_date(i + 1)
                # print('pay_day', payment_date)
                over_due = current_today - payment_date
                overdue_days = None if over_due.days < 0 else over_due.days
                if overdue_days and schedule_balance > 0:
                    try:
                        officer = loan.officer.biodata.name
                    except AttributeError:
                        officer = None
                    payment_schedule.append(
                        {
                            'applicant': loan.applicant.id,
                            'client': loan.client,
                            'payment_date': payment_date,
                            'overdue_days': overdue_days,
                            'amount_outstanding': schedule_balance,
                            'loan': loan.id,
                            'officer': officer
                        }
                    )
        return payment_schedule

    def convert_to_df(self, loan_data):
        df = pd.DataFrame(loan_data)
        return df

    def get_aging_loan_report(self, loan_data):
        df = loan_data
        print('dataframe', df)
        df['payment_date'] = pd.to_datetime(df['payment_date'])

        def calculate_loan_age_0(row):
            # print('my float', row)
            if row['overdue_days'] <= 0:
                return row['amount_outstanding'] <= 0
            else:
                return 0

        def calculate_loan_age_30(row):
            if row['overdue_days'] > 0 <= 30:
                return row['amount_outstanding']
            else:
                return 0

        def calculate_loan_age_60(row):
            # print(row['overdue_days'])
            if row['overdue_days'] >= 30 <= 60:
                return row['amount_outstanding']
            else:
                return 0

        def calculate_loan_age_90(row):
            # print(row['overdue_days'])
            if row['overdue_days'] > 60 <= 90:
                return row['amount_outstanding']
            else:
                return 0

        def calculate_loan_age_over_90(row):
            # print(row['overdue_days'])
            if row['overdue_days'] > 90:
                return row['amount_outstanding']
            else:
                return 0

        # z = lambda y: print(y)'60'
        df['Current'] = df.apply(calculate_loan_age_0, axis=1)
        df['1-30'] = df.apply(calculate_loan_age_30, axis=1)
        df['31-60'] = df.apply(calculate_loan_age_60, axis=1)
        df['61-90'] = df.apply(calculate_loan_age_90, axis=1)
        df['90+'] = df.apply(calculate_loan_age_over_90, axis=1)
        ds = df.groupby(['client', 'applicant'], as_index=False)['amount_outstanding'].sum()

        return ds

    def aging_summary(self, df):
        # Calculate aging of each invoice

        # Display aging report
        aging_report = df.groupby(pd.cut(df['overdue_days'], [0, 30, 60, 90, float('inf')], right=False))[
            'amount_outstanding'].sum()
        aging_report = pd.DataFrame(aging_report).rename(columns={'amount_outstanding': 'total_amount'}).reset_index()
        aging_report['aging'] = aging_report['overdue_days'].astype(str).apply(
            lambda x: x.replace('[', '').replace(')', '').replace(',', ' -'))
        aging_report = aging_report[['aging', 'total_amount']]
        return aging_report.to_dict('records')

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        member = self.request.GET.get('member') if self.request.GET.get('member') != '' else None
        officer = self.request.GET.get('officer') if self.request.GET.get('officer') != '' else None
        print('officer')
        user = self.request.user
        bizness = biz_data(user)
        branch = branch_id(user)
        officers = Staff.objects.filter(branch_id=branch, is_active=True)
        print('de officers', officers)
        ab = AccountBroker.objects.filter(the_account=OuterRef('pk'), business=bizness)  # get account broker
        summary = []
        owner_name = ab.values('members__biodata__name')[:1]
        member_account = MemberAccount.objects.all().annotate(owner=Subquery(owner_name))
        owner = member_account.filter(members=OuterRef('pk')).values('owner')[:1]
        loans = Loans.objects.filter(loan_status=3, branch=branch).annotate(
            interest_rate=F('loan_type__interest_rate')).select_related(
            'applicant').prefetch_related('loan_trans').annotate(client=Subquery(owner)).annotate(
            formula=F('loan_type__formula'))
        this_member = loans.annotate(name=F('applicant__biodata__name')).values('applicant_id', 'name')
        if member is not None or officer is not None:
            loans = Loans.objects.filter(Q(applicant_id=member) | Q(officer_id=officer), loan_status=3,
                                         branch=branch).annotate(
                interest_rate=F('loan_type__interest_rate')).select_related(
                'applicant').prefetch_related('loan_trans').annotate(
                client=Subquery(owner)).annotate(formula=F('loan_type__formula'))

        aging_loans = sorted(self.get_aging_loans(loans), key=lambda x: x['overdue_days'])

        aging_dataframe = self.convert_to_df(aging_loans)
        ind = []
        if not aging_dataframe.empty:
            ind = self.get_aging_loan_report(aging_dataframe)
            ind = ind.to_dict(orient='records')
            summary = self.aging_summary(aging_dataframe)
            aging_dataframe['payment_date'] = aging_dataframe['payment_date'].astype(str)
        # ind = aging_dataframe.groupby('client')
        # print(json.dumps({client: group.to_dict('list') for client, group in ind}))
        context['title'] = 'Aging Reports'
        context['aging_summary'] = summary
        context['loans_summary'] = aging_loans
        context['clients'] = ind
        context['members'] = this_member
        context['officers'] = officers
        return context


@method_decorator(login_required, name='dispatch')
class AgingReportsV2(View):
    template_name = 'loans/loan_aging_report_v2.html'

    def get(self, request, *args, **kwargs):
        business_context = get_business_loans_context_data(request)
        business_filter = business_context['business_filter']

        if business_context == '':
            title = 'Aging report for all branches'
        else:
            the_current_branch_name = business_context['the_current_branch_name']
            title = f'Aging report for {the_current_branch_name}'
        member = self.request.GET.get('member') if self.request.GET.get('member') != '' else None
        officer = self.request.GET.get('officer') if self.request.GET.get('officer') != '' else None
        the_data = get_aging_report_data_v2(request, member, officer)

        # pagination
        page = request.GET.get('page', 1)
        the_active_tab = request.GET.get('the_active_tab', 'home')
        # ----------- individual arrears ---------------------------
        if the_active_tab == 'ind_summ':
            page_ind_summary = page
        else:
            page_ind_summary = 1
        if not isinstance(the_data['df_individual_arrears'], str):
            pg_df_individual_arrears = the_data['df_individual_arrears'].to_dict('records')
        else:
            pg_df_individual_arrears =[]

        # print(pg_df_individual_arrears[0])
        paginator = Paginator(pg_df_individual_arrears, 10)

        try:
            pg_df_individual_arrears = paginator.page(page_ind_summary)
        except PageNotAnInteger:
            pg_df_individual_arrears = paginator.page(1)
        except EmptyPage:
            pg_df_individual_arrears = paginator.page(paginator.num_pages)

        # update the the_data
        the_data['df_individual_arrears'] = pg_df_individual_arrears

        # ----------------- df_overdue_payments --------------------
        if the_active_tab == 'home':
            page_ind_payments = page
        else:
            page_ind_payments = 1
        the_df_overdue_payments = the_data['df_overdue_payments']
        if not isinstance(the_df_overdue_payments, str):
            the_df_overdue_payments = the_df_overdue_payments.explode('overdue_payments')
            # Convert each entry in overdue_payments (dicts) into separate columns
            payments_df = pd.json_normalize(the_df_overdue_payments['overdue_payments'])
            # Merge the flattened payments with the parent DataFrame
            merged_df = pd.concat([the_df_overdue_payments.reset_index(drop=True), payments_df], axis=1)
            flattened_data = merged_df.to_dict('records')
        else:
            the_df_overdue_payments = pd.DataFrame(columns=['column1', 'column2', 'column3'])
            flattened_data = []


        # print('LENGTH',len(flattened_data))
        paginator = Paginator(flattened_data, 10)

        try:
            pg_df_overdue_payments = paginator.page(page_ind_payments)
        except PageNotAnInteger:
            pg_df_overdue_payments = paginator.page(1)
        except EmptyPage:
            pg_df_overdue_payments = paginator.page(paginator.num_pages)

        # update the the_data
        the_data['df_overdue_payments'] = pg_df_overdue_payments


        # members and Officers
        officers = Staff.objects.filter(branch_id=branch_id(request.user), is_active=True)
        members = Member.objects.filter(branch_id=branch_id(request.user), is_active=True)

        # selected member
        selected_member = None
        if member is not None:
            selected_member = int(member)

        # selected staff
        selected_staff = None
        if officer is not None:
            selected_officer = int(officer)

        if member is None:
            member=''
        if officer is None:
            officer=''
        return render(request, self.template_name, locals())


class AgentsDisburseReportView(PermView):
    perm_name = "view_all_loan_reports"
    template_name = "loans/agents_disbursement_report.html"

    def get(self, request, *args, **kwargs):
        business_context = get_business_loans_context_data(request)
        business_filter = business_context['business_filter']

        if business_context['the_current_branch_name'] == 'all branches':
            title = "Loan Officers Disbursement report for all branches"
        else:
            the_current_branch_name = business_context['the_current_branch_name']
            title = f"Loan Officers Disbursement report for {the_current_branch_name}"

        all_staff = Staff.objects.filter(branch__business_id__in=business_context['data_context'])
        staff_data = []
        for staff in all_staff:
            staff_loans = Loans.objects.filter(officer=staff, loan_status__in=[3, 4, 8]).exclude(
                loan_summary_details=None)
            if staff_loans.exists():
                # total disbursed
                all_staff_loans = list(staff_loans.values_list('loan_summary_details', flat=True))

                df = pd.DataFrame(eval(item) for item in all_staff_loans)
                if not df.empty:
                    df = df[df['is_error'] == 'no']
                    all_loans_count = len(df)
                    total_amt_disbursed = df['principal_amt'].sum()
                    total_loans_payable = df['loan_payable'].sum()
                else:
                    all_loans_count = 0
                    total_amt_disbursed = 0
                    total_loans_payable = 0


                the_running_loans = list(
                    staff_loans.filter(loan_status=3).values_list('loan_summary_details', flat=True))
                running_df = pd.DataFrame(eval(item) for item in the_running_loans)

                if not running_df.empty:
                    running_df = running_df[running_df['is_error'] == 'no']

                    running_count = len(running_df)
                    rem_running = running_df['loan_balance'].sum()
                else:
                    running_count = len(running_df)
                    rem_running = 0

                staff_data.append({
                    'officer': staff_loans.first().officer.biodata.name,
                    'total_loans_disbursed': all_loans_count,
                    'total_amount_disbursed': total_amt_disbursed,
                    'total_running_loans': running_count,
                    'running_loan_balance': rem_running,
                    'total_loans_payable': total_loans_payable,
                    'branch': staff.branch.name
                })

        return render(request, self.template_name, locals())


class PortifolioAtRiskView(PermView):
    perm_name = "view_all_loan_reports"
    template_name = "loans/par_report.html"

    def get(self, request, *args, **kwargs):
        business_context = get_business_loans_context_data(request)
        business_filter = business_context['business_filter']

        if business_context['the_current_branch_name'] == 'all branches':
            title = "Portifolio at risk(PAR) report for all branches"
        else:
            the_current_branch_name = business_context['the_current_branch_name']
            title = f"Portifolio at risk(PAR) report for {the_current_branch_name}"


        all_staff = Staff.objects.filter(branch__business_id__in=business_context['data_context'])
        staff_data = []
        for staff in all_staff:
            staff_loans = Loans.objects.filter(officer=staff, loan_status=3)
            if staff_loans.exists():
                # total disbursed
                all_staff_loans = list(staff_loans.values_list('loan_summary_details', flat=True))
                df = pd.DataFrame(eval(item) for item in all_staff_loans)
                df_individual_arrears = df[df['is_error'] == 'no']

                df_individual_arrears['amount_due'], df_individual_arrears['last_date_found'], df_individual_arrears[
                    'total_days_overdue'] = zip(*df_individual_arrears.apply(calculate_amount_due_PAR, axis=1))
                total_principal_in_arrears = df_individual_arrears['amount_due'].sum()
                portifolio = df_individual_arrears['principal_amt'].sum()
                no_of_loans_in_arrears = (df_individual_arrears['amount_due'] > 0).sum()
                par = round(((total_principal_in_arrears / portifolio) * 100), 2)

                staff_data.append({
                    'officer': staff_loans.first().officer.biodata.name,
                    'total_principal_in_arrears': round(total_principal_in_arrears, 2),
                    'portifolio': round(portifolio, 2),
                    'no_of_loans_in_arrears': no_of_loans_in_arrears,
                    'par': par,
                    'branch': staff.branch.name
                })

        return render(request, self.template_name, locals())

        # par_report

@method_decorator(login_required, name='dispatch')
class DisbursementReport(View):
    template_name = 'loans/disbursement_report.html'

    def get(self, request, *args, **kwargs):
        business_context = get_business_loans_context_data(request)
        business_filter = business_context['business_filter']

        if business_context['the_current_branch_name'] == 'all branches':
            title = "Disbursement report for all branches"
        else:
            the_current_branch_name = business_context['the_current_branch_name']
            title = f"Disbursement report for {the_current_branch_name}"
        # title = 'Disbursement report'
        start_date_str = request.GET.get('start_date', '')
        end_date_str = request.GET.get('end_date', '')
        officer = request.GET.get('officer', '')

        start_date = datetime.datetime.strptime(start_date_str, '%Y-%m-%d').date() if start_date_str else None
        end_date = datetime.datetime.strptime(end_date_str, '%Y-%m-%d').date() if end_date_str else None
        #get all loans with status above 3
        all_loans = Loans.objects.filter(branch__business_id__in=business_context['data_context'], loan_status__in=[3,4, 8, 7,6])
        if start_date:
            all_loans = all_loans.filter(schedule_start__gte=start_date)
        if end_date:
            all_loans = all_loans.filter(schedule_start__lte=end_date)
        
        officer_name = ''
        if officer:
            officer_name = Staff.objects.get(id=officer).biodata.name
            all_loans = all_loans.filter(officer_id=officer)


        total_amt = all_loans.aggregate(given_out = Sum('amount_approved'))['given_out']
        if total_amt is None:
            total_amt = 0
        total_count = all_loans.count()
        all_staff = Staff.objects.filter(branch__business_id__in=business_context['data_context'])

        # Pagination
        page = request.GET.get('page', 1)
        paginator = Paginator(all_loans, 10)

        try:
            all_loans = paginator.page(page)
        except PageNotAnInteger:
            all_loans = paginator.page(1)
        except EmptyPage:
            all_loans = paginator.page(paginator.num_pages)

        return render(request, self.template_name, locals())


@method_decorator(login_required, name='dispatch')
class SingleLoanAgingDataView(View):

    def get(self, request, pk, *args, **kwargs):
        the_loan = Loans.objects.filter(id=pk)
        if not the_loan.exists():
            return HttpResponse(f"error|loan nolonger exists")

        the_loan = the_loan.first()
        # first update the loan details summary
        if the_loan.loan_status > 2:
            update_loan_summary_details(the_loan)
        # get the schedules - determine the next schedule for payment
        updated_loan = Loans.objects.get(id=the_loan.id)
        loan_summarised_details = updated_loan.loan_summary_details
        if loan_summarised_details is not None and loan_summarised_details != '':
            loan_summarised_details = json.loads(loan_summarised_details)
            # check for error
            if loan_summarised_details['is_error'] == 'no':
                # get schedules
                the_schedules = loan_summarised_details['ammotization']
                first_balance_greater_than_zero = next(
                    (entry for entry in the_schedules if entry["schedule_balance"] > 0), None
                )
                first_balance_greater_than_zero.pop('schedule_date')

                data_summary_list = [loan_summarised_details]
                df_individual_arrears = pd.DataFrame(data_summary_list)
                df_individual_arrears['amount_due'], df_individual_arrears['last_date_found'], df_individual_arrears[
                    'total_days_overdue'] = zip(*df_individual_arrears.apply(calculate_amount_due_v2, axis=1))

                the_arrears_dict = df_individual_arrears.to_dict(orient="records")[0]
                arrears_data = json.dumps({
                    'amount_due': the_arrears_dict['amount_due'],
                    # 'last_date_found': the_arrears_dict['last_date_found'],
                    'total_days_overdue': the_arrears_dict['total_days_overdue'],

                })
                current_schedule_string = json.dumps(first_balance_greater_than_zero)
                return HttpResponse(f"ok|{current_schedule_string}|{arrears_data}")


            else:
                return HttpResponse(f"error|loan has unresolved issue.Please contact support")
        else:
            return HttpResponse(f"error|loan has unresolved issue.Please contact support")

        # get the total amount in arrears