from django.contrib import messages
from django.contrib.auth.decorators import login_required
from django.http import HttpResponseRedirect
from django.shortcuts import render
from django.urls import reverse_lazy
from django.utils.decorators import method_decorator
from django.views import View
from django.views.generic import ListView
from django.views.generic.edit import FormView

from loans.forms.settings import LoanSettingsForm
from loans.models import Loans, LoanTopups
from sacco.models import LoanSettings
from sacco.utils import businessdata


@method_decorator(login_required, name='dispatch')
class LoanSettingsView(FormView):
    template_name = 'new_loans/loan_settings.html'
    form_class = LoanSettingsForm
    success_url = reverse_lazy('new_loans_settings')

    def get_form_kwargs(self):
        kwargs = super(LoanSettingsView, self).get_form_kwargs()
        kwargs['request'] = self.request
        return kwargs

    def get_context_data(self, **kwargs):
        context = super(LoanSettingsView, self).get_context_data()

        title = 'Loan settings'
        loan_settings = LoanSettings.objects.filter(business=businessdata(self.request)).first()
        context['title'] = title
        context['loan_settings'] = loan_settings
        return context

    def form_valid(self, form):
        form.save()
        messages.success(self.request, 'success', extra_tags='Loan settings updated successfully')
        return super().form_valid(form)
    def form_invalid(self, form):
        print('invalid')
        return super().form_invalid(form)


@method_decorator(login_required, name='dispatch')
class AppliedLoansView(ListView):
    template_name = 'new_loans/applied_loans.html'
    context_object_name = 'loan_applications'
    model = Loans

    def get_queryset(self):
        query = self.request.GET.get('filter')
        queryset = Loans.objects.filter(loan_status__lte=Loans.LOAN_STATUS.APPROVED,
                                        branch__business=businessdata(self.request))
        if query and query != "all":
            queryset = Loans.objects.filter(branch__business=businessdata(self.request), loan_status=query)
        return queryset

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        query = self.request.GET.get('filter')
        title = 'Loan Applications'
        if query and query == '0':
            title = 'Pending Loan application'
        elif query and query == '1':
            title = 'Loan application submitted to the committee for action'
        elif query and query == '2':
            title = 'Approved Loan application pending disbursement'
        context['title'] = title
        return context


class DeleteLoanView(View):
    def post(self):
        pk = self.kwargs['pk']
        return HttpResponseRedirect(reverse_lazy('', args=[pk]))


@method_decorator(login_required, name='dispatch')
class ClearedLoans(View):
    template_name = 'new_loans/cleared_loans.html'

    def get(self, request, *args, **kwargs):
        title = 'Cleared Applications'
        settings = LoanSettings.objects.filter(business=businessdata(request)).first()
        decimals = settings.decimals
        cleared_and_closed_loans = Loans.objects.filter(branch__business=businessdata(request), loan_status=4)
        return render(request, self.template_name, locals())


# running loans
@method_decorator(login_required, name='dispatch')
class RunningLoans(ListView):
    template_name = 'new_loans/running_loans.html'
    context_object_name = 'running_loans'

    def get_queryset(self):
        running_loans = Loans.objects.filter(branch__business=businessdata(self.request), loan_status=3)
        return running_loans

    def get_context_data(self, *args, **kwargs):
        context = super().get_context_data(**kwargs)
        settings = LoanSettings.objects.filter(business=businessdata(self.request)).first()
        decimals = settings.decimals
        context['title'] = 'Running Loans'
        context['decimals'] = decimals
        context['settings'] = settings
        return context

@method_decorator(login_required, name='dispatch')
class TopupReport(ListView):
    template_name = 'new_loans/loan_topups.html'
    context_object_name = 'loan_topups'

    def get_queryset(self):
        loan_topups = LoanTopups.objects.filter(branch__business=businessdata(self.request), approved=True)
        return loan_topups

    def get_context_data(self, *args, **kwargs):
        context = super().get_context_data(**kwargs)
        context['title'] = 'Loan topups'
        return context