import datetime
import json
import traceback

from django.contrib import messages
from django.contrib.auth.decorators import login_required
from django.db import transaction
from django.db.models import F, Q, Sum, FloatField, OuterRef, Subquery
from django.db.models.functions import Coalesce
from django.http import HttpResponseRedirect, JsonResponse
from django.middleware import csrf
from django.shortcuts import get_object_or_404
from django.template.loader import render_to_string
from django.urls import reverse_lazy, reverse
from django.utils.decorators import method_decorator
from django.views import View
from django.views.generic import FormView, DetailView

from accounts.models import Business
from accounts.permission_mixin import BusinessUserMixin
from commons.logger import logger
from loans.forms.loan_application import LoanApplicationForm, CommitteeActionForm, DisburseLoanForm, AddGuarantorForm, \
    EditLoanForm, DeleteAdjustmentForm
from loans.models import IntervalChoices, LoanTypes, Loans, ApplicationAccountOrLoanType, LoanUpload, \
    Loanguarantors, Loansecurities
from loans.utils.amortisation import loan_payment_details
from loans.utils.others import convert_to_year_month, convert_to_month_week, convert_to_week_days
from sacco.models import AccountBroker, LoanSettings, OtherSettings, Member, NotiSettings
from sacco.utils import biz_id, biz_staff, biz_data, checkAndSendMessage, sendTransEmail, \
    branch_id
from transactions.models import Transactions, Account, AccountCategory


@method_decorator(login_required, name='dispatch')
class ApplyLoanView(FormView):
    template_name = 'new_loans/apply_loan.html'
    form_class = LoanApplicationForm
    success_url = reverse_lazy('new_applied_loans')

    def get_form_kwargs(self):
        kwargs = super(ApplyLoanView, self).get_form_kwargs()
        kwargs['request'] = self.request
        return kwargs

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        interval_choices = [{'value': choice[0], 'label': choice[1]} for choice in IntervalChoices.choices]
        members = AccountBroker.objects.filter(members__biodata__business=biz_id(self.request.user)).annotate(
            account_id=F('the_account_id')).annotate(
            name=F('members__biodata__name')).annotate(
            contact=F('members__biodata__contact')).annotate(
            account_type=F('the_account__account_type__name')).annotate(
            account=F('the_account')
        )
        context['title'] = 'Loan Application'
        context['intervals'] = interval_choices
        context['members'] = members
        context['loan_types'] = LoanTypes.objects.filter(business=biz_id(self.request.user), is_active=True)
        return context

    def form_valid(self, form):
        form.save()
        messages.success(self.request, 'success', extra_tags='Loan Application submitted successfully')
        return super().form_valid(form)

    def form_invalid(self, form):
        print('invalid')
        return super().form_invalid(form)


@method_decorator(login_required, name='dispatch')
class LoanDetailView(BusinessUserMixin, DetailView):
    template_name = 'new_loans/loan_details.html'
    model = Loans
    context_object_name = 'detail'

    def get_queryset(self):
        # l = Loans.objects.filter().select_related('loan_type_id', 'account').annotate().values()
        return super().get_queryset().select_related('loan_type').prefetch_related(
            'docs', 'securities', 'loan_trans', 'guarantors', 'loan_charges')

    def charges(self, obj) -> [dict]:
        gen_cgs = obj.loan_charges.all().select_related('charge')
        actualcharges = []
        for charge in gen_cgs:
            actualcharges.append({"name": charge.charge.charge, "amount": charge.amount})
        return actualcharges

    def paid_amount(self, obj) -> float:
        paid = obj.loan_trans.filter(
            Q(transaction_type='Loan repayment') | Q(transaction_type='Loan interest')
        ).aggregate(
            paid=Coalesce(Sum('reporting_amount'), 0.0, output_field=FloatField()))['paid']
        return paid

    def principal_paid(self, obj) -> float:
        paid = obj.loan_trans.filter(transaction_type='Loan repayment').aggregate(
            paid=Coalesce(Sum('reporting_amount'), 0.0, output_field=FloatField()))['paid']
        return paid

    def receivable(self, obj):
        expected = obj.loan_trans.filter(transaction_type='give loan').aggregate(
            paid=Coalesce(Sum('reporting_amount'), 0.0, output_field=FloatField()))['paid']
        return expected

    def check_charges(self, charges) -> bool:
        if len(charges) > 0:
            return True
        return False

    def get_disbursable(self, obj, totalcharges) -> float:
        approved_amount = obj.amount_approved
        disbusable = approved_amount
        if obj.loan_status < Loans.LOAN_STATUS.APPROVED:
            if not obj.charges_paid and obj.is_topup:
                disbusable = approved_amount - totalcharges
            elif obj.charges_paid and obj.is_topup:
                disbusable = approved_amount
            elif not obj.charges_paid and obj.is_topup:
                disbusable = approved_amount - totalcharges
            elif obj.charges_paid and obj.is_topup:
                disbusable = approved_amount - totalcharges
            elif not obj.charges_paid and obj.is_topup:
                disbusable = approved_amount - totalcharges
        else:
            if not obj.charges_paid and obj.is_topup:
                disbusable = approved_amount - 0
            elif not obj.charges_paid and not obj.is_topup:
                disbusable = approved_amount
            elif obj.charges_paid and obj.is_topup:
                disbusable = approved_amount - 0
            elif obj.charges_paid and not obj.is_topup:
                disbusable = approved_amount

        return disbusable

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        http = 'http://' if not self.request.is_secure() else 'https://'
        baseUrl = self.request.headers['HOST']
        base_url = http + baseUrl
        loan = self.get_object()
        de_biz = biz_id(self.request.user)
        loan_settings = LoanSettings.objects.filter(business=de_biz).first()
        docs = loan.docs.all()
        must_upload = True

        if loan_settings.upload_minutes and docs.count() < 1:
            must_upload = False
        recevables_account = Account.objects.filter(name='Loan Receivables', business=de_biz)[0]

        credits = Transactions.objects.filter(Q(loan=loan, account_cr=recevables_account.id),
                                              ~Q(transaction_type='Closing Loan')).aggregate(
            principlecredits=Sum('reporting_amount', default=0.0))['principlecredits']

        debits = Transactions.objects.filter(loan_id=loan, account_dr=recevables_account.id).aggregate(
            principledebits=Sum('reporting_amount', default=0.0))['principledebits']
        if credits is None:
            principalbal = debits
        else:
            principalbal = float(debits) - float(credits)
        principalbal = float(principalbal)

        charges = self.charges(loan)
        total_charge = sum(item["amount"] for item in charges)
        rate = loan.rate / 100
        disbursed = 0
        payable = 0
        sum_interest = 0
        cleared_percent = 0
        securities = loan.securities
        applicant_savings = loan.account.balance or 0.0
        share_price = OtherSettings.objects.filter(business=de_biz).first().share_price
        member_shares = (loan.applicant.shares * share_price) or 0.0
        needed_value = int(loan.loan_type.security_ratio) / 100 * int(loan.amount_requested)
        total_security = securities.aggregate(
            total=Coalesce(Sum('value'), 0.0, output_field=FloatField()))['total']
        nsecurities = securities.count()
        nguarantors = loan.guarantors.count()
        use_shares = loan_settings.use_shares
        requested_duration = f'{loan.requested_duration} {loan.loan_type.formula}'
        approved_duration = f'{loan.approved_duration} {loan.loan_type.formula}'
        paidamount = self.paid_amount(loan)
        remaining_percent = 0
        final_payment = None

        if loan.loan_type.formula == LoanTypes.LOAN_FORMULA.PER_ANNAM and loan.interval == IntervalChoices.YEARS:
            requested_duration = convert_to_year_month(loan.requested_duration)
            if loan.loan_status >= 2:
                approved_duration = convert_to_year_month(loan.approved_duration)

        if loan.loan_type.formula == LoanTypes.LOAN_FORMULA.PER_ANNAM and loan.interval == IntervalChoices.MONTHS:
            requested_duration = convert_to_year_month(loan.requested_duration)
            if loan.loan_status >= 2:
                approved_duration = convert_to_year_month(loan.approved_duration)

        elif loan.loan_type.formula == LoanTypes.LOAN_FORMULA.PER_MONTH and loan.interval == IntervalChoices.YEARS:
            requested_duration = convert_to_month_week(loan.requested_duration)
            if loan.loan_status >= 2:
                approved_duration = convert_to_month_week(loan.approved_duration)

        elif loan.loan_type.formula == LoanTypes.LOAN_FORMULA.PER_MONTH and loan.interval == IntervalChoices.MONTHS:
            requested_duration = convert_to_month_week(loan.requested_duration)
            if loan.loan_status >= 2:
                approved_duration = convert_to_month_week(loan.approved_duration)

        elif loan.loan_type.formula == LoanTypes.LOAN_FORMULA.PER_MONTH and loan.interval == IntervalChoices.DAYS:
            requested_duration = convert_to_month_week(loan.requested_duration)
            if loan.loan_status >= 2:
                approved_duration = convert_to_month_week(loan.approved_duration)

        elif loan.loan_type.formula == LoanTypes.LOAN_FORMULA.PER_WEEK and loan.interval == IntervalChoices.WEEKS:
            requested_duration = convert_to_week_days(loan.requested_duration)
            if loan.loan_status >= 2:
                approved_duration = convert_to_week_days(loan.approved_duration)

        elif loan.loan_type.formula == LoanTypes.LOAN_FORMULA.PER_WEEK and loan.interval == IntervalChoices.DAYS:
            requested_duration = convert_to_week_days(loan.requested_duration)
            if loan.loan_status >= 2:
                approved_duration = convert_to_week_days(loan.approved_duration)

        if use_shares:
            total_security_value = total_security + applicant_savings + member_shares
        else:
            total_security_value = total_security + applicant_savings
        if loan.loan_status > Loans.LOAN_STATUS.APPROVED:
            # duration = loan.approved_duration if loan.loan_status.
            # rate, amount, period, schedule, loan_sched,
            amortization = loan_payment_details(loan.rate_type, rate, loan.amount_approved, loan.approved_duration,
                                                loan.interval, loan.loan_type.formula, acc=loan.account, loan=loan)
            schedules = amortization.gen_schedules()
            payable = f"{schedules['installment'].sum():.2f}"
            sum_interest = schedules['interest_paid'].sum() if schedules is not None else 0
            disbursed = self.get_disbursable(loan, total_charge)
            # get loan balance
            loan_balance = float(payable) - paidamount

            cleared_percent = (paidamount / float(payable)) * 100
            remaining_percent = (loan_balance / float(payable)) * 100
            final_payment = schedules.iloc[-1]['payment_date']
        elif loan.loan_status == Loans.LOAN_STATUS.APPROVED:
            disbursed = loan.amount_approved
        members_accts = Member.objects.filter(branch__business_id=de_biz)
        print(members_accts.count())
        bankaccounts = Account.objects.filter(Q(business=de_biz),
                                              Q(category__name='Bank') | Q(category__name='Mobile Money') | Q(
                                                  category__name='Cash')).values()
        loantypes = LoanTypes.objects.filter(business=de_biz, is_active=True).values()

        title = 'Loan application details'
        if loan.is_topup and int(loan.loan_status) < 3:
            title = 'Loan topup application details'
        context['totaladjustments'] = loan.loan_trans.filter(transaction_type='Loan Adjustment').aggregate(
            adjustments=Coalesce(Sum('reporting_amount', output_field=FloatField()), 0.00))['adjustments']
        context['title'] = title
        context['applicant'] = loan.applicant.biodata.name
        context['minguarantors'] = loan_settings.minimum_guarantors
        context['decimals'] = loan_settings.decimals
        context['top_up_limit'] = loan_settings.top_up_limit
        context['min_approvals'] = loan_settings.min_approvals
        context['nonemember'] = loan_settings.allow_others
        context['use_shares'] = loan_settings.use_shares
        context['actualcharges'] = charges
        context['totalcharges'] = total_charge
        context['disbusable'] = disbursed
        context['paidamount'] = paidamount
        context['totalpayable'] = payable
        context['totalinterest'] = sum_interest
        context['must_upload'] = must_upload
        context['nsecurities'] = nsecurities
        context['nguarantors'] = nguarantors
        context['total_security_value'] = total_security_value
        context['needed_value'] = needed_value
        context['base_url'] = base_url
        context['members_accts'] = members_accts
        context['docs'] = docs
        context['loantypes'] = loantypes
        context['bankaccounts'] = bankaccounts
        context['has_charges'] = self.check_charges(charges)
        context['requested_duration'] = requested_duration
        context['approved_duration'] = approved_duration
        context['cleared_percent'] = cleared_percent
        context['remaining_percent'] = remaining_percent
        context['receivable'] = self.receivable(loan)
        context['principal_paid'] = self.principal_paid(loan)
        context['final_payment'] = final_payment
        context['principalbal'] = principalbal
        return context


@method_decorator(login_required, name='dispatch')
class EditLoanView(BusinessUserMixin, View):

    def post(self, request, pk):
        loan = get_object_or_404(Loans, id=pk)
        user = self.request.user
        try:
            form = EditLoanForm(data=self.request.POST, loan=loan, user=user)
            if form.is_valid():
                form.save()
                messages.success(self.request, 'success', extra_tags='Successfully updated Loan')
        except Exception as _e:
            messages.error(self.request, 'error', extra_tags='Failed to make update')
            logger.error(str(_e))
        return HttpResponseRedirect(reverse('new_loan_details', args=[pk]))


@method_decorator(login_required, name='dispatch')
class CommitteeActionView(FormView):
    template_name = 'new_loans/loan_details.html'
    form_class = CommitteeActionForm

    def get_loan(self, pk):
        loan = get_object_or_404(Loans, id=pk)
        return loan

    def get_form_kwargs(self):
        kwargs = super(CommitteeActionView, self).get_form_kwargs()
        kwargs['request'] = self.request
        return kwargs

    def form_valid(self, form):
        pk = self.kwargs['pk']
        instance = self.get_loan(pk)
        comaction = form.cleaned_data['comaction']
        amounts_approved = form.cleaned_data['a_approved']
        approved_rate = form.cleaned_data['approved_rate']
        approved_date = form.cleaned_data['approved_date']
        approved_duration = form.cleaned_data['approved_duration']
        print('my', comaction)
        if comaction == 1:  # Approve
            if instance.is_topup is True:
                try:
                    old_loan = Loans.objects.filter(account_id=instance.account_id, loan_status=3)[0]
                    old_loan.loan_status = 8  # ToopedUp
                    old_loan.save()
                except IndexError:
                    messages.error(self.request, 'Its not a valid Topup')

                #     =================SAVE IN THE TOPUPS TABLE========================

            charges = ApplicationAccountOrLoanType.objects.filter(loan_type_id=instance.loan_type,
                                                                  general_charge__status=1,
                                                                  general_charge__application="l")
            actualcharges = []
            totalcharges = 00
            for charge in charges:
                ispercentage = charge.general_charge.is_percentage
                if ispercentage == 1:
                    chargeamount = (float(charge.general_charge.amount) * float(amounts_approved)) / 100

                else:
                    chargeamount = charge.general_charge.amount

                actualcharges.append({"name": charge.general_charge.charge, "amount": chargeamount})

                totalcharges = totalcharges + chargeamount
            instance.loan_status = 2
            instance.charges = json.dumps(actualcharges)
            instance.approved_duration = approved_duration
            instance.rate = approved_rate
            instance.amount_approved = amounts_approved
            instance.approved_on = approved_date
            instance.save()
            # form.save(instance=self.get_loan(pk))
            messages.success(self.request, 'success',
                             extra_tags=f'Loan application of {amounts_approved} has been approved successfully')
            if comaction == 2:
                instance.loan_status = 5
                instance.save()

                messages.success(self.request, 'success',
                                 extra_tags=f'Loan application of {instance.amount_requested} rejected successfully because the applicant failed to meet the minimum requirements')
                return HttpResponseRedirect(reverse_lazy('new_applied_loans'))

                # ===================== APPPLICATION IS BOUNCED=====================
            if comaction == 3:
                instance.loan_status = 0
                messages.success(self.request, 'success',
                                 extra_tags=f'Loan application of {instance.amount_requested} recalled for a fresh appraisal')
                return HttpResponseRedirect(reverse_lazy('new_applied_loans'))
        return HttpResponseRedirect(reverse('new_loan_details', args=[pk]))

    def form_invalid(self, form):
        pk = self.kwargs['pk']
        messages.error(self.request, 'error', extra_tags='Error in form Submission')
        return HttpResponseRedirect(reverse('new_loan_details', args=[pk]))


@method_decorator(login_required, name='dispatch')
class SubmitApproval(View):
    def post(self, request, pk):
        loan = get_object_or_404(Loans, id=pk)
        try:
            loan.loan_status = Loans.LOAN_STATUS.APPRAISED
            loan.save()
            messages.success(self.request, 'success', extra_tags='Submitted successfully')
            return HttpResponseRedirect(reverse_lazy('new_loan_details', args=[pk]))
        except AttributeError as e:
            logger.error(str(e))
        except (Exception,) as e:
            logger.error(str(e))
        return HttpResponseRedirect(reverse_lazy('new_loan_details', args=[pk]))


@method_decorator(login_required, name='dispatch')
class DisburseLoanView(FormView):
    form_class = DisburseLoanForm
    template_name = 'new_loans/loan_details.html'

    @transaction.atomic
    def form_valid(self, form):
        pk = self.kwargs['pk']
        amount_approved = form.cleaned_data.get('damount')
        ddate = form.cleaned_data.get('d_date')
        credited = form.cleaned_data.get('credited')
        narration = form.cleaned_data.get('narration')
        chargespaid = form.cleaned_data.get('chargespaid')
        istopup = form.cleaned_data.get('istopup')
        oldloanid = form.cleaned_data.get('oldloanid')
        oldbal = form.cleaned_data.get('oldbalance')
        chargesaccount = form.cleaned_data.get('chargesaccoun')
        rec_no = datetime.datetime.now().strftime("%y%m%d%H%M%S")
        voucher = form.cleaned_data.get('voucher', rec_no)
        loan = Loans.objects.prefetch_related('guarantors', 'branch', 'loan_trans', 'loan_charges').get(id=pk)
        user = self.request.user
        business = biz_data(user)
        staff = biz_staff(user)
        branch = branch_id(user)
        chargesp = chargespaid

        if istopup is True or istopup == 'True':
            istopup = True
        # ================== IF THE CHARGES ARE PAID BEFORE DISBURSEMENT========================

        # ===== Check if the ledgers exist =====
        checkrecievables = Account.objects.filter(name='Loan Receivables', business=business).exists()
        checkrecharges = Account.objects.filter(name='Loan Charges', business=business).exists()
        Account.objects.filter(name='Loan Fines', business=business).exists()

        if not checkrecharges:
            rcategeory2 = AccountCategory.objects.filter(name='Loan incomes', business=business)[0]
            Account.objects.create(name='Loan Charges', business_id=business,
                                                  category_id=rcategeory2.id, added_by=staff)

        if not checkrecievables:
            rcategeory = AccountCategory.objects.filter(name='Account Receivables', business=business)[0]
            Account.objects.create(name='Loan Receivables', business=business,
                                                 category_id=rcategeory.id, added_by=staff)

        # =====================GET LEDGER ACCOUNTS============================
        laonacct = Account.objects.filter(name='Loan Receivables', business=business)[0]
        reserve = Account.objects.filter(name='Opening Reserves', business=business)[0]
        chargeacct = Account.objects.filter(name='Loan Charges', business=business)[0]

        # print('CHARGES Are ',chargespaid)
        if int(chargespaid) == 1:
            chargesp = 1
            if chargesaccount is None:
                messages.error(self.request, 'error', extra_tags="Charges account not selected")

        # ================IF CHARGES ARE TOPPED UP ON THE LOAN=================================
        elif int(chargespaid) == 2:
            chargesp = 2
            chargesaccount = laonacct.id

        # =================== IF THE CHARGES RE DEDUCTED FROM THE LOAN=========================
        elif int(chargespaid) == 3:
            chargesp = 2
            chargesaccount = laonacct.id
        # =================== =MemberAccount=========================
        elif int(chargespaid) == 4:
            memberledger = Account.objects.filter(member_id=loan.account_id)[0]
            chargesaccount = memberledger.id

            chargesp = 1
        if chargesp == 1:
            chargesp = True
        else:
            chargesp = False

        loan.loan_status = 3  # Disbursed
        loan.charges_paid = chargesp
        loan.schedule_start = ddate
        loan.voucher = voucher
        loan.save()

        # ==========================Save loan transaction=======================
        # determine the loan amount basing
        loan_report_amt = amount_approved
        if chargespaid == 2:
            loan_report_amt = loan.amount_approved

        Transactions.objects.create(reporting_amount=loan_report_amt, narration=narration,
                                    account_cr_id=credited, account_dr_id=laonacct.id, tx_date=ddate,
                                    loan=loan, transaction_type='give loan',
                                    branch_id=branch)

        if istopup is True:
            # ======================OLD BALANCE TRANSACTIONS ====================================
            # =========================CREDITING  IT ON THE NEW LOAN==================
            Transactions.objects.create(reporting_amount=oldbal,
                                                        narration='Closing The load due to topup',
                                                        account_cr_id=laonacct.id, account_dr_id=reserve.id,
                                                        tx_date=ddate,
                                                        loan_id=oldloanid, transaction_type='Closing Loan',
                                                        branch_id=branch)

            Transactions.objects.create(reporting_amount=oldbal,
                                                        narration='Closing The load due to topup',
                                                        account_cr_id=reserve.id, account_dr_id=laonacct.id,
                                                        tx_date=ddate,
                                                        loan=loan, transaction_type='give loan',
                                                        branch_id=branch)

        actualchargesx = loan.charges
        actualcharges = json.loads(actualchargesx)

        for chg in actualcharges:
            chargeamount = chg['amount']
            chargename = chg['name']
            narration = chargename
            chargedebit = chargesaccount

            # try:
            Transactions.objects.create(reporting_amount=chargeamount,
                                                     transaction_type='Loan charge',
                                                     narration=narration,
                                                     account_cr_id=chargeacct.id,
                                                     account_dr_id=chargedebit,
                                                     tx_date=ddate,
                                                     receipt=rec_no,
                                                     reference=voucher,
                                                     loan=loan,
                                                     branch_id=branch)

        notify_type = NotiSettings.objects.filter(business=biz_data(self.request.user)).first()
        template_file_link = 'sacco/emails/transactional.html'
        bizz_id = biz_id(self.request.user)
        this_biz = Business.objects.filter(id=bizz_id).first()
        # get guarantors
        gs = loan.guarantors.annotate(name=F('biodata__name')
                                      ).annotate(contact=F('biodata__contact')
                                                 ).annotate(email=F('biodata__email')
                                                            ).values('name', 'email', 'contact')
        for guarantorxx in gs:
            html_body = f"<p>Dear {guarantorxx['name']},</p> <p>This is to inform you that the loan requested by {loan.applicant.biodata.name} of {loan.amount_approved} has been disbursed. <br>We hope that you have a good experience with our services and <b>{this_biz.name}</b> will do it’s best to please you. We will be looked forward to serving you in adherence to our quality standards.</p><p>Thanking you and with profound regards.</p>"
            message = f"Dear {0},\nThis is to inform you that the loan requested by {loan.applicant.biodata.name} of {loan.amount_approved} has been disbursed"

            if guarantorxx['contact']:
                if notify_type.notification_type == 1:  # Send by SMS only
                    checkAndSendMessage(self.request.user, message, guarantorxx['contact'],
                                        guarantorxx['name'])

            elif notify_type.notification_type == 2:  # Send by Email only
                if guarantorxx['email']:
                    sendTransEmail(template_file_link, html_body, this_biz.name, 'Account Opening Successful',
                                   guarantorxx['email'])

            elif notify_type.notification_type == 3:  # Send by both SMS and Email
                if guarantorxx['contact']:
                    checkAndSendMessage(self.request.user, message, guarantorxx['contact'],
                                        guarantorxx['name'])

                if guarantorxx['email']:
                    sendTransEmail(template_file_link, html_body, this_biz.name, 'Account Opening Successful',
                                   guarantorxx['email'])

        messages.success(self.request, 'success',
                         extra_tags=f'Loan amount of {loan.amount_approved} has been disbursed successfully')

        # =====================================================DISBURSE LOAN===================================

        return HttpResponseRedirect(reverse('new_loan_details', args=[pk]))

    def form_invalid(self, form):
        pk = self.kwargs['pk']
        print(form.errors)
        messages.error(self.request, 'error', extra_tags='Failed to disburse')
        return HttpResponseRedirect(reverse('new_loan_details', args=[pk]))


class DeleteAdjustmentView(FormView):
    form_class = DeleteAdjustmentForm
    template_name = 'new_loans/loan_details.html'

    def form_valid(self, form):
        pk = self.kwargs['pk']
        del_adj = form.cleaned_data
        transacton = Transactions.objects.filter(id=del_adj)
        transacton.delete()
        messages.success(self.request, 'error', extra_tags='Successfully deleted adjustment')
        return HttpResponseRedirect(reverse('new_loan_details', args=[pk]))


@method_decorator(login_required, name='dispatch')
class LoanScheduleView(View):

    def get_principal(self, loan) -> float:
        pp = loan.amount_approved
        loan_status = [Loans.LOAN_STATUS.DISBURSED, Loans.LOAN_STATUS.CLOSED, Loans.LOAN_STATUS.WAIVED_OFF,
                       Loans.LOAN_STATUS.WRITTEN_OFF, Loans.LOAN_STATUS.TOPPED_UP]
        if loan.loan_status in loan_status:
            loanacct = Account.objects.filter(name='Loan Receivables', business=biz_data(self.request.user)).first()
            adjustment_added = loan.loan_trans.filter(
                Q(account_dr=loanacct.id), Q(transaction_type='Principal Adjustment')).aggregate(
                total=Coalesce(Sum('reporting_amount', output_field=FloatField()), 0.00))['total']
            adjustment_down = loan.loan_trans.filter(
                Q(account_cr=loanacct.id), Q(transaction_type='Principal Adjustment')).aggregate(
                total=Coalesce(Sum('reporting_amount', output_field=FloatField()), 0.00))['total']

            # adjustments = loan.loan_trans.objects.filter(transaction_type='Loan Adjustment')
            totaldebit = loan.loan_trans.filter(Q(account_dr_id=loanacct.id),
                                                ~Q(transaction_type='Principal Adjustment')).aggregate(
                total=Coalesce(Sum('reporting_amount', output_field=FloatField()), 0.00))['total']
            # incase the is removal of an extra charge (this is a fix for the extra charge that was being added to loan amt ap
            # approved due to 'added to principal'
            extra_charge_removed = loan.loan_trans.filter(transaction_type='Remove extra loan charge').aggregate(
                ex_charge=Coalesce(Sum('reporting_amount', output_field=FloatField()), 0.00))['ex_charge']
            pp = float(totaldebit) - float(adjustment_down - adjustment_added) - extra_charge_removed
        return pp

    def get(self, request, pk):
        loan = get_object_or_404(Loans, id=pk)
        try:
            # rate, amount, period, schedule
            rate = loan.rate / 100
            principal = self.get_principal(loan)
            interval = loan.interval
            formulae = loan.loan_type.formula
            period = loan.approved_duration
            rate_type = loan.rate_type
            stringified = 'Not yet disbursed'
            if loan.loan_status in [Loans.LOAN_STATUS.DISBURSED, Loans.LOAN_STATUS.CLOSED]:
                get_amortisation = loan_payment_details(rate_type, rate, principal, period, interval, formulae,
                                                        acc=loan.account, loan=loan)
                amortisation = get_amortisation.gen_schedules()
                context = {
                    'status': {name: value for value, name in Loans.LOAN_STATUS.choices},
                    'detail': loan,
                    'schedule': amortisation.to_dict('records') if amortisation is not None else [],
                    'payable': amortisation['installment'].sum() if get_amortisation is not None else 0,
                    'disbursed': loan.amount_approved
                }
                stringified = render_to_string('new_loans/includes/schedule.html', context)
            return JsonResponse({'data': stringified, 'status': 'success'})
        except AttributeError as e:
            logger.error(str(e))
        except (Exception,) as e:
            logger.error(str(e))
        return JsonResponse({'data': 'error', 'status': 'failed'})


@method_decorator(login_required, name='dispatch')
class PaymentHistoryView(View):
    def get(self, request, pk):
        loan = Loans.objects.prefetch_related('loan_trans').get(id=pk)
        csrf_token = csrf.get_token(request)
        ts = loan.loan_trans.filter(receipt=OuterRef('receipt'), loan=loan,
                                    transaction_type__in=['Loan repayment', 'Loan interest']).values('id')[:1]

        repayments = (
            loan.loan_trans
            .filter(loan=loan, transaction_type__in=['Loan repayment', 'Loan interest']).only('id', 'receipt',
                                                                                              'tx_date')
            .values('receipt').annotate(txt_date=F('tx_date'))
            .annotate(paidamount=Sum('reporting_amount')).annotate(pk=Subquery(ts))
        )
        print(repayments)
        user_role = request.user.staff.position
        perms_in_role = list(user_role.permissions.all().values_list('item_name', flat=True))
        context = {
            'loan': loan,
            'repayments': repayments,
            'user_perms': perms_in_role,
            'csrf': csrf_token
        }
        stringified = render_to_string('new_loans/includes/payment_history.html', context)
        return JsonResponse({'data': stringified, 'status': 'success'})


@method_decorator(login_required, name='dispatch')
class SecurityView(View):
    def get(self, request, pk):
        loan = get_object_or_404(Loans, id=pk)
        user = self.request.user
        loan_settings = LoanSettings.objects.filter(business=biz_id(user)).first()
        securities = loan.securities.all()
        nsecurities = securities.count()
        maxi_securities = loan.loan_type.maximum_securities
        use_shares = loan_settings.use_shares
        total_security = securities.aggregate(
            total=Coalesce(Sum('value'), 0.0, output_field=FloatField()))['total']

        applicant_savings = loan.account.balance or 0
        share_price = OtherSettings.objects.filter(business=biz_id(user)).first().share_price
        member_shares = (loan.applicant.shares * share_price) or 0
        if use_shares:
            total_security_value = total_security + applicant_savings + member_shares
        else:
            total_security_value = total_security + applicant_savings
        needed_value = int(loan.loan_type.security_ratio) / 100 * int(loan.amount_requested)
        context = {
            'loan': loan,
            'securities': securities,
            'nsecurities': nsecurities,
            'maxi_securities': maxi_securities,
            'total_security_value': total_security_value,
            'needed_value': needed_value,
            'use_shares': use_shares,
            'applicant_savings': applicant_savings,
        }
        stringified = render_to_string('new_loans/includes/securities.html', context)
        return JsonResponse({'data': stringified, 'status': 'success'})


@method_decorator(login_required, name='dispatch')
class AddSecurityView(View):

    def post(self, request, pk):
        loan = get_object_or_404(Loans, id=pk)
        try:
            name = request.POST['name']
            value = request.POST['value']
            value = float(value.replace(',', ''))
            proofs = self.request.FILES.get('proof', )
            description = request.POST['description']
            security = Loansecurities.objects.create(loan=loan, name=name, value=value, file=proofs,
                                                     description=description)
            messages.success(request, 'success',
                             extra_tags=f'Loan Security of {security.name} valued at {security.value} added successfully')
        except Exception as e:
            logger(str(e))
            messages.error(self.request, 'error', extra_tags='Failed to add security')
        return HttpResponseRedirect(reverse('new_loan_details', args=[pk]))


@method_decorator(login_required, name='dispatch')
class EditSecurityView(View):

    def post(self, request, pk, pk2):
        loan = get_object_or_404(Loans, id=pk)
        security = get_object_or_404(Loansecurities, id=pk2)
        name = request.POST['name']
        value = request.POST['value']
        value = float(value.replace(',', ''))
        proofs = self.request.FILES.get('proof', )
        description = request.POST['description']
        security.name = name
        security.value = value
        security.description = description
        security.save()
        messages.success(request, 'success',
                         extra_tags=f'Loan Security of {security.name} Edited successfully with the value of {security.value}')
        return HttpResponseRedirect(reverse('new_loan_details', args=[pk]))


class DeleteSecurityView(View):
    def post(self, request, pk, pk2):
        loan = get_object_or_404(Loans, id=pk)
        security = get_object_or_404(Loansecurities, id=pk2)
        security.delete()
        messages.success(request, 'success',
                         extra_tags=f'Successfully deleted this security')
        return HttpResponseRedirect(reverse('new_loan_details', args=[loan.id]))


@method_decorator(login_required, name='dispatch')
class GuarantorsView(View):
    def get(self, request, pk):
        loan = get_object_or_404(Loans, id=pk)
        guarantors = loan.guarantors.all()
        loan_setings = LoanSettings.objects.filter(business=biz_id(request.user))[0]
        minguarantors = loan_setings.minimum_guarantors
        context = {
            'loan': loan,
            'guarantors': guarantors,
            'minguarantors': minguarantors
        }
        stringified = render_to_string('new_loans/includes/guarantors.html', context)
        return JsonResponse({'data': stringified, 'status': 'success'})


@method_decorator(login_required, name='dispatch')
class AddGuarantorView(View):
    def post(self, request, pk):
        loan = get_object_or_404(Loans, id=pk)
        try:
            form = AddGuarantorForm(self.request.POST, request=self.request, loan=loan)
            if form.is_valid():
                form.save()
                messages.success(self.request, 'success', extra_tags='Successfully added a guarantor')
                return HttpResponseRedirect(reverse('new_loan_details', args=[pk]))
        except Exception as e:
            logger.error(str(e))
        messages.error(self.request, 'danger', extra_tags='Failed to add a guarantor')
        return HttpResponseRedirect(reverse('new_loan_details', args=[pk]))


@method_decorator(login_required, name='dispatch')
class DeleteGuarantorView(View):

    def get(self, request, pk, pk2):
        guarantors = Loanguarantors.objects.filter(loan_id=pk, id=pk2).first()
        if guarantors:
            guarantors.delete()
            messages.success(self.request, 'success', extra_tags='Successfully deleted guarantor')
            return HttpResponseRedirect(reverse('new_loan_details', args=[pk]))
        messages.error(self.request, 'danger', extra_tags='Error in deleting guarantor')
        return HttpResponseRedirect(reverse('new_loan_details', args=[pk]))


class AddLoanDocumentView(View):
    def post(self, request, pk):
        loan = get_object_or_404(Loans, id=pk)
        try:
            name = request.POST['name']
            file = request.FILES.get('file', None)
            description = request.POST['description']
            file_uploaded = LoanUpload.objects.create(loan=loan, file_name=name, file=file,
                                                      added_by=biz_staff(request.user))
            messages.success(request, 'success', extra_tags=f'{file_uploaded.file_name} uploaded successfully')
        except (Exception,) as e:
            logger.error(str(e))
            messages.error(request, 'error', extra_tags='Error occurred while add file')
        return HttpResponseRedirect(reverse('new_loan_details', args=[pk]))


class EditDocumentView(View):
    def post(self, request, pk, pk2):
        loan = get_object_or_404(Loans, id=pk)
        document = get_object_or_404(LoanUpload, id=pk2)
        document.file_name = request.POST['name']
        # document.file = request.FILES.get('file', None)
        description = request.POST['description']
        document.save()
        messages.success(request, 'success', extra_tags=f'{document.file_name} successfully updated')
        return HttpResponseRedirect(reverse('new_loan_details', args=[loan.id]))


class DeleteDocument(View):

    def post(self, request, pk, pk2):
        loan = get_object_or_404(Loans, id=pk)
        document = get_object_or_404(LoanUpload, id=pk2)
        document.delete()
        messages.success(request, 'success', extra_tags='Successfully deleted document')
        return HttpResponseRedirect(reverse('new_loan_details', args=[loan.id]))


class Getdetail(View):
    def get(self, request, pk, *args, **kwargs):
        loan_type = LoanTypes.objects.get(id=pk)
        formulas = loan_type.LOAN_FORMULA.choices
        result_dict = {num: text for num, text in formulas}
        context = {
            'interest_rate': loan_type.interest_rate,
            'interval_name': loan_type.get_interval_display(),
            'interval': loan_type.interval,
            'loan_formula': loan_type.get_formula_display(),
            'max_duration': loan_type.maximum_period,
            'formula': result_dict,
            'period': loan_type.get_period
        }
        return JsonResponse(context)


class LoanTopUp(FormView):
    form_class = ''
