import datetime
from fractions import Fraction

from django import forms

from django.db import transaction
from django.db.models import F

from accounts.models import Biodata
from loans.models import Loans, LoanTypes, Loanguarantors, ApplicationAccountOrLoanType, GeneralCharge, LoanCharges
from loans.utils.amortisation import loan_payment_details
from sacco.models import AccountBroker, NotiSettings, Member
from sacco.utils import branch_id, biz_staff, biz_data, checkAndSendMessage, sendTransEmail
from transactions.models import Account, AccountCategory, Transactions


class LoanApplicationForm(forms.Form):
    amount = forms.CharField()
    account = forms.CharField()
    adate = forms.DateField()
    interest = forms.FloatField()
    duration = forms.IntegerField()
    sub_duration = forms.IntegerField(required=False)
    rate_type = forms.CharField()
    loan_type = forms.CharField()
    payment_interval = forms.IntegerField()

    def __init__(self, *args, **kwargs):
        self.request = kwargs.pop('request', None)
        super(LoanApplicationForm, self).__init__(*args, **kwargs)

    def clean_amount(self):
        cleaned_data = self.cleaned_data
        amount = cleaned_data['amount']
        return amount.replace(',', '')

    @transaction.atomic
    def save(self):
        cleaned_data = self.cleaned_data
        amount = cleaned_data['amount']
        account = cleaned_data['account']
        application_date = cleaned_data['adate']
        interest = cleaned_data['interest']
        duration = cleaned_data['duration']
        sub_duration = cleaned_data['sub_duration']
        rate_type = cleaned_data['rate_type']
        loan_type = cleaned_data['loan_type']
        payment_interval = cleaned_data['payment_interval']

        if sub_duration is not None:
            month_year = Fraction(sub_duration, 12)
            duration = duration + month_year

        applicant = AccountBroker.objects.filter(the_account_id=account).first()
        loan_type = LoanTypes.objects.filter(id=loan_type).first()
        loan = Loans.objects.create(account_id=applicant.the_account.id, applicant_id=applicant.members.id,
                                    branch_id=branch_id(self.request.user), amount_requested=amount, rate=interest,
                                    rate_type=rate_type,
                                    requested_on=application_date, loan_type_id=loan_type.id, interval=payment_interval,
                                    requested_duration=duration,
                                    added_by=biz_staff(self.request.user), is_old=False)
        gen_cgs = ApplicationAccountOrLoanType.objects.filter(loan_type=loan_type, general_charge__status=True,
                                                              general_charge__application=GeneralCharge.APPLICATION.LOANS).prefetch_related(
            'general_charge')
        loan_charges = []

        for charge in gen_cgs:
            if charge.general_charge.is_percentage:
                charge_amount = (float(charge.general_charge.amount) * float(loan.amount_requested)) / 100
            else:
                charge_amount = charge.general_charge.amount
            # print(charge_amount)
            loan_charges.append({"general_charge_id": charge.general_charge.id, "amount": charge_amount})
        charges = [LoanCharges(amount=charge['amount'], charge_id=charge['general_charge_id'], loan=loan)for charge in loan_charges]
        LoanCharges.objects.bulk_create(charges)
        return loan


class CommitteeActionForm(forms.Form):
    comaction = forms.IntegerField(required=False)
    a_approved = forms.CharField(required=False)
    approved_rate = forms.FloatField(required=False)
    approved_date = forms.DateField(required=False)
    approved_duration = forms.FloatField(required=False)
    sub_duration = forms.FloatField(required=False)
    remarks = forms.CharField(required=False)

    def __init__(self, *args, **kwargs):
        self.request = kwargs.pop('request', None)
        super(CommitteeActionForm, self).__init__(*args, **kwargs)


class DeleteAdjustmentForm(forms.Form):
    del_adj = forms.IntegerField()


class DisburseLoanForm(forms.Form):
    approved_amount = forms.FloatField()
    totalcharges = forms.FloatField()
    ddate = forms.DateField()
    credited = forms.IntegerField()
    narration = forms.CharField(required=False)
    chargespaid = forms.IntegerField(required=False)
    istopup = forms.CharField(required=False)
    voucher = forms.CharField(required=False)
    damount = forms.CharField()
    chargesaccoun = forms.IntegerField(required=False)


class AddGuarantorForm(forms.Form):
    gtype = forms.CharField(required=False)
    name = forms.CharField(required=False)
    gender = forms.CharField(required=False)
    dob = forms.CharField(required=False)
    code = forms.CharField(required=False)
    contact = forms.CharField(required=False)
    code1 = forms.CharField(required=False)
    other_contact = forms.CharField(required=False)
    email = forms.CharField(required=False)
    marital_status = forms.CharField(required=False)
    country = forms.CharField(required=False)
    location = forms.CharField(required=False)
    nin = forms.CharField(required=False)
    account = forms.CharField(required=False)

    def __init__(self, *args, **kwargs):
        self.request = kwargs.pop('request', None)
        self.loan = kwargs.pop('loan', None)
        super(AddGuarantorForm, self).__init__(*args, **kwargs)


    def save(self, commit=True):
        cleaned_data = self.cleaned_data
        with transaction.atomic():
            if cleaned_data['gtype'] == 2:
                bio = Biodata.objects.create(name=cleaned_data['name'], gender=cleaned_data['gender'],
                                             dob=cleaned_data['dob'], contact=cleaned_data['contact'],
                                             other_contact=cleaned_data['other_contact'], email=cleaned_data['email'],
                                             marital_status=cleaned_data['marital_status'],
                                             country=cleaned_data['country'], location=cleaned_data['location'],
                                             nin=cleaned_data['nin'], business=branch_id(self.request.user))
                guarantor = Loanguarantors.objects.create(loan=self.loan, biodata=bio,
                                                          added_by=biz_staff(self.request.user))
            else:
                try:
                    member = Member.objects.get(id=cleaned_data['account'])
                except Member.DoesNotExist:
                    member = None
                if member:
                    if member.is_group is False:
                        guarantor = Loanguarantors.objects.create(loan=self.loan, biodata_id=cleaned_data['account'],
                                                                  added_by=biz_staff(self.request.user))
                    else:
                        guarantor = Loanguarantors.objects.create(loan=self.loan, group=member, added_by=biz_staff(self.request.user))
        return guarantor


class EditLoanForm(forms.Form):
    loan_type = forms.IntegerField()
    amount = forms.FloatField()
    adate = forms.DateField()
    interest = forms.IntegerField()
    duration = forms.FloatField()
    payment_method = forms.IntegerField()
    rate_type = forms.IntegerField()

    def __init__(self, *args, **kwargs):
        self.loan = kwargs.pop('loan', None)
        self.user = kwargs.pop('user', None)
        super(EditLoanForm, self).__init__(*args, **kwargs)

    def save(self, commit=True):
        cleaned_data = self.cleaned_data
        loan = self.loan
        if commit:
            amount = cleaned_data['amount']
            adate = cleaned_data['adate']
            interest = cleaned_data['interest']
            duration = cleaned_data['duration']
            payment_method = cleaned_data['payment_method']
            rate_type = cleaned_data['rate_type']
            loan.loan_type_id = cleaned_data['loan_type']
            loan.amount_requested = amount
            loan.requested_on = adate
            loan.rate = interest
            loan.rate_type = rate_type
            loan.requested_duration = duration
            loan.repayment_method = payment_method
            loan.save()
        return loan


class TopUpForm(forms.Form):
    account = forms.CharField()
    amount = forms.CharField()
    d_date = forms.CharField()
    interest = forms.CharField()
    duration = forms.FloatField()
    rate_type = forms.CharField()
    payment_method = forms.CharField()
    loantype = forms.CharField()

    def clean_amount(self):
        amount = str(self.cleaned_data['amount']).replace(',', '')
        return int(amount)


class EditRunningLoanForm(forms.Form):
    amount = forms.FloatField()

    def __init__(self, *args, **kwargs):
        self.staff = kwargs.pop('staff', None)
        self.loan = kwargs.pop('loan', None)
        super(EditRunningLoanForm, self).__init__(*args, **kwargs)

    def save(self, commit=True):
        cleaned_data = self.cleaned_data
        loan = self.loan
        if commit:
            loan.amount_approved = cleaned_data['amount']
            loan.save()
        return loan