import json
import traceback
from datetime import datetime

import xlwt as xlwt
from django.db.models import ExpressionWrapper, Value, Func, CharField, F, Case, When, Sum, FloatField, Subquery, \
    OuterRef, Q
from django.db.models.functions import Concat, Coalesce
from django.http import HttpResponse
from django.template import loader
from django.views import View

from accounts.models import Business, Staff
from loans.models import *
from loans.utils.loan_details import get_aging_report_data_v2, get_business_loans_context_data
from sacco.models import OtherSettings, WithdrawUpload, SavingUpload, GroupMember, AccountBroker
from sacco.utils import businessdata, branchdata, biz_staff_branch
from transactions.downloads import render_to_pdf
from transactions.models import Transactions, SharesTransactions
from transactions.reports import today

style = xlwt.easyxf('font: color-index black, bold on; pattern: pattern solid, fore_colour white; align: wrap on,'
                    'vert centre, horiz center; borders: top_color black, bottom_color black,right_color black, '
                    'left_color black, left thin, right thin, top thin, bottom thin;')

style0 = xlwt.easyxf(
    'font: color-index white, bold on; pattern: pattern solid, fore_colour black; align: wrap off,vert centre, '
    'horiz center; borders: top_color black, bottom_color black,right_color black, '
    'left_color black, left thin, right thin, top thin, bottom thin')

style1 = xlwt.easyxf(
    'font: color-index black; pattern: fore_colour white; align: wrap off; borders: top_color black, bottom_color '
    'black,right_color black, left_color black, left thin, right thin, top thin, bottom thin')


class ApplicationsPdf(View):
    template_name = 'loans/loan_applications_pdf.html'

    def get(self, request, *args, **kwargs):
        title = 'Loan application'
        loanApplications = Loans.objects.raw(
            "SELECT l. *, b.the_account_id as accountid, b.members_id as memberid, a.acc_number, m.biodata_id, m.id, c.name AS member_name FROM account_broker b, member_accounts a, sacco_member m, biodata c, loans l where b.the_account_id = a.id and b.business_id = %s and b.members_id = m.id and m.biodata_id = c.id and l.account_id = a.id and l.branch_id = %s and (l.loan_status = 0 or l.loan_status = 1 or l.loan_status = 2)" % (
                businessdata(request), branchdata(request)))

        file = render_to_pdf(self.template_name, locals())
        resp = HttpResponse(file.getvalue(), content_type="application/pdf")
        resp['Content-Disposition'] = 'attachment; filename=Loanapplications'
        return resp


class Runningpdf(View):
    template_name = 'loans/running_pdf.html'

    def get(self, request, *args, **kwargs):
        title = 'Running Loans'
        business = Business.objects.filter(id=businessdata(request)).first()

        # print('account running', running_loans.account.account_type)

        running_loans = Loans.objects.raw(
            "SELECT l. *, b.the_account_id as accountid, b.members_id as memberid, a.acc_number, m.biodata_id, m.id, c.name AS member_name FROM account_broker b, member_accounts a, sacco_member m, biodata c, loans l where b.the_account_id = a.id and b.business_id = %s and b.members_id = m.id and m.biodata_id = c.id and l.account_id = a.id and l.branch_id = %s and (l.loan_status = 3)" % (
                businessdata(request), branchdata(request)))

        loans = (Loans.objects.filter(branch=branchdata(request), loan_status=3, )
                         .annotate(principal_paid=Sum('loan_trans__reporting_amount'))
                         .annotate(
            applicant_name=F('applicant__biodata__name'),
            acc_number=F('account__acc_number'),
            applicant_phone=F('applicant__biodata__contact'),
            applicant_email=F('applicant__biodata__email'),
            loan_officer=F('officer__biodata__name'),
            applicant_group=Subquery(
                GroupMember.objects.filter(
                    member=OuterRef('applicant')
                ).values('group__name')[:1]))
                         .annotate(principal_balance=F('amount_approved') - F('principal_paid'))
                         .values('applicant_name', 'loan_number', 'acc_number', 'applicant_phone', 'applicant_email',
                                 'amount_approved',
                                 'principal_balance',
                                 'balance', 'schedule_start', 'approved_duration', 'amount_requested', 'rate',
                                 'loan_officer', 'approved_on', 'applicant_group'))

        loans = loans.annotate(
            started=ExpressionWrapper(
                Func(F('schedule_start'), Value('%d/%m/%Y'), function='DATE_FORMAT'),
                output_field=CharField()),
            disbursement_date=ExpressionWrapper(
                Func(F('approved_on'), Value('%d/%m/%Y'), function='DATE_FORMAT'),
                output_field=CharField()),
            duration=Concat(F('approved_duration'), Value(' '), Case(
                When(interval=0, then=Value('Day(s)')),
                When(interval=1, then=Value('Week(s)')),
                When(interval=2, then=Value('Fortnight(s)')),
                When(interval=4, then=Value('Year(s)')),
                When(interval=3, then=Value('Month(s)')),
            ), output_field=CharField())
        ).values_list('applicant_name', 'loan_number', 'acc_number', 'amount_approved', 'rate', 'duration', 'balance',
                      'disbursement_date', 'loan_officer', 'applicant_group')

        file = render_to_pdf(self.template_name, locals())
        resp = HttpResponse(file.getvalue(), content_type="application/pdf")
        d4 = today.strftime("%b-%d-%Y")
        file_name = f'{business.short_name} Running loans - {d4}'
        resp['Content-Disposition'] = 'attachment; filename="{}.pdf"'.format(file_name.upper())
        return resp


class Runningpffdf(View):
    template_name = 'loans/running_pdf.html'
    title = 'Running Loans'

    def get(self, request, *args, **kwargs):
        loanApplications = Loans.objects.raw(
            "SELECT l. *, b.the_account_id as accountid, b.members_id as memberid, a.acc_number, m.biodata_id, m.id, c.name AS member_name FROM account_broker b, member_accounts a, sacco_member m, biodata c, loans l where b.the_account_id = a.id and b.business_id = %s and b.members_id = m.id and m.biodata_id = c.id and l.account_id = a.id and l.branch_id = %s and (l.loan_status = 3)" % (
                businessdata(request), branchdata(request)))

        file = render_to_pdf(self.template_name, locals())
        resp = HttpResponse(file.getvalue(), content_type="runningloans/pdf")
        resp[('Content-Dis'
              'position')] = 'attachment; filename=Runningloans'
        return resp


class SomeView(View):
    template_name = 'loans/loan_applications_pdf.html'
    title = 'Running Loans'

    def get(self, request, *args, **kwargs):
        template_name = 'fdf'
        loanApplications = Loans.objects.raw(
            "SELECT l. *, b.the_account_id as accountid, b.members_id as memberid, a.acc_number, m.biodata_id, m.id, c.name AS member_name FROM account_broker b, member_accounts a, sacco_member m, biodata c, loans l where b.the_account_id = a.id and b.business_id = %s and b.members_id = m.id and m.biodata_id = c.id and l.account_id = a.id and l.branch_id = %s and (l.loan_status = 0 or l.loan_status = 1 or l.loan_status = 2)" % (
                businessdata(request), branchdata(request)))
        # Create the HttpResponse object with the appropriate CSV header.
        response = HttpResponse(
            content_type='text/csv',
            headers={'Content-Disposition': 'attachment; filename="somefilename.csv"'},
        )

        # The data is hard-coded here, but you could load it from a database or
        # some other source.
        csv_data = (
            ('First row', 'Foo', 'Bar', 'Baz'),
            ('Second row', 'A', 'B', 'C', '"Testing"', "Here's a quote"),
        )

        t = loader.get_template(template_name)
        c = {'data': csv_data}
        response.write(t.render(c))
        return response


def export_application(request):
    if request.method == "POST":
        pass
    else:
        loanApplications = Loans.objects.raw(
            "SELECT l. *, b.the_account_id as accountid, b.members_id as memberid, a.acc_number, m.biodata_id, m.id, c.name AS member_name FROM account_broker b, member_accounts a, sacco_member m, biodata c, loans l where b.the_account_id = a.id and b.business_id = %s and b.members_id = m.id and m.biodata_id = c.id and l.account_id = a.id and l.branch_id = %s and (l.loan_status = 0 or l.loan_status = 1 or l.loan_status = 2)" % (
                businessdata(request), branchdata(request)))

        business_context = get_business_loans_context_data(request)
        business_filter = business_context['business_filter']

        create_or_view_all_loans_for_officers = False
        if request.user.staff.position.permissions.filter(item_name='create_or_view_all_loans_for_officers').exists():
            create_or_view_all_loans_for_officers = True

        if create_or_view_all_loans_for_officers:
            loan_applications = Loans.objects.filter(
                loan_status__lte=2, branch__business_id__in=business_context['data_context']
            )
        else:
            loan_applications = Loans.objects.filter(
                loan_status__lte=2, branch__business_id__in=business_context['data_context'], officer=request.user.staff
            )

        # filters - loan number, name status
        loan_status_filter = request.GET.get('loan_status_filter', '')
        member_name = request.GET.get('member_name', '')
        loan_number = request.GET.get('loan_number', '')

        if loan_status_filter:
            loan_applications = loan_applications.filter(loan_status=int(loan_status_filter))
        if member_name:
            loan_applications = loan_applications.filter(applicant__biodata__name__icontains=member_name)
        if loan_number:
            loan_applications = loan_applications.filter(loan_number__icontains=loan_number)

        try:
            response = HttpResponse(content_type='application/ms-excel')
            response['Content-Disposition'] = 'attachment; filename="Loan applications.xls"'

            wb = xlwt.Workbook(encoding='utf-8')
            ws = wb.add_sheet('Loan applications')

            ws.col(1).width = int(20 * 500)
            ws.col(2).width = int(20 * 500)
            ws.col(3).width = int(20 * 400)
            ws.col(4).width = int(20 * 360)
            ws.col(5).width = int(20 * 260)
            ws.col(6).width = int(20 * 260)
            ws.col(7).width = int(20 * 360)
            ws.col(8).width = int(20 * 260)

            style1 = xlwt.easyxf(
                'font: color-index black, bold on; pattern: pattern solid, fore_colour white; ')

            row_num = 1
            ronum = 0
            date_format = xlwt.XFStyle()
            date_format.num_format_str = 'yyyy-mm-dd'
            font_style = xlwt.XFStyle()
            font_style.font.bold = True

            font_stylet = xlwt.XFStyle()
            font_stylet.font.bold = True
            # font_stylet.height = 320

            columns = [
                'SN',
                'MEMBER NAME',
                'DATE',
                'AMOUNT REQUESTE',
                'INTEREST RATE',
                'DURATION',
                'STATUS',
                'BRANCH'

            ]
            ws.write_merge(0, 0, 0, 7, "applications", font_stylet)
            for col_num in range(len(columns)):
                ws.write(row_num, col_num, columns[col_num], font_style)

            font_style = xlwt.XFStyle()
            date_format.num_format_str = 'yyyy-mm-dd'
            font_style.alignment.wrap = False
            font_style.alignment.vert = xlwt.Alignment.HORZ_RIGHT
            for c in loan_applications:
                row_num += 1
                ronum = ronum + 1
                if c.loan_status == 0:
                    loanstate = 'Applied'

                if c.loan_status == 1:
                    loanstate = 'Appraised'

                if c.loan_status == 2:
                    loanstate = 'Approved'

                row = [
                    ronum,
                    c.applicant.biodata.name,
                    c.requested_on.strftime("%Y-%m-%d"),
                    c.amount_requested,
                    c.rate,
                    c.requested_duration,
                    loanstate,
                    c.branch.name

                ]

                for col_num in range(len(row)):
                    ws.write(row_num, col_num, row[col_num], font_style)

            wb.save(response)

            return response

        except Exception as ex:
            print(str(ex))
            traceback.print_exc()

            return HttpResponse(str(ex))


def RunningLoansExcel(request):
    response = HttpResponse(content_type='application/ms-excel')
    # ---------------------------------------------------------------------------------
    business_context = get_business_loans_context_data(request)

    create_or_view_all_loans_for_officers = False

    if request.user.staff.position.permissions.filter(item_name='create_or_view_all_loans_for_officers').exists():
        create_or_view_all_loans_for_officers = True

    running_loans = []
    if create_or_view_all_loans_for_officers:
        running_loans = Loans.objects.filter(branch__business_id__in=business_context['data_context'],
                                             loan_status=3).annotate(
            applicant_group=Subquery(
                GroupMember.objects.filter(
                    member=OuterRef('applicant')
                ).values('group__name')[:1])
        )

    else:
        running_loans = Loans.objects.filter(
            branch__business_id__in=business_context['data_context'], loan_status=3, officer=request.user.staff
        ).annotate(
            applicant_group=Subquery(
                GroupMember.objects.filter(
                    member=OuterRef('applicant')
                ).values('group__name')[:1])
        )

    # Filtering
    loan_officer = request.GET.get('loan_officer', '')
    start_date = request.GET.get('start_date', '')
    end_date = request.GET.get('end_date', '')

    if loan_officer:
        running_loans = running_loans.filter(officer_id=loan_officer)
    if start_date:
        running_loans = running_loans.filter(schedule_start__gte=start_date)
    if end_date:
        running_loans = running_loans.filter(schedule_start__lte=end_date)


    # --------------------------------------------------------------------------------

    business = Business.objects.filter(id=businessdata(request)).first()
    # branch = biz_staff_branch(request.user)

    d1 = today.strftime("%b-%d-%Y")

    file_name = f'{business.short_name} Running loans - {d1}'

    response['Content-Disposition'] = 'attachment; filename="{}.xls"'.format(file_name.capitalize())

    heading = f'{business.name} Running loans as of {d1}'
    if loan_officer:
        the_officer_name = running_loans.first().officer.biodata.name
        heading = f'{heading} for {the_officer_name}'
    if start_date:
        heading = f'{heading} from {start_date}'
    if end_date:
        heading = f'{heading} up to {end_date}'

    sheet_name = f'{d1} Running loans'

    wb = xlwt.Workbook(encoding='utf-8')

    ws = wb.add_sheet(sheet_name.capitalize())

    # Sheet header, first row
    row_num = 2

    columns = ['Name', 'Loan number', 'Sacco Number', 'Total Principal', 'Rate (%)', 'Duration', 'Total Payable',
               'Disbursement Date', 'Loan Officer', 'Group', 'Principal balance', 'Interest balaance', 'Branch']

    the_len = len(columns)

    ws.write_merge(0, 1, 0, the_len - 1, '{}'.format(heading.upper()), style)

    for col_num in range(len(columns)):
        ws.write(row_num, col_num, columns[col_num], style0)
        ws.row(2).height_mismatch = True
        ws.row(2).height = 256 * 2
        ws.col(0).width = int(20 * 350)
        ws.col(1).width = int(20 * 200)
        ws.col(2).width = int(20 * 250)
        ws.col(3).width = int(20 * 120)
        ws.col(4).width = int(20 * 200)
        ws.col(5).width = int(20 * 200)
        ws.col(5).width = int(20 * 200)

    running_loans = running_loans.annotate(principal_paid_total=Sum('loan_trans__reporting_amount')).annotate(
                        applicant_name=F('applicant__biodata__name'),
                        acc_number=F('account__acc_number'),
                        applicant_phone=F('applicant__biodata__contact'),
                        applicant_email=F('applicant__biodata__email'),
                        loan_officer=F('officer__biodata__name'),
                        applicant_group=Subquery(
                            GroupMember.objects.filter(
                                member=OuterRef('applicant')
                            ).values('group__name')[:1])).annotate(principal_balance_total=F('amount_approved') - F('principal_paid_total')).values('applicant_name', 'loan_number', 'acc_number', 'applicant_phone', 'applicant_email',
                             'amount_approved',
                             'principal_balance_total',
                             'balance', 'schedule_start', 'approved_duration', 'amount_requested', 'rate',
                             'loan_officer', 'approved_on', 'applicant_group', 'principal_balance', 'interest_balance', 'branch__name')

    saving_rows = running_loans.annotate(
        started=ExpressionWrapper(
            Func(F('schedule_start'), Value('%d/%m/%Y'), function='DATE_FORMAT'),
            output_field=CharField()),
        disbursement_date=ExpressionWrapper(
            Func(F('approved_on'), Value('%d/%m/%Y'), function='DATE_FORMAT'),
            output_field=CharField()),
        duration=Concat(F('approved_duration'), Value(' '), Case(
            When(interval=0, then=Value('Day(s)')),
            When(interval=1, then=Value('Week(s)')),
            When(interval=2, then=Value('Fortnight(s)')),
            When(interval=4, then=Value('Year(s)')),
            When(interval=3, then=Value('Month(s)')),
        ), output_field=CharField())
    ).values_list('applicant_name', 'loan_number', 'acc_number', 'amount_approved', 'rate', 'duration', 'balance',
                  'disbursement_date', 'loan_officer', 'applicant_group', 'principal_balance_total', 'interest_balance', 'branch__name')



    for saving_row in saving_rows:
        # print(saving_row)
        row_num += 1
        for col_num in range(len(saving_row)):
            ws.write(row_num, col_num, saving_row[col_num])

    wb.save(response)

    return response


def ExpensesExcel(request):
    response = HttpResponse(content_type='application/ms-excel')
    business = Business.objects.filter(id=businessdata(request)).first()
    branch = biz_staff_branch(request.user)

    d1 = today.strftime("%b-%d-%Y")
    file_name = f'{business.short_name} Expenses - {d1}'
    response['Content-Disposition'] = 'attachment; filename="{}.xls"'.format(file_name.capitalize())
    heading = f'{business.name} Expenses as of {d1}'
    sheet_name = f'{d1} Expenses'
    wb = xlwt.Workbook(encoding='utf-8')
    ws = wb.add_sheet(sheet_name.capitalize())

    # Sheet header, first row
    row_num = 2

    columns = ['Date', 'Reference', 'Amount', 'Expense Account', 'Narration']

    the_len = len(columns)
    ws.write_merge(0, 1, 0, the_len - 1, '{}'.format(heading.upper()), style)
    for col_num in range(len(columns)):
        ws.write(row_num, col_num, columns[col_num], style0)
        ws.row(2).height_mismatch = True
        ws.row(2).height = 256 * 2

        ws.col(0).width = int(20 * 200)
        ws.col(1).width = int(20 * 200)
        ws.col(2).width = int(20 * 200)
        ws.col(3).width = int(20 * 250)
        ws.col(4).width = int(20 * 500)

    expense_transactions = Transactions.objects.filter(branch=branch, transaction_type='Expense Invoice')
    saving_rows = expense_transactions.annotate(
        txn_date=ExpressionWrapper(Func(F('tx_date'), Value('%d/%m/%Y'), function='DATE_FORMAT'),
                                   output_field=CharField()),

    ).values_list('txn_date', 'receipt', 'reporting_amount', 'account_dr__name', 'narration')
    for saving_row in saving_rows:
        row_num += 1
        for col_num in range(len(saving_row)):
            ws.write(row_num, col_num, saving_row[col_num])

    wb.save(response)
    return response


def ShareHoldersExcel(request, business_context=None):
    response = HttpResponse(content_type='application/ms-excel')
    start_date_str = request.GET.get('date_from', '')
    end_date_str = request.GET.get('date_to', '')
    business = Business.objects.filter(id=businessdata(request)).first()
    branch = biz_staff_branch(request.user)
    business_setting = OtherSettings.objects.filter(business=businessdata(request)).first()

    d1 = today.strftime("%b-%d-%Y")
    file_name = f'{business.short_name} Share Holders - {d1}'
    sheet_name = f'{d1} Share Holders'

    if start_date_str and end_date_str:
        d1 = f'{start_date_str} to {end_date_str}'
        file_name = f'{business.short_name} Share Holders - {d1}'
        sheet_name = f'{d1}'
    response['Content-Disposition'] = 'attachment; filename="{}.xls"'.format(file_name.capitalize())
    heading = f'{business.name} Share Holders as of {d1}'
    wb = xlwt.Workbook(encoding='utf-8')
    ws = wb.add_sheet(sheet_name.capitalize(), cell_overwrite_ok=True)

    # Sheet header, first row
    row_num = 2

    columns = ['Name', 'Telephone', 'Account Number', 'Shares', 'Share Capital', 'Shareholding start date']

    the_len = len(columns)
    ws.write_merge(0, 1, 0, the_len - 1, '{}'.format(heading.upper()), style)
    for col_num in range(len(columns)):
        ws.write(row_num, col_num, columns[col_num], style0)
        ws.row(2).height_mismatch = True
        ws.row(2).height = 256 * 2

        ws.col(0).width = int(20 * 400)
        ws.col(1).width = int(20 * 250)
        ws.col(2).width = int(20 * 100)
        ws.col(3).width = int(20 * 250)
        ws.col(4).width = int(20 * 200)

    # share_holders = Member.objects.filter(member_type='s', biodata__business=businessdata(request))
    #
    # shared_member = []


    business_context = get_business_loans_context_data(request)

    # Optional: parse to date objects
    if start_date_str:
        start_date = datetime.strptime(start_date_str, "%Y-%m-%d").date()
    if end_date_str:
        end_date = datetime.strptime(end_date_str, "%Y-%m-%d").date()

    # Start building filters
    buyer_filter = Q(shares_buyer__branch__business_id__in=business_context['data_context'])
    seller_filter = Q(shares_seller__branch__business_id__in=business_context['data_context'])

    if start_date_str:
        buyer_filter &= Q(shares_buyer__date__gte=start_date)
        seller_filter &= Q(shares_seller__date__gte=start_date)

    if end_date_str:
        buyer_filter &= Q(shares_buyer__date__lte=end_date)
        seller_filter &= Q(shares_seller__date__lte=end_date)

    member_filter = Q(
        member_type='s',
        biodata__business_id__in=business_context['data_context']
    )
    if start_date_str:
        member_filter &= Q(share_holder_join_date__gte=start_date)
    if end_date_str:
        member_filter &= Q(share_holder_join_date__lte=end_date)

    business_setting = OtherSettings.objects.filter(business=businessdata(request)).first()

    # Optimized query
    shared_member = (
    Member.objects.filter(member_filter).select_related('biodata').prefetch_related('accounts').annotate(
        total_purchased=Coalesce(
            Sum('shares_buyer__shares', filter=buyer_filter),
            Value(0.0), output_field=FloatField()
        ),
        total_sold=Coalesce(
            Sum('shares_seller__shares', filter=seller_filter),
            Value(0.0), output_field=FloatField()
        )
    ).annotate(
        net_shares=F('total_purchased') - F('total_sold'),
        account_number=Subquery(
            AccountBroker.objects.filter(
                members_id=OuterRef('id')
            ).values('the_account__acc_number')[:1]
        )
    ).annotate(
        member_name=F('biodata__name'),
        member_contact=F('biodata__contact'),
        shares_value=F('net_shares') * business_setting.share_price
    )
    .values(
        'member_name', 'member_contact', 'net_shares', 'account_number', 'shares_value', 'share_holder_join_date'
    ))
    columns = ['member_name', 'member_contact','account_number', 'net_shares', 'shares_value', 'share_holder_join_date']

    # write columns, start from row 1
    for i, row in enumerate(shared_member, 4):
        for j, col in enumerate(columns):
            ws.write(i, j, row[col])

    wb.save(response)
    return response


def FailedRepaymentExcel(request, pk):
    response = HttpResponse(content_type='application/ms-excel')
    try:
        upload = LoanRepaymentUpload.objects.get(id=pk)
    except Exception as e:
        print('Error', e)
        return 'Records no longer exist'

    file_name = f'Failed repayments registered on {str(upload.date_added)}'
    sheet_name = f'failed repayments details'

    response['Content-Disposition'] = 'attachment; filename="{}.xls"'.format(file_name.capitalize())
    heading = f'FAILED REPAYMENTS'
    wb = xlwt.Workbook(encoding='utf-8')
    ws = wb.add_sheet(sheet_name.capitalize(), cell_overwrite_ok=True)

    # Sheet header, first row
    row_num = 2

    columns = ['Loan number', 'Details']

    the_len = len(columns)
    ws.write_merge(0, 1, 0, the_len - 1, '{}'.format(heading.upper()), style)
    for col_num in range(len(columns)):
        ws.write(row_num, col_num, columns[col_num], style0)
        ws.row(2).height_mismatch = True
        ws.row(2).height = 256 * 2

        ws.col(0).width = int(20 * 400)
        ws.col(1).width = int(20 * 250)
        ws.col(2).width = int(20 * 100)
        ws.col(3).width = int(20 * 250)
        ws.col(4).width = int(20 * 200)

    rows = json.loads(upload.failed_loans)['failed_loans']

    for row in rows:
        row_num += 1
        ws.write(row_num, 0, row['loan number'])
        ws.write(row_num, 1, row['reason'])

    wb.save(response)
    return response


def FailedLoansUploadedExcel(request, pk):
    response = HttpResponse(content_type='application/ms-excel')
    try:
        upload = UploadLoans.objects.get(id=pk)
    except Exception as e:
        print('Error', e)
        return 'Records no longer exist'

    file_name = f'Failed loans registered on {str(upload.upload_date)}'
    sheet_name = f'failed loans upload details'

    response['Content-Disposition'] = 'attachment; filename="{}.xls"'.format(file_name.capitalize())
    heading = f'FAILED LOANS UPLOAD'
    wb = xlwt.Workbook(encoding='utf-8')
    ws = wb.add_sheet(sheet_name.capitalize(), cell_overwrite_ok=True)

    # Sheet header, first row
    row_num = 2

    columns = ['Account number', 'Details']

    the_len = len(columns)
    ws.write_merge(0, 1, 0, the_len - 1, '{}'.format(heading.upper()), style)
    for col_num in range(len(columns)):
        ws.write(row_num, col_num, columns[col_num], style0)
        ws.row(2).height_mismatch = True
        ws.row(2).height = 256 * 2

        ws.col(0).width = int(20 * 400)
        ws.col(1).width = int(20 * 250)
        ws.col(2).width = int(20 * 100)
        ws.col(3).width = int(20 * 250)
        ws.col(4).width = int(20 * 200)

    rows = json.loads(upload.failed_loans)['failed_loans']

    for row in rows:
        row_num += 1
        ws.write(row_num, 0, row['account number'])
        ws.write(row_num, 1, row['reason'])

    wb.save(response)
    return response


def FailedWithdrawExcel(request, pk):
    response = HttpResponse(content_type='application/ms-excel')
    try:
        upload = WithdrawUpload.objects.get(id=pk)
    except Exception as e:
        print('Error', e)
        return 'Records no longer exist'

    file_name = f'Failed withdraws registered on {str(upload.date_added)}'
    sheet_name = f'failed withdraw details'

    response['Content-Disposition'] = 'attachment; filename="{}.xls"'.format(file_name.capitalize())
    heading = f'FAILED WITHDRAWS'
    wb = xlwt.Workbook(encoding='utf-8')
    ws = wb.add_sheet(sheet_name.capitalize(), cell_overwrite_ok=True)

    # Sheet header, first row
    row_num = 2

    columns = ['Account number', 'Details']

    the_len = len(columns)
    ws.write_merge(0, 1, 0, the_len - 1, '{}'.format(heading.upper()), style)
    for col_num in range(len(columns)):
        ws.write(row_num, col_num, columns[col_num], style0)
        ws.row(2).height_mismatch = True
        ws.row(2).height = 256 * 2

        ws.col(0).width = int(20 * 400)
        ws.col(1).width = int(20 * 250)
        ws.col(2).width = int(20 * 100)
        ws.col(3).width = int(20 * 250)
        ws.col(4).width = int(20 * 200)

    rows = json.loads(upload.failed_withdraws)['failed_withdraw']

    for row in rows:
        row_num += 1
        ws.write(row_num, 0, row['account number'])
        ws.write(row_num, 1, row['reason'])

    wb.save(response)
    return response


def FailedSavingExcel(request, pk):
    response = HttpResponse(content_type='application/ms-excel')
    try:
        upload = SavingUpload.objects.get(id=pk)
    except Exception as e:
        print('Error', e)
        return 'Records no longer exist'

    file_name = f'Failed Savings registered on {str(upload.date_added)}'
    sheet_name = f'failed savings details'

    response['Content-Disposition'] = 'attachment; filename="{}.xls"'.format(file_name.capitalize())
    heading = f'FAILED SAVINGS'
    wb = xlwt.Workbook(encoding='utf-8')
    ws = wb.add_sheet(sheet_name.capitalize(), cell_overwrite_ok=True)

    # Sheet header, first row
    row_num = 2

    columns = ['Account number', 'Details']

    the_len = len(columns)
    ws.write_merge(0, 1, 0, the_len - 1, '{}'.format(heading.upper()), style)
    for col_num in range(len(columns)):
        ws.write(row_num, col_num, columns[col_num], style0)
        ws.row(2).height_mismatch = True
        ws.row(2).height = 256 * 2

        ws.col(0).width = int(20 * 400)
        ws.col(1).width = int(20 * 250)
        ws.col(2).width = int(20 * 100)
        ws.col(3).width = int(20 * 250)
        ws.col(4).width = int(20 * 200)

    rows = json.loads(upload.failed_savings)['failed_saving']

    for row in rows:
        row_num += 1
        ws.write(row_num, 0, row['account number'])
        ws.write(row_num, 1, row['reason'])

    wb.save(response)
    return response


class OtherLoansReport(View):
    def get(self, request, *args, **kwargs):
        return


class AgingReportExcelV2(View):
    def get(self, request):

        title = 'Loan aging report'
        member = self.request.GET.get('member') if self.request.GET.get('member') != '' and self.request.GET.get(
            'member') != 'None' else None
        officer = self.request.GET.get('officer') if self.request.GET.get('officer') != '' and self.request.GET.get(
            'officer') != 'None' else None
        report = self.request.GET.get('report') if self.request.GET.get('report') != '' else None
        the_data = get_aging_report_data_v2(request, member, officer)

        # check if data is empty
        if the_data['all_loans'] == 'empty':
            is_empty = True
        else:

            is_empty = False

        is_empty = False

        if report == 'overdue_payments':
            columns = ['Client', 'Contact', 'Officer in charge', 'Payment date', 'Overdue(days)', 'Expected amount']
            if not is_empty:
                the_data = the_data['df_overdue_payments'].values.tolist()
            else:
                the_data = []


        elif report == 'individual_arrears':
            columns = ['Client', 'Contact', 'Amount']
            if not is_empty:
                the_data = the_data['df_individual_arrears'].values.tolist()
            else:
                the_data = []

        else:
            columns = ['Client', 'Contact', 'Officer in charge', 'Payment date', 'Overdue(days)', 'Expected amount']
            if not is_empty:
                the_data = the_data['df_overdue_payments'].values.tolist()
            else:
                the_data = []

        mem_name = ''
        if member is not None:
            mem_name = Member.objects.get(id=member).biodata.name
        officer_name = ''
        if officer is not None:
            officer_name = Staff.objects.get(id=officer).biodata.name

        response = HttpResponse(content_type='application/ms-excel')
        if report == 'individual_arrears':
            file_name = f'Individual arrears'
        else:
            file_name = f'Overdue payments'

        response['Content-Disposition'] = 'attachment; filename="{}.xls"'.format(file_name.capitalize())
        heading = file_name
        if officer_name != '':
            heading = f'{file_name} for loan officer - {officer_name}'
        if officer_name != '' and mem_name != '':
            heading = f'{file_name} and member - {mem_name}'
        elif mem_name != '':
            heading = f'{file_name} for member - {mem_name}'

        row_num = 2

        sheet_name = file_name
        wb = xlwt.Workbook(encoding='utf-8')
        ws = wb.add_sheet(sheet_name.capitalize())

        the_len = len(columns)

        ws.write_merge(0, 1, 0, the_len - 1, '{}'.format(heading.upper()), style)
        for col_num in range(len(columns)):
            ws.write(row_num, col_num, columns[col_num], style0)
            ws.row(2).height_mismatch = True
            ws.row(2).height = 256 * 2

            ws.col(0).width = int(20 * 400)
            ws.col(1).width = int(20 * 250)
            ws.col(2).width = int(20 * 100)
            ws.col(3).width = int(20 * 250)
            ws.col(4).width = int(20 * 200)
            ws.col(4).width = int(20 * 200)

        # print(the_data[0])
        # print('REPORT',report)
        if report == 'overdue_payments':
            the_count = 0
            # the_prnt_data = the_data[0]
            # del the_prnt_data[16]
            # del the_prnt_data[-1]
            # print(the_prnt_data)
            for actual_row in the_data:
                # convert the string row.overdue_payments to a list
                the_over_due_payments = actual_row[-1]
                the_count = the_count + len(the_over_due_payments)

                for holders_row in the_over_due_payments:
                    row_num += 1
                    # print('ACT', actual_row[-2])
                    # first write the first three
                    ws.write(row_num, 0, actual_row[2])
                    try:
                        ws.write(row_num, 1,
                                 actual_row[-2] if actual_row[-2] != None and actual_row[-2] != 'None' else '')
                    except Exception as e:
                        ws.write(row_num, 1, '')
                    # print('actual_row[16]',actual_row[16])
                    ws.write(row_num, 2, actual_row[-4])

                    ws.write(row_num, 3, holders_row['payment_date'].strftime('%Y-%m-%d'))
                    ws.write(row_num, 4, holders_row['days_overdue'])
                    ws.write(row_num, 5, holders_row['expected_amount'])
        else:
            for holders_row in the_data:
                row_num += 1
                ws.write(row_num, 0, holders_row[2])
                ws.write(row_num, 1, holders_row[-4] if holders_row[-4] != None and holders_row[-4] != 'None' else '')
                ws.write(row_num, 2, holders_row[-3])

        wb.save(response)
        return response

def disbursement_report_excel(request):
    response = HttpResponse(content_type='application/ms-excel')
    business = Business.objects.filter(id=businessdata(request)).first()
    branch = biz_staff_branch(request.user)

    business_context = get_business_loans_context_data(request)
    business_filter = business_context['business_filter']

    start_date_str = request.GET.get('start_date', '')
    end_date_str = request.GET.get('end_date', '')
    officer = request.GET.get('officer', '')


    # if start_date_str is None or start_date_str == 'None':
    #     start_date_str = ''
    # if end_date_str is None or end_date_str == 'None':
    #     end_date_str = ''

    start_date = datetime.strptime(start_date_str, '%Y-%m-%d').date() if start_date_str else None
    end_date = datetime.strptime(end_date_str, '%Y-%m-%d').date() if end_date_str else None
    #get all loans with status above 3
    all_loans = Loans.objects.filter(branch__business_id__in=business_context['data_context'], loan_status__in=[3,4, 8, 7,6])
    if start_date is not None:
        all_loans = all_loans.filter(schedule_start__gte=start_date)
    if end_date is not None:
        all_loans = all_loans.filter(schedule_start__lte=end_date)

    officer_name = ''
    if officer and officer != 'None':
        officer_name = Staff.objects.get(id=officer).biodata.name
        all_loans = all_loans.filter(officer_id=officer)
    

    general_wording_start=f' from {start_date_str}' if start_date_str is not None and start_date_str else ''
    general_wording_end=f' up to {end_date_str}' if end_date_str is not None and end_date_str else ''
    officer_wording = f' for  {officer_name}' if officer_name else ''

    file_name = f'Disbursement report{general_wording_start}{general_wording_end}{officer_wording}'
    response['Content-Disposition'] = 'attachment; filename="{}.xls"'.format(file_name.capitalize())
    heading = file_name
    sheet_name = 'Disbursement report'
    wb = xlwt.Workbook(encoding='utf-8')
    ws = wb.add_sheet(sheet_name.capitalize())

    # Sheet header, first row
    row_num = 2

    columns = ['Member', 'Contact', 'Amount approved', 'Duration', 'Officer', 'Schedule start', 'Status', 'Branch']

    the_len = len(columns)
    ws.write_merge(0, 1, 0, the_len - 1, '{}'.format(heading.upper()), style)
    for col_num in range(len(columns)):
        ws.write(row_num, col_num, columns[col_num], style0)
        ws.row(2).height_mismatch = True
        ws.row(2).height = 256 * 2

        ws.col(0).width = int(20 * 400)
        ws.col(1).width = int(20 * 250)
        ws.col(2).width = int(20 * 100)
        ws.col(3).width = int(20 * 250)
        ws.col(4).width = int(20 * 200)
        ws.col(5).width = int(20 * 200)
        ws.col(6).width = int(20 * 200)

    all_loans = all_loans.annotate(
        loan_status_word=Case(
            When(loan_status=0, then=Value('Applied')),
            When(loan_status=1, then=Value('Appraised')),
            When(loan_status=2, then=Value('Approved')),
            When(loan_status=3, then=Value('Disbursed')),
            When(loan_status=4, then=Value('Closed')),
            When(loan_status=5, then=Value('Rejected')),
            When(loan_status=6, then=Value('Waived off')),
            When(loan_status=7, then=Value('Written off')),
            When(loan_status=8, then=Value('Topped up')),
            default=Value('Unknown'),  # Set default value if none of the conditions match
            output_field=CharField()  # Specify the output field type
    )).values_list(
        'applicant__biodata__name', 
        'applicant__biodata__contact',
        'amount_approved',
        'approved_duration',
        'officer__biodata__name',
        'schedule_start',
        'loan_status_word',
        'branch__name'
        
    )
    
    for holders_row in all_loans:
        row_num += 1
        for col_num in range(len(holders_row)):
            ws.write(row_num, col_num, holders_row[col_num])

    wb.save(response)
    return response