import json
import os
import traceback

import pandas as pd
from datetime import datetime, timedelta

import xlwt
from django.contrib import messages
from django.core.paginator import Paginator, PageNotAnInteger, EmptyPage
from django.db import transaction
from django.db.models import F, OuterRef, Subquery, Q, Case, When, Value, IntegerField
from django.http import HttpResponseRedirect, HttpResponse
from django.shortcuts import render, get_object_or_404, redirect
from django.urls import reverse_lazy
from django.views import View

from accounts.models import Business, Staff, Branch
from client_custom.namayingo.forms import AddGroupForm
from client_custom.namayingo.models import UploadedMemberExcel, ExcelApproval, RequiredUploadApprovals
from client_custom.namayingo.utils import bulk_deposit_from_excel, validate_excel_structure, recording_share_details, \
    recording_loan_repayments_and_fines
from commons.logger import logger
from loans.models import Loanguarantors, Loans, LoanFines, LoanFinePayments
from loans.utils.loan_details import get_business_loans_context_data
from sacco.constants import DELETE
from sacco.forms import AddGroupMemberForm
from sacco.models import GroupMember, ActivityLog, Member, ReasonMemberRemoval, OtherSettings, MemberAccount, \
    AccountBroker, AccountTypes, SavingsSharesRecalculate
from sacco.utils import branch_id, businessdata, biz_data, is_excel_doc, biz_staff, session_business_data, \
    determine_business_id_list
from sacco.views import PermView
from transactions.models import Account, Transactions, SharesTransactions


style = xlwt.easyxf('font: color-index black, bold on; pattern: pattern solid, fore_colour white; align: wrap on,'
                    'vert centre, horiz center; borders: top_color black, bottom_color black,right_color black, '
                    'left_color black, left thin, right thin, top thin, bottom thin;')

style0 = xlwt.easyxf(
    'font: color-index white, bold on; pattern: pattern solid, fore_colour black; align: wrap off,vert centre, '
    'horiz center; borders: top_color black, bottom_color black,right_color black, '
    'left_color black, left thin, right thin, top thin, bottom thin')




class DeleteGroupMember(PermView):
    perm_name = 'delete_member'

    def post(self, request, pk, pk1, *args, **kwargs):
        # print(f'pk:{pk}')
        # print(f'pk1:{pk1}')
        business_context = get_business_loans_context_data(request)
        member = get_object_or_404(GroupMember, member__branch=business_context['branch'], id=pk)

        check_loans = Loans.objects.filter(loan_status=3, group_guaratorship=True, applicant=member.member )
        if check_loans.exists():
            loan_status = True
        else:
            loan_status = False

        # print(f'EXPECTED loan status:{loan_status}')

        if member.is_signatory is True:
            messages.error(request, 'error', extra_tags='Can not delete a signatory. First make member ordinary')
            return HttpResponseRedirect(reverse_lazy('namayingo_group_members', args=[pk1]))
        
        if loan_status:
            messages.error(request, 'error', extra_tags='Can not delete a Member with running loan')
            return HttpResponseRedirect(reverse_lazy('namayingo_group_members', args=[pk1]))
        
        reason = self.request.POST.get('reason')
        ReasonMemberRemoval.objects.create(reason=reason, member=member)
        member.delete()
        messages.success(request, 'success', extra_tags='Successfully deleted Member')
        message = f"{request.user.staff.biodata.name} deleted Member"
        ActivityLog.objects.create(actor=request.user,
                                   action_type=DELETE,
                                   title='Group delete',
                                   branch=business_context['branch'].id,
                                   remarks=message)
        return HttpResponseRedirect(reverse_lazy('namayingo_group_members', args=[pk1]))


class GroupMembers(PermView):
    perm_name = 'view_member'
    template_name = 'namayingo/group_members.html'

    def get(self, request, pk, *args, **kwargs):
        business_context = session_business_data(request)
        data_context = determine_business_id_list(request.user.staff, business_context)
        try:
            group = Member.objects.filter(is_group=True).prefetch_related('members').get(id=pk)
        except Member.DoesNotExist:
            return render(request, 'errors/404.html', status=404)
        except Member.MultipleObjectsReturned:
            group = Member.objects.filter(is_group=True, is_active=True, id=pk).prefetch_related('members').first()
        except Exception:
            return render(request, 'errors/500.html', status=500)
        form = AddGroupMemberForm
        group_membership = GroupMember.objects.filter(group=group)
        account_subquery = AccountBroker.objects.filter(members=OuterRef('pk')).values('the_account__acc_number')[:1]
        members = Member.objects.filter(groups__in=group_membership).annotate(account_number=Subquery(account_subquery),
                                                              ur_name=F('biodata__name'), grp_mem_id=F('groups__id'), role=F('groups__role'))

        sacco_members = Member.objects.filter(is_active=True,branch__business_id__in=data_context['all_ids']).exclude(
            id__in=members.values_list('id')).annotate(account_number=Subquery(account_subquery),
                                                              ur_name=F('biodata__name'))
        title = f'{group.name} Members'

        http = 'http://' if not request.is_secure() else 'https://'
        baseUrl = request.headers['HOST']
        base_url = http + baseUrl

        context = {
            'base_url': base_url, 
            'group': group,
            'members': members,
            'member_count': members.count(),
            'title': title,
            'sacco_members': sacco_members.values('id', 'ur_name', 'account_number'),
            'form': form
        }
        return render(request, self.template_name, context)

class BulkDeposit(PermView):
    perm_name = 'can_deposit'
    template_name = 'namayingo/member_bulk_deposit.html'

    def get_group(self, request, pk):
        try:
            group = Member.objects.filter(is_group=True).prefetch_related('members').get(id=pk)
            return group
        except Member.DoesNotExist:
            return render(request, 'errors/404.html', status=404)
        except Member.MultipleObjectsReturned:
            group = Member.objects.filter(is_group=True, is_active=True, id=pk).prefetch_related('members').first()
            return group
        except Exception as e:
            logger.error(str(e))
            return render(request, 'errors/500.html', status=500)

    def get(self, request, pk, *args, **kwargs):
        title = 'Make Bulk deposits'
        group = self.get_group(request, pk)
        accounts = Account.objects.filter(business=businessdata(request),
                                          category__name__in=['Mobile Money', 'Cash', 'Bank'])
        context = {
            'title': title,
            'group': group,
            'accounts': accounts,
        }
        return render(self.request, self.template_name, context)

    def post(self, request, pk, *args, **kwargs):
        group = self.get_group(request, pk)
        try:
            files = self.request.FILES.get('member_depos')
            # depo_date = self.request.POST.get('depo_date')
            account = self.request.POST.get('account')
            user = self.request.user
            business = biz_data(self.request.user)
            expected_columns = ['ACCOUNT', 'SAVING', 'DATE']
            is_valid = validate_excel_structure(files, expected_columns)
            if is_valid:
                bulk_deposit_from_excel(files, account, group, user, business)
                messages.success(self.request, 'success', extra_tags='Successfully uploaded')
            else:
                messages.error(self.request, 'error', extra_tags='Please check excel format')
        except Exception as e:
            # print(traceback.print_exc())
            messages.error(self.request, 'error', extra_tags='Error occurred while uploading excel')

        return HttpResponseRedirect(reverse_lazy('namayingo_make_bulk_deposit', args=[group.id]))


class UploadMemberData(PermView):
    perm_name = 'handle_group_transactional_data'

    def get_group(self, request, pk):

        try:
            group = Member.objects.filter(is_group=True).prefetch_related('members').get(id=pk)
            return group
        except Member.DoesNotExist:
            return render(request, 'errors/404.html', status=404)
        except Member.MultipleObjectsReturned:
            group = Member.objects.filter(is_group=True, is_active=True, id=pk).prefetch_related('members').first()
            return group
        except Exception as e:
            logger.error(str(e))
            return render(request, 'errors/500.html', status=500)

    @staticmethod
    def under_score(string):
        return string.lower()

    def get(self, request, pk, *args, **kwargs):
        business_context = get_business_loans_context_data(request)

        if business_context['branch'] is None:
            messages.error(
                request, "error",
                extra_tags="Action not allowed in the central view. Please logout and login into the respective branch"
            )
            return HttpResponseRedirect(request.META.get("HTTP_REFERER"))
        group = self.get_group(request, pk)
        title = f'{group.name} - Group transactional data'
        accounts = Account.objects.filter(business=business_context['branch'].business,
                                          category__name__in=['Mobile Money', 'Cash', 'Bank'])
        grp_accounts = group.owners.all()
        grp_accounts_details = Account.objects.filter(member_id__in=grp_accounts.values_list('the_account__id', flat=True))
        
        can_approve_group_transactional_data = False
        if request.user.staff.position.permissions.filter(item_name='can_approve_group_transactional_data').exists():
            can_approve_group_transactional_data = True

        uploads = UploadedMemberExcel.objects.filter(business=business_context['branch'].business, group=group).order_by('-id')

        # NOTE:temporary for now (Requested by client) -------------------------------------------------
        # TODO: Remove once done
        other_accounts = Account.objects.filter(business=business_context['branch'].business, name__icontains='group collection',
                                          category__name__in=['Mobile Money', 'Cash', 'Bank'])
        # ---------------------------------------------------------------------------------------------------

        context = {
            'title': title,
            'group': group,
            'accounts': accounts,
            'uploads': uploads,
            'grp_accounts': grp_accounts_details,
            'can_approve_group_transactional_data': can_approve_group_transactional_data,
            'other_accounts':other_accounts
        }
        return render(self.request, 'namayingo/member_data.html', context)

    def post(self, request, pk, *args, **kwargs):
        group = self.get_group(request, pk)
        business_context = get_business_loans_context_data(request)
        try:
            files = self.request.FILES.get('member_data')
            account = self.request.POST.get('account')
            trans_date = self.request.POST.get('trans_date')
            business = business_context['branch'].business
            staff = biz_staff(self.request.user)
            expected_columns = ['FULLNAME', 'ACCOUNT', 'SAVING', 'PRINCIPAL', 'INTEREST', 'SHARES', 'FINES']
            is_valid = validate_excel_structure(files, expected_columns)
            if is_valid:
                if is_excel_doc(files):
                    if account != 'm':  # not paying with member account
                        UploadedMemberExcel.objects.create(file=files, uploaded_by=staff, payment_account_id=account,
                                                           business=business, group=group, transactions_date=trans_date)  # Set for approval
                    else:
                        UploadedMemberExcel.objects.create(file=files, uploaded_by=staff, payment_account_id=account,
                                                           business=business, group=group, member_account='m',transactions_date=trans_date)
                    messages.success(self.request, 'success', extra_tags='Successfully uploaded')
                else:
                    messages.error(self.request, 'error', extra_tags='This is not an excel file')
            else:
                messages.error(self.request, 'error', extra_tags='Please check excel format')
        except Exception as e:
            print(traceback.print_exc())
            messages.error(self.request, 'error', extra_tags='Error occurred while uploading excel')

        return HttpResponseRedirect(reverse_lazy('namayingo_upload_data', args=[group.id]))


class CreateGroupView(PermView):
    perm_name = 'add_group'

    def get(self, request):
        business_context = get_business_loans_context_data(request)

        if business_context['branch'] is None:
            messages.error(
                request, "error",
                extra_tags="Action not allowed in the central view. Please logout and login into the respective branch"
            )
            return HttpResponseRedirect(request.META.get("HTTP_REFERER"))

        biz = business_context['branch'].business
        branch = business_context['branch'].id
        auto_acc_no = Business.objects.get(id=branch).auto_acc_no
        acc_types = AccountTypes.objects.filter(status=True, business=business_context['branch'].business)
        title = 'Create Group'
        context = {
            'biz': biz,
            'branch': branch,
            'title': title,
            'auto_acc_no': auto_acc_no,
            'acc_types': acc_types,
        }

        return render(self.request, 'namayingo/add_group.html', context)

    def post(self, request):
        business_context = get_business_loans_context_data(request)
        form = AddGroupForm(self.request.POST, user=self.request.user, branch=business_context['branch'])
        if form.is_valid():
            form.save()
            messages.success(self.request, 'success', extra_tags='Successfully added group')
        else:
            # print(form.errors)
            messages.error(self.request, 'success', extra_tags='Error submitting form')
        return HttpResponseRedirect(reverse_lazy('namayingo_create_group'))


class UploadsWaitingView(PermView):
    perm_name = 'can_deposit'

    def get(self, request):
        business_context = get_business_loans_context_data(request)

        if business_context['branch'] is None:
            messages.error(
                request, "error",
                extra_tags="Action not allowed in the central view. Please logout and login into the respective branch"
            )
            return HttpResponseRedirect(request.META.get("HTTP_REFERER"))

        biz = business_context['branch'].business.id
        uploads = UploadedMemberExcel.objects.filter(business=biz)

        context = {
            'title': 'Group member transactions uploads',
            'uploads': uploads
        }
        return render(self.request, 'namayingo/waiting_approval.html', context)


class UploadsWaitingDetailView(PermView):
    perm_name = 'can_approve_group_transactional_data'

    def get(self, request, pk):
        business_context = get_business_loans_context_data(request)

        if business_context['branch'] is None:
            messages.error(
                request, "error",
                extra_tags="Action not allowed in the central view. Please logout and login into the respective branch"
            )
            return HttpResponseRedirect(request.META.get("HTTP_REFERER"))

        biz = business_context['branch'].business.id
        uploads = UploadedMemberExcel.objects.filter(business_id=biz).get(id=pk)
        approvals = ExcelApproval.objects.filter(upload__business=biz, upload=uploads)
        # Load XLSX file
        try:
            df = pd.read_excel(uploads.file.path, sheet_name='Sheet1', dtype={'ACCOUNT': str})
            # Convert to HTML
            html_output = df.to_html()
        except ValueError:
            html_output = None

        #check if the logged in user already approved the upload
        check_approval = ExcelApproval.objects.filter(admin=request.user.staff, upload=uploads)
        # print(check_approval)
        has_approved = False
        if check_approval.exists():
            has_approved = True


        context = {
            'title': 'Uploads detail',
            'uploads': uploads,
            'data': html_output,
            'business': biz,
            'approvals': approvals,
            'has_approved': has_approved
        }
        return render(self.request, 'namayingo/waiting_approva_detaill.html', context)


class ApproveUploadView(PermView):
    perm_name = 'can_deposit'

    @transaction.atomic
    def post(self, request, pk):
        business_context = get_business_loans_context_data(request)
        business_context_2 = session_business_data(request)
        data_context = determine_business_id_list(request.user.staff, business_context_2)
        business_context_revised = get_business_loans_context_data(request)

        if business_context['branch'] is None:
            messages.error(
                request, "error",
                extra_tags="Action not allowed in the central view. Please logout and login into the respective branch"
            )
            return HttpResponseRedirect(request.META.get("HTTP_REFERER"))


        biz = business_context['branch'].business.id
        staff = biz_staff(self.request.user)
        business = business_context['branch'].business
        status = request.POST.get('status')
        upload = UploadedMemberExcel.objects.filter(business=biz).get(id=pk)
        approvals = upload.approvals.count()
        try:
            req_approvals = RequiredUploadApprovals.objects.get(business=biz).number
        except RequiredUploadApprovals.DoesNotExist:
            req_approvals = 0
        if status and status == 'ACCEPT':
            if approvals <= req_approvals and upload.status != 'APPROVED':
                this_staff, created = ExcelApproval.objects.get_or_create(admin=staff, upload=upload)
                if approvals == req_approvals:
                    upload.status = 'APPROVED'
                    upload.save()
                    df = pd.read_excel(upload.file.path, dtype={'ACCOUNT': str}, engine='openpyxl')
                    # df = pd.read_excel(uploads.file.path, sheet_name='Sheet1', dtype={'ACCOUNT': str})
                    df.fillna('', inplace=True)
                    df.columns.astype(str)
                    df.columns = df.columns.str.strip()
                    # to_dict = df.to_dict('records')

                    # making bulk deposit
                    savings_details = bulk_deposit_from_excel(request, df, upload.payment_account.id, upload.group, upload.transactions_date)

                    # recording share transactions
                    shares_payment_details = recording_share_details(request, df, upload.payment_account.id, upload.group, upload.transactions_date)

                    # loan payment - principal, interest and fines
                    loan_payment_details = recording_loan_repayments_and_fines(request, df, upload.payment_account.id, upload.group, upload.transactions_date)

                    all_created_trans = savings_details['added_transactions'] + shares_payment_details['all_capital_transactions'] + loan_payment_details['added_transactions']

                    all_failed_records = savings_details['failed_savings'] + shares_payment_details['failed_shares'] + loan_payment_details['failed_loan_payments']

                    #update the upload record with these details
                    UploadedMemberExcel.objects.filter(id=upload.id).update(
                        failed_records=json.dumps(all_failed_records),
                        failed_number = len(all_failed_records),
                        added_transactions = json.dumps(all_created_trans),
                        added_loan_fines = json.dumps(loan_payment_details['fines_created']),
                        added_loan_fines_payments = json.dumps(loan_payment_details['fine_payments_created']),
                        added_shares = json.dumps(shares_payment_details['all_share_transactions'])
                    )

                    if business_context_revised['branch'] is None:
                        for biz_id in data_context['all_ids']:
                            the_biz_branch = Branch.objects.filter(business_id=biz_id).first()
                            SavingsSharesRecalculate.objects.create(
                                branch=the_biz_branch,
                                update_type='shares'
                            )
                            SavingsSharesRecalculate.objects.create(
                                branch=the_biz_branch,
                                update_type='savings'
                            )
                    else:
                        SavingsSharesRecalculate.objects.create(
                            branch=business_context_revised['branch'],
                            update_type='shares'
                        )
                        SavingsSharesRecalculate.objects.create(
                            branch=business_context_revised['branch'],
                            update_type='savings'
                        )

                    messages.success(request, 'success', extra_tags='Successfully approved excel')
            else:
                messages.error(request, 'error', extra_tags=f'Only {req_approvals} approvals needed')
        elif status and status == 'REJECT':
            upload.status = 'REJECTED'
            upload.save()
            messages.success(request, 'success', extra_tags='Rejected successfully')

        return HttpResponseRedirect(reverse_lazy('uploads_preview', args=[upload.id]))


class UpdateFileView(PermView):

    def post(self, request, pk):
        # print('its me running')
        business_context = get_business_loans_context_data(request)
        biz = business_context['branch'].business.id
        new_file = self.request.FILES['uploaded']
        # print('my file', new_file)
        upload = UploadedMemberExcel.objects.filter(business=biz).get(id=pk)
        try:
            file_path = upload.file.path
        except ValueError as e:
            logger.error(traceback.print_exc())
            file_path = None
        try:
            if file_path:
                os.remove(file_path)
            upload.file = new_file
            upload.save()
            # print(upload.file_url)
            messages.success(request, 'success', extra_tags=f"The file {file_path} has been successfully updated.")
        except FileNotFoundError:
            messages.error(request, 'error', extra_tags=f"The file {file_path} does not exist.")
        except PermissionError:
            logger.error(traceback)
            messages.error(request, 'error', extra_tags='Error occurred while uploading file')
            print(f"You do not have permission to delete the file {file_path}.")
        except Exception as e:
            messages.error(request, 'error', extra_tags='Error occurred while uploading file')
            logger.error(traceback.print_exc())
        return HttpResponseRedirect(reverse_lazy('uploads_preview', args=[upload.id]))
    


class DeleteGroupView(PermView):
    perm_name = 'delete_group'

    def get(self, request, pk, *args, **kwargs):
        business_context = get_business_loans_context_data(request)
        # get the group
        the_group = Member.objects.filter(is_active=True, is_group=True, id=pk)
        if not the_group.exists():
            messages.error(request, 'error', extra_tags='Group nolonger exists')
            return HttpResponseRedirect(reverse_lazy('sacco_groups'))
        
        the_group_name = the_group.first().name
        the_group_id = the_group.first().id
        
        # get the members - the ids
        all_members_ids=list(GroupMember.objects.filter(group=the_group.first()).values_list('member__id', flat=True))
        #check if it has members and check if they have loans
        check_loans = Loans.objects.filter(applicant_id__in =all_members_ids, group_guaratorship=True, loan_status__lt=4)
        if check_loans.exists():
            messages.error(request, 'error', extra_tags='Some members in the group still have running loans or loan applications where the group is a guarantor')
            return HttpResponseRedirect(reverse_lazy('sacco_groups'))
        
        #remove the group
        # Member
        # Account broker
        acc_broker = AccountBroker.objects.filter(members=the_group.first())
        # the account
        for broker in acc_broker:
            actual_acc_transactions = Account.objects.filter(member_id=broker.the_account.id)
            if actual_acc_transactions.exists():
                actual_acc_transactions=actual_acc_transactions.first()
                #check if there are transations available
                check_trans = Transactions.objects.filter(Q(account_dr=actual_acc_transactions) | Q(account_cr=actual_acc_transactions))
                if check_trans.exists():
                    messages.error(request, 'error', extra_tags='Group cant be removed because there are transactions affiliated with its accounts')
                    return HttpResponseRedirect(reverse_lazy('sacco_groups'))
                
        today = datetime.today()
        todays_date = today.strftime("%Y-%m-%d")
                
        the_group.update(is_active=False, name=f'{the_group.first().name} (Group deleted) on {str(todays_date)}')
        GroupMember.objects.filter(group_id=the_group_id).delete()

        message = f'Group with the name "{the_group_name}" deleted'
        
        ActivityLog.objects.create(actor=request.user,
                                   action_type=DELETE,
                                   title='Group delete',
                                   branch=business_context['branch'].id
                                   ,
                                   remarks=message)
        messages.success(request, 'success', extra_tags='Group successfully removed')
        return HttpResponseRedirect(reverse_lazy('sacco_groups'))


class ReverseUploadedData(PermView):
    perm_name = 'handle_group_transactional_data'

    def get(self, request, pk):
        business_context = get_business_loans_context_data(request)
        the_upload = UploadedMemberExcel.objects.filter(id=pk)
        if not the_upload.exists():
            messages.error(request, 'error', extra_tags='Upload nolonger exists')
            return redirect(request.META.get('HTTP_REFERER', '/'))

        the_upload = the_upload.first()

        # reverse all transactions
        if the_upload.added_transactions is not None and the_upload.added_transactions != '':
            Transactions.objects.filter(id__in=json.loads(the_upload.added_transactions)).delete()
        # reverse share transaction
        if the_upload.added_shares is not None and the_upload.added_shares != '':
            SharesTransactions.objects.filter(id__in=json.loads(the_upload.added_shares)).delete()
        # reverse loan fines
        if the_upload.added_loan_fines is not None and the_upload.added_loan_fines != '':
            LoanFines.objects.filter(id__in=json.loads(the_upload.added_loan_fines)).delete()
        # reverse loan fine payments
        if the_upload.added_loan_fines_payments is not None and the_upload.added_loan_fines_payments != '':
            LoanFinePayments.objects.filter(id__in=json.loads(the_upload.added_loan_fines_payments)).delete()

        # delete the upload
        UploadedMemberExcel.objects.filter(id=the_upload.id).delete()
        messages.success(request, 'success', extra_tags='Upload successfully removed')
        return redirect(request.META.get('HTTP_REFERER', '/'))


class DownloadExcelFailedUploadData(PermView):
    def get(self, request, pk):
        response = HttpResponse(content_type='application/ms-excel')

        the_upload = UploadedMemberExcel.objects.filter(id=pk)
        if not the_upload.exists():
            messages.error(request, 'error', extra_tags='Upload nolonger exists')
            return redirect(request.META.get('HTTP_REFERER', '/'))

        the_upload=the_upload.first()


        file_name = f'Failed group upload records registered on {str(the_upload.created_at)}'
        sheet_name = f'failed group upload details'

        response['Content-Disposition'] = 'attachment; filename="{}.xls"'.format(file_name.capitalize())
        heading = f'FAILED GROUP UPLOADS'
        wb = xlwt.Workbook(encoding='utf-8')
        ws = wb.add_sheet(sheet_name.capitalize(), cell_overwrite_ok=True)

        # Sheet header, first row
        row_num = 2

        columns = ['Account number', 'Section', 'Reason']

        the_len = len(columns)
        ws.write_merge(0, 1, 0, the_len - 1, '{}'.format(heading.upper()), style)
        for col_num in range(len(columns)):
            ws.write(row_num, col_num, columns[col_num], style0)
            ws.row(2).height_mismatch = True
            ws.row(2).height = 256 * 2

            ws.col(0).width = int(20 * 400)
            ws.col(1).width = int(20 * 250)
            ws.col(2).width = int(20 * 100)
            ws.col(3).width = int(20 * 250)
            ws.col(4).width = int(20 * 200)

        rows = json.loads(the_upload.failed_records)

        for row in rows:
            row_num += 1
            ws.write(row_num, 0, row['account number'])
            ws.write(row_num, 1, row['failed section'])
            ws.write(row_num, 2, row['reason'])

        wb.save(response)
        return response


class AllGroupsUploadMemberData(PermView):
    perm_name = 'handle_group_transactional_data'

    @staticmethod
    def under_score(string):
        return string.lower()

    def get(self, request, *args, **kwargs):
        business_context = get_business_loans_context_data(request)
        query_params = request.GET.copy()
        query_params.pop('page', None)

        if business_context['branch'] is None:
            messages.error(
                request, "error",
                extra_tags="Action not allowed in the central view. Please logout and login into the respective branch"
            )
            return HttpResponseRedirect(request.META.get("HTTP_REFERER"))
        title = f'All Groups transactional data'

        can_approve_group_transactional_data = False
        if request.user.staff.position.permissions.filter(item_name='can_approve_group_transactional_data').exists():
            can_approve_group_transactional_data = True

        uploads = UploadedMemberExcel.objects.filter(business=business_context['branch'].business).annotate(
            status_order=Case(
                When(status='PENDING', then=Value(1)),
                When(status='APPROVED', then=Value(2)),
                When(status='REJECTED', then=Value(3)),
                default=Value(4),
                output_field=IntegerField()
            )
        ).order_by('status_order')

        loan_officers = Staff.objects.filter(branch__business_id__in=business_context['data_context'])

        groups = Member.objects.filter(is_active=True, is_group=True)

        loan_officer = request.GET.get('loan_officer')
        start_date = request.GET.get('start_date')
        end_date = request.GET.get('end_date')
        status_filter = request.GET.get('status_filter')
        group_filter = request.GET.get('group_filter')

        if loan_officer:
            uploads = uploads.filter(uploaded_by_id=loan_officer)
        if status_filter:
            uploads = uploads.filter(status=status_filter)
        if group_filter:
            uploads = uploads.filter(group__id=group_filter)
        if start_date:
            uploads = uploads.filter(transactions_date__gte=start_date)
        if end_date:
            uploads = uploads.filter(transactions_date__lte=end_date)

        page = request.GET.get('page', 1)
        paginator = Paginator(uploads, 10)

        try:
            uploads = paginator.page(page)
        except PageNotAnInteger:
            uploads = paginator.page(1)
        except EmptyPage:
            uploads = paginator.page(paginator.num_pages)

        context = {
            'title': title,
            'uploads': uploads,
            'can_approve_group_transactional_data': can_approve_group_transactional_data,
            'start_number': 0 if int(page) == 1 else (int(page) - 1) * 10,
            'loan_officers': loan_officers,
            'loan_officer' : loan_officer,
            'start_date' : start_date,
            'groups' : groups,
            'end_date' : end_date,
            'status_filter': status_filter,
            'group_filter': group_filter,
            'query_params': query_params.urlencode(),
        }
        return render(self.request, 'namayingo/all_groups_member_data.html', context)