import datetime
import json
import random

import pandas as pd
from datetime import date

from django.db.models import Sum

from loans.models import Loans, LoanFinePayments, LoanFines
from loans.utils.loan_details import update_loan_summary_details, get_business_loans_context_data
from sacco.models import MemberAccount, AccountBroker, OtherSettings, GroupMember
from sacco.rq_task import bulk_member_deposit, bulk_member_share_purchase, repayments
from sacco.utils import userdata, branchdata, businessdata, branch_id
from transactions.models import Transactions, Account, SharesTransactions


def validate_excel_structure(file_path, expected_columns):
    # Read the Excel file into a DataFrame
    df = pd.read_excel(file_path)
    # Remove leading and trailing whitespaces from column names
    df.columns = df.columns.str.strip()
    # Get the columns from the DataFrame
    actual_columns = df.columns.tolist()

    # Check if the actual columns match the expected columns
    if set(actual_columns) == set(expected_columns):
        # print("Excel structure is valid.")
        return True
    else:
        # print("Excel structure is not valid. Expected columns:", expected_columns, "Actual columns:", actual_columns)
        return False


def is_excel_doc(file):
    excelSigs = [
        ('xlsx', b'\x50\x4B\x05\x06', 2, -22, 4),
        ('xls', b'\x09\x08\x10\x00\x00\x06\x05\x00', 0, 512, 8),  # Saved from Excel
        ('xls', b'\x09\x08\x10\x00\x00\x06\x05\x00', 0, 1536, 8),  # Saved from LibreOffice Calc
        ('xls', b'\x09\x08\x10\x00\x00\x06\x05\x00', 0, 2048, 8)  # Saved from Excel then saved from Calc
    ]

    for sigType, sig, whence, offset, size in excelSigs:
        file.seek(offset, whence)
        byts = file.read(size)

        if byts == sig:
            return True
    return False

def check_group_member(group, member):
    check_member = GroupMember.objects.filter(group=group, member=member)
    if not check_member.exists():
        return f'Member does not belong in {group.name}'
    return 'ok'

def validate_section_required_data(row, checklist):
    if row['ACCOUNT'].strip() == '':
        return 'Invalid account provided'

    for field in checklist:
        # remove quomas from the amt
        amt = str(row[field]).replace(',', '')
        if amt.strip() != '':
            try:
                # Try converting the value to an integer
                float_value = float(amt)
            except ValueError:
                return f'invalid {field} provided'
    return 'ok'


def bulk_deposit_from_excel(request, excel_data, debit_acc, group, transactions_date):
    business_context = get_business_loans_context_data(request)
    # account_dict = [{'account_number': str(row['ACCOUNT']).strip(), 'amount': str(row['SAVING']).strip()} for index, row
    #                 in enumerate(to_dict)]
    # bulk_member_deposit(user, account_dict, account, business)
    failed_savings = []
    added_transactions = []
    for index, row in excel_data.iterrows():
        # validate data
        row_validity = validate_section_required_data(row, ['SAVING'])
        acc_no = row['ACCOUNT'].strip()
        if row_validity != 'ok':
            failed_savings.append({
                'account number': acc_no,
                'failed section': 'savings',
                'reason': row_validity
            })
            continue

        # check if the account exists
        the_mem_broker = AccountBroker.objects.filter(the_account__acc_number=acc_no,
                                                      members__branch=business_context['branch'])
        if not the_mem_broker.exists():
            failed_savings.append({
                'account number': acc_no,
                'failed section': 'savings',
                'reason': 'Provided account not available or invalid'
            })
            continue

        the_mem_broker = the_mem_broker.first()
        member_belonging = check_group_member(group, the_mem_broker.members)
        if member_belonging != 'ok':
            failed_savings.append({
                'account number': acc_no,
                'failed section': 'savings',
                'reason': member_belonging
            })
            continue

        # excel savings
        excel_saving = str(row['SAVING']).replace(',', '')
        if excel_saving.strip() == '':
            excel_saving = 0
        else:
            excel_saving = float(excel_saving)



        #get tha actual transactions account for the member
        the_trans_acc = Account.objects.filter(member=the_mem_broker.the_account, business=business_context['branch'].business)
        if not the_trans_acc.exists():
            failed_savings.append({
                'account number': acc_no,
                'failed section': 'savings',
                'reason': 'Member account has an issue'
            })
            continue

        the_trans_acc = the_trans_acc.first()

        if excel_saving != 0:
            receipt = datetime.datetime.now().strftime("%y%m%d%H%M%S")
            random_number = random.randint(10, 99)
            receipt = f'{receipt}{str(the_trans_acc.id)}{str(random_number)}'

            savings_trans = Transactions.objects.create(
                branch=business_context['branch'],
                transaction_type='Deposit',
                account_dr_id=debit_acc,
                account_cr=the_trans_acc,
                narration=f'Deposit of {excel_saving} to {acc_no}',
                reporting_amount=excel_saving,
                added_by=request.user.staff,
                tx_date=date.today() if transactions_date is None else transactions_date,
                receipt=receipt,
            )
            added_transactions.append(savings_trans.id)

            # update the savings on member acc
            # credits = Transactions.objects.filter(account_cr=the_trans_acc).aggregate(
            # total=Sum('reporting_amount'))['total']
            # if credits is None:
            #     credits = 0
            # debits = Transactions.objects.filter(account_dr=the_trans_acc).aggregate(
            # total=Sum('reporting_amount'))['total']
            # if debits is None:
            #     debits = 0
            # balance = credits - debits
            # MemberAccount.objects.filter(id=the_trans_acc.member.id).update(balance=balance)

    return {
        'failed_savings': failed_savings,
        'added_transactions': added_transactions
    }


def recording_share_details(request, excel_data, debit_acc, group, transactions_date):
    business_context = get_business_loans_context_data(request)

    failed_shares = []
    all_share_transactions = []
    all_capital_transactions = []
    shares_account = Account.objects.filter(business=business_context['branch'].business, category__name='Shares').first()
    for index, row in excel_data.iterrows():
        # validate data
        row_validity = validate_section_required_data(row,['SHARES'])
        acc_no = row['ACCOUNT'].strip()
        if row_validity != 'ok':
            failed_shares.append({
                'account number': acc_no,
                'failed section': 'shares',
                'reason': row_validity
            })
            continue

        #check if the account exists
        the_mem_broker = AccountBroker.objects.filter(the_account__acc_number=acc_no,
                                                   members__branch=business_context['branch'])
        if not the_mem_broker:
            failed_shares.append({
                'account number': acc_no,
                'failed section': 'shares',
                'reason': 'Provided account not available or invalid'
            })
            continue

        the_mem_broker = the_mem_broker.first()
        member_belonging = check_group_member(group, the_mem_broker.members)
        if member_belonging != 'ok':
            failed_shares.append({
                'account number': acc_no,
                'failed section': 'shares',
                'reason': member_belonging
            })
            continue

        # excel shares
        excel_share = str(row['SHARES']).replace(',', '')
        if excel_share.strip() == '':
            excel_share = 0
        else:
            excel_share = float(excel_share)

        # record share transaction
        if excel_share != 0:
            business_setting = OtherSettings.objects.filter(business=business_context['branch'].business).first()
            if business_setting.share_price < 1:
                failed_shares.append({
                    'account number': acc_no,
                    'failed section': 'shares',
                    'reason': 'Share price is not set'
                })
                continue
            else:
                # determine amount for transaction
                bought_amt = float(excel_share) * float(business_setting.share_price)

            narrative = f'Sale of {str(excel_share)} shares to {str(the_mem_broker.members.biodata.name)} at {str(business_setting.share_price)} each.'

            shares_record = SharesTransactions.objects.create(
                buyer=the_mem_broker.members,
                shares=excel_share,
                date=date.today() if transactions_date is None else transactions_date,
                narration=narrative,
                branch=business_context['branch']
            )

            all_share_transactions.append(shares_record.id)

            #record actual transaction
            # transaction
            new_tx = Transactions.objects.create(
                branch=business_context['branch'],
                transaction_type="Sell Shares",
                account_dr_id=debit_acc,
                account_cr=shares_account,
                shares=shares_record,
                narration=narrative,
                tx_date=date.today() if transactions_date is None else transactions_date,
                reporting_amount=bought_amt,
                added_by=request.user.staff
            )

            all_capital_transactions.append(new_tx.id)



    return{
        'failed_shares': failed_shares,
        'all_share_transactions': all_share_transactions,
        'all_capital_transactions': all_capital_transactions
    }



def recording_loan_repayments_and_fines(request, excel_data, debit_acc, group, transactions_date):
    business_context = get_business_loans_context_data(request)
    # loop through records having figure on repayment and fine
    loan_receivables_acc = Account.objects.filter(
        name="Loan Receivables", business=business_context['branch'].business
    ).first()

    interest_acc = Account.objects.filter(
        name="Loan interest receivables", business=business_context['branch'].business
    ).first()

    fines_acc = Account.objects.filter(
        name="Loan Fines", business=business_context['branch'].business
    ).first()

    ln_interest_revenue_acc = Account.objects.filter(
        name="Loan Interest", business=business_context['branch'].business
    ).first()

    failed_loan_payments = []
    added_transactions = []
    fines_created = []
    fine_payments_created = []
    for index, row in excel_data.iterrows():
        # get the repayment
        acc_no = row['ACCOUNT'].strip()

        row_validity = validate_section_required_data(row, ['PRINCIPAL', 'INTEREST', 'FINES'])
        if row_validity != 'ok':
            failed_loan_payments.append({
                'account number': acc_no,
                'failed section': 'principal, interest and fine',
                'reason': row_validity
            })
            continue

        # excel principal
        excel_principal = str(row['PRINCIPAL']).replace(',', '')
        if excel_principal.strip() == '':
            excel_principal = 0
        else:
            excel_principal = float(excel_principal)

        # excel interest
        excel_interest = str(row['INTEREST']).replace(',', '')
        if excel_interest.strip() == '':
            excel_interest = 0
        else:
            excel_interest = float(excel_interest)

        # excel fine
        excel_fine = str(row['FINES']).replace(',', '')
        if excel_fine.strip() == '':
            excel_fine = 0
        else:
            excel_fine = float(excel_fine)

        # Check if the member acc exists
        the_mem_acc = AccountBroker.objects.filter(the_account__acc_number=acc_no, members__branch= business_context['branch'])
        if not the_mem_acc:
            failed_loan_payments.append({
                'account number': acc_no,
                'failed section': 'principal, interest and fine',
                'reason': 'Provided account not available or invalid'
            })
            continue

        the_mem_acc = the_mem_acc.first()
        member_belonging = check_group_member(group, the_mem_acc.members)
        if member_belonging != 'ok':
            failed_loan_payments.append({
                'account number': acc_no,
                'failed section': 'principal, interest and fine',
                'reason': member_belonging
            })
            continue


        # look for the loan
        get_loans = Loans.objects.filter(account__acc_number=acc_no, loan_status=3, branch=business_context['branch'])
        if get_loans.exists():
            the_loan = get_loans.first()
            # print(the_loan.loan_number, the_loan.id)
        else:
            # the_loan = None
            failed_loan_payments.append({
                'account number': acc_no,
                'failed section': 'principal, interest and fine',
                'reason': 'Individual has no active loan'
            })
            continue

        if the_loan is not None:
            # get loan summary details
            loan_summary_details = the_loan.loan_summary_details
            if loan_summary_details is not None and loan_summary_details != '':
                loan_summary_details = json.loads(loan_summary_details)
            else:
                # loan_summary_details = None
                continue

            is_error_summary = loan_summary_details["is_error"]
            if is_error_summary == 'yes':
                # loan has some issues
                failed_loan_payments.append({
                    'account number': acc_no,
                    'failed section': 'principal, interest and fine',
                    'reason': 'Current running loan has some issues'
                })
                continue

            else:
                principal_balance = loan_summary_details['principal_balance']
                interest_balance = loan_summary_details['interest_balance']

                # check if provided figures dont exceed the principal and interest balances
                if excel_principal > principal_balance:
                    failed_loan_payments.append({
                        'account number': acc_no,
                        'failed section': 'principal, interest and fine',
                        'reason': 'Provided principal is greater than principal balance on the loan',
                    })
                    continue

                if excel_interest > interest_balance:
                    failed_loan_payments.append({
                        'account number': acc_no,
                        'failed section': 'principal, interest and fine',
                        'reason': 'Provided interest is greater than interest balance on the loan',
                    })
                    continue

                #every thing is ok -- Adding payments
                # using the default receipting format as used before
                receipt = datetime.datetime.now().strftime("%y%m%d%H%M%S")
                random_number = random.randint(10, 99)
                receipt = f'{receipt}{str(the_loan.id)}{str(random_number)}'

                # principal payment
                if excel_principal != 0:
                    repayment_trans = Transactions.objects.create(
                        reporting_amount=excel_principal,
                        narration=f"Loan payment of {str(excel_principal)} as principal for loan: {the_loan.loan_number} given to {the_loan.applicant.biodata.name}({the_loan.account.acc_number})",
                        account_dr_id=debit_acc,
                        account_cr=loan_receivables_acc,
                        loan=the_loan,
                        tx_date=date.today() if transactions_date is None else transactions_date,
                        receipt=receipt,
                        transaction_type="Loan repayment",
                        added_by_id=userdata(request),
                        branch=business_context['branch'],
                    )
                    added_transactions.append(repayment_trans.id)

                # interest payment
                if excel_interest != 0:
                    # add the expected interest
                    exp_interest = Transactions.objects.create(
                        reporting_amount=excel_interest,
                        narration=f"Expected interest of {excel_interest} on loan: {the_loan.loan_number} given to {the_loan.applicant.biodata.name}({the_loan.account.acc_number})",
                        account_dr=interest_acc,
                        account_cr=ln_interest_revenue_acc,
                        loan=the_loan,
                        tx_date=date.today() if transactions_date is None else transactions_date,
                        transaction_type="Expected interest",
                        added_by_id=userdata(request),
                        branch=business_context['branch'],
                    )
                    added_transactions.append(exp_interest.id)

                    # actual payment of interest
                    interest_trans = Transactions.objects.create(
                        reporting_amount=excel_interest,
                        narration=f"Loan payment of {str(excel_interest)} as interest for loan: {the_loan.loan_number} given to {the_loan.applicant.biodata.name}({the_loan.account.acc_number})",
                        account_dr_id=debit_acc,
                        account_cr=interest_acc,
                        loan=the_loan,
                        tx_date=date.today() if transactions_date is None else transactions_date,
                        receipt=receipt,
                        transaction_type="Loan interest",
                        added_by_id=userdata(request),
                        branch=business_context['branch'],
                    )
                    added_transactions.append(interest_trans.id)

                # Create the fine
                if excel_fine != 0:
                    fine_date_realised = date.today().strftime('%Y-%m-%d')
                    the_fine_record = LoanFines.objects.create(
                        loan=the_loan,
                        amount=excel_fine,
                        added_by_id=userdata(request),
                        narration=f'Fine effected on {fine_date_realised}',
                    )
                    fines_created.append(the_fine_record.id)

                    # create the fine payment
                    the_fine_payment_record = LoanFinePayments.objects.create(
                        loan=the_loan,
                        amount=excel_fine,
                        added_by_id=userdata(request),
                    )
                    fine_payments_created.append(the_fine_payment_record.id)

                    # Actual fine transaction
                    fines_transaction = Transactions.objects.create(
                        reporting_amount=excel_fine,
                        narration=f"Loan payment of {str(excel_fine)} as a fine for loan: {the_loan.loan_number} given to {the_loan.applicant.biodata.name}({the_loan.account.acc_number})",
                        account_dr_id=debit_acc,
                        account_cr=fines_acc,
                        loan=the_loan,
                        tx_date=date.today() if transactions_date is None else transactions_date,
                        receipt=receipt,
                        transaction_type="Loan Fines",
                        added_by_id=userdata(request),
                        branch=business_context['branch'],
                    )

                    added_transactions.append(fines_transaction.id)

            # update the loan_summary_details
            update_loan_summary_details(the_loan)

    # return the results
    return {
        'added_transactions': added_transactions,
        'failed_loan_payments': failed_loan_payments,
        'fines_created': fines_created,
        'fine_payments_created': fine_payments_created
    }
