import datetime
import traceback

from django import forms
from django.core.exceptions import ValidationError
from django.db import transaction
from django.utils.translation import gettext as _
from django_countries.fields import CountryField

from commons.logger import logger
from sacco.models import OtherSettings, Member, MemberAccount, AccountBroker
from sacco.utils import biz_data, biz_staff, biz_staff_branch, gen_account, check_account
from transactions.models import AccountCategory, Account
from utils.general import proper_dial


class AddGroupForm(forms.Form):
    name = forms.CharField(required=False)
    dob = forms.DateField()
    contact = forms.CharField(required=False)
    other_contact = forms.CharField(required=False)
    email = forms.EmailField(required=False)
    desc = forms.CharField(required=False)
    code = CountryField().formfield(required=False)
    code1 = CountryField().formfield(required=False)
    address = forms.CharField(widget=forms.TextInput, required=False)
    account_type = forms.CharField(required=False)
    account_number = forms.CharField(widget=forms.TextInput, required=False)

    def __init__(self, *args, **kwargs):
        self.user = kwargs.pop('user', None)
        self.group = kwargs.pop('group', None)
        self.branch = kwargs.pop('branch', None)
        super(AddGroupForm, self).__init__(*args, **kwargs)

    def the_business(self):
        biz = self.branch.business
        return biz

    def clean_contact(self):
        cleaned_data = self.cleaned_data
        code = cleaned_data.get('code')
        contact = cleaned_data.get('contact')
        if contact:
            primary_c = proper_dial(contact, code)
        else:
            primary_c = contact
        return primary_c

    def clean_other_contact(self):
        cleaned_data = self.cleaned_data
        code = cleaned_data.get('code')
        contact = cleaned_data.get('other_contact')
        if contact:
            primary_c = proper_dial(contact, code)
        else:
            primary_c = contact
        return primary_c

    def clean_account_number(self):
        cleaned_data = self.cleaned_data
        account_number = cleaned_data.get('account_number')
        # print('aa', type(account_number))
        # try:
        if self.the_business().auto_acc_no is True:

            if account_number == '' or account_number is None:
                account_number = gen_account(self.user)
        else:
            if account_number != '' or account_number is not None:
                if check_account(account_number, self.user) is True:
                    raise ValidationError(_('Account already exists'), code='AccountNumber')
        return account_number

    def gen_member(self, name, contact, email, location, user, upload_date, branch):
        """Creating new member"""
        try:
            member = Member.objects.create(name=name, contact=contact, email=email, created_by=biz_staff(user),
                                           member_type='o', is_group=True, location=location,
                                           branch=branch,
                                           date_joined=upload_date if upload_date else datetime.date.today())
            return member
        except Exception as e:
            logger.error(str(e))
            raise ValidationError(_("Error creating member: " + str(e)), code='member_creation')

    @transaction.atomic
    def save(self, commit=True):
        user = self.user
        cleaned_data = super().clean()
        name = cleaned_data.get('name')
        dob = cleaned_data.get('dob')
        email = cleaned_data.get('email')
        location = cleaned_data.get('location')
        contact = cleaned_data.get('contact')
        account_number = cleaned_data.get('account_number')
        account_type = cleaned_data.get('account_type')

        try:
            with transaction.atomic():
                setting_status = OtherSettings.objects.filter(business=self.branch.business).first()
                logger.error(f'Settings status {str(setting_status)}')

                logger.error(f'Biodata has been created')
                member = self.gen_member(name, contact, email, location, user, dob, self.branch)
                logger.error(f'Member has been created')
                account = MemberAccount.objects.create(account_type_id=account_type, acc_number=account_number)
                logger.error(f'Account created')
                broker = AccountBroker.objects.create(members=member, the_account=account, business=biz_data(user))
                logger.error(f'Broker created')
                cat, created = AccountCategory.objects.get_or_create(business=biz_data(user), name="Members",
                                                                     cat_type='liability', dr_cr='cr')
                logger.error(f'Cat {str(cat)}')
                member_acc = Account.objects.create(category=cat, business=biz_data(user), member=account,
                                                    added_by=biz_staff(user))
                logger.error(f'Group acc {str(member_acc)} created')
                return member
        except Exception as e:
            logger.error(traceback.print_exc())
            logger.error(f'Error in reg forms: {e}')
            # print(traceback.print_exc())
            # print('exception', str(e))
            raise e


# class EditGroupForm(forms.Form):
#     name = forms.CharField(required=False)
#     dob = forms.DateField(
#         widget=forms.SelectDateWidget(
#             attrs={'class': 'form-control', 'style': 'width:33.33333333333333%; ' 'display:inline' '-block;'},
#             years=range(1920, datetime.date.today().year + 1)), initial=timezone.now())
#
#     contact = forms.CharField(required=False)
#     other_contact = forms.CharField(required=False)
#     email = forms.EmailField(required=False)
#     account_type = forms.CharField(required=False)
#     desc = forms.CharField(required=False)
#     code = CountryField().formfield(required=False)
#     code1 = CountryField().formfield(required=False)
#     location = forms.CharField(widget=forms.TextInput, required=False)
#     account_number = forms.CharField(widget=forms.TextInput, required=False)

class UpdateFileForm(forms.Form):
    uploaded = forms.FileField()

