import traceback
from datetime import datetime

from django.db.models import F, Q
from rest_framework import status
from rest_framework.exceptions import ValidationError
from rest_framework.response import Response
from rest_framework.views import APIView

from accounts.models import Business
from apis.serializers.auth_serializer import UserSerializer
from apis.serializers.core_serializer import TransactionsSerializer, ShareTransSerializer, LoansSerializer, \
    MemberSerializer, BiodataSerializer
from loans.models import Loans
from sacco.models import AccountBroker, UserMember, OtherSettings, MemberAccount, FinancialYear
from transactions.models import Transactions, SharesTransactions, Account
from transactions.reports import this_financial_year


class BaseAPI(APIView):
    def get_member(self, request):
        user = request.user
        try:
            member = UserMember.objects.get(user=user).member
            return member
        except (UserMember.DoesNotExist, AttributeError):
            # member = None
            raise ValidationError


class TransactionsAPI(BaseAPI):

    def get(self, request):
        accts = []
        if self.get_member(request):
            member_accounts = AccountBroker.objects.filter(members=self.get_member(request)).select_related(
                'the_account')
            for account in member_accounts:
                accts.append(account.the_account)
            member_withs = Transactions.objects.filter(account_dr__member__in=accts
                                                       ).annotate(account=F('account_dr__member__acc_number')).values(
                'account', 'transaction_type', 'reporting_amount', 'narration', 'date_added',
                'tx_date', 'id', 'receipt')
            member_deps = Transactions.objects.filter(account_cr__member__in=accts
                                                      ).annotate(account=F('account_cr__member__acc_number')).values(
                'transaction_type', 'reporting_amount', 'narration', 'date_added',
                'account', 'tx_date', 'id', 'receipt')
            transactions = member_withs.union(member_deps).order_by('-tx_date', '-id')
            return Response(
                {
                    "code": 1,
                    'transactions': TransactionsSerializer(transactions, many=True).data,
                }, status=status.HTTP_200_OK)
        return Response({"code": 0, "response": "Member doesnt exist"}, status=status.HTTP_400_BAD_REQUEST)


class AccountBalanceAPI(BaseAPI):
    def get(self, request):
        accs = []
        if self.get_member(request):
            member_accounts = AccountBroker.objects.filter(members=self.get_member(request)
                                                           ).select_related('the_account')
            for acc in member_accounts:
                accs.append(
                    {
                        "id": acc.the_account.id,
                        "account_number": acc.the_account.acc_number,
                        "type": acc.the_account.account_type.name,
                        "balance": acc.the_account.balance
                    }
                )
            accounts = {"accounts": accs}
            return Response({"code": 1, "response": accounts}, status=status.HTTP_200_OK)
        return Response({"code": 0, "response": "Member doesnt exist"}, status=status.HTTP_400_BAD_REQUEST)


class LoansAPI(BaseAPI):
    def get(self, request):
        accts = []
        if self.get_member(request):
            member_accounts = AccountBroker.objects.filter(members=self.get_member(request)).select_related(
                'the_account')
            for account in member_accounts:
                accts.append(account.the_account)
            member_loans = Loans.objects.filter(account__in=accts)

            return Response({"code": 1, "loans": LoansSerializer(member_loans, many=True).data},
                            status=status.HTTP_200_OK)
        return Response({"code": 0, "response": "Member doesnt exist"}, status=status.HTTP_400_BAD_REQUEST)


class SavingsAPI(BaseAPI):
    def get(self, request):
        accts = []
        if self.get_member(request):
            member_accounts = AccountBroker.objects.filter(members=self.get_member(request)).select_related(
                'the_account')
            for account in member_accounts:
                accts.append(account.the_account)
            initial_deposit = Transactions.objects.filter(transaction_type='Initial deposit',
                                                          account_cr__member__in=accts).order_by('-date_added')
            other_deposits = Transactions.objects.filter(Q(account_cr__member__in=accts) | Q(
                account_dr__member__in=accts)).filter(transaction_type='Deposit').order_by('-date_added')

            deposits = initial_deposit.union(other_deposits).order_by('-tx_date', '-id')
            return Response(
                {
                    "code": 1,
                    'transactions': TransactionsSerializer(deposits, many=True).data,

                }, status=status.HTTP_200_OK)
        return Response({"code": 0, "response": "Member doesnt exist"}, status=status.HTTP_400_BAD_REQUEST)


class WithdrawsAPIView(BaseAPI):
    def get(self, request):
        accts = []
        if self.get_member(request):
            member_accounts = AccountBroker.objects.filter(members=self.get_member(request)
                                                           ).select_related('the_account')
            for account in member_accounts:
                accts.append(account.the_account)
            withdraws = Transactions.objects.filter(transaction_type='Withdraw'
                                                    ).filter(
                Q(account_dr__member__in=accts) | Q(account_cr__member__in=accts)).order_by('-tx_date', '-id')
            return Response({"code": 1, 'transactions': TransactionsSerializer(withdraws, many=True).data,
                             }, status=status.HTTP_200_OK)
        return Response({"code": 0, "response": "Member doesnt exist"}, status=status.HTTP_400_BAD_REQUEST)


class SharesViewAPI(BaseAPI):
    def get(self, request):
        accts = []
        member = self.get_member(request)
        if member:
            member_accounts = AccountBroker.objects.filter(members=member).select_related('the_account')
            for account in member_accounts:
                accts.append(account.the_account)
            shares_trans = SharesTransactions.objects.filter(Q(seller=member) | Q(buyer=member)).order_by(
                '-date')
            share_price = OtherSettings.objects.filter(business=member.biodata.business).first()
            return Response(
                {
                    "code": 1,
                    'share_transactions': ShareTransSerializer(shares_trans, many=True).data,
                    'total_shares': member.shares,
                    'share_price': share_price.share_price

                }, status=status.HTTP_200_OK)
        return Response({"code": 0, "response": "Member doesnt exist"}, status=status.HTTP_400_BAD_REQUEST)


class TransfersAPIView(BaseAPI):
    def get(self, request):
        accts = []
        if self.get_member(request):
            member_accounts = AccountBroker.objects.filter(members=self.get_member(request)
                                                           ).select_related('the_account')
            for account in member_accounts:
                accts.append(account.the_account)
            transfers = Transactions.objects.filter(transaction_type='Transfer', account_dr__member__in=accts).order_by(
                '-tx_date', '-id')
            return Response({"code": 1, 'transactions': TransactionsSerializer(transfers, many=True).data,
                             }, status=status.HTTP_200_OK)
        return Response({"code": 0, "response": "Member doesnt exist"}, status=status.HTTP_400_BAD_REQUEST)


class ProfileViewAPI(BaseAPI):

    def get(self, request):
        user = request.user
        member = self.get_member(request)
        if member:
            try:
                data = {
                    "user": UserSerializer(user).data,
                    "member_detail": MemberSerializer(member).data,
                    "biodata": BiodataSerializer(member.biodata).data,
                    "business": BiodataSerializer(member.biodata.business).data,
                    "base_url": 'http://' + request.headers['Host'],
                }
                return Response({"code": 1, "data": data}, status=200)
            except AttributeError:
                raise ValidationError
        return Response({'code': 0, 'data': 'No profile attached to this member'}, status=status.HTTP_400_BAD_REQUEST)


class RepaymentAPIView(BaseAPI):
    def get(self, request):
        accts = []
        if self.get_member(request):
            member_accounts = AccountBroker.objects.filter(members=self.get_member(request)
                                                           ).select_related('the_account')
            for account in member_accounts:
                accts.append(account.the_account)
            repayments = Transactions.objects.filter(loan__isnull=False
                                                     ).filter(Q(account_dr__member__in=accts) |
                                                              Q(account_dr__member__in=accts)).order_by('-tx_date',
                                                                                                        '-id')
            return Response({"code": 1, 'transactions': TransactionsSerializer(repayments, many=True).data,
                             }, status=status.HTTP_200_OK)

        return Response({"code": 0, "response": "Member doesnt exist"}, status=status.HTTP_400_BAD_REQUEST)


class MemberStatementAPI(BaseAPI):
    def post(self, request):

        il = []
        try:
            business_id = request.data['business']

            # business_id = request.POST.get('business')
            business = Business.objects.get(id=business_id)
            # return Response({"code": 0, "response": business.name}, status=200)
        except Exception as e:
            traceback.print_exc()
            return Response({"code": 0, "response": "SACCO Not found"}, status=status.HTTP_204_NO_CONTENT)

        # get account
        try:
            member_account_id = request.data['account']

            member_account = MemberAccount.objects.get(id=member_account_id)
        except:
            return Response({"code": 0, "response": "Member Not found"}, status=status.HTTP_204_NO_CONTENT)

        account = Account.objects.filter(business=business, member=member_account)

        if account.count() == 0:
            return Response({"code": 0, "response": "Account Not found"}, status=status.HTTP_204_NO_CONTENT)
        else:
            account = account.first()

        # make GL
        if 'start_date' in request.data:
            st_date = request.data['start_date']
        else:
            dates = FinancialYear.objects.filter(status=True, business=business).last()
            st_date = dates.start_date

        if 'end_date' in request.data:
            en_date = request.data['end_date']
        else:
            dates = FinancialYear.objects.filter(status=True, business=business).last()
            en_date = datetime.today().date()

        # get gl where account is this

        if st_date is None and en_date is None:
            gl = Transactions.objects.filter(
                Q(account_dr=account, financial_year=this_financial_year(request)) | Q(account_cr=account,
                                                                                       financial_year=this_financial_year(
                                                                                           request)))
        elif st_date is None and en_date is not None:
            gl = Transactions.objects.filter(
                (Q(account_dr=account) | Q(account_cr=account)), Q(tx_date__lte=en_date)
            )
        elif st_date is not None and en_date is None:
            gl = Transactions.objects.filter(
                (Q(account_dr=account) | Q(account_cr=account)), Q(tx_date__gte=st_date)
            )
            ob_gl_dr = Transactions.objects.filter(
                Q(account_dr=account), Q(tx_date__lt=st_date)
            )
            ob_gl_cr = Transactions.objects.filter(
                Q(account_cr=account), Q(tx_date__lt=st_date)
            )
            ob_cr = 0
            ob_dr = 0
            total_ob = 0
            for x in ob_gl_dr:
                ob_dr += x.reporting_amount

            for x in ob_gl_cr:
                ob_cr += x.reporting_amount
            if account.category.dr_cr == 'cr':
                total_ob = ob_cr - ob_dr
                r_balance = total_ob

                obj = {
                    'gl': None,
                    'account_dr': None,
                    'account_cr': None,
                    'amount_cr': r_balance,
                    'amount_dr': 0,
                    'is_opening_balance': True,
                    'balance': r_balance
                }
                il.append(obj)
            else:
                total_ob = ob_dr - ob_cr
                r_balance = total_ob
                obj = {
                    'gl': None,
                    'account_dr': None,
                    'account_cr': None,
                    'amount_cr': 0,
                    'amount_dr': r_balance,
                    'is_opening_balance': True,
                    'balance': r_balance

                }
                il.append(obj)

        elif st_date is not None and en_date is not None:
            gl = Transactions.objects.filter(
                (Q(account_dr=account) | Q(account_cr=account)),
                Q(tx_date__range=(st_date, en_date))
            )
            ob_gl_dr = Transactions.objects.filter(
                Q(account_dr=account), Q(tx_date__lt=st_date)
            )
            ob_gl_cr = Transactions.objects.filter(
                Q(account_cr=account), Q(tx_date__lt=st_date)
            )
            ob_cr = 0
            ob_dr = 0
            total_ob = 0
            for x in ob_gl_dr:
                ob_dr += x.reporting_amount

            for x in ob_gl_cr:
                ob_cr += x.reporting_amount
            if account.category.dr_cr == 'cr':
                total_ob = ob_cr - ob_dr
                r_balance = total_ob
                obj = {
                    'gl': None,
                    'account_dr': None,
                    'account_cr': None,
                    'amount_cr': r_balance,
                    'amount_dr': 0,
                    'is_opening_balance': True,
                    'balance': r_balance

                }
                il.append(obj)

            else:
                total_ob = ob_dr - ob_cr
                r_balance = total_ob
                obj = {
                    'gl': None,
                    'account_dr': None,
                    'account_cr': None,
                    'amount_cr': 0,
                    'amount_dr': r_balance,
                    'is_opening_balance': True,
                    'balance': r_balance

                }
                il.append(obj)

        if account.category.dr_cr == 'cr':
            for g in gl:
                # for each result, check if its dr or cr
                if g.account_dr == account:
                    r_balance = r_balance - g.reporting_amount
                    # record to temp
                    obj = {
                        'gl': {
                            'id': g.id,
                            'txno': g.txno,
                            'branch': g.branch.name,
                            'transaction': g.transaction_type,
                            'tx_date': g.tx_date
                        },
                        'account_dr': g.account_dr.name,
                        'account_cr': g.account_cr.name,
                        'amount_cr': 0,
                        'amount_dr': g.reporting_amount,
                        'is_opening_balance': False,
                        'balance': r_balance

                    }
                    il.append(obj)

                if g.account_cr == account:
                    r_balance = r_balance + g.reporting_amount
                    # record to temp
                    obj = {
                        'gl': {
                            'id': g.id,
                            'txno': g.txno,
                            'branch': g.branch.name,
                            'transaction': g.transaction_type,
                            'tx_date': g.tx_date
                        },
                        'account_dr': g.account_dr.name,
                        'account_cr': g.account_cr.name,
                        'amount_cr': g.reporting_amount,
                        'amount_dr': 0,
                        'is_opening_balance': False,
                        'balance': r_balance

                    }
                    il.append(obj)

        else:
            for g in gl:
                # for each result, check if its dr or cr
                if g.account_dr == account:
                    r_balance = r_balance + g.reporting_amount
                    # record to temp
                    obj = {
                        'gl': {
                            'id': g.id,
                            'txno': g.txno,
                            'branch': g.branch.name,
                            'transaction': g.transaction_type,
                            'tx_date': g.tx_date
                        },
                        'account_dr': g.account_dr.name,
                        'account_cr': g.account_cr.name,
                        'amount_cr': 0,
                        'amount_dr': g.reporting_amount,
                        'is_opening_balance': False,
                        'balance': r_balance

                    }
                    il.append(obj)

                if g.account_cr == account:
                    r_balance = r_balance - g.reporting_amount
                    # record to temp
                    obj = {
                        'gl': {
                            'id': g.id,
                            'txno': g.txno,
                            'branch': g.branch.name,
                            'transaction': g.transaction_type,
                            'tx_date': g.tx_date
                        },
                        'account_dr': g.account_dr.name,
                        'account_cr': g.account_cr.name,
                        'amount_cr': g.reporting_amount,
                        'amount_dr': 0,
                        'is_opening_balance': False,
                        'balance': r_balance

                    }
                    il.append(obj)

        totalcr = 0
        totaldr = 0
        for i in il:
            totalcr += i['amount_cr']
            totaldr += i['amount_dr']

        st_date = datetime.strptime(str(st_date), "%Y-%m-%d").date()
        en_date = datetime.strptime(str(en_date), "%Y-%m-%d").date()
        context = {'individual_gl': il, 'total_cr': totalcr,
                   "total_dr": totaldr,
                   'start_date': st_date, "end_date": en_date}

        return Response({"statement": context, }, status=status.HTTP_200_OK)

    def get(self, request):
        return Response({"code": 0, "response": "Please Make a POST"}, status=status.HTTP_400_BAD_REQUEST)
