from django.contrib import messages
from django.contrib.auth import authenticate, login, logout
from django.contrib.auth.decorators import login_required
from django.contrib.auth.forms import PasswordResetForm
from django.contrib.auth.mixins import LoginRequiredMixin
from django.contrib.auth.tokens import default_token_generator
from django.contrib.auth.views import PasswordResetView
from django.contrib.messages.views import SuccessMessageMixin
from django.core.mail import BadHeaderError, send_mail
from django.http import HttpResponseRedirect
from django.shortcuts import render, redirect
from django.template.loader import render_to_string
from django.urls import reverse_lazy
from django.utils.encoding import force_bytes
from django.utils.http import urlsafe_base64_encode
from django.views import View
from django.views.generic import TemplateView

from accounts.forms import LoginForm, NewPasswordForm
from accounts.models import User
from accounts.permission_mixin import BusinessUserMixin
from commons.logger import logger
from sacco.constants import UPDATE, AUTHENTICATION
from sacco.models import GeneratedPassword, ActivityLog
from sacco.utils import biz_staff


class Login(View):
    form_class = LoginForm
    template_name = 'accounts/login.html'
    created_email = 'mufukoadmin/emails/action_email.html'

    def get(self, request, *args, **kwargs):
        title = 'Login'
        form = self.form_class()
        return render(request, self.template_name, locals())

    def post(self, request, *args, **kwargs):
        form = self.form_class(request.POST)
        if form.is_valid():
            username = form.cleaned_data['username']
            password = form.cleaned_data['password']

            user = authenticate(request, username=username, password=password)

            if user is not None:
                if user.is_active:
                    user.backend = 'accounts.backends.EmailOrUsernameModelBackend'
                    login(request, user)
                    if user.is_nugsoft:
                        return HttpResponseRedirect(reverse_lazy('nug_admin'))
                    else:
                        if user.staff and user.staff.is_active and user.staff.biodata.business is not None and user.staff.biodata.business.status == 1:
                            # print('Remaining days', user.staff.biodata.business.licence.remaining_days)
                            if int(user.staff.biodata.business.licence.remaining_days) > 0:
                                try:
                                    passw = GeneratedPassword.objects.get(staff=user.staff)
                                    user_mail = passw.staff.user.id
                                    request.session['staff'] = user_mail
                                    if passw.is_used and passw.can_login:
                                        # return HttpResponseRedirect(reverse_lazy('index'))
                                        return HttpResponseRedirect(reverse_lazy('business_selector'))
                                    else:
                                        messages.error(request, 'Your are using the system generated password and maybe '
                                                                'compromised. Please choose your new password to proceed')
                                        return HttpResponseRedirect(reverse_lazy('staff_new_password'))
                                except GeneratedPassword.DoesNotExist:
                                    messages.error(request, 'This did not pass through the right processes as there '
                                                            'verification has failed.')
                                    return HttpResponseRedirect(reverse_lazy('login'))
                                except Exception as e:
                                    messages.error(request, str(e))
                                    return HttpResponseRedirect(reverse_lazy('login'))
                            else:
                                # ======= send email here ===========
                                mail_content = f"<p>Hello <b>Admin</b>, <p>The login attempt of <b>{user.staff.biodata.name}</b> into <b>{user.staff.biodata.business.name}</b> account on Mfuko Plus has failed. This is because the business account license expired on <b>{user.staff.biodata.business.licence.expiry_date}</b></p>"
                                subject = f"{user.staff.biodata.business.name} license notice"
                                new_business_mail = {
                                    'site_name': 'Mfuko Plus',
                                    'subject': subject,
                                    'body': mail_content,
                                    'protocol': 'http',
                                }
                                email_html = render_to_string(self.created_email, new_business_mail)

                                send_mail(subject, '', '',
                                          [user.staff.biodata.business, 'macdanson2@gmail.com', 'info@mfuko.net'],
                                          html_message=email_html,
                                          fail_silently=False)
                                messages.error(request, f'Sorry! You can not login to your account because your business account expired')
                                return HttpResponseRedirect(reverse_lazy('login'))
                        else:
                            messages.error(request, 'Sorry! It seems your business account is suspended, contact the admin')
                            return HttpResponseRedirect(reverse_lazy('login'))
                else:
                    messages.error(request, "Your account is not activated")
                    return HttpResponseRedirect(reverse_lazy('login'))
            else:
                messages.error(request, 'Email or Password provided does not match any user records. Check and try '
                                        'again')
                return HttpResponseRedirect(reverse_lazy('login'))
        else:
            messages.error(request, 'Form Validation')
            return HttpResponseRedirect(reverse_lazy('login'))


class NewUser(View):
    template_name = 'accounts/new-user.html'
    form_class = NewPasswordForm
    success_url = reverse_lazy('password_reset_done')

    def get(self, request, *args, **kwargs):
        title = 'Set new password'
        form = self.form_class
        return render(request, self.template_name, locals())

    def post(self, request, *args, **kwargs):
        if 'staff' in request.session:
            user_id = request.session.get('staff')
            staff = User.objects.filter(id=user_id).first()

            # print(staff)
            newPassform = self.form_class(request.POST)
            if newPassform.is_valid():
                password1 = newPassform.cleaned_data['new_password1']
                password2 = newPassform.cleaned_data['new_password2']
                if password1 and password2:
                    if password1 == password2:
                        passw = GeneratedPassword.objects.get(staff=staff.staff)
                        staff.set_password(password1)
                        staff.save()
                        passw.is_used = True
                        passw.save()
                        # Record the audit for the new user setting own password
                        message = f"{staff.staff.biodata.name} Successfully updated " \
                                  f"password from the default that is system generated"
                        ActivityLog.objects.create(actor=staff, action_type=UPDATE,
                                                   branch=staff.staff.branch_id, remarks=message)
                        staff.backend = 'accounts.backends.EmailOrUsernameModelBackend'
                        login(request, staff)
                        # add the required data to session
                        return HttpResponseRedirect(reverse_lazy('business_selector'))
                    else:
                        messages.error(request, 'Passwords dont match')
                    # login(request, staff)
                    messages.success(request, 'Successfully updated password')
                    return HttpResponseRedirect(reverse_lazy('staff_new_password'))
            else:
                messages.error(request, 'Form not valid')
                return HttpResponseRedirect(reverse_lazy('staff_new_password'))
        else:
            messages.error(request, 'Try verifying your user details')
            return HttpResponseRedirect(reverse_lazy('login'))


class ResetPasswordView(SuccessMessageMixin, PasswordResetView):
    template_name = 'accounts/password_reset/recover-password.html'
    html_template = 'accounts/emails/password_reset_email.html'
    # plain_template_name = 'accounts/emails/password_reset_email.txt'
    subject_template_name = 'users/password_reset_subject'
    success_url = reverse_lazy('password_reset_done')
    form_class = PasswordResetForm

    def get(self, request, *args, **kwargs):
        title = 'Recover password'
        form = self.form_class()
        return render(request, self.template_name, locals())

    def post(self, request, *args, **kwargs):
        domain = request.headers['Host']
        password_reset_form = self.form_class(request.POST)
        if password_reset_form.is_valid():
            data = password_reset_form.cleaned_data['email']
            # associated_users = User.objects.filter(Q(email=data) | Q(staff__biodata__email=data))
            try:
                user = User.objects.get(email=data) or User.objects.get(staff__biodata__email=data)
            except User.DoesNotExist:
                user = None
            except User.MultipleObjectsReturned:
                user = User.objects.filter(staff__biodata__email=data).first()
            if user:
                subject = "Password Reset Request"
                mail_content = "<p>Hello " + user.email + ",</p> <p>We have received your request to reset your" \
                                                          " password. To initiate the password reset process " \
                                                          "for your account at " + domain + ", Mfuko Account," \
                                                                                            " click the " \
                                                                                            "link <b></b></p>"
                c = {
                    "email": user.email,
                    'domain': domain,
                    'site_name': 'Mfuko Plus',
                    "uid": urlsafe_base64_encode(force_bytes(user.pk)),
                    "body": mail_content,
                    "salute": 'Sincerely',
                    'token': default_token_generator.make_token(user),
                    'protocol': 'http' if not request.is_secure() else 'https',
                }
                msg_html = render_to_string(self.html_template, c)
                # Record the audit for the user requesting to change the password
                if hasattr(user, 'staff'):
                    if hasattr(user.staff, 'biodata'):
                        message = f"{user.staff.biodata.name} requested to reset password " \
                                  f"and an email with instructions were sent"
                        ActivityLog.objects.create(actor=user, action_type=AUTHENTICATION,
                                                   branch=user.staff.branch_id, remarks=message)
                try:
                    rec_email = user.email if not user.staff or user.staff.biodata is None else user.staff.biodata.email
                    send_mail(subject, '', '', [rec_email], html_message=msg_html)
                except BadHeaderError as e:
                    # print('ths er', str(e))
                    logger.error(str(e))
                except AttributeError as e:
                    # print('that er', str(e))
                    logger.error(str(e))
                except Exception as e:
                    # print('ihave er', str(e))
                    logger.error(str(e))
                return HttpResponseRedirect(self.success_url)
            else:
                messages.error(request, 'Email does not exist')
                return HttpResponseRedirect(reverse_lazy('forgot_password'))


@login_required
def auth_logout(request):
    logout(request)
    return HttpResponseRedirect(reverse_lazy('login'))

class BusinessSelectorView(BusinessUserMixin ,View):
    template_name = 'accounts/business_selector.html'

    def get(self, request, *args, **kwargs):
        title = "Select Business"

        staff = biz_staff(request.user)

        staff_businesses = staff.businesses.all()

        return render(request, self.template_name, locals())

    def dispatch(self, request, *args, **kwargs):
        user_businesses = request.user.staff.businesses.all()
        # check if the person has businesses added
        if user_businesses.count() == 0:
            # add the original business
            request.user.staff.businesses.add(request.user.staff.branch.business)

        user_businesses = request.user.staff.businesses.all()
        # If user has access to only one business, redirect to main dashboard
        if not request.user.staff.can_access_central and user_businesses.count() == 1:
            request.session['selected_business_id'] = user_businesses.first().id
            return redirect('index')  # Replace with your dashboard URL name
        return super().dispatch(request, *args, **kwargs)

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context['businesses'] = self.request.user.staff.businesses.all()
        return context

    def post(self, request, *args, **kwargs):
        selected = request.POST.get('business_selection')
        if selected == 'central':
            request.session['selected_business_id'] = 'central'
        else:
            try:
                business_id = int(selected)
                if request.user.staff.businesses.filter(id=business_id).exists():
                    request.session['selected_business_id'] = business_id
            except (ValueError, TypeError):
                pass
        return redirect('index')
