from django.contrib.auth.signals import user_logged_in, user_login_failed, user_logged_out
from django.dispatch import receiver

from sacco.constants import LOGIN, LOGIN_FAILED, LOGOUT
from sacco.models import ActivityLog


def get_client_ip(request):
    if request:
        x_forwarded_for = request.META.get("HTTP_X_FORWARDED_FOR")
        return (
            x_forwarded_for.split(",")[0]
            if x_forwarded_for
            else request.META.get("REMOTE_ADDR")
        )


@receiver(user_logged_in)
def log_user_login(sender, request, user, **kwargs):
    if hasattr(user, 'staff'):
        try:
            this_user = user.staff.biodata.name
            branch_id = user.staff.branch_id
        except AttributeError as e:
            this_user = ''
            branch_id = None
    else:
        this_user = None
        branch_id = None
    log_title = 'Login Successful'
    message = f"{this_user} is logged in with ip:{get_client_ip(request)}"
    ActivityLog.objects.create(actor=user, action_type=LOGIN, title=log_title, remarks=message, branch=branch_id)


@receiver(user_login_failed)
def log_user_login_failed(sender, credentials, request, **kwargs):
    message = f"Login Attempt Failed for email {credentials.get('email')} with ip: {get_client_ip(request)}"
    log_title = 'Failed login attempt'
    ActivityLog.objects.create(action_type=LOGIN_FAILED, title=log_title, remarks=message)


@receiver(user_logged_out)
def log_user_out(sender, request, user, **kwargs):
    if user.staff:
        branch_id = user.staff.branch_id
        user_detail = user.staff.biodata.name
    else:
        branch_id = None
        user_detail = request.user
    log_title = f'Logout'
    message = f"The following user {user_detail} has logged out"
    ActivityLog.objects.create(action_type=LOGOUT, title=log_title, remarks=message,
                               actor=request.user, branch=branch_id)
