import uuid

from django.contrib.contenttypes.models import ContentType
from django.core.validators import MinLengthValidator, MaxLengthValidator
from django.db import models
from django.contrib.auth.models import (BaseUserManager, AbstractBaseUser)
from django.db.models.signals import post_save
from django.dispatch import receiver
from django.urls import reverse
from django_countries.fields import CountryField
from django_resized import ResizedImageField

from sacco.constants import CREATE
from sacco.models import ActivityLog
from utils.file_uploads import logo_files, certificate_files, bio_photo, sig_files
from utils.general import convert_national


class Salutation(models.Model):
    name = models.CharField(max_length=255)

    class Meta:
        verbose_name = 'Member Salutation'
        db_table = 'member_salutation'

    def __str__(self):
        return self.name


class UserManager(BaseUserManager):
    def create_user(self, username, password=None):
        """
        Creates and saves a User with the given email, date of
        birth and password.
        """
        if not username:
            raise ValueError('Users must have an email address')
        user = self.model(username=username)
        user.set_password(password)
        user.save(using=self._db)
        return user

    def create_superuser(self, username, password):
        """
        Creates and saves a superuser with the given email, phone number and password.
        """
        user = self.create_user(
            username=username,
            password=password,
        )
        user.is_admin = True
        user.is_active = True
        user.is_superuser = True
        user.user_type = 'admin'
        user.save(using=self._db)
        return user


class User(AbstractBaseUser):
    is_active = models.BooleanField(default=True, null=True, verbose_name='Active')
    is_admin = models.BooleanField(default=False, null=True, verbose_name='Admin')
    is_superuser = models.BooleanField(default=False, null=True, verbose_name='Super User')
    is_nugsoft = models.BooleanField(default=False, null=True, verbose_name='Nugsoft Admin')
    username = models.CharField(verbose_name='Username', max_length=255, unique=True)
    email = models.EmailField(blank=True, null=True, unique=True)
    staff = models.OneToOneField('Staff', blank=True, null=True, on_delete=models.CASCADE)

    objects = UserManager()
    USERNAME_FIELD = 'username'
    REQUIRED_FIELDS = []

    def __str__(self):
        return str(self.username)

    def has_perm(self, perm, obj=None):
        """Does the user have a specific permission?"""
        # Simplest possible answer: Yes, always
        return True

    def has_module_perms(self, app_label):
        """Does the user have permissions to view the app `app_label`?"""

        # Simplest possible answer: Yes, always
        return True

    @property
    def is_staff(self):
        """Is the user a member of staff?"""
        # Simplest possible answer: All admins are staff
        return self.is_admin

    class Meta:
        db_table = 'users'
        ordering = ['-id']

    def save_related(self):
        self.staff.save()
        self.save()


class SubscriptionPackage(models.Model):
    name = models.CharField(max_length=255, blank=True, null=True)
    limit = models.IntegerField(default=0)
    date_created = models.DateField(auto_now_add=True)
    status = models.BooleanField(default=True)

    class Meta:
        db_table = 'subscription_packages'
        verbose_name = 'Subscription Package'
        verbose_name_plural = 'Subscription Packages'

    def __str__(self):
        return self.name


class Biz_status(models.IntegerChoices):
    Active = 1, 'Active'
    Suspended = 0, 'Suspended'
    Deactivated = 2, 'Deactivated'


class Biz_type(models.IntegerChoices):
    TraditionalSACOO = 1, 'Traditional SACOO'
    LendingCompany = 2, 'Lending Company'
    InvestmentClub = 3, 'Investment Club'
    AssetFinancing = 4, 'Asset Financing'


class Business(models.Model):
    INSTANCE_TYPES = (
        ('namayingo', 'namayingo'),
        ('nugsoft', 'nugsoft'),
        ('not connected', 'not connected'),
    )

    code = models.UUIDField(unique=True, blank=True, null=True)
    name = models.CharField(max_length=255, blank=True, null=True)
    address = models.TextField(blank=True, null=True)
    short_name = models.CharField(max_length=100, blank=True, null=True)
    contact = models.CharField(max_length=100, blank=True, null=True)
    email = models.EmailField(blank=True, null=True)
    logo = models.ImageField(blank=True, null=True, upload_to=logo_files)
    certificate = models.ImageField(blank=True, null=True, upload_to=certificate_files)
    launch_date = models.DateTimeField(blank=True, null=True)
    certificate_no = models.CharField(blank=True, null=True, max_length=255)
    website = models.URLField(blank=True, null=True)
    clients = models.IntegerField(blank=True, null=True, default=0)
    type = models.IntegerField(blank=True, null=True, choices=Biz_type.choices)
    status = models.IntegerField(blank=True, null=True, choices=Biz_status.choices, default=1)
    package = models.ForeignKey(SubscriptionPackage, on_delete=models.CASCADE, blank=True, null=True)
    date_created = models.DateField(auto_now_add=True)
    auto_acc_no = models.BooleanField(default=False, verbose_name='Auto generate Account Number')
    # share_status = models.BooleanField(default=False, verbose_name='Membership Share status')
    does_withdraw = models.BooleanField(default=True, verbose_name='Does withdraw')
    does_deposit = models.BooleanField(default=True, verbose_name='Does deposit')
    does_transfer = models.BooleanField(default=True, verbose_name='Does transfer')
    has_shares = models.BooleanField(default=True, verbose_name='Has shares')
    synced = models.BooleanField(default=False, verbose_name='Synced to the portal')
    created_by = models.ForeignKey('Staff', on_delete=models.SET_NULL, null=True, blank=True)
    business_number = models.CharField(max_length=9, blank=True, null=True, validators=[
        MinLengthValidator(9, message='Length has to be 9'),
        MaxLengthValidator(9, message='Length has to be 9')])
    instance_type = models.CharField(max_length=30, default='not connected', choices=INSTANCE_TYPES)

    def save(self, *args, **kwargs):
        if not self.id:
            self.code = uuid.uuid4()
        return super(Business, self).save(*args, **kwargs)

    class Meta:
        db_table = 'businesses'
        verbose_name = 'Business'
        verbose_name_plural = 'Businesses'

    def __str__(self):
        if self.name:
            return self.name
        return ''


    @property
    def logo_url(self):
        if self.logo and hasattr(self.logo, 'url'):
            return self.logo.url


@receiver(post_save, sender=Business)
def create_business(sender, created, instance, **kwargs):
    if created:
        if instance.created_by:
            creator = instance.created_by.biodata.name
            branch_id = instance.created_by.branch_id
        else:
            branch_id = None
            creator = instance.created_by
        log_title = 'Business created'
        message = f"{creator} created business with name {instance.name}"

        ct = ContentType.objects.get_for_model(instance)
        ActivityLog.objects.create(actor=creator, title=log_title, action_type=CREATE, remarks=message,
                                   branch=branch_id, content_type=ct)


class Branch_status(models.IntegerChoices):
    Active = 1, 'Active'
    Suspended = 0, 'Suspended'
    Deactivated = 2, 'Deactivated'


class Branch(models.Model):
    code = models.UUIDField(unique=True, blank=True, null=True)
    name = models.CharField(max_length=255, blank=True, null=True)
    address = models.TextField(blank=True, null=True)
    contact = models.CharField(max_length=20, blank=True, null=True)
    status = models.IntegerField(blank=True, null=True, choices=Branch_status.choices, default=1)
    date_created = models.DateField(auto_now_add=True)
    created_by = models.ForeignKey('Staff', related_name='creator', on_delete=models.SET_NULL, null=True, blank=True)
    business = models.ForeignKey('accounts.Business', related_name='branches', on_delete=models.CASCADE)
    is_main = models.BooleanField(default=False)
    synced = models.BooleanField(default=False, verbose_name='Synced to the portal')

    class Meta:
        db_table = 'branches'
        verbose_name = 'Branch'
        verbose_name_plural = 'Branches'

    def save(self, *args, **kwargs):
        if not self.id:
            self.code = uuid.uuid4().hex
        return super(Branch, self).save(*args, **kwargs)

    def __str__(self):
        return f'{self.name} - {self.business.name}'


@receiver(post_save, sender=Branch)
def create_branch(sender, created, instance, **kwargs):
    if created:
        if instance.created_by:
            branch_id = instance.created_by.branch_id

            creator = instance.created_by
            name = creator.biodata.name
        else:
            name = None
            branch_id = None
        log_title = 'Branch Created'
        message = f"{name} created branch with name {instance.name}"
        ct = ContentType.objects.get_for_model(instance)
        ActivityLog.objects.create(actor=instance.created_by, title=log_title, action_type=CREATE, remarks=message,
                                   branch=branch_id, content_type=ct)


class Biodata(models.Model):
    GENDER_CHOICES = (
        ('M', 'Male'),
        ('F', 'Female')
    )

    MARITAL_CHOICES = (
        (1, 'Single'),
        (2, 'Married'),
        (3, 'Divorced'),
        (4, 'Cohabitation'),
        (5, 'Widowed'),
    )

    name = models.CharField(max_length=255, blank=True, null=True)
    gender = models.CharField(max_length=1, choices=GENDER_CHOICES, blank=True, null=True)
    marital_status = models.IntegerField(blank=True, null=True, choices=MARITAL_CHOICES)
    dob = models.DateField(null=True, blank=True)
    nin = models.CharField(null=True, blank=True, max_length=40)
    contact = models.CharField(null=True, blank=True, max_length=40)
    other_contact = models.CharField(null=True, blank=True, max_length=45)
    mobile_money = models.CharField(null=True, blank=True, max_length=40)
    country = CountryField(blank_label='(select country)', blank=True, null=True)
    title = models.ForeignKey(Salutation, blank=True, null=True, on_delete=models.CASCADE)
    email = models.EmailField(blank=True, null=True)
    nok = models.CharField(max_length=255, blank=True, null=True)
    nok_contacts = models.CharField(max_length=20, blank=True, null=True)
    nok_contacts2 = models.CharField(max_length=20, blank=True, null=True)
    photo = models.ImageField(upload_to=bio_photo, blank=True, null=True)
    signature = ResizedImageField(size=[100, 100], upload_to=sig_files, blank=True, null=True)
    date_created = models.DateField(auto_now_add=True)
    location = models.TextField(blank=True, null=True)
    created_by = models.ForeignKey('Staff', related_name='created', on_delete=models.SET_NULL, null=True, blank=True)
    business = models.ForeignKey('accounts.Business', on_delete=models.CASCADE, blank=True, null=True)
    employee_no = models.CharField(max_length=255, blank=True, null=True)
    synced = models.BooleanField(default=False, verbose_name='Synced to the portal')
    member_upload = models.UUIDField(blank=True, null=True)

    def __str__(self):
        return str(self.name)

    class Meta:
        db_table = 'biodata'
        verbose_name = 'Biodata'
        verbose_name_plural = 'Biodata'
        # unique_together = ('contact', 'business',)

    @property
    def photo_url(self):
        if self.photo and hasattr(self.photo, 'url'):
            return self.photo.url

    @property
    def sign_url(self):
        if self.signature and hasattr(self.signature, 'url'):
            return self.signature.url

    @property
    def local_contact(self):
        try:
            return convert_national(self.contact)
        except (Exception,) as e:
            return None

    @property
    def local_other_contact(self):
        try:
            return convert_national(self.other_contact)
        except (Exception,) as e:
            return None

    @property
    def local_mm(self):
        try:
            return convert_national(self.mobile_money)
        except (Exception,) as e:
            return None


class Staff(models.Model):
    biodata = models.OneToOneField(Biodata, blank=True, null=True, on_delete=models.CASCADE)
    branch = models.ForeignKey(Branch, related_name='office', on_delete=models.CASCADE, blank=True, null=True)
    position = models.ForeignKey('Position', related_name='positions', on_delete=models.SET_NULL, null=True, blank=True)
    role = models.ForeignKey('Role', related_name='roles', on_delete=models.SET_NULL, null=True, blank=True)
    on_payroll = models.BooleanField(default=False)
    is_active = models.BooleanField(default=True)
    is_contact = models.BooleanField(default=False, verbose_name="Is this staff a contact person?")
    can_access_central = models.BooleanField(default=False)
    can_migrate_staff = models.BooleanField(default=False)
    businesses = models.ManyToManyField(
        Business,
        related_name='staff_members',
        blank=True
    )


    def __unicode__(self):
        return str(self.biodata.name)

    def save_related(self):
        self.biodata.save()
        if hasattr(self, 'user'):
            self.user.save()
        self.save()

    class Meta:
        db_table = 'staffs'
        verbose_name = 'Staff'
        verbose_name_plural = 'Staffs'

    @property
    def get_absolute_url(self):
        return reverse('staff_detail', kwargs={'pk': self.id})


class Payroll(models.Model):
    staff = models.ForeignKey(Staff, on_delete=models.CASCADE, related_name='staff')
    gross_pay = models.IntegerField(default=0, null=True, blank=True)
    net_pay = models.IntegerField(default=0, null=True, blank=True)
    paye = models.IntegerField(default=0, null=True, blank=True)
    nssf = models.IntegerField(default=0, null=True, blank=True)
    branch = models.ForeignKey(Branch, on_delete=models.CASCADE, blank=True, null=True)
    created_date = models.DateField(auto_now_add=True)
    update_on = models.DateTimeField(blank=True, null=True)
    is_deleted = models.BooleanField(default=False)

    def __str__(self):
        return self.gross_pay

    class Meta:
        db_table = 'payroll'
        verbose_name = 'payroll'
        verbose_name_plural = 'payroll'


class PayrollPayment(models.Model):
    PAYMENT_TYPE = (
        (1, 'Salary'),
        (2, 'Allowance'),
        (3, 'Advance')
    )
    payroll = models.ForeignKey(Payroll, on_delete=models.CASCADE)
    gross_pay = models.FloatField(default=0, null=True, blank=True)
    net_pay = models.FloatField(default=0, null=True, blank=True)
    paye = models.FloatField(default=0, null=True, blank=True)
    nssf = models.FloatField(default=0, null=True, blank=True)
    branch = models.ForeignKey(Branch, on_delete=models.CASCADE, blank=True, null=True)
    month = models.DateField(null=True)
    type = models.IntegerField(default=1, choices=PAYMENT_TYPE)
    created_date = models.DateField(auto_now_add=True)
    update_on = models.DateTimeField(blank=True, null=True)

    def __str__(self):
        return self.gross_pay

    class Meta:
        db_table = 'payroll_payments'
        verbose_name = 'payroll_payment'
        verbose_name_plural = 'payroll_payments'


class Permissions(models.Model):
    description = models.CharField(max_length=255, null=True, blank=True, unique=True)
    item_name = models.CharField(max_length=255, null=True, blank=True, unique=True)

    def __str__(self):
        return self.item_name if not self.description else self.description

    class Meta:
        db_table = 'permissions'
        verbose_name_plural = 'Permissions'

    def save(self, *args, **kwargs):
        if not self.id:
            if self.description:
                name = self.description.lower()
                self.item_name = name.replace(' ', '_')
            return super(Permissions, self).save(*args, **kwargs)


class Position(models.Model):
    title = models.CharField(max_length=255, blank=True)
    business = models.ForeignKey('accounts.Business', on_delete=models.CASCADE)
    created_date = models.DateField(auto_now_add=True)
    update_on = models.DateTimeField(blank=True, null=True)
    created_by = models.ForeignKey('accounts.Staff', related_name='admin',
                                   on_delete=models.SET_NULL, null=True, blank=True)
    permissions = models.ManyToManyField(Permissions, blank=True)

    def __str__(self):
        return self.title

    class Meta:
        db_table = 'positions'
        verbose_name = 'Position'
        verbose_name_plural = 'Positions'


class Role(models.Model):
    title = models.CharField(max_length=255, blank=True)
    business = models.ForeignKey('accounts.Business', related_name='company', on_delete=models.CASCADE)
    created_date = models.DateField(auto_now_add=True)
    created_by = models.ForeignKey('accounts.Staff', related_name='staffs', on_delete=models.SET_NULL, null=True,
                                   blank=True)

    def __str__(self):
        return self.title

    class Meta:
        db_table = 'user_roles'
        verbose_name = 'User Role'
        verbose_name_plural = 'User Roles'


class EmailResetCode(models.Model):
    user = models.OneToOneField(User, on_delete=models.CASCADE)
    code = models.IntegerField(blank=True, null=True,
                               validators=[
                                   MinLengthValidator(6, message='Length has to be 6'),
                                   MaxLengthValidator(6, message='Length has to be 6')
                               ]
                               )

    class Meta:
        db_table = 'email_reset_code'
