from django import forms
from django.contrib.auth import password_validation
from django.contrib.auth.forms import SetPasswordForm
from django.core.exceptions import ValidationError
from django.utils.translation import gettext as _

from accounts.models import Staff
from sacco.models import GeneratedPassword


class LoginForm(forms.Form):
    username = forms.CharField(widget=forms.TextInput, required=True)
    password = forms.CharField(widget=forms.PasswordInput, required=True)


class MySetPasswordForm(SetPasswordForm):
    error_messages = {
        'staff_absent': _('Staff does not exist'),
    }

    def save(self, *args, commit=True, **kwargs):
        user = super().save(commit=False)
        user.is_active = True
        if commit:
            user.save()
            try:
                staff = Staff.objects.get(user=user)
            except Staff.DoesNotExist:
                raise ValidationError(
                    self.error_messages['staff_absent'],
                    code='staff_absent',
                )
            GeneratedPassword.objects.filter(staff=staff).update(is_used=True, can_login=True)
        return user


class NewPasswordForm(forms.Form):
    """
    A form that lets a user change set their password without entering the old
    password
    """
    error_messages = {
        'password_mismatch': _('The two password fields didn’t match.'),
    }
    new_password1 = forms.CharField(
        label=_("New password"),
        widget=forms.PasswordInput(attrs={'autocomplete': 'new-password'}),
        strip=False,
        help_text=password_validation.password_validators_help_text_html(),
    )
    new_password2 = forms.CharField(
        label=_("New password confirmation"),
        strip=False,
        widget=forms.PasswordInput(attrs={'autocomplete': 'new-password'}),
    )

    def clean_new_password2(self):
        password1 = self.cleaned_data.get('new_password1')
        password2 = self.cleaned_data.get('new_password2')
        if password1 and password2:
            if password1 != password2:
                raise ValidationError(
                    self.error_messages['password_mismatch'],
                    code='password_mismatch',
                )
        return password2
