from django import forms
from django.contrib import admin
from django.contrib.auth.admin import UserAdmin as BaseUserAdmin
from django.contrib.auth.forms import ReadOnlyPasswordHashField

from .models import *


class UserChangeForm(forms.ModelForm):
    """A form for updating users. Includes all the fields on
    the user, but replaces the password field with admin's
    password hash display field.
    """
    password = ReadOnlyPasswordHashField(label="Password",
                                         help_text=(
                                             "Raw passwords are not stored, so there is no way to see this "
                                             "user's password, but you can change the password using "
                                             "<a href=\"../password/\">this form</a>."
                                         ),
                                         )

    class Meta:
        model = User
        fields = '__all__'

    def clean_password(self):
        # Regardless of what the user provides, return the initial value.
        # This is done here, rather than on the field, because the
        # field does not have access to the initial value
        return self.initial["password"]


class UserAdmin(BaseUserAdmin):
    form = UserChangeForm
    list_display = ('email', 'username', 'is_active')
    list_filter = ('is_admin', 'is_active')
    fieldsets = (
        (None, {'fields': ('username', 'password')}),
        ('Personal info', {'fields': ('email',)}),
        ('Permissions', {'fields': ('is_admin', 'is_superuser', 'is_active', 'is_nugsoft', 'staff')}),
        # ('Group Permissions', {'fields': ('user_role',)})
    )
    # add_fieldsets is not a standard ModelAdmin attribute. UserAdmin
    # overrides get_fieldsets to use this attribute when creating a user.
    add_fieldsets = (
        (None, {'classes': ('wide',), 'fields': ('email', 'username', 'password1', 'password2')}),)
    search_fields = ['username']
    ordering = ['-id']
    list_per_page = 50
    filter_horizontal = ()


@admin.register(Biodata)
class BioAdmin(admin.ModelAdmin):
    search_fields = ['name']

admin.site.register(User, UserAdmin)
admin.site.register(ActivityLog)
